(* $Id: screrge.ml 44 2008-10-20 00:21:41Z ohl $ *)
(* Copyright (C) 2008 by Thorsten Ohl <ohl@physik.uni-wuerzburg.de>

   This is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   It is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  *)  

let version () =
  prerr_endline "screrge: an experimental iRex DR1000S scribble merger";
  prerr_endline "$Id: screrge.ml 44 2008-10-20 00:21:41Z ohl $";
  prerr_endline "";
  prerr_endline "  ********************************************************";
  prerr_endline "  *                                                      *";
  prerr_endline "  * This is unreleased code and might damage your files! *";
  prerr_endline "  *                                                      *";
  prerr_endline "  *              BACKUPS ARE YOUR FRIENDS                *";
  prerr_endline "  *                                                      *";
  prerr_endline "  ********************************************************";
  prerr_endline ""

let license () =
  prerr_endline "";
  version ();
  prerr_endline "  Copyright (C) 2008 by Thorsten Ohl <ohl@physik.uni-wuerzburg.de>";
  prerr_endline "";
  prerr_endline "  This is free software; you can redistribute it and/or modify it";
  prerr_endline "  under the terms of the GNU General Public License as published by ";
  prerr_endline "  the Free Software Foundation; either version 2, or (at your option)";
  prerr_endline "  any later version.";
  prerr_endline "";
  prerr_endline "  It is distributed in the hope that it will be useful, but";
  prerr_endline "  WITHOUT ANY WARRANTY; without even the implied warranty of";
  prerr_endline "  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the ";
  prerr_endline "  GNU General Public License for more details.";
  prerr_endline "";
  prerr_endline "  You should have received a copy of the GNU General Public License";
  prerr_endline "  along with this program; if not, write to the Free Software";
  prerr_endline "  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.";
  prerr_endline ""

type options =
    { mutable db_file : string;
      mutable suffix : string;
      mutable overwrite : bool;
      mutable verbose : bool }

let defaults =
  { db_file = "metadata.db";
    suffix = "_merged";
    overwrite = false;
    verbose = false }

let merged_filename options name =
  if Filename.check_suffix name ".pdf" then
    Filename.chop_suffix name ".pdf" ^ options.suffix ^ ".pdf"
  else
    name ^ options.suffix ^ ".pdf"

module SMap = Map.Make (struct type t = int let compare = compare end)

let add_scribble page scribble map =
  try
    SMap.add page (scribble :: SMap.find page map) map
  with
  | Not_found -> SMap.add page [scribble] map

let make_scribble_map scribbles =
  List.fold_left
    (fun map s ->
      match s.Metadata.start_anchor with
      | Some (Metadata.PDF_Page page) -> add_scribble page (Scribble.parse s.Metadata.data) map
      | Some (Metadata.Unknown_Anchor _)
      | None -> map)
    SMap.empty (List.rev scribbles)

let find_scribbles page map =
  try SMap.find page map with Not_found -> [] 

let ops page scribbles =
  List.concat (List.map (ThoPDF.ops_of_scribble (ThoPDF.viewport_of_page page)) scribbles)

let merge_file options pdf_file =
  let pdf_file' = merged_filename options pdf_file in
  if not options.overwrite && Sys.file_exists pdf_file' then
    print_endline ("screrge: " ^ pdf_file' ^ " already exists!")
  else
    match Metadata.scribbles ~db:options.db_file pdf_file with
    | [] ->
	print_endline ("screrge: not scribbles for " ^ pdf_file ^ " in " ^ options.db_file)
    | scribbles ->
	if options.verbose then
	  print_endline ("screrge: " ^ pdf_file ^ " -> " ^ pdf_file' ^ " ...");
	let pdf = Pdfread.pdf_of_file pdf_file in
	let scribble_map = make_scribble_map scribbles in
	let n = ref 0 in
	let pages =
	  List.map
	    (fun page ->
	      incr n;
	      match find_scribbles !n scribble_map with
	      | [] -> page
	      | scribble_list ->
		  if options.verbose then
		    Printf.printf "screrge: file %s, page %d has %d scribbles\n"
		      pdf_file !n (List.length scribble_list);
		  ThoPDF.enclose_page_with_ops [] page (ops page scribble_list))
	    (Pdfdoc.pages_of_pagetree pdf) in
	let doc = Pdfdoc.change_pages pdf pages in
	Pdfwrite.pdf_to_file doc pdf_file'

let merge_files options =
  let annotations = Metadata.annotations ~db:options.db_file () in
  let scribbled_files =
    List.fold_right
      (fun ann acc ->
	match ann.Metadata.annotation_type with
	| Metadata.Scribble -> ann.Metadata.filename :: acc
	| _ -> acc)
      annotations [] in
  List.iter (merge_file options) scribbled_files

let _ =
  let pdf_files = ref []
  and options = defaults in
  let usage = "usage: screrge [options] [pdf files]" in
  Arg.parse
    [ ("-db", Arg.String (fun s -> options.db_file <- s), "database file (default: metadata.db)");
      ("-suffix", Arg.String (fun s -> options.suffix <- s), "suffix for merged files");
      ("-overwrite", Arg.Unit (fun _ -> options.overwrite <- true), "clobber output files");
      ("-verbose", Arg.Unit (fun _ -> options.verbose <- true), "be more verbose");
      ("-version", Arg.Unit (fun _ -> version (); exit 0), "print version and exit");
      ("-license", Arg.Unit (fun _ -> license (); exit 0), "print copyright/license and exit") ]
    (fun s -> pdf_files := s :: !pdf_files)
    usage;
  begin match !pdf_files with
  | [] -> merge_files options
  | files -> List.iter (merge_file options) (List.rev files)
  end


