(* $Id: metadata.ml 35 2008-10-19 14:34:00Z ohl $ *)
(* Copyright (C) 2008 by Thorsten Ohl <ohl@physik.uni-wuerzburg.de>

   This is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   It is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  *)  


(* $ echo '.dump file_metadata' | sqlite3 metadata.db 
   CREATE TABLE file_metadata (
     file_id             INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
     filename            VARCHAR(250) NOT NULL,
     directory_path      VARCHAR(250) NOT NULL DEFAULT '.',
     sort_priority       INTEGER NOT NULL DEFAULT 0,
     is_directory        INTEGER NOT NULL DEFAULT 0 CHECK (is_directory IN (0,1)),
     is_hidden           INTEGER NOT NULL DEFAULT 0 CHECK (is_hidden IN (0,1)),
     is_template         INTEGER NOT NULL DEFAULT 0 CHECK (is_template IN (0,1)),
     file_type           VARCHAR(250) NOT NULL,
     file_size           INTEGER NOT NULL DEFAULT 0,
     file_last_modified  INTEGER NOT NULL DEFAULT 0,
     title               TEXT,
     author              TEXT,
     number_of_pages     INTEGER
   );

   $ echo '.dump annotations' | sqlite3 metadata.db
   CREATE TABLE annotations (
     annotation_id    INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
     annotation_type  TEXT NOT NULL,
     file_id          INTEGER NOT NULL,
     layer            TEXT NOT NULL,
     file_position    INTEGER NOT NULL,
     title            TEXT,
     start_anchor     TEXT,
     end_anchor       TEXT,
     data             BLOB
   );
*)

let sqlite3_open file =
  if Sys.file_exists file then
    let db = Sqlite3.db_open file in
    match Sqlite3.errcode db with
    | Sqlite3.Rc.OK -> db
    | _ ->
	ignore (Sqlite3.db_close db);
	failwith ("sqlite3_open " ^ file ^ " " ^ Sqlite3.errmsg db)
  else
    failwith ("sqlite3_open " ^ file ^ " doesn't exist!")

let get_integer stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.INT n -> Int64.to_int n
  | Sqlite3.Data.NULL -> invalid_arg "get_integer: unexpected NULL"
  | _ -> invalid_arg "get_integer: incorrect column type"

let get_integer_option stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.INT n -> Some (Int64.to_int n)
  | Sqlite3.Data.NULL -> None
  | _ -> invalid_arg "get_integer: incorrect column type"

let get_text stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.TEXT s -> s
  | Sqlite3.Data.NULL -> invalid_arg "get_text: unexpected NULL"
  | _ -> invalid_arg "get_text: incorrect column type"

let get_text_option stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.TEXT s -> Some s
  | Sqlite3.Data.NULL -> None
  | _ -> invalid_arg "get_text_option: incorrect column type"

let get_blob stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.BLOB s -> s
  | Sqlite3.Data.NULL -> invalid_arg "get_blob: unexpected NULL"
  | _ -> invalid_arg "get_blob: incorrect column type"

let get_blob_option stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.BLOB s -> Some s
  | Sqlite3.Data.NULL -> None
  | _ -> invalid_arg "get_blob_option: incorrect column type"

let get_data stmt c =
  Sqlite3.Data.to_string_debug (Sqlite3.column stmt c)

type file_type =
  | PDF
  | Unknown_File_Type of string

let file_type_to_string = function
  | PDF -> "PDF"
  | Unknown_File_Type s -> "unkown file type '" ^ s ^ "'"

let get_file_type stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.TEXT "pdf" -> PDF
  | Sqlite3.Data.TEXT s -> Unknown_File_Type s
  | _ -> invalid_arg "get_text: incorrect column type"

type annotation_type =
  | Scribble
  | Unknown_Annotation_Type of string

let annotation_type_to_string = function
  | Scribble -> "scribble"
  | Unknown_Annotation_Type s -> "unkown annotation type '" ^ s ^ "'"

let get_annotation_type stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.TEXT "scribble" -> Scribble
  | Sqlite3.Data.TEXT s -> Unknown_Annotation_Type s
  | _ -> invalid_arg "get_text: incorrect column type"

type anchor =
  | PDF_Page of int
  | Unknown_Anchor of string

let anchor_to_string = function
  | PDF_Page n -> "pdf:/page:" ^ string_of_int n
  | Unknown_Anchor s -> "unkown anchor '" ^ s ^ "'"

let anchor_of_string s =
  let len = String.length s in
  if len <= 10 then
    Unknown_Anchor s
  else if String.compare (String.sub s 0 10) "pdf:/page:" = 0 then
    try
      PDF_Page (int_of_string (String.sub s 10 (len - 10)))
    with
    | Failure "int_of_string" -> Unknown_Anchor s
  else
    Unknown_Anchor s

let get_anchor_option stmt c =
  match Sqlite3.column stmt c with
  | Sqlite3.Data.TEXT s -> Some (anchor_of_string s)
  | Sqlite3.Data.NULL -> None
  | _ -> invalid_arg "get_text: incorrect column type"

(* The following is not tail recursive, but we won't deal with long lists
   and can avoid a [List.rev] this way. *)

let rec rows' cons stmt =
  match Sqlite3.step stmt with
  | Sqlite3.Rc.ROW ->
      (* NB: [cons stmt :: rows' stmt] won't work,
	 because [cons stmt] would not be evaluated first. *)
      let row = cons stmt in
      row :: rows' cons stmt
  | _ -> []

let rows cons stmt =
  let result = rows' cons stmt in
  match Sqlite3.finalize stmt with
  | Sqlite3.Rc.OK -> result
  | _ -> raise (Sqlite3.Error "finalize failed!")

(* The following works equally well, but suffers from nested exceptions:

     let rec rows cons stmt =
       match Sqlite3.step stmt with
       | Sqlite3.Rc.ROW ->
           (* NB: [cons stmt :: rows stmt] won't work,
     	 because [cons stmt] would not be evaluated first. *)
           let row = cons stmt in
           row :: rows cons stmt
       | _ ->
           begin match Sqlite3.finalize stmt with
           | Sqlite3.Rc.OK -> []
           | _ -> raise (Sqlite3.Error "finalize failed!")
           end
 *)  

type scribble =
    { file : string;
      file_type : file_type;
      layer : string;
      file_position : int;
      title : string option;
      start_anchor : anchor option;
      end_anchor : anchor option;
      data : string }

let scribbles ?(db = "metadata.db") filename =
  let scribble_of_row stmt =
    { file = get_text stmt 0;
      file_type = get_file_type stmt 1;
      layer = get_text stmt 2;
      file_position = get_integer stmt 3;
      title = get_text_option stmt 4;
      start_anchor = get_anchor_option stmt 5;
      end_anchor = get_anchor_option stmt 6;
      data = get_blob stmt 7 } in
  let db_conn = sqlite3_open db in
  let stmt =
    Sqlite3.prepare db_conn
      "select m.filename, m.file_type,
              a.layer, a.file_position, a.title,
              a.start_anchor, a.end_anchor, a.data
         from file_metadata as m, annotations as a using (file_id)
        where m.filename = ?1
          and a.annotation_type = 'scribble'
          and a.data not null" in
  match Sqlite3.bind stmt 1 (Sqlite3.Data.TEXT filename) with
  | Sqlite3.Rc.OK ->
      let result = rows scribble_of_row stmt in
      ignore (Sqlite3.db_close db_conn);
      result
  | _ -> raise (Sqlite3.Error "scribbles: can't bind variable")

type annotation =
    { filename : string;
      annotation_type : annotation_type }

let annotations ?(db = "metadata.db") () =
  let annotation_of_row stmt =
    { filename = get_text stmt 0;
      annotation_type = get_annotation_type stmt 1 } in
  let db_conn = sqlite3_open db in
  let stmt =
    Sqlite3.prepare db_conn
      "select m.filename, a.annotation_type
       from file_metadata as m, annotations as a using (file_id)" in
  let result = rows annotation_of_row stmt in
  ignore (Sqlite3.db_close db_conn);
  result
