#!/bin/bash
printf "\nKobo Boot Image Packer v0.1c (c) May 2019 GPLv2 by BloodRagg\n\n"
[ "${2/=}" = "$2" ] && { cat <<EOF
Usage: ${0##*/} [bootimg] mbr=file ...

Options:

  mbr=file              master boot record (required for new image)
  ubootbin=file         u-boot executable
  ubootenv=file         u-boot environment
  hwconfig=file         ntx hardware configuration
  uimage=file           uImage with Linux kernel
  initrd=file           init ram disk
  waveform=file         e-ink waveform (screen calibration)
  serial=file           serial number
  dtb=file              Linux device-tree blob
  fw=file               firmware
  data1=file            data (unknown blob)
  data2=file            data (unknown blob)

EOF
exit 1;}

grab(){ tail -c+$(($2+1)) $1 | head -c$3;}

insert(){
 #notrunc,fsync: because function can also be used on image files. (notrunc,fsync applies to files only)
 local bootimg=$1 offset=$2 maxsize=$3 filename=$4
 local size=$(wc -c $filename|awk '{print $1}')
  [ "$size" -gt "$maxsize" ] && { echo "$filename: too big, maxsize=$maxsize";exit 1;}
    dd if=/dev/zero of=$bootimg bs=1 seek=$offset count=$maxsize conv=notrunc,fsync 2>/dev/null  #wipe space
    dd if=$filename of=$bootimg bs=1 seek=$offset count=$size conv=notrunc,fsync 2>/dev/null     #insert file
}

write_magicsize(){
 local bootimg=$1 offset=$2 filename=$3
 local size=$(printf "%08x\n" $(wc -c $filename|awk '{print $1}'))
 local magic="\xff\xf5\xaf\xff\x78\x56\x34\x12\x${size:6:2}\x${size:4:2}\x${size:2:2}\x${size:0:2}\x0\x0\x0\x0"
 printf "$magic" | dd of=$bootimg seek=$offset bs=1 conv=notrunc,fsync 2>/dev/null  #insert magicsize
}

mkubootenv(){
    local file="$1"
    local bin=$(tempfile)
    local imgsize=$((0x20000))
    local crcsize=4
    local envsize=$(wc -c $file|awk '{print $1}')
    local padsize=$((imgsize-crcsize-envsize))

    tr '\n' '\0' < $file > $bin
    dd if=/dev/zero bs=$padsize count=1 2>/dev/null>> $bin
    local crc=$(crc32 $bin)
    printf "\x${crc:6:2}\x${crc:4:2}\x${crc:2:2}\x${crc:0:2}" > $bin
    tr '\n' '\0' < $file >> $bin
    dd if=/dev/zero bs=$padsize count=1 2>/dev/null>> $bin
    echo $bin
}

bootimg=$1
[ -r "$bootimg" ] && get_mbr_from="$bootimg"
[ "${2#mbr=}" = "${2}" ] || mbr=${2#mbr=}
[ -r "$mbr" ] && get_mbr_from="$mbr"
[ -n "$get_mbr_from" ] || { echo "error: no master boot record found";exit 1;}
sectors=$(grab $get_mbr_from $((0x1c6)) 4|hexdump -ve '/4 "%d"')
[ -n "$sectors" ] || { echo "error: no master boot record found";exit 1;}
sectorsize=512
mbr_start_p1=$((sectors*sectorsize))
[ $mbr_start_p1 -gt $((102400*512)) ] && { echo "error: mbr_start too big - $((mbr_start_p1/1024/1024))mb";exit 1;}

[ -r "$bootimg" ] || { 
  dd if=/dev/zero of=$bootimg bs=$mbr_start_p1 count=1 conv=fsync 2>/dev/null
}

map=(0x00000000 0x00000200 0x00000400 0x0007FFF0 0x00080BF0 0x000a0c00 0x000c0000 0x000e0000 0x000f0000 0x00100000 0x00400000 0x006FFFF0 $mbr_start_p1)
name=(mbr serial ubootbin hwconfig fw dtb ubootenv data1 data2 uimage initrd waveform mbr_start_p1)

echo "Building Image"
for option in $*;do
  file=${option#*=}
  [ "${option%=*}" = "$option" ] || printf "  adding: $file (${option/=*})\n"
  [ -r "$file" ] || { echo "${option%=*}: error reading $file" ;exit 1;}
  case $option in
    mbr=*)             insert $bootimg $((${map[0]})) $((${map[1]}-${map[0]})) $file;;
    serial=*)          insert $bootimg $((${map[1]})) $((${map[2]}-${map[1]})) $file;;
    ubootbin=*)        insert $bootimg $((${map[2]})) $((${map[3]}-${map[2]})) $file;;
    hwconfig=*)        write_magicsize $bootimg $((${map[3]})) $file
                       insert $bootimg $((${map[3]}+16)) $((${map[4]}-${map[3]}-16)) $file;;
    fw=*)              write_magicsize $bootimg $((${map[4]})) $file
                       insert $bootimg $((${map[4]}+16)) $((${map[5]}-${map[4]}-16)) $file;;
    dtb=*)             write_magicsize $bootimg $((${map[4]})) $file
                       insert $bootimg $((${map[5]}+16)) $((${map[6]}-${map[5]}-16)) $file;;
    ubootenv=*)        [ "${file%.txt}" = "$file" ] || file=$(mkubootenv $file)
                       insert $bootimg $((${map[6]})) $((${map[7]}-${map[6]})) $file;;
    data1=*)           insert $bootimg $((${map[7]})) $((${map[8]}-${map[7]})) $file;;
    data2=*)           insert $bootimg $((${map[8]})) $((${map[9]}-${map[8]})) $file;;
    uimage=*)          insert $bootimg $((${map[9]})) $((${map[10]}-${map[9]})) $file;;
    initrd=*)          insert $bootimg $((${map[10]})) $((${map[11]}-${map[10]})) $file;;
    waveform=*)        write_magicsize $bootimg $((${map[11]})) $file
                       insert $bootimg $((${map[11]}+16)) $((${map[12]}-${map[11]}-16)) $file;; 
  esac
done
printf "Done.\n\nImagefile: $bootimg\n\n"

exit 0
