import os
import sys
import time
import cairo
import numpy
import djvu.decode
from calibre.customize import FileTypePlugin

class DJVUCovers(FileTypePlugin):

    name                = "Automatic DJVU Covers"
    description         = "Automatically converts the first page of newly added .djvu books to png and adds it as a cover."
    supported_platforms = ["linux"]
    author              = "Hazel Brenner"
    version             = (1, 0, 0)
    file_types          = set(['djvu'])
    on_postimport       = True

    def postimport(self, id, fmt, db):
        db = db.new_api
        path = "/tmp/book" + str(int(time.time())) + ".djvu"
        db.copy_format_to(id, fmt, path)
        context = Context()

        cover_path = context.process(path)
        cover = open(cover_path, 'rb').read()
        bidm = {id: cover}
        db.set_cover(bidm)

        os.remove(cover_path)
        os.remove(path)


class Context(djvu.decode.Context):
    def __init__(self):
        self.cairo_pixel_format = cairo.FORMAT_ARGB32
        self.djvu_pixel_format = djvu.decode.PixelFormatRgbMask(0xFF0000, 0xFF00, 0xFF, bpp=32)
        self.djvu_pixel_format.rows_top_to_bottom = 1
        self.djvu_pixel_format.y_top_to_bottom = 0

    def handle_message(self, message):
        if isinstance(message, djvu.decode.ErrorMessage):
            print >> sys.stderr, message
            # Exceptions in handle_message() are ignored, so sys.exit()
            # wouldn't work here.
            os._exit(1)

    def process(self, path):
        self.document = self.new_document(djvu.decode.FileURI(path))
        self.document.decoding_job.wait()

        return self.generate_cover()

    def generate_cover(self):
        page = self.document.pages[0]

        page_job = page.decode(wait=True)
        width, height = page_job.size
        rect = (0, 0, width, height)
        bytes_per_line = cairo.ImageSurface.format_stride_for_width(self.cairo_pixel_format, width)
        color_buffer = numpy.zeros((height, bytes_per_line), dtype=numpy.uint32)

        page_job.render(djvu.decode.RENDER_COLOR, rect, rect, self.djvu_pixel_format, row_alignment=bytes_per_line, buffer=color_buffer)

        color_buffer ^= 0xFF000000
        surface = cairo.ImageSurface.create_for_data(color_buffer, self.cairo_pixel_format, width, height)

        filename = "/tmp/cover" + str(int(time.time())) + ".png"

        surface.write_to_png(filename)

        return filename
