#!/usr/bin/python
# -*- coding: utf-8 -*-

# HTMLgen Output Plugin for Sigil, by PeTrDu
# Shared under GNU General Public License version 3 (GPL-v3)

# Icon thanks to BeckyEbook
# French translation thanks to Arios
# German translation thanks to brucewelch
# Italian translation thanks to Auramazda
# Polish translation thanks to bravosx

import sys, os, re, datetime, shutil

try:
    from configparser import ConfigParser
except ImportError:
    from ConfigParser import ConfigParser

PY2 = sys.version_info[0] == 2
try:
    if PY2:
        import Tkinter as tkinter
        import tkFileDialog as tkinter_filedialog
    else:
        import tkinter
        import tkinter.filedialog as tkinter_filedialog
except ImportError:
    pass

iswin = sys.platform.startswith('win')
isosx = sys.platform.startswith('darwin')

if iswin:
    if PY2:
        import _winreg as winreg
    else:
        import winreg

filesdir4htm= ''
filesdirFull=''
plugin_dir  = ''
plugin_name = ''
opffolderpath = ''
logmessages = {}
dic_href_id = {}
current_file_id = ''
didFailCopy = False

def setlogmessages(msgs):
    lang_options = ['htmlfile','saveas','copied', 'w_nocopy','langload','notkinkr',
        'pyvers','readmeta', 'askfn','cancel', 'openfn','e_openfn',
        'header', 'reading','htmsaved','newfoldr', 'w_mkdir', 'w_nodir',
        'timespnt','htm_warn', 'htm_ok']
    lenopt=len(lang_options)
    lenmsg=len(msgs)
    max=lenmsg if lenmsg<=lenopt else lenopt
    for cnt in range(0,max):
        logmessages.update({lang_options[cnt]:msgs[cnt]})

def setaddonlanguage(lang):
    lng=lang.lower()[:2]
    if lng=='eu': lng='es'
    if lng=='ca': setlogmessages([
        "Fitxer HTML","Anomena i desa l'HTML...","Fitxer copiat: %s",
        "[ADVERTÈNCIA] No s'ha pogut copiar el fitxer: %s",
        "Idioma carregat: Català","[ERROR] No s'ha trobat el mòdul 'tkinker' de python.",
        "Utilitzant Python %s","Llegint metadades de l'epub...",
        "Demanant fitxer de sortida...","Operació cancel·lada.",
        "Obrint el fitxer de sortida: %s","[ERROR] No s'ha pogut obrir el fitxer de sortida.",
        "Preparant per escriure la capçalera de l'html...",
        "Llegint %s...","Fitxer HTML desat.","Carpeta creada: %s",
        "[ADVERTÈNCIA] No s'ha pogut crear el directori de sortida: %s",
        "[ADVERTÈNCIA] No s'ha trobat el directori de sortida: %s",
        "Temps empleat: %s seg.","S'ha desat el fitxer HTML amb errors o advertències.",
        "S'ha desat el fitxer HTML amb èxit."])
    elif lng=='de': setlogmessages([
        "HTML-Datei","Speichern als HTML...","Datei kopiert: %s",
        "[WARNING] Datei kann nicht geöffnet werden: %s",
        "Geladene Sprache: Deutsch","[ERROR] 'tkinker' Python-Modul nicht gefunden.",
        "Verwende Python %s","Lese epub-Metadaten...",
        "Benötige Dateinamen für die Ausgabe...","Vorgang abgebrochen.",
        "Öffne die ausgegebene Datei: %s","[ERROR] Konnte die ausgegebene Datei nicht öffnen.",
        "Bereit zur Erstellung des html-Kopfs...",
        "Lese %s...","HTML-Datei gespeichert.","Ordner erstellt: %s",
        "[WARNING] Konnte Ausgabe-Verzeichnis nicht erstellen: %s",
        "[WARNING] Ausgabe-Verzeichnis nicht gefunden: %s",
        "Benötigte Zeit: %s sec.","HTML-Datei mit Warnung oder Fehlern gespeichert.",
        "HTML-Datei erfolgreich gespeichert."])
    elif lng=='es': setlogmessages([
        "Archivo HTML","Guardar como HTML...","Archivo copiado: %s",
        "[ADVERTENCIA] No se ha podido copiar el archivo: %s",
        "Idioma cargado: Español","[ERROR] No se ha encontrado el módulo 'tkinker' de python.",
        "Usando Python %s","Leyendo metadatos del epub...",
        "Preguntando el archivo de salida...","Operación cancelada.",
        "Abriendo archivo de salida: %s","[ERROR] No se ha podido abrir el archivo de salida.",
        "Preparando para escribir la cabecera del html...",
        "Leyendo %s...","Archivo HTML guardado.","Carpeta creada: %s",
        "[ADVERTENCIA] No se ha podido crear el directorio de salida: %s",
        "[ADVERTENCIA] No se ha encontrado el directorio de salida: %s",
        "Tiempo usado: %s seg.","Archivo HTML guardado con errores o advertencias.",
        "Archivo HTML guardado con éxito."])
    elif lng=='fr': setlogmessages([
        "Fichier HTML","Enregistrer au format HTML...","Fichier %s copié.",
        "[ATTENTION] Impossible de copier le fichier %s",
        "Langue chargée: Français","[ERREUR] le module python 'tkinker'est introuvable.",
        "Utilisation de Python %s","Lecture des métadonnées du epub...",
        "Demande d'un nom pour l'enregistrement du fichier...","Opération annulée.",
        "Ouverture du fichier de sortie %s","[ERREUR] Impossible d'ouvrir ce nom de fichier.",
        "Préparation à l'écriture de l'entête html...",
        "Lecture de %s...","Fichier HTML enregistré.","Le dossier %s a été créé.",
        "[ATTENTION] Impossible de créer le dossier de sortie %s",
        "[ATTENTION] Le dossier de sortie %s est introuvable.",
        "Temps écoulé: %s sec.","Fichier HTML enregistré avec des mises en garde ou des erreurs.",
        "Fichier HTML enregistré avec succès."])
    elif lng=='gl': setlogmessages([
        "Ficheiro HTML","Gardar como HTML...","Ficheiro copiado: %s",
        "[ADVERTENCIA] Non se puido copiar o ficheiro: %s",
        "Idioma cargado: Galego","[ERRO] Non se atopou o módulo 'tkinker' de python.",
        "Empregando Python %s","Lendo metadatos do epub...",
        "Inquirindo o ficheiro de saída...","Operación cancelada.",
        "Abrindo ficheiro de saída: %s","[ERRO] Non se puido abrir o ficheiro de saída.",
        "Preparando para escribir a cabeceira do html...",
        "Lendo %s...","Ficheiro HTML gardado.","Carpeta creada: %s",
        "[ADVERTENCIA] Non se puido crear o directorio de saída: %s",
        "[ADVERTENCIA] Non se atopou o directorio de saída: %s",
        "Tempo empregado: %s seg.","Ficheiro HTML gardado con erros ou advertencias.",
        "Ficheiro HTML gardado con éxito."])
    elif lng=='it': setlogmessages([
        "HTML file","Salva come HTML...","File copiato: %s",
        "[ATTENZIONE] Non è stato possibile copiare il file: %s",
        "Linguaggio caricato: Italiano","[ERRORE] Il modulo python 'tkinker' non è stato trovato.",
        "Versione di Python %s","Lettura dei metadati...",
        "Dare un nome al file di output...","Operazione cancellata.",
        "Apertura file output: %s","[ERRORE] Impossibile aprire il file di output.",
        "Preparazione per la scrittura degli html header...",
        "Lettura di %s...","File HTML salvato.","Cartella creata: %s",
        "[ATTENAZIONE] Impossibile creare la cartella di output: %s",
        "[ATTENZIONE] Cartella di output non trovata: %s",
        "Tempo impiegato: %s sec.","File HTML salvato con avvertimenti o errori.",
        "file HTML salvato correttamente."])
    elif lng=='pl': setlogmessages([
        "Plik HTML","Zapisz jako HTML ...","Kopiuj plik: %s"
        ,"[OSTRZEŻENIE] Nie można skopiować pliku: %s"
        ,"Załadowany język: polski","[BŁĄD] Nie znaleziono modułu python \"tkinker\"."
        ,"Używanie Pythona %s","Czytanie metadanych epub ..."
        ,"Pytanie o wyjściową nazwę pliku ...","Operacja anulowana."
        ,"Otwieranie pliku wyjściowego: %s","[BŁĄD] Nie można otworzyć pliku wyjściowego."
        ,"Przygotowanie do napisania nagłówka html ..."
        ,"Czytanie %s ...","Zapisano plik HTML.","Folder został utworzony: %s"
        ,"[OSTRZEŻENIE] Nie można utworzyć folderu wyjściowego: %s"
        ,"[OSTRZEŻENIE] Nie znaleziono folderu wyjściowego: %s"
        ,"Upływający czas: %s sek.","Plik HTML zapisany z ostrzeżeniami lub błędami."
        ,"Plik HTML został pomyślnie zapisany."])
    else: setlogmessages([
        "HTML file","Save as HTML...","Copied file: %s",
        "[WARNING] Cannot copy the file: %s",
        "Language loaded: English","[ERROR] 'tkinker' python module not found.",
        "Using Python %s","Reading epub metadata...",
        "Asking for output filename...","Operation canceled.",
        "Opening output filename: %s","[ERROR] Couldn't open output filename.",
        "Getting ready to write the html header...",
        "Reading %s...","HTML file saved.","Folder created: %s",
        "[WARNING] Couldn't create output directory: %s",
        "[WARNING] Output directory not found: %s",
        "Time spent: %s sec.","HTML file saved with warnings or errors.",
        "HTML file saved successfully."])

def getSigilUILang(bk):
    sigiluilang=''
    try:
        if isinstance(bk.sigil_ui_lang,str):
            sigiluilang=bk.sigil_ui_lang
        else:
            sigiluilang=bk.sigil_ui_lang()
    except:
        config = ConfigParser()
        try:
            config.read(os.path.join(plugin_dir,'..','sigil.ini'))
        except:
            config.read(os.path.join(plugin_dir,'..','sigil.ini'),encoding='utf-8')
        sigiluilang=config.get('user_preferences', 'ui_language')
    return sigiluilang

def getPluginVersion():
    try:
        file = open(os.path.join(plugin_dir,plugin_name,'plugin.xml'), "r")
        m=re.search('(?i)<version>([^<]*)</version>',file.read())
        file.close()
        if m:
            return m.group(1)
    except:
        pass
    return ''

def getOpfFolderPath(ebook_root):
    try:
        file=open(os.path.join(ebook_root,'META-INF',"container.xml"), 'r')
        m=re.search(r'(?is)<rootfile\b[^>]+\bfull-path="([^>"]+)[\\/][^>"\\/]+"',file.read())
        file.close()
        if m:
            return os.path.join(ebook_root,m.group(1))
    except:
        pass
    return ebook_root

def getDesktopPath():
    home = os.path.expanduser('~')
    desktop = os.path.join(home, 'Desktop')
    if not os.path.isdir(desktop) and iswin:
        try:
            regkey = winreg.OpenKey(winreg.HKEY_CURRENT_USER, \
                r'Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders')
            value, regtype = winreg.QueryValueEx(regkey, 'Desktop')
            winreg.CloseKey(regkey)
            if regtype==1 and len(value)>0 and os.path.isdir(value):
                desktop = value
        except:
            pass
    return desktop if os.path.isdir(desktop) else home

def set_filesdir_strings(filename):
    global filesdirFull
    global filesdir4htm
    dotpos=filename.rfind('.')
    if dotpos==-1:
        filesdirFull = filename + '_files'
    else:
        filesdirFull = filename[:dotpos] + '_files'
    filesdir4htm=os.path.split(filesdirFull)[-1]
    filesdir4htm=filesdir4htm.replace('%','%25').replace('&','&amp;') # special html chars % &

def makeCssChanges(text):
    text=text.rstrip()
    # url(../Fonts/xxxxx.otf) >>> url("dir_files/xxxxx.otf")
    text=re.sub('(?i)(url[(][\'"])\.\.[\\/](Fonts|Images)[\\/]([^)\n\r"]+)[\'"][)]', 'url("' + filesdir4htm + '/\\3")', text)
    text=re.sub('(?i)(url[(])\.\.[\\/](Fonts|Images)[\\/]([^)\n\r"]+)[)]', 'url("' + filesdir4htm + '/\\3")', text)

    return text.replace('\n','\n\t\t').replace('\r','')

def hreflinkreplace(matchobj):
    t_ini,t_file,t_id,t_end = matchobj.group(1,3,4,5)
    repl = t_ini + '#'
    if len(t_file)>0:
        if t_file in dic_href_id:
            repl += dic_href_id[t_file]
        else:
            repl += t_file
    else:
        repl += current_file_id
    if len(t_id)>0:
        repl += t_id.replace('#','.')
    return repl+t_end

def makeHtmChanges(text):
    # get only the body
    m=re.search('(?is)<body[^>]*>(.+)</body', text)
    if not m:
        return ''
    text=m.group(1)

    # id="idtag" >>> id="filename.xml.idtag'
    text=re.sub('(?i)(<[^>]* id=)"([^"]+")', '\\1"' + current_file_id + '.\\2', text)

    # <a href="path/file.xhtml#idtag" >>> href="#file.xhtml.idtag"
    text=re.sub(r'(?i)(<a\b[^>]* href=")([^:"#]+[\\/]|)([^:"\\/#]+|)(#[^:"]*|)(")', hreflinkreplace, text)

    # src/href=images/audio/video >>> "filesdir4htm/*"
    text=re.sub(r'(?i)(<[^>]*\bsrc)="\.\.[\\/](Images|Audio|Video)[\\/]', '\\1="' + filesdir4htm + '/', text)
    text=re.sub(r'(?i)(<[^>]*\bhref)="\.\.[\\/](Images|Audio|Video)[\\/]', '\\1="' + filesdir4htm + '/', text)

    # Compatibility fixes
    text=re.sub(r'(<[^>]*)\bxml:(lang=")', r'\1\2', text) # xml:lang= >>> lang=
    text=re.sub(r'(?i)<(h\d)\b([^>]*)/>', r'<\1\2></\1>', text) # old firefox issue
    text=re.sub(r'(?i)<(article|aside|details|div|figcaption|figure|footer|header|nav|section|summary)\b([^>]*)( id="[^>"]+")([^>]*>)',
                r'<\1\2\4<a\3></a>', text) # ids ignored in LO

    return text.replace('\r','')

def getdefaultfilename(resm):
    deffilename=''
    if resm:
        if 'creator_aut' in resm:
            deffilename=resm['creator_aut']
        elif 'creator' in resm:
            deffilename=resm['creator']
        if 'title' in resm:
            if deffilename!='':
                deffilename += ' - '
            deffilename += resm['title']

    # chars not allowed in filename \/:*?"<>\|
    deffilename=re.sub('[\\\\/:*?"<>\|]','',deffilename).strip().replace('&amp;','&')

    return plugin_name if deffilename=='' else deffilename

def getresfrommetadataxml(metadataxml):
    res = {}
    for m in re.finditer('<(dc|dcns):(title|creator|language)( [^<>]*|)>([^<>]+)</', metadataxml):
        dtag,dtagdata,dvalue = m.group(2,3,4)
        if dtag=='creator' and ':role="' in dtagdata:
            m2 = re.search('.*:role="([^"<>]+)"', dtagdata)
            if m:
                dtag += '_' + m2.group(1)
        res[dtag] = dvalue
    return res

def filecopybin(href):
    try:
        fname=os.path.split(href)[-1]
        filesource=os.path.join(opffolderpath,href)
        filedest=os.path.join(filesdirFull,fname)
        shutil.copyfile(filesource, filedest)
        print(logmessages['copied'] % fname)
    except:
        global didFailCopy
        didFailCopy=True
        print(logmessages['w_nocopy'] % href)

def run(bk):
    timecount=datetime.datetime.now()

    global plugin_name
    global plugin_dir
    global opffolderpath

    plugin_name=bk._w.plugin_name
    plugin_dir=bk._w.plugin_dir
    opffolderpath=getOpfFolderPath(bk._w.ebook_root)
    plugin_version=getPluginVersion()

    print(plugin_name + ' v' + plugin_version)

    setaddonlanguage(getSigilUILang(bk))
    print(logmessages['langload'])

    if 'tkinter' not in sys.modules and 'Tkinter' not in sys.modules:
        print(logmessages['notkinkr'])
        return -1
    print(logmessages['pyvers'] % sys.version)

    print(logmessages['readmeta'])
    metadataxml=bk.getmetadataxml()
    resm=getresfrommetadataxml(metadataxml)

    print(logmessages['askfn'])
    prefs = bk.getPrefs()
    if not 'lastfoldersaveas' in prefs or not os.path.isdir(prefs['lastfoldersaveas']):
        prefs['lastfoldersaveas']=getDesktopPath()
    root = tkinter.Tk()
    root.withdraw()
    root.title(plugin_name)
    iconfile=os.path.join(plugin_dir,plugin_name,'plugin.png')
    if not PY2 and not isosx and os.path.isfile(iconfile):
        icon_img = tkinter.PhotoImage(file=iconfile)
        root.tk.call('wm', 'iconphoto', root._w, icon_img)
    fileoptions = { 'defaultextension':'.html',
                    'initialfile'     :getdefaultfilename(resm),
                    'initialdir'      :prefs['lastfoldersaveas'],
                    'title'           :logmessages['saveas'],
                    'filetypes'       :[(logmessages['htmlfile'],'*.html')],
                    'parent'          :root }
    timecount=datetime.datetime.now()-timecount
    filename = tkinter_filedialog.asksaveasfilename(**fileoptions)
    timecount=datetime.datetime.now()-timecount
    root.destroy()
    if filename:
        filename=filename.strip()
        if len(filename)<1:
            filename=None
        else:
            prefs['lastfoldersaveas']=os.path.dirname(filename)
    if not filename:
        print(logmessages['cancel'])
        return 0
    bk.savePrefs(prefs)

    set_filesdir_strings(filename)

    print(logmessages['openfn'] % filename)
    try:
        if PY2:
            htmlout = open(filename, 'w')
        else:
            htmlout = open(filename, 'w',encoding='utf-8')
    except:
        print(logmessages['e_openfn'])
        return -1

    print(logmessages['header'])
    outdata = '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN">\n'
    outdata += '<html>\n<head>\n'
    outdata += '\t<meta http-equiv="content-type" content="text/html; charset=utf-8">\n'
    outdata += '\t<title>'
    if 'title' in resm:
        outdata += resm['title']
    outdata += '</title>\n'
    if 'creator_aut' in resm:
        outdata += '\t<meta name="author" content="' + resm['creator_aut'] + '"/>\n'
    outdata += '\t<meta name="generator" content="' + plugin_name + ' ' + plugin_version + '"/>\n'
    outdata += datetime.datetime.now().strftime('\t<meta name="created" content="%Y-%m-%dT%H:%M:%S">\n')

    if PY2:
        outdata = outdata.encode('utf-8','ignore')
    htmlout.write(outdata)

    # Reading css...
    isFirstFile=True
    for id, href in bk.css_iter():
        print(logmessages['reading'] % href)
        indata = bk.readfile(id)
        if not isinstance(indata,(str)):
            outdata = '\t\t' + indata.decode('utf-8')
        else:
            outdata = '\t\t' + indata
        outdata = makeCssChanges(outdata)
        if isFirstFile:
            outdata = '\t<style>\n' + outdata
        isFirstFile=False
        if PY2:
            outdata=outdata.encode('utf-8','ignore')
        htmlout.write(outdata)
        indata = None

    outdata = '\n\t</style>\n</head>\n<body' if not isFirstFile else '</head>\n<body'

    if 'language' in resm:
        outdata += ' lang="' + resm['language'] + '"'
    outdata += '>\n'
    htmlout.write(outdata)

    # making dictionary { xhtml_filename : ID_in_epub }
    global dic_href_id
    dic_href_id = {}
    spine_list = []
    for id, linear, href in bk.spine_iter():
        slashpos = href.rfind('\\') + 1
        slashpos2 = href.rfind('/') + 1
        if slashpos2>slashpos: slashpos=slashpos2
        dic_href_id[href[slashpos:]] = id
        spine_list += [(id,href[slashpos:])]

    # Reading xhtml...
    isFirstFile=True
    global current_file_id
    for (id,href) in spine_list:
        print(logmessages['reading'] % id)
        if isFirstFile:
            outdata =   '  <p id="' + id + '"></p>\n'
            isFirstFile=False
        else:
            outdata = '  <hr id="' + id + '" />\n'
        htmlout.write(outdata)

        if href.lower().endswith('.svg'):
            outdata  = '  <div class="margin:0;padding:0;">\n'
            outdata += '    <img style="max-width:100%;" alt="" src="' + os.path.join(filesdir4htm,href) + '"/>\n'
            outdata += '  </div>\n'
        else:
            indata = bk.readfile(id)
            current_file_id = id
            outdata=makeHtmChanges(indata)
        if PY2:
            outdata=outdata.encode('utf-8','ignore')
        htmlout.write(outdata)

    outdata = '</body>\n</html>'
    htmlout.write(outdata)
    htmlout.close()
    print(logmessages['htmsaved'])

    hrefsToCopy=[]
    for id, href, mime in bk.image_iter():
        hrefsToCopy += [href]
    for id, href, mime in bk.font_iter():
        hrefsToCopy += [href]
    for id, href, mime in bk.media_iter():
        hrefsToCopy += [href]

    global didFailCopy
    isWarning=False
    didFailCopy=False
    didFailMakedir=False
    if len(hrefsToCopy)>0:
        if not os.path.exists(filesdirFull):
            try:
                os.mkdir(filesdirFull)
                print(logmessages['newfoldr'] % filesdirFull)
            except:
                didFailMakedir=True
        if os.path.isdir(filesdirFull):
            for href in hrefsToCopy:
                filecopybin(href)
            if didFailCopy:
                isWarning=True
        else:
            isWarning=True
            if didFailMakedir==True:
                print(logmessages['w_mkdir'] % filesdirFull)
            else:
                print(logmessages['w_nodir'] % filesdirFull)

    timecount=datetime.datetime.now()-timecount
    print(logmessages['timespnt'] % '{0:.3f}'.format(timecount.total_seconds()))

    if isWarning:
        print('\n' + logmessages['htm_warn'])
    else:
        print('\n' + logmessages['htm_ok'])

    return -1 if isWarning else 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
