

from __future__ import (unicode_literals, division, absolute_import, print_function)

import csv
import os
import platform
import psutil
import re
import shutil
import cStringIO
import subprocess
import time
import zipfile

from calibre.utils.config_base import tweaks

from .ion_text import IonText
from .kpf_container import DICTIONARY_RULES_FILENAME
from .misc import (create_temp_dir, file_read_binary, file_write_binary, json_deserialize, json_serialize,
            locale_decode, locale_encode, natural_sort_key, quote_name, truncate_list, user_home_dir,
            windows_user_dir, IS_MACOS, IS_WINDOWS, LOCALE_ENCODING)
from .previewer_prep_epub import EpubPrep
from .yj_versions import GENERIC_CREATOR_VERSIONS

__license__   = "GPL v3"
__copyright__ = "2018, John Howell <jhowell@acm.org>"

PREPARE_EPUBS_FOR_PREVIEWER = True
FORCED_CLEANED_FILENAME = None
STOP_ONCE_INPUT_PREPARED = False
MAC_TEST = False
OVERRIDE_PREVIEWER_PATH = None
SAVE_INTERMEDIATE_FILES = False

MAX_CONVERSION_RETRIES = 5

CONVERSION_SLEEP_SEC = 0.1
COMPLETION_SLEEP_SEC = 1.0

TAIL_COUNT = 30
TAIL_EXTRA = 20

MAX_GUIDANCE = 30

CONVERSION_LOGS = [
    "log_conversion_app.txt",
    "log_wordConversion.txt",
    "log_stripsource.txt",
    "log_processEpub.txt",
    "log_conversion.txt",
    "log_cap.txt",
    "log_yjImgOptimiser.txt",
    "log_createMM.txt",
    "log_kglogtoion.txt",
    "log_posMap.txt",
    "log_locMap.txt",
    "log_iontocsv.txt",
    "log_metriccollector.txt",
    "execution environment",
    ]

CURRENT_LINE = [0]
ERROR_MESSAGE_HANDLING = [
    ("log_conversion_app.txt", r"Timeout\.", CURRENT_LINE),
    ("log_conversion_app.txt", r"Error: Could not find or load", CURRENT_LINE),

    ("log_cap.txt", r"IsLiveable FAILED", ["Invalid 'style/tag' combinations:", -3]),

    ("log_conversion.txt", r"UnsatisfiedLinkError", CURRENT_LINE),
    ("log_conversion.txt", r"java\.lang\.IllegalArgumentException:", CURRENT_LINE),
    ("log_conversion.txt", r"^Error(\(YJEpubAdapter\))?:E00601: ", [0, 1, 2, 3]),
    ("log_conversion.txt", r"^Error(\([A-Za-z]+\))?:E[0-9]+:", CURRENT_LINE),
    ("log_conversion.txt", r"[ERROR] : PHANTOMJS:", CURRENT_LINE),
    ("log_conversion.txt", r"NullPointerException", CURRENT_LINE),
    ("log_conversion.txt", r"Internal error occured\.", CURRENT_LINE),

    ("log_yjImgOptimiser.txt", r"Exception in thread", CURRENT_LINE),

    ("log_createMM.txt", "^Error(core)", [0, 2]),
    ("log_createMM.txt", r"^Error(\([A-Za-z]+\))?:E[0-9]+:", CURRENT_LINE),

    ("log_locMap.txt", r"^Error(\([A-Za-z]+\))?:E[0-9]+:", CURRENT_LINE),

    ("log_posMap.txt", r"^Error(\([A-Za-z]+\))?:E[0-9]+:", CURRENT_LINE),
    ("log_posMap.txt", r"IllegalArgumentException:", ["kcfpositionmapcreator:", 0]),
    ("log_posMap.txt", r"UNKNOWN_RUNTIME_ERROR", ["kcfpositionmapcreator:", 0]),
    ("log_posMap.txt", r"EXCEPTION_ACCESS_VIOLATION", ["kcfpositionmapcreator:", 0]),
    ("log_posMap.txt", r"Unknown runtime exception", ["kcfpositionmapcreator:", 0]),

    ("log_cap.txt", r"Exception in thread", CURRENT_LINE),
    ("log_conversion.txt", r"Exception in thread", CURRENT_LINE),
    ("log_posMap.txt", r"Exception: java\.lang\.", CURRENT_LINE),
    ("log_posMap.txt", r"Exception in thread", ["kcfpositionmapcreator:", 0]),
    ("log_processEpub.txt", r"^Error(\([A-Za-z]+\))?:E[0-9]+:", CURRENT_LINE),

    ("log_conversion_app.txt", r"java\.lang\.NoClassDefFoundError", CURRENT_LINE),
    ("log_conversion_app.txt", r"java\.lang\.Error", CURRENT_LINE),
    ("log_conversion_app.txt", r"INFO: Error in creating mastermobi", CURRENT_LINE),
    ("log_conversion_app.txt", r"INFO: Error in source to YJ conversion", CURRENT_LINE),
    ("log_conversion_app.txt", r"INFO: Input file not found", CURRENT_LINE),
    ("log_conversion_app.txt", r"Exception in thread", CURRENT_LINE),

    ("log_processEpub.txt", r"UnsatisfiedLinkError", CURRENT_LINE),
    ("log_processEpub.txt", r"Error occurred during initialization of VM", [0, 1]),
    ("log_stripsource.txt", r": No source found in this book\.", CURRENT_LINE),
    ]

RETRY_ERRORS = [
    ("IMG_CONV_UNKNWON_ERROR", "image converter", False),
    ("Image to Jpeg conversion failed with exit code", "image to jpeg conversion", False),
    ("Error in creating mastermobi", "kindlegen", False),
    ("Error(YJHtmlPreprocessor):E00814:", "HTML preprocessor", False),
    ("Error:E00814:", "HTML preprocessor", False),
    ("ErrorCode: ACCURARY_THRESHOLD_ERROR",
            "location map mismatch", True),
    ("Error:E141502: Error occurred while generating position map: ErrorCode: UNKNOWN_RUNTIME_ERROR",
            "position map creation run time", True),
    ("kcfpositionmapcreator: Exception in thread \"main\" java.lang.OutOfMemoryError",
            "position map out of memory", True),
    ("kcfpositionmapcreator: #  EXCEPTION_ACCESS_VIOLATION",
            "position map creation access", True),
    ]

EXTRA_GUIDANCE = [
    ({104, 403, 808, 20113, 30005, 134002}, "Some combinations of style property values are not allowed. Check and modify CSS styles."),
    ({202, 228, 230, 830, 120027}, "The organization of content within the book is incorrect. Check and modify html."),
    ({215},("Possibly a comic with panels has page-spread properties defined in the OPF without also "
            "setting the orientation. Add <meta name=\"orientation-lock\" content=\"portrait\"/> to the OPF metadata.")),
    ({226, 248, 111001, 111002}, "The languages supported are limited. More may be added in the future."),
    ({229, 401}, "There are limitations on comics and manga. They must contain only images, no text."),
    ({252, 1005, 24010}, ("NCX TOC entry may point to an element with style display:none or to a body element or "
            "when a referenced file name has no extension or is missing.")),
    ({256, 493, 30007, 120019, 120024, 120041, 120042, 120043, 120044, 140140, 20118},
            "Complex table formats are not allowed. Split up or simply the table."),
    ({262, 274, 285, 289, 295, 141103}, "Only simple SVG containing a reference to a single image is allowed."),
    ({601}, "Possibly caused by attempts to convert a dictionary or an image with a reference to a Windows drive letter."),
    ({606}, "May indicate that the book is too large to handle. Split it into smaller pieces."),
    ({803}, "Possibly caused by the presence of an 'onload' attribute."),
    ({804}, "Possibly caused by an image with the id of 'body' or the presence of mathml."),
    ({24011}, "TOC does not reflect the book organization."),
    ({120003}, "Possibly caused by multiple data-app-amzn-magnify anchors being contained within the same outer div."),
    ({131003}, "Check images for problems and replace bad ones."),
    ({141502}, ("Possibly caused by media queries that result in a significant content difference between "
            "different formats. Another possibility is missing/incorrect character encoding declarations.")),
    ]

IGNORABLE_WARNINGS = {

    1,

    24,
    2001,
    110101,
    }

UNKNOWN_VERSION_PREFIX = "unknown"

class WindowsApp(object):
    executable_ext = ".exe"
    path_seperator = ";"

    def os_exec_converter(self):

        system_root = os.environ.get(b"SystemRoot") or os.environ.get(b"windir")
        if not system_root:
            return ConversionResult(error_msg="SystemRoot environment variable is missing")

        system_drive = os.environ.get(b"SystemDrive") or system_root[:2]

        user_profile = os.environ.get(b"USERPROFILE") or locale_encode(user_home_dir())
        if not user_profile:
            return ConversionResult(error_msg="USERPROFILE environment variable is missing")

        temp_dir = os.environ.get(b"TEMP") or os.environ.get(b"TMP")
        if not temp_dir:
            return ConversionResult(error_msg="TEMP environment variable is missing")

        path = self.join_search_path(b"./lib", b"./jre/bin", b"./bin", os.environ.get(b"Path", b""))

        self.env = {

            b"JAVA_HOME": locale_encode(os.path.join(self.converter_working_dir, "jre")),
            b"Path": path,
            b"SystemDrive": system_drive,
            b"SystemRoot": system_root,
            b"TEMP": temp_dir,
            b"TMP": temp_dir,
            b"USERPROFILE": user_profile,
            b"windir": system_root,
            }

        self.java_tool_options = []

        log4j_file_name = os.path.join(self.data_dir, "log4j.properties")
        log4j_config = "log4j.rootLogger=DEBUG, A1\nlog4j.appender.A1=%s\nlog4j.appender.A1.layout=%s\n" % (
                self.log4j_obfuscated_ConsoleAppender_class, self.log4j_obfuscated_TTCCLayout_class)

        file_write_binary(log4j_file_name, log4j_config.encode("utf8"))
        self.java_tool_options.append("-Dlog4j.configuration=" + shell_quote("file:" + log4j_file_name))

        for ver in ["", self.program_version.replace(".", "")]:
            log4j_file = "log4j%s.properties" % ver
            if os.path.isfile(os.path.join(self.converter_working_dir, log4j_file)):
                break

        if os.path.isfile(os.path.join(self.converter_working_dir, "bin", "KindleImageComparator.exe")):

            self.java_tool_options.append("-DImageComparator.root=./bin")

        self.prep_converter_app_env()

        if self.java_tool_options:
            self.env[b"JAVA_TOOL_OPTIONS"] = locale_encode(" ".join(self.java_tool_options))

        self.exec_converter_app()

class MacOSApp(object):
    executable_ext = WindowsApp.executable_ext if MAC_TEST else ""
    path_seperator = ":"

    def os_exec_converter(self):
        self.env = {}
        for env_var in [b"HOME", b"LOGNAME", b"SHELL", b"TMPDIR", b"USER"]:
            if env_var in os.environ:
                self.env[env_var] = os.environ.get(env_var)

        self.env[b"JAVA_HOME"] = locale_encode(os.path.join(self.converter_working_dir, "jre"))

        self.java_tool_options = []

        self.prep_converter_app_env()

        if self.java_tool_options:
            self.env[b"JAVA_TOOL_OPTIONS"] = locale_encode(" ".join(self.java_tool_options))

        self.exec_converter_app()

class LinuxApp(object):
    executable_ext = ""
    path_seperator = ":"

    def __init__(self, *args, **kwargs):
        raise Exception("Kindle Previewer 3 is not supported under Linux")

class KindlePreviewerCommon(object):
    program_name = "Kindle Previewer 3"
    tool_name = "KPR"
    min_supported_version = "3.24.0"

    def __init__(self, *args, **kwargs):
        self.program_path = OVERRIDE_PREVIEWER_PATH or self.program_path
        self.program_path = tweaks.get("kfx_output_previewer_path", self.program_path)

    def exec_converter(self):
        if self.program_version_sort >= natural_sort_key("3.23.0"):
            self.log4j_obfuscated_ConsoleAppender_class = "f.a.d.g"
            self.log4j_obfuscated_TTCCLayout_class = "f.a.d.I"
        else:

            self.log4j_obfuscated_ConsoleAppender_class = "d.a.b.g"
            self.log4j_obfuscated_TTCCLayout_class = "d.a.b.I"

        self.converter_working_dir = os.path.join(self.program_path, "lib", "fc")

        retry_count = 0
        result = self.exec_converter_once()

        while (result.kpf_data is None) and (retry_count <= MAX_CONVERSION_RETRIES) and not SAVE_INTERMEDIATE_FILES:
            for retry_error,error_desc,no_loc_map in RETRY_ERRORS:
                if retry_error in result.error_msg:
                    break
            else:
                break

            self.log.info("Erratic %s error occurred -- Retrying conversion" % error_desc)
            retry_count += 1

            if no_loc_map:
                self.no_loc_map = True

            result = self.exec_converter_once()

        return result

class KindlePreviewerWindows(WindowsApp, KindlePreviewerCommon):
    program_versions = {
        16263168: "3.0.0",
        16229376: "3.1.0",
        17240064: "3.2.0",
        17299456: "3.3.0",
        18409984: "3.4.0",
        18294784: "3.5.0",
        20320256: "3.6.0",
        20335616: "3.7.0",
        20336128: "3.7.1",
        20561920: "3.8.0",
        20816896: "3.9.0",
        21291008: "3.10.1",
        21503488: "3.11.0",
        21699072: "3.12.0",
        21701120: "3.13.0",
        21845504: "3.14.0",
        21826560: "3.15.0",
        21918208: "3.16.0",
        22158848: "3.17.0",
        22113280: "3.17.1",
        24826344: "3.20.0",
        24829416: "3.20.1",
        24845288: "3.21.0",
        24932280: "3.22.0",
        25197496: "3.23.0",
        25367992: "3.24.0",
        28348344: "3.25.0",
        28277176: "3.27.0",
        28413880: "3.28.0",
        28437944: "3.28.1",
        }

    def __init__(self, *args, **kwargs):
        self.program_path = os.path.join(windows_user_dir(local_appdata=True), "Amazon", "Kindle Previewer 3")
        KindlePreviewerCommon.__init__(self)

    def exec_converter_app(self):
        argv = [
            self.converter_java_path,
            "-Dfile.encoding=UTF8",
            "-cp", self.classpath,
            self.conversion_app_class,
            ]

        for arg in self.converter_app_args():
            argv.append(b64(arg))

        argv.append("--is-base64-encoded")

        self.argv = locale_encode(argv)

        self.process = subprocess.Popen(self.argv, stdout=self.out_file, stderr=subprocess.STDOUT,
                            cwd=locale_encode(self.converter_working_dir), env=self.env)

class KindlePreviewerMacOS(MacOSApp, KindlePreviewerCommon):
    program_versions = {
        39253104: "3.0.0",
        39247040: "3.1.0",
        39405692: "3.2.0",
        38926032: "3.3.0",
        60363396: "3.4.0",
        58373708: "3.5.0",
        60552820: "3.6.0",
        60556308: "3.7.0",
        60941076: "3.8.0",
        60849600: "3.9.0",
        61310668: "3.10.1",
        61641952: "3.11.0",
        61868392: "3.12.0",
        61971840: "3.13.0",
        62280808: "3.14.0",
        62463396: "3.15.0",
        62595768: "3.16.0",
        62932776: "3.17.0",
        62183980: "3.17.1",
        67303184: "3.20.0",
        67305144: "3.20.1",
        65788280: "3.21.0",
        65986852: "3.22.0",
        66364496: "3.23.0",
        67069284: "3.24.0",
        70183476: "3.25.0",
        67716468: "3.26.0",
        66488936: "3.27.0",
        66751500: "3.28.0",
        66697784: "3.28.1",
        }

    def __init__(self, *args, **kwargs):
        self.program_path = "/Applications/Kindle Previewer 3.app/Contents/MacOS"
        KindlePreviewerCommon.__init__(self)

    def exec_converter_app(self):
        self.argv = [
            "./jre/bin/java",
            "-Dfile.encoding=UTF8",
            "-cp", shell_quote(self.classpath),
            self.conversion_app_class,
            ]

        for arg in self.converter_app_args():
            self.argv.append(shell_quote(arg))

        command = " ".join(self.argv)

        self.process = subprocess.Popen(
            locale_encode(command), stdout=self.out_file, stderr=subprocess.STDOUT,
            cwd=locale_encode(self.converter_working_dir),
            env=self.env, shell=True)

class ConversionResult(object):
    def __init__(self, kpf_data=None, error_msg="", logs="", guidance="", cleaned_epub_data=None):
        self.kpf_data = kpf_data
        self.error_msg = error_msg
        self.logs = logs
        self.guidance = guidance
        self.cleaned_epub_data = cleaned_epub_data

class ConversionApp(object):
    def __init__(self):
        pass

    def convert(self, infile, log, timeout_sec, tail_logs, no_prep, no_loc_map, no_img_opt, cleaned_filename):
        self.infile = infile
        self.log = log
        self.timeout_sec = timeout_sec
        self.tail_logs = tail_logs
        self.no_prep = no_prep
        self.no_loc_map = no_loc_map
        self.no_img_opt = no_img_opt

        self.program_version = self.get_program_version()
        self.program_version_sort = natural_sort_key(self.program_version)

        if (self.min_supported_version and (not self.program_version.startswith(UNKNOWN_VERSION_PREFIX)) and
                self.program_version_sort < natural_sort_key(self.min_supported_version)):
            return ConversionResult(error_msg="Unsupported %s version %s installed (version %s or newer required)" % (
                    self.program_name, self.program_version, self.min_supported_version))

        self.data_dir = create_temp_dir()
        self.unique_cnt = 0
        self.in_file_name = os.path.abspath(self.infile)

        if self.infile.endswith(".epub"):
            epub_prep = EpubPrep(self.log, self.infile)

            if PREPARE_EPUBS_FOR_PREVIEWER and not self.no_prep:

                in_file_name_ = re.sub(r"[^a-zA-Z0-9 .:/\\_+-]", "", self.infile)
                if not re.match(r"^[a-zA-Z]", in_file_name_): in_file_name_ = "f" + in_file_name_
                self.in_file_name = os.path.join(self.data_dir, os.path.basename(in_file_name_))

                epub_prep.prepare(self.in_file_name, self.conversion_app_name == "EpubAdapterApp",
                        fix_gif=self.program_version_sort < natural_sort_key("3.23.0"),
                        copy_page_numbers=self.program_version_sort < natural_sort_key("3.23.0"))

                if cleaned_filename:
                    file_write_binary(cleaned_filename, file_read_binary(self.in_file_name))
                    self.log.info("Saved cleaned conversion input file to %s" % cleaned_filename)

            self.is_dictionary = epub_prep.is_dictionary
            self.is_kim = epub_prep.is_kim
        else:
            self.is_dictionary = self.is_kim = False

        if STOP_ONCE_INPUT_PREPARED:
            result = ConversionResult(error_msg="Conversion disabled")
        else:
            self.log.info("Converting %s to KPF" % quote_name(self.infile))
            result = self.exec_converter()

        try:
            shutil.rmtree(self.data_dir)
        except:
            pass

        return result

    def get_program_version(self):
        main_program_path = os.path.join(self.program_path, "%s%s" % (self.program_name, self.executable_ext))

        if not os.path.isfile(main_program_path):
            return UNKNOWN_VERSION_PREFIX

        program_len = os.path.getsize(main_program_path)
        return self.program_versions.get(program_len, "%s_%d" % (UNKNOWN_VERSION_PREFIX, program_len))

    def exec_converter_once(self):
        if not os.path.isdir(self.converter_working_dir):
            return ConversionResult(error_msg="%s not installed as expected. (%s missing)" % (self.program_name, self.converter_working_dir))

        self.converter_java_path = os.path.join(self.converter_working_dir, "jre", "bin", "java%s" % self.executable_ext)
        if not os.path.isfile(self.converter_java_path):
            return ConversionResult(error_msg="%s not installed as expected. (%s missing)" % (self.program_name, self.converter_java_path))

        main_jar_name = "EpubToKFXConverter"
        for fn in os.listdir(os.path.join(self.converter_working_dir, "lib")):
            if fn.startswith(main_jar_name) and fn.endswith(".jar"):
                self.converter_jar_fn = fn
                break
        else:
            return ConversionResult(error_msg="%s not installed as expected. (%s missing)" % (self.program_name, main_jar_name))

        for fn in ["yjhtmlcleanerapp", "htmlcleanerapp"]:
            if os.path.isfile(os.path.join(self.converter_working_dir, "bin", fn + self.executable_ext)):
                self.html_cleaner_fn = fn
                break
        else:
            return ConversionResult(error_msg="%s not installed as expected. (htmlcleanerapp missing)" % self.program_name)

        self.classpath = "lib/*"

        unique_dir = os.path.join(self.data_dir, "%04x" % self.unique_cnt)
        self.unique_cnt += 1
        os.mkdir(unique_dir)

        self.out_dir = os.path.join(unique_dir, "conv_resources")
        os.mkdir(self.out_dir)

        self.out_file_name = os.path.join(unique_dir, "conversion.log")
        self.out_file = open(self.out_file_name, "wb")

        self.log.info("Launching %s %s (version %s)" % (self.program_name, self.conversion_app_name, self.program_version))

        self.os_exec_converter()

        start_time = time.time()
        timeout = failure = False

        while self.process.poll() is None:
            if self.timeout_sec and (time.time() - start_time > self.timeout_sec):
                timeout = True

                parent = psutil.Process(self.process.pid)
                children = parent.children(recursive=True)
                for child in children:
                    child.kill()
                psutil.wait_procs(children, timeout=5)
                parent.kill()
                parent.wait(5)
                self.out_file.write(b"\nTimeout. Conversion did not complete within %d seconds. Process terminated\n" % self.timeout_sec)

            time.sleep(CONVERSION_SLEEP_SEC)

        if self.process.returncode:
            self.out_file.write(b"\nFailure. Process return code %s\n" % unicode(self.process.returncode))
            failure = True

        self.out_file.close()

        time.sleep(COMPLETION_SLEEP_SEC)

        log_data = {}
        log_data["log_conversion_app.txt"] = file_read_utf8(self.out_file_name)

        exe_env = []
        exe_env.append("platform: %s, architecture: %s, locale: %s" % (platform.platform(), unicode(platform.architecture()), LOCALE_ENCODING))
        exe_env.append("cwd: %s, program_path: %s" % (self.converter_working_dir, self.program_path))
        exe_env.append("argv:")
        for arg in self.argv:
            exe_env.append("  %s" % b64_dump(arg))
        exe_env.append("env:")
        for k,v in sorted(self.env.items()):
            exe_env.append("  %s = %s" % (str_dump(k), str_dump(v)))
        log_data["execution environment"] = "\n".join(exe_env)

        conversion_guidance_data = {}

        for dirpath, dns, fns in os.walk(self.out_dir):
            for fn in fns:

                if (fn.startswith("log_") and fn.endswith(".txt")) or fn in CONVERSION_LOGS:
                    if fn not in CONVERSION_LOGS:
                        self.log.warning("Unexpected conversion log found: %s" % fn)

                    try:
                        log_data[fn] = file_read_utf8(os.path.join(dirpath, fn))
                    except:
                        pass

                if fn == "tempInput.json" and "ionToCSV" in dirpath:
                    temp_input = json_deserialize(file_read_utf8(os.path.join(dirpath, fn)).decode("utf8"))
                    for name,loc in temp_input["messageIonFilePaths"].items():
                        conversion_guidance_data[name] = file_read_utf8(loc)

        error_msg = None

        for log_file, message_pat, lines in ERROR_MESSAGE_HANDLING:
            log = log_data.get(log_file, None)

            if (not log) and self.conversion_app_name != "KFXGenApp" and log_file == "log_conversion.txt":
                log = log_data.get("log_conversion_app.txt", None)

            if log and re.search(message_pat, log, flags=re.MULTILINE):
                log_list = log.split("\n")
                for i,log_line in enumerate(log_list):
                    if re.search(message_pat, log_line):
                        msgs = []
                        for offset_or_msg in lines:
                            if isinstance(offset_or_msg, unicode):
                                msgs.append(offset_or_msg)
                            else:
                                line_num = i + offset_or_msg
                                if line_num >= 0 and line_num < len(log_list):
                                    msgs.append(log_list[line_num])

                        error_msg = re.sub(r"Converting File: .+?; ","", " ".join(msgs))
                        break
                else:
                    raise Exception("Failed to locate conversion log line with message: %s" % message_pat)

            if error_msg:
                break

        if os.path.isfile(os.path.join(self.out_dir, "book", "book.kdf")) and not (timeout or failure):
            error_msg = ""
            kpf_data = package_kpf(self.log, self.out_dir, self.tool_name, self.program_version,
                            report_missing_kcb=(self.conversion_app_name != "EpubAdapterApp"))
        else:
            kpf_data = None

            if not error_msg:
                error_msg = "Unknown (See log)"

        if error_msg:
            error_msg = "Kindle Previewer error: %s" % error_msg

        logs = []
        logs.append(error_msg or "Successful conversion to KPF")
        logs.append("\n\n")

        for fn in sorted(log_data.keys(), key=log_sort_key):
            logs.append("******************** " + fn + " ********************\n")
            logs.append(tail(log_data[fn]) if self.tail_logs else log_data[fn])
            logs.append("\n")

        guidance_entries = []
        conversion_log_csv_fn = "%s-ConversionLog.csv" % (os.path.splitext(os.path.basename(self.in_file_name))[0])
        conversion_log_csv_file = os.path.join(self.out_dir, conversion_log_csv_fn)

        csv_field_names = EXPECTED_CSV_FIELD_NAMES = ["Type", "Description", "Source File", "Line Number", "Recommended Fix"]

        if os.path.isfile(conversion_log_csv_file):
            try:
                with open(conversion_log_csv_file) as csvfile:
                    for row in csv.reader(csvfile):
                        if len(row) < 2:
                            pass
                        elif row[0] == "Type":
                            csv_field_names = row

                            if row != EXPECTED_CSV_FIELD_NAMES:
                                self.log.warning("Unexpected conversion summary header: %s" % unicode(row))
                        else:
                            field = {}
                            for name,val in zip(csv_field_names, row):
                                field[name.decode("utf8")] = val.decode("utf8") if val and val != b"None" else ""

                            guidance_lines = []
                            guidance_lines.append("%s: %s" % (field.pop("Type", ""), field.pop("Description", "")))

                            source_file = field.pop("Source File", "")

                            if source_file:
                                msg = "    Source File: %s" % source_file

                                line_number = field.pop("Line Number", "")
                                if line_number:
                                    msg += " (Line %s)" % line_number

                                guidance_lines.append(msg)

                            for k,v in sorted(field.items()):
                                if v:
                                    guidance_lines.append("    %s: %s" % (k, v))

                            guidance_entries.append("\n".join(guidance_lines) + "\n")
            except Exception as e:
                self.log.warning("Exception occurred processing conversion summary: %s" % unicode(e))

        for guidance_name,guidance_data in conversion_guidance_data.items():

            try:
                guidance = IonText(self.log).deserialize_multiple_values(guidance_data)

                if (len(guidance) == 2 and
                        isinstance(guidance[0], dict) and guidance[0].get("type") == "metadata" and
                        isinstance(guidance[1], dict) and guidance[1].get("type") == "message_store"):
                    for data in guidance[1].get("data", []):
                        message_exposure_level = data.pop("message_exposure_level", "")

                        if message_exposure_level == "internal":
                            guidance_lines = []
                            code = data.pop("code", "")

                            guidance_lines.append("%s %s%s: %s" % (message_exposure_level.capitalize(),
                                    data.pop("type", "").capitalize(),
                                    " " + code if code else "", data.pop("message", "")))

                            source_location = data.pop("source_location", None)
                            if source_location:
                                if source_location.get("type", "") == "filename_line_no_based":
                                    filename = source_location.get("filename")
                                    if filename:
                                        msg = "    Source File: %s" % filename

                                        line_no = source_location.get("line_no")
                                        if line_no:
                                            msg += " (Line %s)" % unicode(line_no)

                                        guidance_lines.append(msg)
                                else:
                                    guidance_lines.append("    Source Location: %s" % unicode(source_location))

                            for k,v in data.items():
                                guidance_lines.append("    %s: %s" % (k.capitalize(), v))

                            int_code = int(code) if re.match("^[0-9]+$", code) else 0

                            for codes,fix in EXTRA_GUIDANCE:
                                if int_code in codes:
                                    guidance_lines.append("    Recommended Fix: %s" % fix)

                            guidance_entries.append("\n".join(guidance_lines) + "\n")

                        elif message_exposure_level not in ["", "publisher"]:
                            self.log.warning("Guidance %s has unexpected message_exposure_level: %s" % (guidance_name, message_exposure_level))
                            print("data: %s" % unicode(data))
                else:
                    self.log.warning("Guidance %s has unexpected structure" % guidance_name)
            except Exception as e:
                self.log.warning("Exception processing conversion guidance %s (%d bytes): %s" % (
                        guidance_name, len(guidance_data), unicode(e)))

        for fn in sorted(log_data.keys(), key=log_sort_key):
            if fn not in ["log_createMM.txt", "log_kglogtoion.txt"]:
                for msg in sorted(list(set(re.findall(r"Warning(?:\([A-Za-z]+\))?:W[0-9]+:.*", log_data[fn])))):
                    m = re.search(":W([0-9]+):(.*)", msg)
                    msg_num = int(m.group(1))

                    if msg_num == 21:
                        self.log.warning("Previewer failed to process page numbers in source document")

                    if msg_num not in IGNORABLE_WARNINGS:
                        guidance_entries.append("Conversion Warning: %s\n" % m.group(2))

        return ConversionResult(kpf_data=kpf_data, error_msg=error_msg, logs="\n".join(logs),
                guidance="\n".join(truncate_list(guidance_entries, MAX_GUIDANCE)))

    def join_search_path(self, *args):

        pl = []
        for arg in args:
            pl.extend(arg.split(str(self.path_seperator)))

        path_list = []
        for dir in pl:
            if dir and dir not in path_list:
                path_list.append(dir)

        return str(self.path_seperator).join(path_list)

if MAC_TEST or IS_MACOS:
    class KindlePreviewer(KindlePreviewerMacOS):
        pass

elif IS_WINDOWS:
    class KindlePreviewer(KindlePreviewerWindows):
        pass

else:
    class KindlePreviewer(LinuxApp):
        pass

class KFXGenApp(KindlePreviewer, ConversionApp):
    conversion_app_name = "KFXGenApp"
    conversion_app_class = "com.amazon.kfxconverter.app.KFXGenApp"

    def __init__(self, *args, **kwargs):
        KindlePreviewer.__init__(self, *args, **kwargs)
        ConversionApp.__init__(self, *args, **kwargs)

    def prep_converter_app_env(self):
        pass

    def converter_app_args(self):

        argv = [
            "-libDir", ".",
            "-tmpDir", self.out_dir,
            "-outDir", self.out_dir,
            "-inputFile", self.in_file_name,
            "-locale", "en",
            "-amzncreator", "\"%s %s\"" % (self.program_name, self.program_version),
            "-debug",
            ]

        loc_map_disabled = False

        if self.is_dictionary:
            self.log.warning("Lookup will not function in dictionaries converted to KFX format")

        if self.program_version_sort >= natural_sort_key("3.23.0"):

            if self.no_img_opt or self.is_kim:
                self.log.info("Image optimization and location map generation disabled")
                argv.append("-disablePostConversionForYJ")
                loc_map_disabled = True

        elif self.no_img_opt:
            raise Exception("Disabling image optimization is only available in Kindle Previewer version 3.23 or higher")

        if self.program_version_sort >= natural_sort_key("3.17.0"):
            if (self.no_loc_map or self.is_dictionary) and not loc_map_disabled:
                self.log.info("Location map generation disabled")
                argv.append("-disableLocMapGeneration")

        if self.program_version_sort >= natural_sort_key("3.15.0"):
            argv.append("-run-conversion-for-yjcoach")

            if self.program_version_sort <= natural_sort_key("3.17.0"):
                argv.append("-allowYJConversionForJP")

        if self.program_version_sort >= natural_sort_key("3.14.0") and self.program_version_sort < natural_sort_key("3.16.0"):
            argv.append("-allowYJConversionForArabic")

        if self.program_version_sort >= natural_sort_key("3.13.0"):
            argv.append("-allowYJConversionForCN")

        if self.program_version_sort >= natural_sort_key("3.11.0"):

            pass

        if self.program_version_sort >= natural_sort_key("3.10.1"):
            argv.append("-allowYJConversionForFL")

        return argv

def convert_nonyj_to_kpf(infile, log, app_name, timeout_sec, tail_logs, no_prep, no_loc_map, no_img_opt, cleaned_filename):
    intype = os.path.splitext(infile)[1]

    if intype == ".last":
        return last_previewer_gui_result_to_kpf(log)

    if not app_name:
        app_name = "KFXGenApp"

    cleaned_filename = FORCED_CLEANED_FILENAME or cleaned_filename

    if False:
        pass

    elif app_name == "KFXGenApp" and intype in [".epub", ".opf", ".doc", ".docx", ".mobi"]:
        conversion_app = KFXGenApp()

    else:
        return ConversionResult(error_msg="Cannot generate KPF from %s file using %s" % (intype, app_name))

    return conversion_app.convert(infile, log, timeout_sec, tail_logs, no_prep, no_loc_map, no_img_opt, cleaned_filename)

def shell_quote(s):
    return "\"%s\"" % s if ((" " in s) or ("*" in s)) and not s.startswith("\"") else s

def b64(s):
    return locale_encode(s).encode("base64").decode("ascii").replace("\n", "")

def b64_dump(s):
    try:
        sb = s.encode("ascii").decode("base64").decode("ascii")

        return "%s    (base64)    %s" % (str_dump(s), str_dump(sb))
    except:
        return str_dump(s)

def str_dump(s):
    if isinstance(s, unicode):
        return s

    ss = []
    for ch in s:
        o = ord(ch)
        if ch == b"\\":
            ss.append("\\\\")
        elif o >= 0x20 and o <= 0x7e:
            ss.append(unichr(o))
        else:
            ss.append("\\x%02x" % o)

    return "".join(ss)

def file_read_utf8(filename):
    return file_read_binary(filename).decode("utf-8", errors="replace").replace("\r", "")

def tail(data):
    lines = data.split("\n")
    removed = len(lines) - TAIL_COUNT
    if removed <= TAIL_EXTRA:
        return data

    return ("(%d lines removed from log)\n" % removed) + "\n".join(lines[-TAIL_COUNT:])

def log_sort_key(log_name):
    if log_name in CONVERSION_LOGS:
        return (CONVERSION_LOGS.index(log_name), "")

    return (99, log_name)

def last_previewer_gui_result_to_kpf(log):
    log.info("Locating most recent Kindle Previewer GUI results")

    for temp_var in [b"TMPDIR", b"TEMP", b"TMP"]:
        if temp_var in os.environ:
            try:
                temp_dir = locale_decode(os.environ.get(temp_var))
                break
            except:
                pass
    else:
        return ConversionResult(error_msg="Failed to locate the TEMP directory")

    last_kdf_dir = None
    last_kdf_time = None

    for dirpath, dns, fns in os.walk(temp_dir):
        for dn in dns:
            if (re.match("^[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}$", dn, flags=re.IGNORECASE) or
                    re.match("^[0-9a-z]{6}$", dn, flags=re.IGNORECASE) or
                    re.match("^[0-9a-f]{8}(-[0-9a-f]{4}){3}-[0-9a-f]{12}$", dn, flags=re.IGNORECASE)):
                kdf_dir = os.path.join(dirpath, dn)

                has_mobi = False
                for kdf_dirpath, kdf_dns, kdf_fns in os.walk(kdf_dir):
                    for kdf_fn in kdf_fns:
                        if kdf_fn.endswith(".mobi"):
                            has_mobi = True
                            break
                    break

                if has_mobi and os.path.isdir(os.path.join(kdf_dir, "conv_metrics")):
                    kdf_time = os.stat(kdf_dir).st_ctime

                    if (not last_kdf_dir) or kdf_time > last_kdf_time:
                        last_kdf_dir = kdf_dir
                        last_kdf_time = kdf_time

        break

    if not last_kdf_dir:
        return ConversionResult(error_msg="No Kindle Previewer KDF database file found. No conversion results are available.")

    if not os.path.isfile(os.path.join(last_kdf_dir, "book", "book.kdf")):
        return ConversionResult(error_msg="The most recent Kindle Previewer conversion failed to produce a KDF database "
                    "(Enhanced Typesetting: Not Supported)")

    log.info("Found %s, created %s" % (last_kdf_dir, time.ctime(last_kdf_time)))

    return ConversionResult(kpf_data=package_kpf(log, last_kdf_dir))

def package_kpf(log, out_dir, default_tool_name="KPR", default_tool_version=None, report_missing_kcb=True):
    if not os.path.isfile(os.path.join(out_dir, "book", "book.kdf")):
        raise Exception("KDF database file not found in %s" % out_dir)

    files = {}

    for kcb_file in [
                os.path.join(out_dir, "book", "book.kcb"),
                os.path.join(out_dir, "run.kcb")]:
        if os.path.isfile(kcb_file):
            kcb = json_deserialize(file_read_binary(kcb_file))
            break
    else:
        if report_missing_kcb:
            log.warning("KCB file is missing after conversion")

        kcb = {"metadata": {
                "book_path": "resources",
                "format": "yj",
                "tool_name": default_tool_name,
                }}

    metadata = kcb["metadata"]

    metadata["book_path"] = "resources"

    if default_tool_version:
        if "tool_version" not in kcb["metadata"]:
            metadata["tool_version"] = default_tool_version

        elif (metadata.get("tool_name"), metadata.get("tool_version")) in GENERIC_CREATOR_VERSIONS:
            metadata["tool_name"] = default_tool_name
            metadata["tool_version"] = default_tool_version

    files["book.kcb"] = json_serialize(kcb)

    book_dir = os.path.join(out_dir, "book")
    for dirpath, dns, fns in os.walk(book_dir):
        dir = dirpath[len(book_dir):].replace("\\", "/")
        for fn in fns:
            if not fn.endswith(".kcb"):
                files["resources" + dir + "/" + fn] = file_read_binary(os.path.join(dirpath, fn))

    for dictionary_rules_path in [
                os.path.join(out_dir, DICTIONARY_RULES_FILENAME),
                os.path.join(out_dir, "conv_tmp", "out", DICTIONARY_RULES_FILENAME)]:
        if os.path.isfile(dictionary_rules_path):
            files["resources" + "/" + DICTIONARY_RULES_FILENAME] = file_read_binary(dictionary_rules_path)
            break

    f = cStringIO.StringIO()

    with zipfile.ZipFile(f, "w",  zipfile.ZIP_DEFLATED) as zf:
        for fn in files:
            zf.writestr(fn, files[fn])

    result_data = f.getvalue()
    f.close()

    return result_data

