#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys, os, glob, uuid, re, zipfile
from subprocess import Popen, PIPE
from xml.dom.minidom import parse, parseString

# add kindleunpack imports
current_dir = os.path.dirname(os.path.abspath(__file__))
sys.path.append(os.path.join(current_dir,  'kindleunpack'))
from mobi_split import mobi_split
from unipath import pathof
from compatibility_utils import PY2, iswindows
from dualmetafix_mmap import DualMobiMetaFix

# simple kindlegen wrapper
def kgWrapper(*args):
    process = Popen(list(args), stdout=PIPE, stderr=PIPE)
    stdout, stderr = process.communicate()
    retcode = process.returncode
    return stdout, stderr, retcode

# main routine    
def Converter():
    if PY2:
        print('This script requires Python 3!')
        return
        
    # log file
    log_file_data = ''
    
    # add fake asin
    add_asin = True

    #----------------------------------
    # get a list of all epub files
    #----------------------------------
    epub_files = glob.glob(os.path.join(current_dir,  '**', '*.epub*'), recursive = True)
    if epub_files == []:
        print('No epub files found.')
        return
    
    # common kindlegen paths
    if iswindows:
        win_kp_path = os.path.join(os.getenv('USERPROFILE'), 'AppData', 'Local', 'Amazon', 'Kindle Previewer 3','lib', 'fc', 'bin', 'kindlegen.exe')
    osx_kp_path = os.path.join('/Applications', 'Kindle Previewer 3.app', 'Contents', 'MacOS', 'lib', 'fc', 'bin', 'kindlegen')
    kg_exe = os.path.join(current_dir, 'kindlegen.exe')
    kg_bin = os.path.join(current_dir, 'kindlegen')
    
    #----------------------------
    # look for kindlegen
    #----------------------------
    if iswindows:
        if os.path.isfile(win_kp_path): 
            kg_path = win_kp_path
        if os.path.isfile(kg_exe):
            kg_path = kg_exe
    elif os.path.isfile(osx_kp_path):
        kg_path = osx_kp_path
    elif os.path.isfile(kg_bin):
        kg_path = kg_bin
    else:
        kg_path = None
    
    # stop execution if kindlegen wasn't found
    if not kg_path:
        print('Kindlegen not found!')
        return
    else:
        print('KindleGen path: ', kg_path)

    #-----------------------
    # process epubs
    #-----------------------
    for epub_file in epub_files:
        current_file = os.path.basename(epub_file)
        print('\nProcessing {} ...'.format(current_file))

        #---------------------
        # run kindlegen
        #---------------------
        args = [kg_path, epub_file, '-c1', '-dont_append_source']
        stdout, stderr, retcode = kgWrapper(*args)
        
        # check return code
        if retcode == 2:
            print('{} failed to convert because of errors!'.format(current_file))
            
            # save errors
            log_file_data += stdout.decode('utf-8')

        else:
            # define paths
            mobi_path = os.path.splitext(epub_file)[0] + '.mobi'
            azw_path = os.path.splitext(epub_file)[0] + '.azw3'
            
            # if kindlegen didn't fail, there should be a .mobi file
            if os.path.isfile(mobi_path):
                if retcode == 0:
                    print('{} built.'.format(os.path.basename(mobi_path)))
                else:
                    print('{} built with WARNINGS!'.format(os.path.basename(mobi_path)))
                
                # save warnings
                if retcode != 0:
                    log_file_data += stdout.decode('utf-8')
                                
                #--------------------
                # add ASIN
                #--------------------
                if add_asin:
                    asin = None
                    # open epub as zip file
                    z = zipfile.ZipFile(epub_file, 'r')
                    for filename in z.namelist():
                        root, ext = os.path.splitext(filename)
                        # look for .opf file
                        if ext == '.opf':
                            opf_data = z.read(filename)
                            dom = parseString(opf_data)
                            # look for ASIN entry
                            for dc_id in dom.getElementsByTagName('dc:identifier'):
                                if dc_id.hasAttribute('opf:scheme'):
                                    opf_scheme = dc_id.getAttribute('opf:scheme')
                                    if opf_scheme == ('MOBI-ASIN'):
                                        asin = dc_id.firstChild.data
                                        break
                    z.close()
                    
                    # create fake ASIN
                    if not asin:
                        asin = str(uuid.uuid4())[24:82]
                    dmf = DualMobiMetaFix(mobi_path, asin)
                    open(pathof(mobi_path + '.tmp'),'wb').write(dmf.getresult())
                    
                    # if DualMobiMetaFix didn't fail, there should be a .temp file
                    if os.path.isfile(str(mobi_path) + '.tmp'):
                        print('ASIN: ' + asin + ' added')

                        # delete original file and rename temp file
                        os.remove(str(mobi_path))
                        os.rename(str(mobi_path) + '.tmp', str(mobi_path))
                    else:
                        print('ASIN couldn\'t be added.')

                #------------------
                # split mobi
                #------------------
                mobisplit = mobi_split(pathof(mobi_path))
                open(azw_path, 'wb').write(mobisplit.getResult8())
                
                # delete temporary .mobi file
                if os.path.isfile(azw_path):
                    os.remove(str(mobi_path))

                # display status message
                    print('{} generated.'.format(os.path.basename(azw_path)))
                else:
                    print('{} NOT generated.'.format(os.path.basename(azw_path)))
            else:
                print('Internal error: {} NOT found!'.format(mobi_path))

    #----------------------
    # write log file
    #----------------------
    if log_file_data != '':
        with open(os.path.join(current_dir, 'errors.txt'), 'w') as errors:
            errors.write(log_file_data)

    print('\nDone')
    
if __name__ == '__main__':
    Converter()
