#!/usr/bin/env python2
# vim:fileencoding=utf-8

'''
This plugin is mostly based on Doitsu's code for Sigil's ACE Plugin.
Also, it's a simplified version. It just enables running ACE directly from the Editor.
At the end, the report will open on your default browser.
The GUI settings is based on KindleUnpack.
'''

from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__ = 'GPL v3'
__copyright__ = '2018, Thiago'

# PyQt libraries
from PyQt5.Qt import QApplication, QAction, QMessageBox, QDialog, Qt, QMenu, QIcon, QPixmap
# Calibre libraries
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.plugin import Tool
from calibre.utils.config import JSONConfig, config_dir
from calibre.constants import iswindows, islinux, isosx

# standard libraries
import os, locale, sys, tempfile, webbrowser, shutil
import os.path
from os.path import expanduser

# DiapDealer's temp folder code
from contextlib import contextmanager

# Get config
import calibre_plugins.ACE.config as cfg

# Get user language
language = locale.getdefaultlocale()

# Localized messages
if language[0] == "pt_BR" or "pt_PT" or "pt":
    msg_RUN = "Executar ACE"
    msg_SAVEPOINT = "Antes de: ACE"
    msg_FORMAT_TITLE = "Formato não suportado"
    msg_FORMAT_MSG = "Não é possível verificar arquivos {} com o ACE."
    msg_ERROR_TITLE = "Erro"
    msg_FATAL_TITLE = "Exceção não tratada"
    msg_FATAL_MSG = "Um erro inesperado ocorreu. Clique em 'Mostrar detalhes' para mais informações."
    msg_REPORT_TRUE = "A verificação do ACE foi concluída!\n" \
                      "O relatório será aberto no seu navegador padrão."
    msg_REPORT_FALSE = "A verificação do ACE foi concluída!\n" \
                       "O relatório foi salvo na pasta: '"
    msg_RERUN_TITLE = 'Reexecutar'
    msg_RERUN_MSG = 'Ocorreu um erro durante a execução.\n' \
                    'Gostaria de executar o ACE novamente?\n' \
                    'Isso pode resolver alguns erros.'

else:
    msg_RUN = "Run ACE"
    msg_SAVEPOINT = "Before: ACE"
    msg_FORMAT_TITLE = "Unsupported file format"
    msg_FORMAT_MSG = "You can't check {} files with ACE."
    msg_ERROR_TITLE = "Error"
    msg_FATAL_TITLE = "Unhandled exception"
    msg_FATAL_MSG = "An unexpected error occurred. Click 'Show details' for more info."
    msg_REPORT_TRUE = "ACE check is finished!\n" \
                      "The report will open on your default browser."
    msg_REPORT_FALSE = "ACE check is finished!\n" \
                       "The report was saved to: '"
    msg_RERUN_TITLE = 'Rerun'
    msg_RERUN_MSG = 'ACE found an error during execution.\n' \
                    'Do you want to try to rerun it?\n' \
                    'This can resolve a few errors.'


@contextmanager
def make_temp_directory():
    import tempfile
    temp_dir = tempfile.mkdtemp()
    yield temp_dir
    shutil.rmtree(temp_dir)


# Simple wrapper for ACE
def ace_wrapper(*args):
    import subprocess
    startupinfo = None
    if islinux:
        process = subprocess.Popen(list(args), stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                   startupinfo=startupinfo, shell=False)
    else:
        process = subprocess.Popen(list(args), stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                   startupinfo=startupinfo, shell=True)
        # Stop the windows console popping up every time the program is run
        if iswindows:
            startupinfo = subprocess.STARTUPINFO()
            startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
            startupinfo.wShowWindow = subprocess.SW_HIDE
    ret = process.communicate()
    return_code = process.returncode
    return ret, return_code


def get_icon(icon_name):

    # Check to see whether the icon exists as a Calibre resource
    # This will enable skinning if the user stores icons within a folder like:
    # ...\AppData\Roaming\calibre\resources\images\Plugin Name\
    icon_path = os.path.join(config_dir, 'resources', 'images', 'ACE',
                             icon_name.replace('images/', ''))
    if os.path.exists(icon_path):
        pixmap = QPixmap()
        pixmap.load(icon_path)
        return QIcon(pixmap)
    # As we did not find an icon elsewhere, look within our zip resources
    return get_icons(icon_name)


class AceTool(Tool):
    
    # Set this to a unique name it will be used as a key
    name = 'ace'
    # If True the user can choose to place this tool in the plugins toolbar
    allowed_in_toolbar = True
    # If True the user can choose to place this tool in the plugins menu
    allowed_in_menu = True

    def create_action(self, for_toolbar=True):
        # Create an action, this will be added to the plugins toolbar and
        # the plugins menu
        ac = QAction(get_icon('images/icon.png'), msg_RUN, self.gui)
        if not for_toolbar:
            # Register a keyboard shortcut for this toolbar action. We only
            # register it for the action created for the menu, not the toolbar,
            # to avoid a double trigger
            self.register_shortcut(ac, 'ACE-tool', default_keys=('Ctrl+Shift+Alt+A',))
        else:
            menu = QMenu()
            ac.setMenu(menu)
            config_menu_item = menu.addAction(_('Customize'), cfg.ConfigWidget)

        ac.triggered.connect(self.ask_user)
        return ac

    def ask_user(self):
        # ----------------------------------------
        # Get preferences
        # ----------------------------------------
        open_report = cfg.plugin_prefs['open_report']
        report_path = cfg.plugin_prefs['report_path']
        debug_mode = cfg.plugin_prefs['debug_mode']

        # ----------------------------------------
        # Create a savepoint
        # ----------------------------------------
        self.boss.add_savepoint(msg_SAVEPOINT)

        # -----------------------------------------
        # Test if the file is an epub
        # -----------------------------------------
        book_type = self.current_container.book_type
        if book_type != 'epub':
            QMessageBox.information(self.gui, msg_FORMAT_TITLE, msg_FORMAT_MSG.format(book_type))
            return

        # ----------------------------------------
        # Create temp directory and Run ACE
        # ----------------------------------------
        with make_temp_directory() as td:
            # Write current container to temporary epub
            epub_path = os.path.join(td, 'temp.epub')
            report_folder = os.path.join(report_path, 'report')
            report_data = os.path.join(report_folder, 'data')
            report_file_name = os.path.join(report_folder, 'report.html')
            if os.path.exists(report_data):
                shutil.rmtree(report_data)
            self.boss.commit_all_editors_to_container()
            self.current_container.commit(epub_path)

            # Define ACE command line parameters
            args = ['ace', '-f', '-o', report_folder, epub_path]

            # ----------------------------------------
            # Run ACE and handle errors
            # ----------------------------------------
            try:
                # Display busy cursor
                QApplication.setOverrideCursor(Qt.WaitCursor)

                # Run ACE
                result, return_code = ace_wrapper(*args)
                stdout = result[0]
                stderr = result[1]

                # Full debug mode (complete log)
                if debug_mode:
                    stdout += stderr
                    QApplication.clipboard().setText(stdout)

                # Hide busy cursor
                QApplication.restoreOverrideCursor()

                if return_code == 1:
                    # If an error is found
                    QMessageBox.warning(self.gui, msg_ERROR_TITLE, stderr)

                    # Ask user to rerun ACE
                    from PyQt5 import QtGui
                    reply = QMessageBox.question(self.gui, msg_RERUN_TITLE, msg_RERUN_MSG,
                                                 QMessageBox.Yes, QMessageBox.No)
                    if reply == QMessageBox.Yes:
                        # Display busy cursor
                        QApplication.setOverrideCursor(Qt.WaitCursor)

                        # ----------------------------------------
                        # Rerun ACE
                        # ----------------------------------------
                        result, return_code = ace_wrapper(*args)
                        stdout = result[0]
                        stderr = result[1]

                        # Hide busy cursor
                        QApplication.restoreOverrideCursor()

                        if return_code == 1:
                            # Exit if error persists
                            QMessageBox.critical(self.gui, msg_ERROR_TITLE, stderr)
                            return
                    else:
                        return
            except NameError:

                # Hide busy cursor
                QApplication.restoreOverrideCursor()

                # Exit if an unexpected error occurs, and report the error to the user
                import traceback
                error_dialog(self.gui, _(msg_FATAL_TITLE),
                    _(msg_FATAL_MSG),
                    det_msg=traceback.format_exc(), show=True)
                return

        # Show report on default browser
        if open_report:
            QMessageBox.information(self.gui, "ACE, by Daisy", msg_REPORT_TRUE)
            url = 'file://' + os.path.abspath(report_file_name)
            webbrowser.open(url)
        else:
            # Point report location
            QMessageBox.information(self.gui, "ACE, by Daisy", msg_REPORT_FALSE + report_folder + "'.")
