# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__docformat__ = 'restructuredtext en'

import os, locale
try:
    from PyQt5.Qt import (QWidget, QLabel, QLineEdit, QPushButton, QCheckBox,
                            QGroupBox, QVBoxLayout, QComboBox)
except ImportError:
    from PyQt4.Qt import (QWidget, QLabel, QLineEdit, QPushButton, QCheckBox,
                            QGroupBox, QVBoxLayout, QComboBox)

from calibre.utils.config import JSONConfig
from calibre.utils.filenames import expanduser
from calibre.gui2 import choose_dir, error_dialog

from calibre_plugins.ACE.__init__ import PLUGIN_NAME, PLUGIN_VERSION

PLUGIN_ICONS = ['images/icon.png']

# This is where all preferences for this plugin will be stored.
plugin_prefs = JSONConfig('plugins/ACE')

# Set default preferences
plugin_prefs.defaults['report_path'] = expanduser('~')
plugin_prefs.defaults['open_report'] = True
plugin_prefs.defaults['debug_mode'] = False

# Get user language
language = locale.getdefaultlocale()

# Localized messages
if language[0] == "pt_BR" or "pt_PT" or "pt":
    msg_REPORT_FOLDER = "Pasta do Relatório"
    msg_REPORT_FOLDER_TEXT = '<p>Pasta para salvar o relatório'
    msg_SELECT_FOLDER = 'Selecionar/Alterar Pasta do Relatório'
    msg_SELECT_FOLDER_HINT = '<p>Selecionar/Alterar a pasta onde o relatório será salvo.'
    msg_OPEN_REPORT = 'Abrir relatório após a verificação'
    msg_OPEN_REPORT_HINT = '<p>Quando desmarcado, o plugin irá mostrar uma mensagem apontando para a pasta do relatório.'
    msg_DEBUG_MODE = 'Modo de Depuração'
    msg_DEBUG_MODE_HINT = '<p>Quando marcado, o registro do ACE será salvo na área de transferência.'
    msg_GET_DIR = 'Selecione a pasta para salvar o Relatório'
    msg_ERRMSG = '<p>O caminho especificado para a Pasta do Relatório não existe.</p>' \
                        '<p>Suas últimas alteraçções de preferência <b>NÃO</b> serão salvas!</p>' + \
                        '<p>Você deve refazer a configuração e garantir que os valores estão corretos.'
else:
    msg_REPORT_FOLDER = 'Report Folder:'
    msg_REPORT_FOLDER_TEXT = '<p>Folder to save the report'
    msg_SELECT_FOLDER = 'Select/Change Report Folder'
    msg_SELECT_FOLDER_HINT = '<p>Select/Change the folder where the report will be saved.'
    msg_OPEN_REPORT = 'Open Report after checking'
    msg_OPEN_REPORT_HINT = '<p>When unchecked, it will display a message pointing to report folder.'
    msg_DEBUG_MODE = 'Debug Mode'
    msg_DEBUG_MODE_HINT = '<p>When checked, ACE log will be saved to clipboard.'
    msg_GET_DIR = 'Select Directory to save Report to'
    msg_ERRMSG = '<p>The path specified for the Report Folder does not exist.</p>' \
                        '<p>Your latest preference changes will <b>NOT</b> be saved!</p>' + \
                        '<p>You should configure again and make sure your settings are correct.'

class ConfigWidget(QWidget):

    def __init__(self):
        QWidget.__init__(self)
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        # --- Directory Options ---
        directory_group_box = QGroupBox(_(msg_REPORT_FOLDER), self)
        layout.addWidget(directory_group_box)
        directory_group_box_layout = QVBoxLayout()
        directory_group_box.setLayout(directory_group_box_layout)

        # Directory path Textbox
        # Load the textbox with the current preference setting
        self.directory_txtBox = QLineEdit(plugin_prefs['report_path'], self)
        self.directory_txtBox.setToolTip(_(msg_REPORT_FOLDER_TEXT))
        directory_group_box_layout.addWidget(self.directory_txtBox)
        self.directory_txtBox.setReadOnly(True)

        # Folder select button
        directory_button = QPushButton(_(msg_SELECT_FOLDER), self)
        directory_button.setToolTip(_(msg_SELECT_FOLDER_HINT))
        # Connect button to the getDirectory function
        directory_button.clicked.connect(self.getDirectory)
        directory_group_box_layout.addWidget(directory_button)

        # Open report checkbox
        self.open_report_check = QCheckBox(_(msg_OPEN_REPORT), self)
        self.open_report_check.setToolTip(_(msg_OPEN_REPORT_HINT))
        directory_group_box_layout.addWidget(self.open_report_check)
        # Load the checkbox with the current preference setting
        self.open_report_check.setChecked(plugin_prefs['open_report'])

        # Debug checkbox
        self.debug_mode_check = QCheckBox(_(msg_DEBUG_MODE), self)
        self.debug_mode_check.setToolTip(_(msg_DEBUG_MODE_HINT))
        directory_group_box_layout.addWidget(self.debug_mode_check)
        # Load the checkbox with the current preference setting
        self.debug_mode_check.setChecked(plugin_prefs['debug_mode'])

    def save_settings(self):
        # Save current dialog sttings back to JSON config file
            plugin_prefs['report_path'] = unicode(self.directory_txtBox.displayText())
            plugin_prefs['open_report'] = self.open_report_check.isChecked()
            plugin_prefs['debug_mode'] = self.debug_mode_check.isChecked()

    def getDirectory(self):
        c = choose_dir(self, _(PLUGIN_NAME + 'dir_chooser'),
                _(msg_GET_DIR))
        if c:
            self.directory_txtBox.setReadOnly(False)
            self.directory_txtBox.setText(c)
            self.directory_txtBox.setReadOnly(True)

    def validate(self):
        # This is just to catch the situation where someone might
        # manually enter a non-existent path in the Default path textbox.
        # Shouldn't be possible at this point.
        if not os.path.exists(self.directory_txtBox.text()):
            errmsg = msg_ERRMSG
            error_dialog(None, _(PLUGIN_NAME + ' v' + PLUGIN_VERSION),
                                    _(errmsg), show=True)
            return False
        return True
