/*
 * xepdmgr.c
 *
 * Daemon able to autonomously update the screen of the iliad
 *
 * History:
 *      23/03/2008 DRF Initial implementation. Almost works (only sees changes
 *                     to root window), ignoring that there is no iliad code yet.
 *      31/08/2008 DRF Implement the communication with the clients (root window
 *                      atoms/properties).Small fix to the Damage part.
 *       6/09/2008 DRF Make the comms with the clients more reliable.
 *       7/09/2008 DRF Implementation of the signal support and timeouts.
 *                     Implement the child process support. Implement the
 *                     messages actions.
 *       1/10/2008 HSL Fix some cut&paste erros causing too much full refreshes
 *                     Add code to kill the busy led, and to enable the keyboard
 *                     button on the tool bar
 *                 DRF Add a --version command-line switch
 *                     Implement the TRACKEDAREAS related part.
 *       4/10/2008 DRF Fix the "wrong number of parametrs" msg in --help.
 *                     Misc. fixes in the TRACKEDAREAS part.
 *
 *
 * Authors: DRF Dario Rodriguez dario@softhome.net
 *          HSL Hansel from mobileread.com
 * (c) 2008 Dario Rodriguez
 * This program is licensed under the terms of the GNU GPL v3 or greater.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <X11/Xlib.h>
#include <X11/extensions/Xdamage.h>
#ifndef DUMMY_BUILD
#include <liberdm/erdm.h>
#include <liberipc/eripcbusyd.h>
#include <liberipc/eripctoolbar.h>
#endif

#define VERSIONMAJOR 1
#define VERSIONMINOR 3

#define SHARED_ATOM_NAME_FOR_COMM "XEPDMGR_CMD"
#define SHARED_ATOM_NAME_FOR_ID "XEPDMGR_NEXTID"
#define MAXCMDSIZE 1024

#define PIXELTHRESHOLD (1024*768/3)

#define REFRESHTIMEOUTMS 500
#define TRACKEDAREASNUMBER 2

#define FDRD 0
#define FDWR 1

typedef struct TypeTrackedArea {
        int x1,y1,x2,y2;
        int pixel_count;
        int merge_with;
        int merge_overhead;
} sTrackedArea;

typedef struct TypeEpdmgr {
        Display *dpy;
        Window root;
        Atom atom;
        Atom atom_id;
        Damage damage;
        int damage_event, damage_error;
        int xfixes_event, xfixes_error;
        int tracked_areas_size;
        int tracked_areas_num;
        sTrackedArea tracked_areas[TRACKEDAREASNUMBER+1];
        int ux1,uy1,ux2,uy2;
        int pixel_count;
        int pending_refresh;
        int auto_disabled;
        int cmd_last_id;
        int cmd_last_lines;
} sEpdmgr;

typedef enum EnumRefreshType {
        refreshFull=0,
        refreshTyping,
        refreshAuto
} eRefreshType;

/* Global vars */
int (*default_x_error_func)(Display *dpy, XErrorEvent *err);
volatile int flag_sigint=0;
volatile int flag_sigchld=0;
char *argv0=NULL;
int signal_fd=-1;

/* Epdmgr struct manipulation */
sEpdmgr *EpdmgrInit(char *Display);
void EpdmgrFini(sEpdmgr *Epdmgr);

/* Message processing and refresh */
void ProcessMessage(sEpdmgr *Epd);
char *RefreshType2Str(eRefreshType RefreshType);
void Refresh(sEpdmgr *Epd, eRefreshType RefreshType);

/* Tracked areas processing */
void TrackedAdd(sEpdmgr *Epd, int x1, int y1, int x2, int y2);
int TrackedGetIntersection(sTrackedArea *t, sTrackedArea *o);
void TrackedMergeOne(sEpdmgr *Epd);

/* Aux funcs */
static int fatal_error(char *string, ...);
static int x_error_func(Display *dpy, XErrorEvent *err);
static int install_signal(int Num,void (*f)(int));
static void sigint(int Num);
static void sigchld(int Num);
static int ReapChildren(int Pid);

/* Program entry point */
int
main(int argc, char *argv[], char *envp[])
{
        sEpdmgr *Epd;
        char *display_name;
        int signal_fdpair[2];
        int dpy_fd, max_fd;
        fd_set read_set;
        XEvent event;
        XDamageNotifyEvent *de;
        int child_pid;
        char Dummy;
        struct timeval pre,post,timeout,pending;
        if(argc<2)
                return(fatal_error("%s: wrong number of parameters. Use --help for help.\n",argv[0]));
        if(strcmp(argv[1],"--help")==0)
                return(fatal_error("Syntax: %s [{ --version | --help }] displayname [program_to_exec [program_params [...]]]\nExample: %s :0 ./myprogram\n",argv[0],argv[0]));
        if(strcmp(argv[1],"--version")==0)
                return(fatal_error("%s: version %i.%i\n",argv[0],VERSIONMAJOR,VERSIONMINOR));
        argv0=argv[0];
        display_name=argv[1];
        if((Epd=EpdmgrInit(display_name))==NULL)
                return(1);
        pipe(signal_fdpair);
        signal_fd=signal_fdpair[FDWR];
        install_signal(SIGINT,sigint);
        Epd->pending_refresh=0;
        if(argc>2) {
                /* Start child process */
                install_signal(SIGCHLD,sigchld);
                if((child_pid=fork())==0) {
                        int fd;
#ifndef DUMMY_BUILD
                        erClientChannel_t erbusyChannel;
                        erClientChannel_t ertoolbarChannel;
                        erIpcStartClient(ER_BUSYD_CHANNEL, &erbusyChannel);
                        busySetBusy(erbusyChannel, ccBusyState_Off);
                        erIpcStartClient(ER_TOOLBAR_CHANNEL, &ertoolbarChannel);
                        tbSelectIconSet(ertoolbarChannel, ER_PDF_VIEWER_UA_ID);
                        tbClearIconSet(ertoolbarChannel, ER_PDF_VIEWER_UA_ID);
#endif
                        /* Close unneeded fds */
                        for(fd=3;fd<1024;fd++)
                                close(fd);
                        execve(argv[2],argv+2,envp);
                        return(fatal_error("%s: error launching child process\n",argv[0]));
                }
        } else
                child_pid=-1;
        while(!flag_sigint) {
                while(XPending(Epd->dpy)) {
                        XNextEvent(Epd->dpy,&event);
                        if(event.type==PropertyNotify && event.xproperty.window==Epd->root && event.xproperty.atom==Epd->atom) {
                                ProcessMessage(Epd);
                        } else if(event.type==Epd->damage_event+XDamageNotify) {
                                de=(XDamageNotifyEvent *) &event;
//                                printf("Damaged: areaxy(%i,%i) areawh(%i,%i)\n",de->area.x,de->area.y,de->area.width,de->area.height);
                                TrackedAdd(Epd,de->area.x,de->area.y,de->area.x+de->area.width-1,de->area.y+de->area.height-1);
                                Epd->pending_refresh=1;
                                /* Reset the timeout */
                                pending.tv_sec=REFRESHTIMEOUTMS/1000,pending.tv_usec=(REFRESHTIMEOUTMS%1000)*1000;
                        }
                }
                dpy_fd=ConnectionNumber(Epd->dpy);
                FD_ZERO(&read_set);
                FD_SET(dpy_fd,&read_set);
                max_fd=dpy_fd;
                FD_SET(signal_fdpair[FDRD],&read_set);
                max_fd=(signal_fdpair[FDRD]>max_fd)?signal_fdpair[FDRD]:max_fd;
                if(Epd->pending_refresh && !Epd->auto_disabled) {
                        gettimeofday(&pre,NULL);
                        timeout.tv_sec=pending.tv_sec;
                        timeout.tv_usec=pending.tv_usec;
                }
                if(select(max_fd+1,&read_set,NULL,NULL,(Epd->pending_refresh && !Epd->auto_disabled)?&timeout:(struct timeval *)NULL)<0)
                        continue;
                /* Check for signal "writes" */
                if(FD_ISSET(signal_fdpair[FDRD],&read_set))
                        read(signal_fdpair[FDRD],&Dummy,1);
                if(Epd->pending_refresh && !Epd->auto_disabled) {
                        gettimeofday(&post,NULL);
                        /* Calc. the delta */
                        if(post.tv_usec>=pre.tv_usec) {
                                timeout.tv_sec=post.tv_sec-pre.tv_sec;
                                timeout.tv_usec=post.tv_usec-pre.tv_usec;
                        } else {
                                timeout.tv_sec=post.tv_sec-pre.tv_sec-1;
                                timeout.tv_usec=(1000000L+post.tv_usec)-pre.tv_usec;
                        }
                        /* decrement the timeout */
                        pending.tv_sec-=timeout.tv_sec;
                        pending.tv_usec-=timeout.tv_usec;
                        if(pending.tv_sec>0 && pending.tv_usec<0) {
                                pending.tv_sec--;
                                pending.tv_usec+=1000000L;
                        }
                        /* Process timeout expired */
                        if(pending.tv_sec<0 || (pending.tv_sec==0 && pending.tv_usec<0))
                                Refresh(Epd,refreshAuto);
                }
                /* Check if child exited */
                if(flag_sigchld) {
                        if(ReapChildren(child_pid)!=-1) {
                                child_pid=-1;
                                ProcessMessage(Epd);
                                break;
                        }
                }
        }
        if(child_pid!=-1) {
                kill(child_pid,SIGQUIT);
                ReapChildren(child_pid);
                child_pid=-1;
        }
        EpdmgrFini(Epd);
        signal_fd=-1;
        close(signal_fdpair[FDRD]);
        close(signal_fdpair[FDWR]);
        return(0);
}

/* Epdmgr struct manipulation */
sEpdmgr *
EpdmgrInit(char *Display)
{
        sEpdmgr *Epd;
        if((Epd=malloc(sizeof(sEpdmgr)))==NULL)
                return(NULL); /* Insufficient memory */
        memset(Epd,0,sizeof(sEpdmgr));
        if((Epd->dpy=XOpenDisplay(Display))==NULL) {
                fatal_error("%s: couldn't open display %s\n",argv0,Display);
                EpdmgrFini(Epd);
                return(NULL); /* Error connecting to display */
        }
        default_x_error_func=XSetErrorHandler(x_error_func);
        if(!XDamageQueryExtension(Epd->dpy, &Epd->damage_event, &Epd->damage_error)) {
                fatal_error("%s: no damage extension\n",argv0);
                EpdmgrFini(Epd);
                return(NULL); /* Error: no XDamage extension */
        }
        if(!XFixesQueryExtension(Epd->dpy, &Epd->xfixes_event, &Epd->xfixes_error)) {
                fatal_error("%s: no damage extension\n",argv0);
                EpdmgrFini(Epd);
                return(NULL); /* Error: no XFixes extension */
        }
        Epd->root=RootWindow(Epd->dpy,DefaultScreen(Epd->dpy));
        XSelectInput (Epd->dpy, Epd->root,
                      SubstructureNotifyMask|
                      ExposureMask|
                      StructureNotifyMask|
                      PropertyChangeMask);
        // Note to self: XDamageDeltaRectangles would also be useful for scribble support
        Epd->damage = XDamageCreate (Epd->dpy, Epd->root, XDamageReportDeltaRectangles);
        Epd->atom=XInternAtom(Epd->dpy, SHARED_ATOM_NAME_FOR_COMM, False);
        XSetSelectionOwner(Epd->dpy, Epd->atom, Epd->root, CurrentTime);
        Epd->atom_id=XInternAtom(Epd->dpy, SHARED_ATOM_NAME_FOR_ID, False);
        XSetSelectionOwner(Epd->dpy, Epd->atom_id, Epd->root, CurrentTime);
        Epd->cmd_last_id=0;
        Epd->cmd_last_lines=0;
        // Clear the command atom
        XChangeProperty(Epd->dpy, Epd->root, Epd->atom, Epd->atom, 8,PropModeReplace, (unsigned char *) "", 1);
        // Set the initial id
        XChangeProperty(Epd->dpy, Epd->root, Epd->atom_id, Epd->atom_id, 8,PropModeReplace, (unsigned char *) ((Epd->cmd_last_id==0)?"1":"0"), 2);
        // Initialize the tracked areas part
        Epd->tracked_areas_size=TRACKEDAREASNUMBER+1;
        Epd->tracked_areas_num=0;
        // Return successfully
        return(Epd);
}

void
EpdmgrFini(sEpdmgr *Epd)
{
        if(Epd==NULL)
                return;
        if(Epd->dpy!=NULL) {
                XCloseDisplay(Epd->dpy);
                Epd->dpy=NULL;
        }
        free(Epd);
}


/* Message processing */
void
ProcessMessage(sEpdmgr *Epd)
{
        char *cmd,*ptr;
        Atom type;
        int format;
        int cur_id;
        int line_no;
        unsigned long size, remain;
        char *start,*end;
        if(XGetWindowProperty(Epd->dpy, Epd->root,Epd->atom,0,MAXCMDSIZE,True,Epd->atom,
             &type,&format,&size,&remain,(unsigned char **)&cmd)!=Success || cmd==NULL)
             return;
#ifdef DEBUG_MESSAGES
        {
                int i;
                printf("Received: \"");
                for(i=0;cmd[i]!='\0';i++) {
                        if(cmd[i]<' ' || cmd[i]=='\"') {
                                printf("\\x%02x",cmd[i]);
                        } else
                                printf("%c",cmd[i]);
                }
                printf("\"\n");
        }
#endif
        // check the id
        ptr=cmd;
        if((cmd[0]=='0' || cmd[0]=='1' || cmd[0]=='?') && cmd[1]=='\n') {
                cur_id=(*cmd=='?')?-1:*cmd-'0';
                ptr+=2;
        } else
                return; /* incorrect format */
        // parse cmd, do action, etc
        for(start=ptr,line_no=0;(end=strchr(start,'\n'))!=NULL;start=end+1,line_no++) {
                if(cur_id==Epd->cmd_last_id && line_no<Epd->cmd_last_lines)
                        continue; /* This line was processed earlier */
                *end='\0';
#ifdef DEBUG_MESSAGES
                printf("                                 Command: \"%s\"\n",start);
#endif
                if(strcmp(start,"Flush")==0) {
                        Refresh(Epd,refreshAuto);
                } else if(strcmp(start,"RefreshFull")==0) {
                        Refresh(Epd,refreshFull);
                } else if(strcmp(start,"RefreshTyping")==0) {
                        Refresh(Epd,refreshTyping);
                } else if(strcmp(start,"Auto(1)")==0) {
                        Epd->auto_disabled=0;
                } else if(strcmp(start,"Auto(0)")==0) {
                        Epd->auto_disabled=1;
                } else {
                        ; /* Ignore the message */
                }
        }
        // Save the just-processed id and line count
        Epd->cmd_last_id=(cur_id!=-1)?cur_id:0;
        Epd->cmd_last_lines=line_no;
        // Set the next id
        XChangeProperty(Epd->dpy, Epd->root, Epd->atom_id, Epd->atom_id, 8,PropModeReplace, (unsigned char *) ((Epd->cmd_last_id==0)?"1":"0"), 2);
        XFree((XPointer)cmd);
}

/* Screen refresh */
char *
RefreshType2Str(eRefreshType RefreshType)
{
        return((RefreshType==refreshAuto)?"Auto":
               (RefreshType==refreshFull)?"Full":
               (RefreshType==refreshTyping)?"Typing":"Unknown");
}
void
Refresh(sEpdmgr *Epd, eRefreshType RefreshType)
{
        XserverRegion parts;
        eRefreshType final_type;
        if(RefreshType==refreshAuto) {
                if(Epd->pixel_count>=PIXELTHRESHOLD)
                        final_type=refreshFull;
                else
                        final_type=refreshTyping;
        } else
                final_type=RefreshType;
        {
                int i;
                sTrackedArea *t;
                for(i=0,t=Epd->tracked_areas+0;i<Epd->tracked_areas_num;i++,t++) {
                        printf("Area %2i: (%i,%i)-(%i,%i) Area:%i\n",i,
                                                               t->x1,t->y1,t->x2,t->y2,t->pixel_count);

                }
        }
        printf("Refresh: %s/%s (%i,%i)-(%i,%i) Area:%i Changed:%i\n",RefreshType2Str(RefreshType),
                                               RefreshType2Str(final_type),
                                               Epd->ux1,Epd->uy1,Epd->ux2,Epd->uy2,(Epd->ux2-Epd->ux1+1)*(Epd->uy2-Epd->uy1+1),Epd->pixel_count);
#ifndef DUMMY_BUILD
        dmDisplay(dmCmdPriorUrgent,(final_type==refreshFull)?dmQFull:dmQTyping);
#endif
        parts=XFixesCreateRegion(Epd->dpy, 0, 0);
        XDamageSubtract(Epd->dpy, Epd->damage, None, parts);
        Epd->pending_refresh=0;
        Epd->ux1=Epd->uy1=Epd->ux2=Epd->uy2=0;
        Epd->pixel_count=0;
        Epd->tracked_areas_num=0;
}

/* Tracked areas processing */
void
TrackedAdd(sEpdmgr *Epd, int x1, int y1, int x2, int y2)
{
        int i,j;
        sTrackedArea *t;
        /* Add the new element */
        t=Epd->tracked_areas+Epd->tracked_areas_num;
        memset(t,0,sizeof(sTrackedArea));
        t->x1=x1,t->y1=y1,t->x2=x2,t->y2=y2;
        t->pixel_count=(x2-x1+1)*(y2-y1+1);
        Epd->tracked_areas_num++;
        /* Merge two areas if we're at the limit */
        if(Epd->tracked_areas_num==Epd->tracked_areas_size)
                TrackedMergeOne(Epd);
        /* Calculate the unified area */
        t=Epd->tracked_areas+0;
        Epd->ux1=t->x1,Epd->uy1=t->y1;
        Epd->ux2=t->x2,Epd->uy2=t->y2;
        Epd->pixel_count=t->pixel_count;
        for(i=1,t++;i<Epd->tracked_areas_num;i++,t++) {
                Epd->ux1=(t->x1<Epd->ux1)?t->x1:Epd->ux1;
                Epd->uy1=(t->y1<Epd->uy1)?t->y1:Epd->uy1;
                Epd->ux2=(t->x2>Epd->ux2)?t->x2:Epd->ux2;
                Epd->uy2=(t->y2>Epd->uy2)?t->y2:Epd->uy2;
                Epd->pixel_count+=t->pixel_count;
                for(j=0;j<i;j++)
                        Epd->pixel_count-=TrackedGetIntersection(Epd->tracked_areas+j,t);
        }
}

int
TrackedGetIntersection(sTrackedArea *t, sTrackedArea *o)
{
        int cy1,cy2;
        int count;
        if(t->x2<o->x1 || o->x2<t->x1 ||
           t->y2<o->y1 || o->y2<t->y1)
                return(0); /* No collision */
        /* Collision */
        /* First we sort t and o so t is to the left of o */
        /* Then, segment1: t */
        /*       segment2: t intersects o */
        /*       segment3: t (if t larger than o) */
        /*                 o (if o larger than t) */
        if(o->x1<t->x1) {
                sTrackedArea *tmp=t;
                t=o;
                o=tmp;
        }
        if(o->x2>t->x2) { /* o larger than t */
                /*
                 *  +-----+
                 *  | t   |
                 *  |   +-----+
                 *  +---|   o |
                 *      +-----+
                 */
                cy1=(t->y1<o->y1)?t->y1:o->y1;
                cy2=(t->y2>o->y2)?t->y2:o->y2;
                count=(t->x2-o->x1+1)*(cy2-cy1+1);
        } else { /* t larger than o */
                /*
                 *  +--------------+
                 *  | t            |
                 *  |   +-----+    |
                 *  +---|   o |----+
                 *      +-----+
                 */
                cy1=(t->y1<o->y1)?t->y1:o->y1;
                cy2=(t->y2>o->y2)?t->y2:o->y2;
                count=(o->x2-o->x1+1)*(cy2-cy1+1);
        }
        return(count);
}

void
TrackedMergeOne(sEpdmgr *Epd)
{
        int i,k;
        sTrackedArea *t,*o;
        int x1,x2,y1,y2,count,overhead; /* current merge*/
        /* First pass, calculate the overheads */
        for(i=0;i<Epd->tracked_areas_num;i++) {
                t=Epd->tracked_areas+i;
                for(k=i+1;k<Epd->tracked_areas_num;k++) {
                        o=Epd->tracked_areas+k;
                        x1=(t->x1<o->x1)?t->x1:o->x1;
                        y1=(t->y1<o->y1)?t->y1:o->y1;
                        x2=(t->x2>o->x2)?t->x2:o->x2;
                        y2=(t->y2>o->y2)?t->y2:o->y2;
                        count=(x2-x1+1)*(y2-y1+1);
                        overhead=count-(t->pixel_count+o->pixel_count-TrackedGetIntersection(t,o));
                        if((k==i+1) || overhead<t->merge_overhead) {
                                t->merge_with=k;
                                t->merge_overhead=overhead;
                        }
                }
        }
        /* Second pass, select the one with least overhead */
        k=0;
        for(i=1;i<(Epd->tracked_areas_num-1);i++) {
                if(Epd->tracked_areas[i].merge_overhead<Epd->tracked_areas[k].merge_overhead)
                        k=i;
        }
        /* Do the merge */
        t=Epd->tracked_areas+k;
        o=Epd->tracked_areas+t->merge_with;
        t->x1=(t->x1<o->x1)?t->x1:o->x1;
        t->y1=(t->y1<o->y1)?t->y1:o->y1;
        t->x2=(t->x2>o->x2)?t->x2:o->x2;
        t->y2=(t->y2>o->y2)?t->y2:o->y2;
        t->pixel_count=(t->x2-t->x1+1)*(t->y2-t->y1+1);
        Epd->tracked_areas_num--;
        if(t->merge_with!=Epd->tracked_areas_num)
                memmove(Epd->tracked_areas+t->merge_with,Epd->tracked_areas+Epd->tracked_areas_num,sizeof(sTrackedArea));
}


/* Aux funcs implementations*/
static int
fatal_error(char *string, ...)
{
        va_list arg_list;
        va_start(arg_list,string);
        vfprintf(stderr,string,arg_list);
        va_end(arg_list);
        fflush(stderr);
        return(1);
}


static int
x_error_func(Display *dpy, XErrorEvent *err)
{
        if(err->error_code==BadWindow) {
                return(0);
        } else if(err->error_code==BadDrawable) {
                return(0);
        }
        return(default_x_error_func(dpy,err));
}

static int
install_signal(int Num,void (*f)(int))
{
        struct sigaction sact;
        sact.sa_handler=f;
        sigemptyset(&sact.sa_mask);
        sact.sa_flags=0;
        return(sigaction(Num,&sact,0));
}

static void
sigint(int Num)
{
        flag_sigint++;
        if(signal_fd!=-1)
                write(signal_fd,".",1);
}

static void
sigchld(int Num)
{
        flag_sigchld++;
        if(signal_fd!=-1)
                write(signal_fd,".",1);
}

static
int ReapChildren(int Pid)
{
        int reaped_pid,status;
        while(1) {
                reaped_pid=waitpid(-1,&status,WNOHANG|WUNTRACED);
                if(reaped_pid<=0)
                        break;
                if(reaped_pid==Pid)
                        return(0);
        }
        return(-1);
}

