//C- -------------------------------------------------------------------
//C- Java DjVu (r) (v. 0.8)
//C- Copyright (c) 2004-2005 LizardTech, Inc.  All Rights Reserved.
//C- Java DjVu is protected by U.S. Pat. No.C- 6,058,214 and patents
//C- pending.
//C-
//C- This software is subject to, and may be distributed under, the
//C- GNU General Public License, Version 2. The license should have
//C- accompanied the software or you may obtain a copy of the license
//C- from the Free Software Foundation at http://www.fsf.org .
//C-
//C- The computer code originally released by LizardTech under this
//C- license and unmodified by other parties is deemed "the LIZARDTECH
//C- ORIGINAL CODE."  Subject to any third party intellectual property
//C- claims, LizardTech grants recipient a worldwide, royalty-free,
//C- non-exclusive license to make, use, sell, or otherwise dispose of
//C- the LIZARDTECH ORIGINAL CODE or of programs derived from the
//C- LIZARDTECH ORIGINAL CODE in compliance with the terms of the GNU
//C- General Public License.   This grant only confers the right to
//C- infringe patent claims underlying the LIZARDTECH ORIGINAL CODE to
//C- the extent such infringement is reasonably necessary to enable
//C- recipient to make, have made, practice, sell, or otherwise dispose
//C- of the LIZARDTECH ORIGINAL CODE (or portions thereof) and not to
//C- any greater extent that may be necessary to utilize further
//C- modifications or combinations.
//C-
//C- The LIZARDTECH ORIGINAL CODE is provided "AS IS" WITHOUT WARRANTY
//C- OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
//C- TO ANY WARRANTY OF NON-INFRINGEMENT, OR ANY IMPLIED WARRANTY OF
//C- MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//C-
//C- In addition, as a special exception, LizardTech Inc. gives permission
//C- to link the code of this program with the proprietary Java
//C- implementation provided by Sun (or other vendors as well), and
//C- distribute linked combinations including the two. You must obey the
//C- GNU General Public License in all respects for all of the code used
//C- other than the proprietary Java implementation. If you modify this
//C- file, you may extend this exception to your version of the file, but
//C- you are not obligated to do so. If you do not wish to do so, delete
//C- this exception statement from your version.
//C- -------------------------------------------------------------------
//C- Developed by Bill C. Riemers, Foxtrot Technologies Inc. as work for
//C- hire under US copyright laws.
//C- -------------------------------------------------------------------
//
package com.lizardtech.djvu;

import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.WeakHashMap;


/**
 * This class implements a random access mapping of input streams and URL 
 * connections.  For input streams and non-http connections, the mapping 
 * is achieved by reading in all the data into memory.  Http streams are
 * mapped by reading in data blocks when needed.
 *
 * @author Bill C. Riemers
 * @version $Revision: 1.13 $
 */
public class DataPool
  extends DjVuObject
{
  //~ Static fields/initializers ---------------------------------------------

  /** The default size of each block. */
  public static final int BLOCKSIZE = 8192;
    
  /** Object for caching raw data. ! */
  public static WeakHashMap cache = new WeakHashMap();

  //~ Instance fields --------------------------------------------------------

  // The end of the stream, or a number larger than the end of the stream.
  private int endOffset=Integer.MAX_VALUE;
  
  // The url we are reading.
  private URL url=null;
      
  // The largest end offset of read data.
  private int currentSize=0;
  
  //~ Constructors -----------------------------------------------------------

  /**
   * Creates a new DataPool object.
   */
  public DataPool()
  {
  }

  //~ Methods ----------------------------------------------------------------

  /**
   * Creates an instance of DataPool with the options interherited from the
   * specified reference.
   * 
   * @param ref Object to interherit DjVuOptions from.
   * 
   * @return a new instance of DataPool.
   */
  public static DataPool createDataPool(final DjVuInterface ref)
  {
    final DjVuOptions options = ref.getDjVuOptions();

    return (DataPool)create(
      options,
      options.getDataPoolClass(),
      DataPool.class);
  }

  /**
   * Initialize this map to read the specified URL. If a cached DataPool for this 
   * URL exists, it will be returned.
   * 
   * @param url the URL to read
   * 
   * @return an initialized DataPool
   */
  public DataPool init(final URL url)
  {
    this.url=url;
    DataPool retval=this;
    if(url != null)
    {
      if (cache.containsKey(url)) retval=(DataPool)cache.get(url);
      else cache.put(url,this);
    }
    return retval;
  }
  
  /**
   * Initialize this map to read the specified stream
   * 
   * @param input the InputStream to read
   * 
   * @return the initialized DataPool
   */
  public DataPool init(final InputStream input)
  {
    return this;
  }
  
  /** 
   * Query the largest read end offset.
   *
   * @return the largest read end offset
   */
  public int getCurrentSize()
  {
    return currentSize;
  }
  
  /**
   * Request the specified block of data. Data may be buffered, or read.
   *
   * @param index the position of the block start position divided by BLOCKSIZE.
   * @param read True if unavailable blocks should be read from the data source.
   *
   * @return a byte array up to size BLOCKSIZE, or null if no data is available.
   */
  public byte [] getBlock(final int index, final boolean read)
  {
	byte[] retval=null; 
	int start=index*BLOCKSIZE;

	if ((index >= 0) && (start < endOffset)) {
	  if (currentSize < start) currentSize=start;
		  
	  System.err.println("Reading block " + Integer.toString(index));

	  retval=new byte[BLOCKSIZE];
	  try {
		FileInputStream input = new FileInputStream(url.getFile());
        input.skip(start);
        int nrread = input.read(retval, 0, BLOCKSIZE);
        input.close();
        if (nrread != BLOCKSIZE) endOffset = start+nrread;
	  } catch (final Throwable exp) {
	    printStackTrace(exp);
	  }
	}
	
    return retval;
  }
    
  /**
   * Query the size of this vector.
   *
   * @return the size of this vector
   */
  public int getEndOffset()
  {
    return endOffset;
  }

}
