#!/usr/bin/env python2
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__docformat__ = 'restructuredtext en'

import os

from PyQt5.Qt                                 import (QProgressDialog, QTimer)
from io                                       import StringIO
from lxml                                     import etree
from calibre_plugins.extract_tocs.unidecode   import unidecode
from calibre.ebooks.oeb.polish.container      import get_container
from calibre.utils.titlecase                  import titlecase

NCX_NS = 'http://www.daisy.org/z3986/2005/ncx/'

class ProgressDialog(QProgressDialog):

    def __init__(self, gui, book_ids):
        self.total_books = len(book_ids)
        QProgressDialog.__init__(self, '', 'Cancel', 0, self.total_books, gui)
        self.gui        = gui
        self.book_ids   = book_ids
        self.db         = self.gui.current_db.new_api
        self.i          = 0
        self.toc_items  = []
        self.setWindowTitle('Extracting ToCs')
        icon = get_icons('images/icon.png')
        self.setWindowIcon(icon)
        self.resize(self.width()*2, self.height())
        QTimer.singleShot(0, self.do_book_action)
        self.exec_()

    def do_book_action(self):
        if self.wasCanceled():
            return self.do_close()
        if self.i >= self.total_books:
            return self.do_close()
        
        book_id = self.book_ids[self.i]
        authors = unidecode(u' & '.join(self.db.field_for('authors', book_id)))
        title   = unidecode(self.db.field_for('title', book_id))
        series  = ''
        index   = ''
        if self.db.field_for('series', book_id):
            series = unidecode(self.db.field_for('series', book_id))
            index  = '%.2f' % self.db.field_for('series_index', book_id)
            
        authors = titlecase(authors.lower()).replace('"', '""')
        title   = titlecase(title.lower()).replace('"', '""')
        series  = titlecase(series.lower()).replace('"', '""')

        epub      = self.db.format(book_id, 'EPUB', as_path=True)
        container = get_container(epub, tweak_mode=True)

        for dir, subdirs, files in os.walk(container.root):
            for f in files:
                if f.endswith('.ncx'):
                  tree    = etree.parse(os.path.join(dir,f))
                  entries = tree.xpath('//ncx:navLabel/ncx:text/text()', namespaces={'ncx':NCX_NS})
                  for entry in entries:
                      str = unidecode(u''+entry).replace('"', '""')
                      self.toc_items.append( (titlecase(str.lower()),
                                              authors, title, series, index) )
        
        self.i += 1
        self.setLabelText(_('Book %d of %d') % (self.i, self.total_books))
        self.setValue(self.i)
        QTimer.singleShot(0, self.do_book_action)

    def do_close(self):
        self.hide()
        self.gui = None
