# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import, print_function)
__license__   = 'GPL v3'
__copyright__ = '2016, DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.39"  #export raw data

import os,sys,apsw
from PyQt5.Qt import (Qt, QDialog, QGridLayout, QVBoxLayout, QHBoxLayout, QFont,
                                        QWidget, QSize, QPushButton, QComboBox, QGroupBox, QSplitter )  # for visualizing
from PyQt5.Qt import QFileDialog,QObject,QApplication  # for exporting a .csv

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.gui2 import gprefs, question_dialog, info_dialog

from calibre_plugins.job_spy.qpainter_charts import Chart, PieChart, ViewerLegend, ViewerPieChart, DataTable
from calibre_plugins.job_spy.config import prefs

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(600, 600)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def save_dialog_geometry(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class VisualizeMetadataDialog(SizePersistedDialog):

    def __init__(self,gui,guidb,icon):

        parent = gui
        unique_pref_name = 'job_spy:visualize_metadata_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.gui = gui
        self.guidb = guidb

        self.setWindowTitle('JS+ GUI Tool: Visualize Metadata')
        self.setWindowIcon(icon)

        self.layout_main = QVBoxLayout()
        self.setLayout(self.layout_main)

        self.setToolTip("<p style='white-space:wrap'>Create a pie chart representing the proportion of books having a particular metadata value.")

        self.vm_groupbox = QGroupBox('Visualize Top 50 Metadata Values:')
        self.vm_groupbox.setAlignment(Qt.AlignCenter)
        self.vm_groupbox.setMaximumWidth(400)
        self.vm_groupbox.setMaximumHeight(150)
        self.vm_groupbox.setToolTip("<p style='white-space:wrap'>Create a pie chart representing the proportion of books having a particular metadata value.")
        self.layout_main.addWidget(self.vm_groupbox)

        self.vm_gb_layout = QVBoxLayout()
        self.vm_groupbox.setLayout(self.vm_gb_layout)

        font = QFont()
        font.setBold(False)
        font.setPointSize(10)

        self.custom_columns_combobox = QComboBox()
        self.custom_columns_combobox.setEditable(False)
        self.custom_columns_combobox.setFont(font)
        self.custom_columns_combobox.setToolTip("<p style='white-space:wrap'>Select the Custom Column or Standard Column for which you would like to view a pie chart.")
        self.vm_gb_layout.addWidget(self.custom_columns_combobox)

        self.custom_columns_combobox.setObjectName("visualize_metadata_combobox")

        self.custom_columns_combobox.addItem('authors')
        self.custom_columns_combobox.addItem('publishers')
        self.custom_columns_combobox.addItem('series')
        self.custom_columns_combobox.addItem('tags')
        self.custom_columns_combobox.addItem('pubdate')
        self.custom_columns_combobox.addItem('last_modified')
        self.custom_columns_combobox.addItem('timestamp')

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            error_dialog(self.gui, _('JS+'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)
            return

        self.custom_column_list = self.get_table_custom_columns(my_db,my_cursor)
        for label in self.custom_column_list:
            self.custom_columns_combobox.addItem(label)
        #END FOR

        self.custom_columns_combobox.setMaxVisibleItems(len(self.custom_column_list))

        self.push_button_visualize_metadata = QPushButton("Visualize Selected Metadata", self)
        self.push_button_visualize_metadata.clicked.connect(self.visualize_metadata)
        self.push_button_visualize_metadata.setFont(font)
        self.push_button_visualize_metadata.setDefault(False)
        self.push_button_visualize_metadata.setToolTip("<p style='white-space:wrap'>Create a pie chart representing the proportion of books having a particular metadata value. ")
        self.vm_gb_layout.addWidget(self.push_button_visualize_metadata)

        self.push_button_export_metadata = QPushButton("Export Visualized Metadata", self)
        self.push_button_export_metadata.clicked.connect(self.export_metadata)
        self.push_button_export_metadata.setFont(font)
        self.push_button_export_metadata.setDefault(False)
        self.push_button_export_metadata.setToolTip("<p style='white-space:wrap'>Export (to a .csv file) the raw data that was used to Visualize the chosen Metadata.")
        self.vm_gb_layout.addWidget(self.push_button_export_metadata)

        self.raw_data = None
        self.export_column_name = None

        #-----------------------------------------------------
        self.push_button_cancel = QPushButton("Exit")
        self.push_button_cancel.clicked.connect(self.save_and_exit)
        self.push_button_cancel.setDefault(True)
        self.push_button_cancel.setToolTip("<p style='white-space:wrap'>Save the current size of this dialog, and then exit.")
        self.vm_gb_layout.addWidget(self.push_button_cancel)
        #------------------------------------------------

        font.setPointSize(6)

        self.chart_splitter = QSplitter()
        self.layout_main.addWidget(self.chart_splitter)

        self.viewer_left = ViewerPieChart()
        self.chart_splitter.addWidget(self.viewer_left)

        self.viewer_right = ViewerLegend()
        self.chart_splitter.addWidget(self.viewer_right)

        #-----------------------------------------------------
        self.resize(self.sizeHint())
        self.resize_dialog()

    #----------------------------------------------------
    #----------------------------------------------------
    #----------------------------------------------------
    #----------------------------------------------------
    #----------------------------------------------------
    def visualize_metadata(self):
        self.save_dialog_geometry()
        selected_column = self.custom_columns_combobox.currentText()
        self.draw_pie_chart(selected_column)
    #----------------------------------------------------
    #----------------------------------------------------
    def draw_pie_chart(self,selected_column):

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            return
       #-----------------------------
        if selected_column.startswith("#"):
            label = selected_column.replace("#","").strip()
            self.export_column_name = label
            tmp_rows = self.get_custom_column_data(my_db,my_cursor,label)
        else:
            tmp_rows = self.get_standard_data(my_db,my_cursor,selected_column)
            self.export_column_name = selected_column
       #-----------------------------
        if len(tmp_rows) == 0:
            info_dialog(self.gui, 'No Data to Visualize','Nothing to show...').show()
            return
       #-----------------------------

        table = DataTable()
        table.addColumn('Legend')
        table.addColumn('#')
        i = 0
        n_rest = 0
        for row in tmp_rows:
            n_count,mydata = row
            if not isinstance(mydata,unicode):
                mydata = str(mydata)
            s = str(n_count)
            n = len(s)
            if n >= 4:
                pass
            elif n == 3:
                s = " " + s
            elif n == 2:
                s = "   " + s
            elif n == 1:
                s = "    " + s
            mydata = s + ": " + mydata
            if i < 50:
                table.addRow([mydata,n_count])
                i = i + 1
            else:
                n_rest = n_rest + n_count
        #END FOR
        if n_rest > 0:
            s = str(n_rest)
            n = len(s)
            if n >= 4:
                pass
            elif n == 3:
                s = " " + s
            elif n == 2:
                s = "   " + s
            elif n == 1:
                s = "    " + s
            mydata =   s + ": " + "All Other Values"
            table.addRow([mydata,n_rest])

        self.raw_data = tmp_rows   #for exporting to a csv if needed...
        del tmp_rows
        #-----------------------------

        chart = PieChart(table)

        self.viewer_left.setGraph(chart)

        self.viewer_left.resize(480,480)       #width,height

        legend = PieChart(table)

        self.viewer_right.setGraph(legend)

        self.viewer_right.resize(480,960)       #width,height

        self.resize(self.sizeHint())
        self.resize_dialog()

        try:
            del table
            del chart
        except:
            pass

        self.show()

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def get_custom_column_data(self,my_db,my_cursor,label):

        mysql = "SELECT id,label,datatype,normalized FROM custom_columns WHERE label = ?"
        my_cursor.execute(mysql,([label]))
        tmp_rows= my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        if len(tmp_rows) == 0:
            my_db.close()
            #~ if DEBUG: print(label + "  has not been configured as a proper custom column in this library")
            return tmp_rows
        for row in tmp_rows:
            id,label,datatype,normalized = row
        #END FOR
        del tmp_rows
        s_id = str(id)
        s_id = s_id.strip()
        #-----------------------------
        if normalized == 1:
            mysql = "SELECT count(book) AS number,(SELECT value FROM custom_column_[N] WHERE id = books_custom_column_[N]_link.value) AS mydata \
                            FROM books_custom_column_[N]_link  GROUP BY mydata ORDER BY number DESC, mydata ASC "
        else:
            if datatype == "datetime":
                mysql = "SELECT count(book) AS number,(substr(value,0,8)) AS value FROM custom_column_[N] GROUP BY value ORDER BY number DESC"
            else:
                mysql = "SELECT count(book) AS number,value FROM custom_column_[N] GROUP BY value ORDER BY number DESC"

        mysql = mysql.replace("[N]",s_id)

        my_cursor.execute(mysql)
        tmp_rows= my_cursor.fetchall()
        my_db.close()
        if not tmp_rows:
            tmp_rows = []
        return tmp_rows                  #  n_count,mydata = row

    #----------------------------------------------------
    #----------------------------------------------------
    def get_standard_data(self,my_db,my_cursor,selected_column):

        if selected_column == "authors":
            mysql1 = "SELECT id,name FROM authors"
            mysql2 = "SELECT count(book) AS number,author FROM books_authors_link GROUP BY author ORDER BY number DESC"
        elif selected_column == "publishers":
            mysql1 = "SELECT id,name FROM publishers"
            mysql2 = "SELECT count(book) AS number,publisher FROM books_publishers_link GROUP BY publisher ORDER BY number DESC"
        elif selected_column == "series":
            mysql1 = "SELECT id,name FROM series"
            mysql2 = "SELECT count(book) AS number,series FROM books_series_link GROUP BY series ORDER BY number DESC"
        elif selected_column == "tags":
            mysql1 = "SELECT id,name FROM tags"
            mysql2 = "SELECT count(book) AS number,tag FROM books_tags_link GROUP BY tag ORDER BY number DESC"
        elif selected_column == "pubdate":
            mysql1 = None
            mysql2 = "SELECT count(id) AS number,(substr(pubdate,0,8)) AS date FROM books GROUP BY date ORDER BY number DESC"
        elif selected_column == "last_modified":
            mysql1 = None
            mysql2 = "SELECT count(id) AS number,(substr(last_modified,0,8)) AS date FROM books GROUP BY date ORDER BY number DESC"
        elif selected_column == "timestamp":
            mysql1 = None
            mysql2 = "SELECT count(id) AS number,(substr(timestamp,0,8)) AS date FROM books GROUP BY date ORDER BY number DESC"
        else:
            tmp_rows = []
            return tmp_rows

        #~ -------------------------------------------
        if mysql1:
            legend_name_dict = {}

            my_cursor.execute(mysql1)
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                my_db.close()
                tmp_rows = []
                return tmp_rows
            for row in tmp_rows:
                id,name = row
                legend_name_dict[id] = name
            #END FOR
            del tmp_rows

            my_cursor.execute(mysql2)
            tmp_rows2 = my_cursor.fetchall()
            my_db.close()
            if not tmp_rows2:
                tmp_rows = []
                return tmp_rows
            tmp_rows = []
            for row in tmp_rows2:
                number,linkid = row
                name = legend_name_dict[linkid]
                item = number,name      #  n_count,mydata = item
                tmp_rows.append(item)
            #END FOR
            del tmp_rows2
            del legend_name_dict
        #~ -------------------------------------------
        else:
            my_cursor.execute(mysql2)
            tmp_rows2 = my_cursor.fetchall()
            my_db.close()
            if not tmp_rows2:
                tmp_rows = []
                return tmp_rows
            tmp_rows = []
            for row in tmp_rows2:
                number,date = row
                item = number,date      #  n_count,mydata = item
                tmp_rows.append(item)
            #END FOR
            del tmp_rows2
        #~ -------------------------------------------
        return tmp_rows

    #----------------------------------------------------
    #----------------------------------------------------
    def get_table_custom_columns(self,my_db,my_cursor):
        self.custom_column_list = []

        mysql = "SELECT label,datatype FROM custom_columns WHERE datatype <> 'composite' AND datatype <> 'comments' ORDER BY label"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        my_db.close()
        for row in tmp_rows:
            label,datatype = row
            label = "#" + label
            self.custom_column_list.append(label)
        del tmp_rows

        return self.custom_column_list
    #----------------------------------------------------
    def apsw_connect_to_library(self):

        my_db = self.gui.library_view.model().db

        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", str(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid

    #----------------------------------------------------
    def save_and_exit(self):
        self.save_dialog_geometry()
        self.close()
    #----------------------------------------------------
    def export_metadata(self):

        if not self.raw_data:
            return

        if not self.export_column_name:
            return

        #~ if DEBUG:
            #~ for row in self.raw_data:
                #~ print(str(row))

        title = "Choose the Directory for the exported .csv file' "

        default_user_csv_directory = prefs['DEFAULT_VISUALIZE_METADATA_EXPORT_DIRECTORY']

        chosen_directory_name = QFileDialog.getExistingDirectory(None,title,default_user_csv_directory,QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks )

        if not chosen_directory_name:
            return

        prefs['DEFAULT_VISUALIZE_METADATA_EXPORT_DIRECTORY'] = chosen_directory_name
        prefs

        outfilename = str("js_exported_") + str(self.export_column_name) + str(".txt")

        export_csv_file_full_path = os.path.join(chosen_directory_name,outfilename)
        export_csv_file_full_path = export_csv_file_full_path.replace(os.sep, '/')
        export_csv_file_full_path = export_csv_file_full_path.decode(filesystem_encoding)

        import csv

        if os.path.exists(chosen_directory_name):
            try:
                with open(export_csv_file_full_path, 'wb') as outfile:
                    outcsv = csv.writer(outfile, delimiter=str(','), quotechar=str('"'))
                    for row in self.raw_data:
                        count,value = row
                        count = str(count)
                        if not isinstance(value,unicode):
                            value = unicode(value)    #e.g. int
                        new = count,value
                        if DEBUG: print(new)
                        new = [s.encode('utf-8') for s in new]
                        outcsv.writerow(new)
                outfile.close()
                msg = 'The export of the raw metadata with counts was successful.<br><br>File path:'+ export_csv_file_full_path + "<br><br>Number of rows exported: " + str(len(self.raw_data))
                info_dialog(self.gui, 'Export Success',msg).show()
            except Exception as e:
                if DEBUG: print("export failure: ", str(e))
                msg = 'The export of the raw metadata failed due to: ' + str(e)
                info_dialog(self.gui, 'Export Failure',msg).show()

            del outfile
            del outcsv

        del csv

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
#END OF visualize_metadata_dialog.py