# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import, print_function)
__license__   = 'GPL v3'
__copyright__ = '2018 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.126"     #New GUI Tool: Identifier Type Removal

import os,sys,apsw
from functools import partial

from PyQt5.Qt import (Qt, QDialog, QLabel,  QFont, QWidget, QApplication, QCheckBox,
                                       QIcon, QGridLayout, QGroupBox, QMargins, QScrollArea,
                                       QTableWidget, QTableWidgetItem, QDialogButtonBox,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout)

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.db.cache import Cache as dbcache
from calibre.gui2 import gprefs, error_dialog, question_dialog, info_dialog

LIBRARY_CODES_PLUGIN_IDENTIFIERS_LIST = ['isbn','issn','isni','lc_authority_name','lccn','loc_lccn','oclc','oclc-owi','viaf_author_id']

ZOTERO_METADATA_IMPORTER_PLUGIN_LIST = ['zkey','zkey_file','zcollection']

MEDIA_FILE_IMPORTER_PLUGIN_LIST = ['zmfi','zmfi_origpath']


#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(200, 200)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self):

        #~ if DEBUG: self.geom = None

        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class RemoveIdTypesDialog(SizePersistedDialog):

    def __init__(self, parent,maingui,restart_dialog):
        unique_pref_name = 'Job_Spy:remove_id_types_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.hide()

        self.maingui = maingui

        self.guidb = self.maingui.library_view.model().db

        self.restart_dialog = restart_dialog

        self.identifier_types_list = []

        self.get_all_identifier_types()

        if len(self.identifier_types_list) == 0:
            error_dialog(self.maingui, _('JS+ GUI Tool'),_('No Identifiers Exist in this Calibre Library.'), show=True)
            self.close()

        mytitle = 'JS+ GUI Tool:  Remove Identifier Types'
        self.setWindowTitle(_(mytitle))

        self.setWindowFlags( Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint | Qt.WindowStaysOnTopHint)              # http://doc.qt.io/qt-5/qt.html#WindowType-enum

        t = "<p style='white-space:wrap'>This Tool is used to delete selected Identifier Types for selected Books.\
                                                             <br><br>The 'Usage', or 'Source', of the displayed Identifier Types in the current Library is indicated if known.\
                                                             <br><br><b><i>Delete Identifier Types at your own risk.<br><br>You have been warned.</i></b>\
                                                             <br><br>After removal, any composite (built from other/virtual) Custom Columns will continue to show their previous Identifier Type Value until Calibre itself decides to refresh its data.  Ditto for the Book Details."

        self.setToolTip(t)

        #--------------------------------------------------
        layout = QVBoxLayout(self)
        self.setLayout(layout)
        #--------------------------------------------------
        column_label_list = []

        column_label_list = []
        column_label_list.append("Type")
        column_label_list.append("Usage")
        column_label_list.append("?")

        self.n_total_cols = 3
        #--------------------------------------------------
        self.n_log_rows = len(self.identifier_types_list)

        self.matrix = QTableWidget(self.n_log_rows,self.n_total_cols)

        self.matrix.setSortingEnabled(False)

        self.matrix.setHorizontalHeaderLabels(column_label_list)


        self.matrix.setColumnWidth(0, 200)
        self.matrix.setColumnWidth(1, 400)
        self.matrix.setColumnWidth(2, 1)

        self.matrix.clearContents()
        #--------------------------------------------------
        #--------------------------------------------------
        myflags = Qt.ItemFlags()
        myflags != Qt.ItemIsEnabled  # not enabled; read-only
        #--------------------------------------------------
        #--------------------------------------------------
        r = 0
        try:
            for type in self.identifier_types_list:

                type_ = QTableWidgetItem(type)
                type_.setFlags(myflags)
                self.matrix.setItem(r,0,type_)

                usage = self.get_usage(type)
                usage_ = QTableWidgetItem(usage)
                usage_.setFlags(myflags)
                self.matrix.setItem(r,1,usage_)

                checkbox_ = QTableWidgetItem()
                checkbox_.setFlags(Qt.NoItemFlags)
                checkbox_.setFlags(Qt.ItemIsEnabled | Qt.ItemIsUserCheckable)
                checkbox_.setCheckState(False)
                checkbox_.setSelected(False)
                self.matrix.setItem(r,2,checkbox_)
                #--------------------------------------
                r = r + 1
                #--------------------------------------
            #END FOR
        except Exception as e:
            if DEBUG: print("class RemoveIdTypesDialog(SizePersistedDialog):", str(e))
            return

        self.n_total_rows = r

        layout.addWidget(self.matrix)

        self.matrix.sortItems(0,Qt.AscendingOrder )

        self.matrix.setSortingEnabled(True)

        self.resize_all_columns()

        #-----------------------------------------------------

        self.bottom_buttonbox = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.bottom_buttonbox.rejected.connect(self.reject)
        layout.addWidget(self.bottom_buttonbox)

        self.push_button_remove_selected_identifier_types = QPushButton(" ", self)
        self.push_button_remove_selected_identifier_types.setText("Remove Selected Types")
        self.push_button_remove_selected_identifier_types.setToolTip("<p style='white-space:wrap'>Remove the above <b>'checked'</b> Identifier Types for the Selected Books.")
        self.push_button_remove_selected_identifier_types.setDefault(False)
        self.push_button_remove_selected_identifier_types.clicked.connect(self.remove_selected_identifier_types)
        self.bottom_buttonbox.addButton(self.push_button_remove_selected_identifier_types,0)

        self.push_button_copy_to_clipboard = QPushButton(" ", self)
        self.push_button_copy_to_clipboard.setText("Export to Clipboard")
        self.push_button_copy_to_clipboard.setToolTip("<p style='white-space:wrap'>The Matrix in its entirety will be copied to the Clipboard in a tab-delimited format.  Paste Special into a Spreadsheet, or Paste into a Text Document.")
        self.push_button_copy_to_clipboard.setDefault(True)
        self.push_button_copy_to_clipboard.clicked.connect(self.copy_log_to_clipboard)
        self.bottom_buttonbox.addButton(self.push_button_copy_to_clipboard,0)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.clip = QApplication.clipboard()

        self.optimize_column_widths()

        self.repaint()

        self.show()
    #-----------------------------------------------------
    def get_usage(self,type):
        type = type.lower()
        if type in LIBRARY_CODES_PLUGIN_IDENTIFIERS_LIST:
            usage = "Library Codes Plug-in  "
        elif type in ZOTERO_METADATA_IMPORTER_PLUGIN_LIST:
            usage = "Zotero Metadata Importer Plug-in  "
        elif type in MEDIA_FILE_IMPORTER_PLUGIN_LIST:
            usage = "Media File Importer Plug-in  "
        elif type.startswith("z"):
            usage = "User-Defined and/or Other Plug-in  "
        else:
            usage = ""
        return usage
    #-----------------------------------------------------
    def resize_all_columns(self):
        self.matrix.resizeColumnsToContents()
        self.save_matrix_listing_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def optimize_column_widths(self):
        self.matrix.resizeColumnsToContents()
        self.save_matrix_listing_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def copy_log_to_clipboard(self):
        #tab delimited, ready to "paste special" into Calc or just paste into text document

        self.matrix.selectAll()

        s = ''
        s = s + "\t".join([str(self.matrix.horizontalHeaderItem(i).text()) for i in xrange(0, self.n_total_cols)])
        s = s +  '\n'
        for r in xrange(0, self.n_total_rows):
            for c in xrange(0, 2):
                try:
                    s = s + str(self.matrix.item(r,c).text()) + "\t"
                except AttributeError:
                    s = s + "\t"
            s = s[:-1] + "\n"        #eliminate last '\t'
        self.clip.setText(s)

        self.save_matrix_listing_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_matrix_listing_dialog_geometry(self):
        self.dialog_closing(None)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_all_identifier_types(self):
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(self.maingui, _('JS+ GUI Tool'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)
        #-------------------------------------
        mysql = "SELECT DISTINCT type FROM identifiers"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        my_db.close()
        if not tmp_rows:
            tmp_rows = []
        if len(tmp_rows) > 0:
            for row in tmp_rows:
                #~ if DEBUG: print("row: ", str(row))
                l = list(row)
                type = l[0]
                self.identifier_types_list.append(type)
                #~ if DEBUG: print("type: ", type)
                del l
            #END FOR
        del tmp_rows
    #-----------------------------------------------------
    #-----------------------------------------------------
    def remove_selected_identifier_types(self):

        self.save_matrix_listing_dialog_geometry()

        selected_books_list = self.get_selected_books()
        if len(selected_books_list) == 0:
            self.hide()
            error_dialog(self.maingui, _('JS+ GUI Tool'),_('No Books Were Selected.'), show=True)
            self.show()
            return

        checked_types_list = self.get_checked_types()
        n = len(checked_types_list)
        if n == 0:
            self.hide()
            error_dialog(self.maingui, _('JS+ GUI Tool'),_('No Identifier Types Were Selected.'), show=True)
            self.show()
            return

        if n == 1:
            msg = "Remove the " + str(n) + " Selected Identifier Type for the Selected Books?  Are you sure?<br>"
        else:
            msg = "Remove the " + str(n) + " Selected Identifier Types for the Selected Books?  Are you sure?<br>"
        for type in checked_types_list:
            msg = msg + "<br>" + type
        #END FOR

        self.hide()

        if question_dialog(self.maingui, "JS+ GUI Tool - Remove Identifier Types", msg):
            self.show()
        else:
            self.show()
            return

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(self.maingui, _('JS+ GUI Tool'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)
        #-------------------------------------
        mysql = "DELETE FROM identifiers WHERE type = ? and book = ?"
        my_cursor.execute("begin")
        for book in selected_books_list:
            book = int(book)
            for type in checked_types_list:
                my_cursor.execute(mysql,(type,book))
                if DEBUG: print("Removed: ", str(type), "  for book: ", str(book))
            #END FOR
        #END FOR
        my_cursor.execute("commit")
        my_db.close()

        self.force_refresh_of_cache(selected_books_list)

        msg = 'Selected Identifier Types have been removed for the Selected Books.'
        self.maingui.status_bar.show_message(_(msg), 10000)

        self.restart_dialog()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_checked_types(self):
        checked_types_list = []

        for r in xrange(0, self.n_total_rows):
            state = self.matrix.item(r,2).checkState()
            #~ print("state: ", str(state))
            if state:
                if state > 0:                                         # 1 = partially checked; 2 = checked
                    type = self.matrix.item(r,0).text()
                    checked_types_list.append(type)
        #END FOR

        return checked_types_list
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_selected_books(self):
        selected_books_list = []

        book_ids_list = map( partial(self.convert_id_to_book), self.maingui.library_view.get_selected_ids() )

        n = len(book_ids_list)
        if n == 0:
            del book_ids_list
            return selected_books_list

        for item in book_ids_list:
            s = str(item['calibre_id'])
            selected_books_list.append(s)
        #END FOR

        del book_ids_list

        return selected_books_list
    #-----------------------------------------------------
    #-----------------------------------------------------
    def convert_id_to_book(self, idval):
        book = {}
        book['calibre_id'] = idval
        return book
    #-----------------------------------------------------
    def force_refresh_of_cache(self,selected_book_list):
        backend = self.maingui.library_view.model().db.backend
        mydbcache = dbcache(self.maingui.library_view.model().db.backend)
        mydbcache.init()
        self.maingui.library_view.model().refresh_ids(selected_book_list)
        self.maingui.tags_view.recount()
    #-----------------------------------------------------
    def apsw_connect_to_library(self):

        path = self.guidb.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", str(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------

#END OF book_identifier_matrix_listing_dialog.py