# !/usr/bin/env python2
# vim:fileencoding=utf-8

from __future__ import (unicode_literals, division, absolute_import, print_function)
__license__ = 'GPL v3'
__copyright__ = '2018, Saume'

# Constant definitions.
PLUGIN_NAME = u"ZIP-RAR to Comic Book"
PLUGIN_VERSION = (1, 0, 0)
PLUGIN_DESC = u"Convert zip and rar compressed archives to comic books (cbz and cbr)."
EXT_ZIP = "zip"
EXT_ZIP_COMIC = "cbz"
EXT_RAR = "rar"
EXT_RAR_COMIC = "cbr"
SUPPORTED_FILE_FORMATS = [EXT_ZIP, EXT_RAR]

# Imports.
from calibre.customize import FileTypePlugin
from calibre.library import db
from calibre.constants import DEBUG
from calibre import prints
import sys
import traceback

def log(*args):
	prints('Saume: ', *args)
	#if DEBUG:
	#	prints('Saume: ', *args)

def GetLibraryPath():
	from calibre.utils.config import prefs
	path = prefs['library_path']
	if path:
		path = path.replace('\\', '/')
		while path.endswith('/'):
			path = path[:-1]
	return path

class SaumeZipToCbz(FileTypePlugin):
	# Plugin metadata.
	name = PLUGIN_NAME
	description = PLUGIN_DESC
	author = u"Saume"
	version = PLUGIN_VERSION
	supported_platforms = ['windows', 'osx', 'linux']
	can_be_disabled = True
	minimum_calibre_version = (3, 28, 0)
	# Plugin functionality.
	file_types = set(SUPPORTED_FILE_FORMATS)
	on_import = True
	on_postimport = True
	on_preprocess = False
	on_postprocess = True
	priority = 1
	
	b_recursion_stopper = False
	
	def is_customizable(self):
		return False
	
	def run(self, path_to_ebook):
		# Prevent recursive calls to run() when adding / importing / converting books from within run().
		if self.b_recursion_stopper:
			return path_to_ebook
		self.b_recursion_stopper = True
		
		try:
			# Get the current library.
			lib_path = GetLibraryPath()
			log("current library: " + lib_path + ".")
			c_db = db(lib_path).new_api
			
			# Get all book ids from the library. Sometimes this returns only an int instead of a collection...
			all_ids = c_db.all_book_ids()
			log("all book ids: " + str(all_ids))
			# Parse all books in the library to see if any book needs processing.
			for b_id in all_ids:
				# Make sure the book id is valid (safety since we can generate a range above).
				if not c_db.has_id(b_id):
					continue
				
				# List all formats for the book, but only if the file exists.
				lst_fmt = c_db.formats(b_id, True)
				
				# Look for the formats we care about.
				has_zip = False
				has_cbz = False
				has_rar = False
				has_cbr = False
				for val in lst_fmt:
					fmt_lc = val.lower()
					
					if fmt_lc == EXT_ZIP:
						has_zip = True
					elif fmt_lc == EXT_ZIP_COMIC:
						has_cbz = True
					elif fmt_lc == EXT_RAR:
						has_rar = True
					elif fmt_lc == EXT_RAR_COMIC:
						has_cbr = True
				
				log("processing book " + str(b_id) + ", which has " + str(len(lst_fmt)) + " formats (zip: " + str(has_zip) + ", cbz: " + str(has_cbz) + ", rar: " + str(has_rar) + ", cbr: " + str(has_cbr) + ").")
				if has_zip and not has_cbz:
					self.AutoConvertAndAdd(b_id, c_db.format_abspath(b_id, EXT_ZIP), EXT_ZIP, EXT_ZIP_COMIC, c_db)
				if has_rar and not has_cbr:
					self.AutoConvertAndAdd(b_id, c_db.format_abspath(b_id, EXT_RAR), EXT_RAR, EXT_RAR_COMIC, c_db)
		except:
			# Make sure to always reset this to False or future calls will do nothing.
			self.b_recursion_stopper = False
			
			# Log the exception for debugging.
			exc_t, exc_v, exc_tr = sys.exc_info()
			lines = traceback.format_exception(exc_t, exc_v, exc_tr)
			e = ""
			for line in lines:
				e += line
			log(e)
		else:
			# Make sure to always reset this to False or future calls will do nothing.
			self.b_recursion_stopper = False
			
		return path_to_ebook
	
	def postimport(self, book_id, book_format, c_db):
		fmt_lc = book_format.lower()
		log("book " + str(book_id) + " imported (format: " + fmt_lc + ").")
		
		if fmt_lc == EXT_ZIP:
			self.AutoConvertAndAdd(book_id, c_db.format_abspath(book_id, book_format), fmt_lc, EXT_ZIP_COMIC, c_db)
		elif fmt_lc == EXT_RAR:
			self.AutoConvertAndAdd(book_id, c_db.format_abspath(book_id, book_format), fmt_lc, EXT_RAR_COMIC, c_db)
	
	def postadd(self, book_id, fmt_map, c_db):
		log("book " + str(book_id) + " added with " + str(len(fmt_map)) + " formats.")
		
		# Keep iterating all formats, if we have both zip and rar, we will want to process both, rather than just break after the first.
		for k in fmt_map:
			fmt_lc = k.lower()
			log("checking format " + fmt_lc + " for book " + str(book_id) + ".")
			
			if fmt_lc == EXT_ZIP:
				self.AutoConvertAndAdd(book_id, c_db.format_abspath(book_id, k), fmt_lc, EXT_ZIP_COMIC, c_db)
			elif fmt_lc == EXT_RAR:
				self.AutoConvertAndAdd(book_id, c_db.format_abspath(book_id, k), fmt_lc, EXT_RAR_COMIC, c_db)
	
	# og_full_path Full path with the extension to the original file to convert from.
	# og_format Lower case extension for the original file, without the dot (".").
	# new_format Lower case extension for the new file, without the dot (".").
	# c_db Reference to the calibre library database (calls c_db.add_format() on it).
	def AutoConvertAndAdd(self, book_id, og_full_path, og_format, new_format, c_db):
		if og_full_path is None:
			log("can't convert book " + str(book_id) + " from " + og_format + " to " + new_format + " because the given path is invalid.")
			return False
		else:
			# Don't replace existing formats, also don't run hooks to avoid fuck-ups with other plugins modifying our data and trying to convert it or some other shit.
			# Additionally, NEVER run hooks from here if the method is called in run(), as it would create an infinite loop.
			r = c_db.add_format(book_id, new_format, og_full_path, False, False)
			log("converted book " + str(book_id) + " (" + og_full_path + ") from " + og_format + " to " + new_format + " (success: " + str(r) + ").")
			return r

# Definitions of _make_file and PersistentTemporaryFile copied from source code.
#
#def _make_file(suffix, prefix, base):
#	suffix, prefix = map(force_unicode, (suffix, prefix))
#	return tempfile.mkstemp(suffix, prefix, dir=base)
#
#class PersistentTemporaryFile(object):
#
#	"""
#	A file-like object that is a temporary file that is available even after being closed on
#	all platforms. It is automatically deleted on normal program termination.
#	"""
#	_file = None
#	
#	def __init__(self, suffix="", prefix="", dir=None, mode='w+b'):
#		if prefix is None:
#			prefix = ""
#		if dir is None:
#			dir = base_dir()
#		fd, name = _make_file(suffix, prefix, dir)
#		
#		self._file = os.fdopen(fd, mode)
#		self._name = name
#		self._fd = fd
#		atexit.register(cleanup, name)
#	
#	def __getattr__(self, name):
#		if name == 'name':
#			return self.__dict__['_name']
#		return getattr(self.__dict__['_file'], name)
#	
#	def __enter__(self):
#		return self
#	
#	def __exit__(self, *args):
#		self.close()
#	
#	def __del__(self):
#		try:
#			self.close()
#		except:
#			pass
