#!/usr/bin/perl

#    impfindGUI.pl
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

use strict;
use warnings;
use Tk;
use Tk::HList;
use Tk::NoteBook;
use File::Find;
use File::Basename;
use File::Spec::Functions qw(curdir rel2abs catfile);
use File::Copy;
use FindBin qw($RealBin);

### Globals ###
my $VERSION = "0.4";
my $RCFILE = catfile($RealBin,"impfindGUI.rc");

# stores all of the information about the books from the last search
my %SearchBookInfo;
my %ShelfBookInfo;

my $CurSearchPath;
my %Checks;
my %Help;

# DefaultPath, ShelfPath, LastPath
my %Config;
 $Config{'LastPath'} = "";
 $Config{'DefaultPath'} = "";
 $Config{'ShelfPath'} = "";

# gui widgets
my $MW; 		#Main Window
my $NoteBook; 	#Tabbed Notebook
my %NoteTabs;	#Individual notebook tabs
my %Labels;		#Label Widgets
my %Entries;	#Entry Widgets
my %HLists;
my %Frames;
my %Radios;

$Help{'General'} = 
"impfindGUI.pl Version $VERSION\n\n" .
"-- Version notes--\n\n" .
"As of this version searching for mobi and lit ".
"files is disabled.  That feature should be in " .
"the next release.\n\n" .
"-- Usage --\n\n" .
"The impfindGUI script works by itself without " .
"relying on the impfind script.  The operation " .
"is fairly straight forward.\n\n" .
"Use the Browse button to select a directory or " .
"you can type a directory into the field.  Then " .
"click the Search button and impfindGUI will find ".
"all of the imp files in the directory and display ".
"Category, Title, Author and path information about ".
"all of the books that were found.\n\n".
"Ticking the Recursive check box will cause the ".
"Search to move through all of the directories below ".
"the selected directory.\n\n".
"Ticking the Keyword Search check box will allow ".
"you to enter a search term that will be matched ".
"against the information for each book and only ".
"those that contain the term will be displayed.\n\n" .
"Clicking on the headers in any of the lists will " .
"sort the list by that column. \n\n" .
"From the Search tab books can be selected by clicking " .
"on them.  They can then be copied to the Book Shelf " .
"directory by clicking the Copy Selected to Book Shelf " .
"button.\n\n" .
"From the Book Shelf tab books can be selected and " .
"deleted from the Book Shelf directory by clicking " .
"the Delete Selected button.  All books can be deleted " .
"from the Book Shelf by clicking the the Delete All " .
"button.\n";

######################
# Subs
#######################

sub _GetImpInfo {

	my ($impfilename) = @_;
	my $start_address = 48;
	my $next_address = 0;
	my @info;
	my @ret_info;
	my $s;

	open IMPFILE, "<$impfilename";
	my $old_rs = $/;
	$/="\0";
	$next_address = $start_address;
	for (my $i = 0; $i < 7; $i++){
		seek IMPFILE, $next_address, 0;
		$s = <IMPFILE>;
		chomp $s;
		push @info,$s;
		$next_address += length($s) + 1;
	}
	close IMPFILE;
	$/ = $old_rs;
	#category
	push @ret_info, $info[1];
	#Title
	push @ret_info, $info[3];
	#Author
	push @ret_info, $info[6];
	#type
	push @ret_info, "Imp";

	return @ret_info;
}

sub _FindBooks {
	my ($dir,$recurse,$types,@books) = @_;
	my @types = split (/:/,$types);
	my @f;
	my $ext;
	my $ndir;
	my $name;
	
	#print "Dir $dir\nRecurse $recurse\nTypes $types\n";
	opendir DIR, $dir or die "Cant open $dir for read: $!\n";
	#print "Searching in $dir\n";
	@f = readdir DIR;
	close DIR;
	foreach $name (@f)
	{
		next if $name eq '.' || $name eq '..';
		#print "Current File $name\n";
		$ndir = catfile($dir,$name);
		if ( (-d $ndir) && $recurse){
			
			#print "Found another Directory $ndir Calling myself\n";
			@books = _FindBooks($ndir,$recurse,$types,@books);
			#print "Coming back up 1 level\n";
		}
		if (! -d $ndir){
			foreach $ext (@types){
				#print "Testing $name against $ext\n";
				push @books,catfile($dir,$name) if $name =~ /$ext$/ ;
			}
		}
	}
	#print "Currently in $dir\n";
	return @books;
}


############################################################
# _LoadHList
# Arguments: List name, number of columns, array with info
############################################################
sub _LoadHList {
	my ($list,$cols,@info) = @_;
	my $rows = @info / $cols;

	$HLists{$list}->delete('all');
	for (my $row_num = 0; $row_num < $rows; $row_num++){
		$HLists{$list}->add($row_num);
		for(my $col_num = 0; $col_num < $cols; $col_num++){
			$HLists{$list}->itemCreate($row_num,$col_num,
										-text => $info[($row_num * $cols) + $col_num]);
		}
	}
}

#############################################################
# Generates an array that can passed to _LoadHList that has
# been sorted.
#############################################################
sub _LoadShelfList {
	#sort by category title author 
	my $sortby = shift;
	my $book;
	my $sortfield;

	my @sortedbooks;
	
	my %sortfields;

	foreach $book (keys %ShelfBookInfo){
		next if exists $sortfields{ $ShelfBookInfo{$book}->{$sortby} };
		$sortfields{ $ShelfBookInfo{$book}->{$sortby} } = 1;
		
	}
	
	foreach $sortfield ( sort( keys %sortfields) ) {
		foreach $book (keys %ShelfBookInfo) {
			if ($ShelfBookInfo{$book}->{$sortby} eq $sortfield){
				push @sortedbooks, $ShelfBookInfo{$book}->{'category'};
				push @sortedbooks, $ShelfBookInfo{$book}->{'title'};
				push @sortedbooks, $ShelfBookInfo{$book}->{'author'};
				push @sortedbooks, $book;
			}
		}
	}

	$Labels{'TotShelfBooks'}->configure(-text => @sortedbooks/4);
	$Labels{'ShelfPath'}->configure(-text => $Config{'ShelfPath'});
	_LoadHList('Shelf',4,@sortedbooks);

}

###########################################################
# Generates an array that can be passed to _LoadHList that
# has been sorted and searched.
###########################################################
sub _LoadSearchList {
	# sort by category title author type
	my $sortby = shift;
	my $book;
	my $sortfield;

	my @sortedbooks;

	my %sortfields;

	foreach $book (keys %SearchBookInfo){
		next if exists $sortfields{ $SearchBookInfo{$book}->{$sortby} };
		$sortfields{ $SearchBookInfo{$book}->{$sortby} } = 1;
		
	}
	
	foreach $sortfield ( sort( keys %sortfields) ) {
		foreach $book (keys %SearchBookInfo) {
			if ($SearchBookInfo{$book}->{$sortby} eq $sortfield){
				push @sortedbooks, $SearchBookInfo{$book}->{'category'};
				push @sortedbooks, $SearchBookInfo{$book}->{'title'};
				push @sortedbooks, $SearchBookInfo{$book}->{'author'};
				push @sortedbooks, $SearchBookInfo{$book}->{'type'};
				push @sortedbooks, $book;
			}
		}
	}

	$Labels{'TotBooks'}->configure(-text => @sortedbooks/5);
	_LoadHList('Search',5,@sortedbooks);
}

#############################################################
# Searches the directory named in the CurPath Entry for any
# books of the type specified by the type check boxes.
# Loads the book information into the SearchBookInfo hash
#############################################################
sub _SearchForBooks {

	my $dir = $Entries{'CurPath'}->get();
	my $recurse = $Checks{'Recurse'};
	my $types;
	my $book;
	my $found_it;
	my $search_term;
	my $field;
	my @books;
	# info order category, title, author, type
	my @info;

	%SearchBookInfo = ();
	
	$types .= ".imp" if $Checks{'FileTypeImp'};
	$types .= ":.mobi:.prc" if $Checks{'FileTypeMobi'};
	$types .= ":.lit" if $Checks{'FileTypeLit'};
	
	@books = _FindBooks($dir,$recurse,$types);
	foreach $book (@books){
		$found_it = 1;
		@info = ();
		if($book =~ /\.imp$/){
			push @info, _GetImpInfo($book);
		}
		if ($Checks{'Keyword'}){
			$found_it = 0;
			$search_term = lc( $Entries{'SearchTerm'}->get() );
			print "Searching for $search_term\n";
			if($Radios{'SearchValue'} eq "All"){
				print "Searching all fields\n";
				foreach (@info){
					$field = lc($_);
					$found_it = 1 if $field =~ /$search_term/;
				}
			}
			if($Radios{'SearchValue'} eq "Category"){
				print "Searching Category\n";
				$field = lc($info[0]);
				print "Comparing $field to $search_term\n";
				$found_it = 1 if $field =~ /$search_term/;
			}
			if($Radios{'SearchValue'} eq "Title"){
				print "Searching Titles\n";
				$field = lc($info[1]);
				print "Comparing $field to $search_term\n";
				$found_it = 1 if $field =~ /$search_term/;
			}
			if($Radios{'SearchValue'} eq "Author"){
				print "Searching Authors\n";
				$field = lc($info[2]);
				print "Comparing $field to $search_term\n";
				$found_it = 1 if $field =~ /$search_term/;
			}
		}
		if($found_it == 1){
			$SearchBookInfo{$book}->{'category'} = shift @info;
			$SearchBookInfo{$book}->{'title'} = shift @info;
			$SearchBookInfo{$book}->{'author'} = shift @info;
			$SearchBookInfo{$book}->{'type'} = shift @info;
		}	
	}
	_LoadSearchList('category');
}

################################################################
# Loads the global hash ShelfBookInfo with the information 
# about the books in the book shelf
################################################################
sub _FindShelfBooks {
	my $dir = $Entries{'ShelfPath'}->get();
	my $book;

	my @books;
	my @info;

	if(! -d $dir){
		$MW->messageBox(-title => "Book Shelf",
						-type => 'ok',
						-message => "The Book Shelf directory does not seem to be set.\n" .
									"This can be set by going to the Configuration Tab.");
		return;
	}

	@books = _FindBooks($dir,1,".imp");
	%ShelfBookInfo = ();
	foreach $book(@books){
		@info = ();
		push @info, _GetImpInfo($book);
		$ShelfBookInfo{$book}->{'category'} = shift @info;
		$ShelfBookInfo{$book}->{'title'} = shift @info;
		$ShelfBookInfo{$book}->{'author'} = shift @info;
	}
	_LoadShelfList('category');
}


sub _CopyToShelf {
	my @indexes = $HLists{'Search'}->selectionGet();
	my @paths;
	my $tot_files;
	my $resp;

	foreach (@indexes){
		push @paths, $HLists{'Search'}->itemCget($_,4,"-text");
	}
	$tot_files = @paths;
	$resp = $MW->messageBox(-title 	=> "Copy to Shelf",
							-type 	=> 'yesno',
							#-font	=> 'ansi 10',
							-message => "Copy $tot_files Files to shelf?\n" . $Config{'ShelfPath'}); 
	if ($resp eq "Yes"){
		foreach (@paths){
			copy ($_, $Config{'ShelfPath'});
		}
	}
	#_FindShelfBooks();
	$HLists{'Search'}->selectionClear();
}

sub _DeleteBooks {
	my $extent = shift;
	my @indexes;
	my @paths;
	my $tot_files;
	my $resp;
	my $c = 0;

	if($extent eq 'sel'){
 		@indexes = $HLists{'Shelf'}->selectionGet() ;
	}
	else {
		while ( $HLists{'Shelf'}->info('exists',$c) ){
			push @indexes,$c;
			$c++;
		}
	}
	foreach (@indexes){
		push @paths, $HLists{'Shelf'}->itemCget($_,3,"-text");
	}
	$tot_files = @paths;
	$resp = $MW->messageBox(-title 	=> "Delete from Shelf",
							-type 	=> 'yesno',
							#-font	=> 'ansi 10',
							-message => "Delete $tot_files Files from shelf?\n" . $Config{'ShelfPath'}); 
	if ($resp eq "Yes"){
		foreach (@paths){
			print "Deleting $_\n";
			unlink $_;
		}
	}
	_FindShelfBooks();
}

sub _LoadRCFile
{
	
	if (-e "$RCFILE")
	{
		open RC, "<$RCFILE" or die "Cant open $RCFILE for reading:$!\n";
		
		while (<RC>)
		{
			chomp;
			s/#.*//;
			s/^\s+//;
			s/\s+$//;
			next unless length;
			my ($var, $value) = split(/\s*=\s*/,$_,2);
			$Config{$var} = $value;
		}
		close RC;
	}
}

sub _WriteRCFile {
	my $k;

	open RC, ">$RCFILE" or die "Cant open $RCFILE for writing:$!\n";
	foreach $k (keys %Config){
		print RC $k . " = " . $Config{$k} . "\n";
	}
	close RC;
}

# Clean up the loose ends
sub _End {
	print "Closing impfindGUI.pl writing $RCFILE\n";
	print "Last Path Searched " . $Config{'DefaultPath'} . "\n";
	print "Book Shelf Path " . $Config{'ShelfPath'} . "\n";
	_WriteRCFile;
	exit;
}

#### Tk Gui ####

# main window
$MW = MainWindow->new(-title => "ImpFindGUI");
# tabbed notebook 
$NoteBook = $MW->NoteBook()->pack(-fill => 'both', -expand => 1);
$NoteTabs{'Search'} = $NoteBook->add("Sheet 1", -label=>"Search");
$NoteTabs{'Shelf'} 	= $NoteBook->add("Sheet 2", 
									-label=>"Book Shelf",
									-raisecmd => \&_FindShelfBooks,
									);
$NoteTabs{'Configure'} = $NoteBook->add("Sheet 5", -label=>"Configure");
$NoteTabs{'About'} = $NoteBook->add("Sheet 6", -label=>"About");

######## Widgets in Tabs #########

#Search Tab
##### Search Frame
$Frames{'Search'} = $NoteTabs{'Search'}->Frame(-relief => 'raised',
												-borderwidth => 2,
											   )->pack(-fill=>'both',-expand=>0,-side=>'top');
$Frames{'Search'}->Label(-text=>"Path")->grid(-column=>0,-row=>0);
$Entries{'CurPath'} = $Frames{'Search'}->Entry(-background => 'white',
												-textvariable => \$Config{'LastPath'},
												)->grid(-column=>1,-row=>0,-stick=>'ew');
$Entries{'CurPath'}->bind('<Return>',\&_SearchForBooks);
$Frames{'Search'}->Button(-text=>"Browse",
						-command => sub {
										$Config{'LastPath'} = catfile($MW->chooseDirectory(-title=>"Choose Search Directory"),"");
										},
						)->grid(-column=>2, -row=>0);

$Frames{'Search'}->Checkbutton(-text => "Recursive",
								-variable => \$Checks{'Recurse'},
								)->grid(-column => 0, -row => 1);
$Frames{'Search'}->Button(-text => "Search",
						  -command => \&_SearchForBooks,
						  )->grid(-column => 1, -row => 1);
$Frames{'Search'}->Label(-text => "Books Found",
						)->grid(-column => 2, -row => 1);
$Labels{'TotBooks'} = $Frames{'Search'}->Label(-text => "0",
											  )->grid(-column => 3, -row => 1);
$Frames{'Search'}->Checkbutton(-text => "Keyword Search",
							   -variable => \$Checks{'Keyword'},
							   -command => sub{
							   					if($Checks{'Keyword'}){
													$Entries{'SearchTerm'}->configure(-state => 'normal');
													$Radios{'SearchAll'}->configure(-state => 'normal');
													$Radios{'SearchCategory'}->configure(-state => 'normal');
													$Radios{'SearchTitle'}->configure(-state => 'normal');
													$Radios{'SearchAuthor'}->configure(-state => 'normal');
												}
												else {
													$Entries{'SearchTerm'}->configure(-state => 'disabled');
													$Radios{'SearchAll'}->configure(-state => 'disabled');
													$Radios{'SearchCategory'}->configure(-state => 'disabled');
													$Radios{'SearchTitle'}->configure(-state => 'disabled');
													$Radios{'SearchAuthor'}->configure(-state => 'disabled');
												}
											   },
								)->grid(-column => 1, -row => 2);
$Frames{'SearchType'} = $Frames{'Search'}->Frame()->grid(-column => 1, -row => 3);
$Radios{'SearchAll'} = $Frames{'SearchType'}->Radiobutton(-text => "All",
														  -value => "All",
														  -variable => \$Radios{'SearchValue'},
														  -state => 'disabled',
														  )->grid(-column => 0, -row => 0);
$Radios{'SearchCategory'} = $Frames{'SearchType'}->Radiobutton(-text => "Category",
															   -value => "Category",
															   -variable => \$Radios{'SearchValue'},
														  		-state => 'disabled',
															   )->grid(-column => 1, -row => 0);
$Radios{'SearchTitle'} = $Frames{'SearchType'}->Radiobutton(-text => "Title",
															-value => "Title",
															-variable => \$Radios{'SearchValue'},
															-state => 'disabled',
															)->grid(-column => 2, -row => 0);
$Radios{'SearchAuthor'} = $Frames{'SearchType'}->Radiobutton(-text => "Author",
															 -value => "Author",
															 -variable => \$Radios{'SearchValue'},
															 -state => 'disabled',
															 )->grid(-column => 3, -row => 0);
$Frames{'BookType'}= $Frames{'Search'}->Labelframe(-text => "Book File Type",
															)->grid(-column => 2, -row => 2, -rowspan => 3);
$Frames{'BookType'}->Checkbutton(-text => "Imp",
								-variable => \$Checks{'FileTypeImp'},
								)->grid(-column => 0, -row => 0, -stick => 'w')->select();
$Frames{'BookType'}->Checkbutton(-text => "Mobi",
								-variable => \$Checks{'FileTypeMobi'},
								-state => 'disabled',
								)->grid(-column => 0, -row => 1, -stick => 'w');
$Frames{'BookType'}->Checkbutton(-text => "Lit",
								-variable => \$Checks{'FileTypeLit'},
								-state => 'disabled',
								)->grid(-column => 0, -row => 2, -stick => 'w');

$Entries{'SearchTerm'} = $Frames{'Search'}->Entry(-state => 'disabled',
												  -background => 'white',
												  )->grid(-column => 1, -row => 4);
$Entries{'SearchTerm'}->bind('<Return>',\&_SearchForBooks);

##### SearchDisplay Frame
$Frames{'SearchDisplay'} = $NoteTabs{'Search'}->Frame(-relief => 'raised', 
														-borderwidth => 3,
													)->pack(-fill => 'both',-expand => 1);
$HLists{'Search'} = $Frames{'SearchDisplay'}->Scrolled(
												"HList",
												-header => 1,
												-columns => 5,
												-selectmode => 'extended',
												-scrollbars => 'osoe',
												-width => 70,
												-background => 'white',
												-selectbackground => 'SeaGreen3',
												)->pack(-expand => 1, -fill => 'both');
$HLists{'SearchCatButt'} = $HLists{'Search'}->Button(-anchor => 'center',
								  			-text => "Category",
								  			-command => sub {_LoadSearchList('category');});

$HLists{'SearchTitleButt'} = $HLists{'Search'}->Button(-anchor => 'center',
								  			-text => "Title",
								  			-command => sub {_LoadSearchList('title');});

$HLists{'SearchAuthButt'} = $HLists{'Search'}->Button(-anchor => 'center',
								  			-text => "Author",
								  			-command => sub {_LoadSearchList('author');});

$HLists{'SearchTypeButt'} = $HLists{'Search'}->Button(-anchor => 'center',
								  			-text => "Type",
								  			-command => sub {_LoadSearchList('type');});
$HLists{'Search'}->header('create',0, 
						-itemtype => 'window',
						-borderwidth => -2,
						-widget => $HLists{'SearchCatButt'});
$HLists{'Search'}->header('create',1,
						-itemtype => 'window',
						-borderwidth => -2,
						-widget => $HLists{'SearchTitleButt'});
$HLists{'Search'}->header('create',2,
						-itemtype => 'window',
						-borderwidth => -2,
						-widget => $HLists{'SearchAuthButt'});
$HLists{'Search'}->header('create',3,
						-itemtype => 'window',
						-borderwidth => -2,
						-widget => $HLists{'SearchTypeButt'});
$HLists{'Search'}->header('create',4,-text => "Path");

# Copy To Shelf button
$NoteTabs{'Search'}->Button(-text => "Copy Selected to Shelf",
							-command => \&_CopyToShelf,
							)->pack(-side => 'bottom');

####### Book Shelf Tab ######
$Frames{'ShelfInfo'} = $NoteTabs{'Shelf'}->Frame()->pack(-fill => 'both', -expand => 0, -side => 'top');
$Frames{'ShelfInfo'}->Label(-text => "Books in Shelf")->grid(-column => 0, -row => 0, -stick => 'w');
$Labels{'TotShelfBooks'} = $Frames{'ShelfInfo'}->Label(-text => '0',)->grid(-column => 1, -row => 0, -stick => 'w');
$Frames{'ShelfInfo'}->Label(-text => "Shelf Path")->grid(-column => 0, -row => 1, -stick => 'w');
$Labels{'ShelfPath'} = $Frames{'ShelfInfo'}->Label(-text => "\/")->grid(-column => 1,
																		-row => 1, 
																		-stick => 'w');
$HLists{'Shelf'} = $NoteTabs{'Shelf'}->Scrolled('HList',
												-header => 1,
												-columns => 4,
												-selectmode => 'extended',
												-scrollbars => 'osoe',
												-background => 'white',
												selectbackground => 'gray',
												)->pack(-expand => 1, -fill => 'both');

$HLists{'ShelfCatButt'} = $HLists{'Shelf'}->Button(-anchor => 'center',
													-text => "Category",
													-command => sub {_LoadShelfList('category')});
$HLists{'ShelfTitleButt'} = $HLists{'Shelf'}->Button(-anchor => 'center',
													-text => "Title",
													-command => sub {_LoadShelfList('title')});
$HLists{'ShelfAuthButt'} = $HLists{'Shelf'}->Button(-anchor => 'center',
													-text => "Author",
													-command => sub {_LoadShelfList('author')});
$HLists{'Shelf'}->header('create',0,
						-itemtype => 'window',
						-borderwidth => -2,
						-widget => $HLists{'ShelfCatButt'});
$HLists{'Shelf'}->header('create',1,
						-itemtype => 'window',
						-borderwidth => -2,
						-widget => $HLists{'ShelfTitleButt'});
$HLists{'Shelf'}->header('create',2,
						-itemtype => 'window',
						-borderwidth => -2,
						-widget => $HLists{'ShelfAuthButt'});
$HLists{'Shelf'}->header('create',3, -text => "Path");

$Frames{'ShelfControls'} = $NoteTabs{'Shelf'}->Frame()->pack(-expand => 1, -fill => 'x');
$Frames{'ShelfControls'}->Button(-text => "Delete All",
								 -command => sub { _DeleteBooks('all') },
								 )->grid(-column => 0, -row => 0);
$Frames{'ShelfControls'}->Button(-text => "Delete Selected",
								 -command => sub{_DeleteBooks('sel')},
								 )->grid(-column => 1, -row => 0);

#### Configure Tab ######
$Frames{'ConfigurePaths'} = $NoteTabs{'Configure'}->Frame()->pack(-expand => 1, -fill => 'y', -side => 'left');
$Frames{'ConfigurePaths'}->Label(-text => "Default Directory")->grid(-column => 0, -row => 0, -stick => 'w');
$Entries{'DefaultPath'} = $Frames{'ConfigurePaths'}->Entry(-background => 'white',
															-textvariable => \$Config{'DefaultPath'},
													)->grid(-column => 1, 
															-row => 0,
															-stick => 'w');
$Frames{'ConfigurePaths'}->Label(-text => "Shelf Path")->grid(-column => 0, -row => 1, -stick => 'w');
$Entries{'ShelfPath'} = $Frames{'ConfigurePaths'}->Entry(-background => 'white',
														-textvariable => \$Config{'ShelfPath'},
														)->grid(-column => 1, -row => 1,-stick => 'w');

#### About Tab #####
$NoteTabs{'About'}->Scrolled("Text",
							 -scrollbars => 'osoe',
							 -background => 'white',
							 -wrap => 'word',
							 )->pack(-expand => 1, -fill => 'both',
							 )->insert('end' ,$Help{'General'});



_LoadRCFile;
MainLoop();
_End();
