#!/usr/bin/env python
# -*- coding: utf-8 -*-
try:
    from sigil_bs4 import BeautifulSoup, Comment
except:
    from bs4 import BeautifulSoup, Comment
import sys, re, os

from tkinter import Tk, BOTH, StringVar, PhotoImage 
from tkinter.ttk import Frame, Button, Label, Entry
isosx = sys.platform.startswith('darwin')

class Dialog(Frame):
    global Cancel
    Cancel = True

    def __init__(self, parent, bk):
        # display the dialog box
        Frame.__init__(self, parent)   
        self.parent = parent
        self.bk = bk
        self.initUI()

    def savevalues(self):
        global Cancel
        Cancel = False
        prefs = self.bk.getPrefs()
        prefs['anchor'] = self.anchor.get()
        prefs['anchorid'] = self.anchorid.get()
        prefs['fndef'] = self.fndef.get()
        prefs['fndefid'] = self.fndefid.get()
        prefs['backlink'] = self.backlink.get()
        self.bk.savePrefs(prefs)
        self.master.quit()
        self.master.destroy()
        
    def initUI(self):
        # define dialog box properties
        self.parent.title("FN Linker")
        self.pack(fill=BOTH, expand=1)
        
        # get preferences
        prefs = self.bk.getPrefs()

        # footnote anchor class
        anchorLabel = Label(self, text="Anchor class: ")
        anchorLabel.place(x=10, y=10)
        self.anchor=StringVar(None)
        if 'anchor' in prefs:
            self.anchor.set(prefs['anchor'])
        else:
            self.anchor.set('noteanchor')
        anchorEntry=Entry(self, textvariable=self.anchor)
        anchorEntry.place(x=130, y=10, width=85)

        # footnote id prefix
        anchorIDLabel = Label(self, text="Anchor ID prefix: ")
        anchorIDLabel.place(x=10, y=30)
        self.anchorid=StringVar(None)
        if 'anchorid' in prefs:
            self.anchorid.set(prefs['anchorid'])
        else:
            self.anchorid.set('bodyftn')
        anchorIDEntry=Entry(self, textvariable=self.anchorid)
        anchorIDEntry.place(x=130, y=30, width=85)
        
        # footnote definition class
        fndefLabel = Label(self, text="Definition class: ")
        fndefLabel.place(x=10, y=50)
        self.fndef=StringVar(None)
        if 'fndef' in prefs:
            self.fndef.set(prefs['fndef'])
        else:
            self.fndef.set('note')
        fndefEntry=Entry(self, textvariable=self.fndef)
        fndefEntry.place(x=130, y=50, width=85)

        # footnote definition id
        fndefIDLabel = Label(self, text="Definition ID prefix: ")
        fndefIDLabel.place(x=10, y=70)
        self.fndefid=StringVar(None)
        if 'fndefid' in prefs:
            self.fndefid.set(prefs['fndefid'])
        else:
            self.fndefid.set('ftn')
        fndefIDEntry=Entry(self, textvariable=self.fndefid)
        fndefIDEntry.place(x=130, y=70, width=85)
        
        # backlink class
        backlinkLabel = Label(self, text="Backlink class: ")
        backlinkLabel.place(x=10, y=90)
        self.backlink=StringVar(None)
        if 'backlink' in prefs:
            self.backlink.set(prefs['backlink'])
        else:
            self.backlink.set('noteSymbol')
        backlinkEntry=Entry(self, textvariable=self.backlink)
        backlinkEntry.place(x=130, y=90, width=85)
        
        # OK and Cancel buttons
        cancelButton = Button(self, text="Cancel", command=self.quit)
        cancelButton.place(x=130, y=120)
        okButton = Button(self, text="OK", command=self.savevalues)
        okButton.place(x=10, y=120)

def run(bk):
    fn_def_id = None
    # set Tk parameters for dialog box
    root = Tk()
    root.geometry("240x160+300+300")
    app = Dialog(root, bk)
    if not isosx:
        icon_img = PhotoImage(file=os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'sigil.png'))
        root.tk.call('wm', 'iconphoto', root._w, icon_img)
    root.mainloop()
    
    if Cancel  == True:
        print('Plugin terminated by user.\nPlease click OK to close the Plugin Runner window.')
        return -1
    
    #--------------------------------------
    # get preferences
    #--------------------------------------
    prefs = bk.getPrefs()
    
    # class for <sup> footnotes
    fn_class = prefs['anchor']

    # id prefix for <sup> footnote anchors
    fn_id = prefs['anchorid']
    
    # class for <p> footnote definitions
    fndef_class = prefs['fndef']

    # id prefix for <p> footnote definitions
    fndef_id = prefs['fndefid']
    
    # class for <a> backlink numbers in footnote definitions file
    backlink_class = prefs['backlink']

    # debug mode
    if 'debug' not in prefs:
        prefs['debug'] = False
        bk.savePrefs(prefs)
    debug = prefs['debug']
    
    # get epub version number
    if bk.launcher_version() >= 20160102:
        epubversion = bk.epub_version()
    else:
        epubversion = BeautifulSoup(bk.get_opf(), 'lxml').find('package')['version']
        
    #--------------------------------
    # get guide/landmark items
    #--------------------------------
    if epubversion.startswith("3"):
        # look for nav.xhtml
        opf_soup = BeautifulSoup(bk.get_opf(), 'lxml')
        nav_item = opf_soup.find('item', {'properties' : 'nav'})
        if nav_item:
            nav_href = nav_item['href']
            nav_id = bk.href_to_id(nav_href)

            # get landmarks from nav document
            landmarks = {}
            nav_soup = BeautifulSoup(bk.readfile(nav_id), 'html.parser')
            for landmark in nav_soup.find_all('a', {'epub:type' : re.compile('.*?')}):
                epub_type = landmark['epub:type']
                href = landmark['href']
                landmarks[epub_type] = href
                
            # check for footnotes landmarks item
            if not 'footnotes' in landmarks and not 'rearnotes' in landmarks:
                print('\nError: Missing footnotes/rearnotes epub3 landmark: Use Add Semantics > Footnotes/Rear Notes to mark the footnotes definitions file.\nPlugin terminated!\nPlease click OK to close the Plugin Runner window.')
                return -1
            else:
                if 'rearnotes' in landmarks:
                    fn_def_id = bk.href_to_id(landmarks['rearnotes'].replace('../', ''))
                if 'footnotes' in landmarks:
                    fn_def_id = bk.href_to_id(landmarks['footnotes'].replace('../', ''))
        else:
            print('\nError: nav document not found!.nPlugin terminated!\nPlease click OK to close the Plugin Runner window.')
            return -1
    
    else:
        opf_guide_items = {}
        for ref_type, title, href in bk.getguide():
            opf_guide_items[ref_type] = href

        # look for notes guide item
        if not 'notes' in opf_guide_items:
            print('Error: Missing Notes guide item. Use Add Semantics > Notes to mark the footnote definitions file.\nPlugin terminated!\nPlease click OK to close the Plugin Runner window.')
            return -1
        else: 
            fn_def_id = bk.href_to_id(opf_guide_items['notes'])


    #-----------------------------------------------
    # process footnote anchors
    #-----------------------------------------------
    sup_counter = 0
    fn_to_href = {}
    context = {}
    file_list = list(bk.text_iter())
        
    # process file list
    print('\nProcessing footnote anchor files...\n')
    for (html_id, href) in file_list:
        html = bk.readfile(html_id)

        # load html code into BeautifulSoup
        soup = BeautifulSoup(html, 'html.parser')
        orig_soup = str(soup)
 
        # look for footnote anchors
        all_tags = soup.find_all(attrs={"class" : fn_class})
        
        # process matches
        for index, each_tag in enumerate(all_tags, start = 1):
            sup_counter += 1
            fn_to_href[sup_counter] = '../' + href + '#' + fn_id + str(sup_counter)

            # footnote reference is wrapped in another tag, e.g. span
            if each_tag.a != None:
                each_tag.a['id'] = fn_id + str(sup_counter)
                each_tag.a['href'] = '../' + bk.id_to_href(fn_def_id) + '#' + fndef_id + str(sup_counter)
                if epubversion.startswith("3"):
                    each_tag.a['epub:type'] = 'noteref'
                each_tag.a.string = str(sup_counter)
            # footnote referrence is another tag, e.g. anchor or sup
            else:
                each_tag.name = 'a'
                each_tag['id'] = fn_id + str(sup_counter)
                each_tag['href'] = '../' + bk.id_to_href(fn_def_id) + '#' + fndef_id + str(sup_counter)
                if epubversion.startswith("3"):
                    each_tag['epub:type'] = 'noteref'
                each_tag.string = str(sup_counter)
            
            # debug: save footnote reference
            if debug == True:
                if each_tag.parent.string != '':
                    context[sup_counter] =  str(each_tag.parent)
                elif each_tag.parent.parent.string != '':
                    context[sup_counter] =  str(each_tag.parent.parent)
                else:
                    context[sup_counter] =  'undefined'
            
        # update footnote anchor file
        if str(soup) != orig_soup:
            # check for missing epub3 namespace attribute
            if epubversion.startswith("3") and 'xmlns:epub' not in soup.html.attrs:
                soup.html['xmlns:epub'] = 'http://www.idpf.org/2007/ops'
            # try Sigil bs4 pretty print
            try:
                bk.writefile(html_id, str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  ")))
            except:
                bk.writefile(html_id, str(soup))
            print('Footnote anchor file: ', href, ' updated')
    
    # terminate plugin if no footnote anchors were found
    if sup_counter == 0:
        print('No <sup class="' + fn_class +'"> tags found! \nPlugin terminated.')
        return -1
    
    #----------------------------------------------
    # count number of footnote definitions
    #----------------------------------------------

    # check endnotes files
    def_counter = 0
    html = bk.readfile(fn_def_id)
    
    # delete original footnote numbers
    html = re.sub(r'(class="' + fndef_class + '"[^>]*>)\d+\.*\s*', r'\1 ', html) 

    # define footnote definition structure
    soup = BeautifulSoup(html, 'html.parser')
    orig_soup = str(soup)
    
    # find all footnote definitions
    all_tags = soup.find_all(attrs={"class" : fndef_class})
    fndef_number = len(all_tags)

    # compare the number of footnote definitions with the number of footnotes
    print('\nChecking number of anchors and definitions...')
    if sup_counter == fndef_number:
        print('Footnote anchors: ', sup_counter)
        print('Footnote definitions: ', fndef_number)
        
        # update notes section
        print('\nProcessing footnote definitions...')
        for index, each_tag in enumerate(all_tags, start = 1):
            def_counter += 1

            # add aside to epub3 books
            if epubversion.startswith("3"):
                aside = soup.new_tag('aside', id=fndef_id + str(def_counter))
                aside['epub:type'] = 'footnote'
                if each_tag.name == 'aside':
                    each_tag['id'] = fndef_id + str(def_counter)
                elif each_tag.parent.name == 'aside':
                    each_tag.parent['id'] = fndef_id + str(def_counter)
                    each_tag.parent['class'] = fndef_class
                    del each_tag['class']
                elif each_tag.parent.parent.name == 'aside':
                    each_tag.parent.parent['id'] = fndef_id + str(def_counter)
                    each_tag.parent.parent['class'] = fndef_class
                    del each_tag['class']
                else:
                    each_tag.wrap(aside)
            else:
                each_tag['id'] = fndef_id + str(def_counter)
            backlink = soup.new_tag('a', href=fn_to_href[def_counter])
            backlink['class'] = backlink_class
            backlink.string = str(def_counter)
            if each_tag.a == None and each_tag.name != 'a':
                each_tag.insert(0, backlink)
            elif each_tag.name == 'a':
                each_tag.replace_with(backlink)
            else:
                each_tag.a.replace_with(backlink)
            
            #debug: insert context as comment tag
            if debug == True:
                existing_comment = each_tag.find(string=lambda text:isinstance(text,Comment))
                if existing_comment:
                    existing_comment.replace_with(Comment(context[index]))
                    print('Comment updated for footnote', index)
                else:
                    # insert comment
                    try:
                        each_tag.insert(0, Comment(context[index]))
                        print('Comment added for footnote', index)
                    except:
                        print('Comment for ', index, 'not found!')
                        
        # update footnote definitions file
        if str(soup) != orig_soup:
            # check for missing epub3 namespace attribute
            if epubversion.startswith("3") and 'xmlns:epub' not in soup.html.attrs:
                soup.html['xmlns:epub'] = 'http://www.idpf.org/2007/ops'
            # try Sigil bs4 pretty print
            try:
                bk.writefile(fn_def_id, str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  ")))
            except:
                bk.writefile(fn_def_id, str(soup))
            print('Footnote definitions file: ', bk.id_to_href(fn_def_id), ' updated.')
        
    else:
        print("\nThe number of footnote anchors and footnote definitions don't match!\nDon't save the file!!!")
        print('Footnote anchors: ', sup_counter)
        print('Footnote definitions: ', fndef_number)
        return -1
    
    print('\nPlease click OK to close the Plugin Runner window.')
    
    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
