#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import, print_function)
from calibre.gui2.actions import InterfaceAction
from calibre.utils.config import JSONConfig


__copyright__ = '2017, Becky <becky@fr.pl>'
__docformat__ = 'restructuredtext en'

from functools import partial
try:
    from PyQt5.Qt import QToolButton, QMenu, QModelIndex
except ImportError:
    from PyQt4.Qt import QToolButton, QMenu


from calibre.db.legacy import LibraryDatabase
from calibre.library import current_library_name

from calibre_plugins.metamover.common_utils import (set_plugin_icon_resources, get_icon, create_menu_action_unique, debug_print)

from calibre.gui2 import (question_dialog, Dispatcher, error_dialog, info_dialog)

from calibre_plugins.metamover.config import get_library_config

import calibre_plugins.metamover.config as cfg


# The class that all interface action plugins must inherit from
from calibre.gui2.actions import InterfaceAction

import re

try:
    load_translations()
except NameError:
    pass # load_translations() added in calibre 1.9

PLUGIN_ICONS = ['images/metamover.png', 'images/metamovermini.png']


class InterfacePlugin(InterfaceAction):

    name = 'METAmover'
    action_spec = ('METAmover', None, _('Copy informations from comment to custom columns'), ())
    prefs = JSONConfig('plugins/METAmover')


    def genesis(self):
        self.is_library_selected = True
        self.menu_actions = []
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        self.build_menus()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))

        ###self.action_pages()
        self.qaction.triggered.connect(self.megazord)


    def build_menus(self):
        m = self.menu
        m.clear()

        ac = create_menu_action_unique(self, m, _('Copy numer of &pages from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_pages),
                                  shortcut_name=_('Copy number of pages to custom column'))
        self.menu_actions.append (ac)
        ac = create_menu_action_unique(self, m, _('Copy &original title from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_originaltitle),
                                  shortcut_name=_('Copy original title to custom column'))
        self.menu_actions.append (ac)
        ac = create_menu_action_unique(self, m, _('Copy &translators from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_translators),
                                  shortcut_name=_('Copy translators to custom column'))
        self.menu_actions.append (ac)

        ac = create_menu_action_unique(self, m, _('Copy &genre from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_genre),
                                  shortcut_name=_('Copy genre to custom column'))
        self.menu_actions.append (ac)

        ac = create_menu_action_unique(self, m, _('Copy book &series from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_seria),
                                  shortcut_name=_('Copy book series to custom column'))
        self.menu_actions.append (ac)

        ac = create_menu_action_unique(self, m, _('Copy other &editions from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_othereditions),
                                  shortcut_name=_('Copy other editions to custom column'))
        self.menu_actions.append (ac)

        ac = create_menu_action_unique(self, m, _('Copy &rating from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_rating),
                                  shortcut_name=_('Copy rating to custom column'))
        self.menu_actions.append (ac)

        ac = create_menu_action_unique(self, m, _('Copy first edition &date from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_firstpubdate),
                                  shortcut_name=_('Copy first edition date to custom column'))
        self.menu_actions.append (ac)

        ac = create_menu_action_unique(self, m, _('Copy first e&dition date in Po&land from comment to custom column'), 'images/metamovermini.png',
                                  triggered=partial(self.action_firstpubdatepl),
                                  shortcut_name=_('Copy first edition date in Poland to custom column'))
        self.menu_actions.append (ac)

        m.addSeparator()
        ac = create_menu_action_unique(self, m, _('Copy LubimyCzytac &ID from comment to identifiers'), 'images/metamovermini.png',
                                  triggered=partial(self.action_id),
                                  shortcut_name=_('Copy LubimyCzytac ID to identifiers'))
        self.menu_actions.append (ac)

        ac = create_menu_action_unique(self, m, _('Delete &HR before comment'), 'images/metamovermini.png',
                                  triggered=partial(self.action_hr),
                                  shortcut_name=_('Remove horizontal rule before comment'))
        self.menu_actions.append (ac)

        #m.addSeparator()
        #ac = create_menu_action_unique(self, m, _('Customize &LubimyCzytac plugin'), 'config.png',
        #                          shortcut=False, triggered=self.konfiguracja_pluginu_lubimyczytac)
        #self.menu_actions.append (ac)

        m.addSeparator()
        ac = create_menu_action_unique(self, m, _('&Customize plugin')+'...', 'config.png',
                                  shortcut=False, triggered=self.show_configuration)
        self.menu_actions.append (ac)
        self.gui.keyboard.finalize()


    def megazord(self):
        from_menu = True
        self.action_pages(from_menu)
        self.action_originaltitle(from_menu)
        self.action_translators(from_menu)
        self.action_genre(from_menu)
        self.action_seria(from_menu)
        self.action_othereditions(from_menu)
        self.action_rating(from_menu)
        self.action_firstpubdate(from_menu)
        self.action_firstpubdatepl(from_menu)
        self.action_id()
        self.action_hr()

        confirmBox = self._is_confirm(cfg.CONFIRM)
        if confirmBox:
            debug_print ("True")
            return info_dialog(self.gui, _('Completed'),
                _('All tasks have been completed (as configured)'), show=True)
        else:
            debug_print ("False")
            return info_dialog(self.gui, _('Completed'),
                _('All tasks have been completed (as configured)'), show=False)

    def action_pages(self, from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy pages to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy pages to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colforpages = self._get_column_validity(cfg.KEY_PAGES_CUSTOM_COLUMN)

        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for pages first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_pages = colforpages

        overwrite_pages = self._is_overwrite_pages(cfg.OVERWRITE_PAGES)
        delete_pages = self._is_delete_pages(cfg.DELETE_PAGES)

        self.dzialaj_pages(book_ids, col_pages, overwrite_pages, delete_pages)


    def dzialaj_pages(self, book_ids, col_pages, overwrite_pages, delete_pages):

        dbA = self.gui.current_db

        db = self.gui.current_db.new_api

        id_aux = {}

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldpages = miA.get(col_pages)

            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('(<p id="strony">Książka ma )(\d+)( )(stron|strony)(\.</p>)',texto)

                if id_aux[id] is not None:
                    if id_aux[id].group(2):
                        debug_print ("Znaleziono liczbe stron w komentarzu: ", id_aux[id].group(2))
                        if oldpages is not None:
                            if overwrite_pages:
                                db.set_field(col_pages, {id:id_aux[id].group(2)})
                        else:
                            db.set_field(col_pages, {id:id_aux[id].group(2)})

                        #Delete unnecessary line in comment
                        if delete_pages:
                            id_aux[id]=re.sub('<p id="strony">Książka ma (\d+) (.+?)\.</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunieta liczba stron: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


    def action_originaltitle(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy original title from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy original title from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colfororgtit = self._get_column_validity(cfg.KEY_ORIGINALTITLE_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for original title first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_orgtit = colfororgtit

        overwrite_orgtit = self._is_overwrite_pages(cfg.OVERWRITE_ORIGINALTITLE)
        delete_orgtit = self._is_delete_pages(cfg.DELETE_ORIGINALTITLE)

        self.dzialaj_originaltitle(book_ids, col_orgtit, overwrite_orgtit, delete_orgtit)


    def dzialaj_originaltitle (self, book_ids, col_orgtit, overwrite_orgtit, delete_orgtit):

        dbA = self.gui.current_db
        db = self.gui.current_db.new_api

        id_aux = {}

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldorigtitle = miA.get(col_orgtit)


            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('(<p id="tytul_oryginalu">Tytuł oryginału: <em>)(.*?)(</em></p>)',texto)

                if id_aux[id] is not None:
                    if id_aux[id].group(2):
                        debug_print ("Znaleziono tytul oryginalu w komentarzu: ", id_aux[id].group(2))
                        if oldorigtitle is not None:
                            if overwrite_orgtit:
                                db.set_field(col_orgtit, {id:id_aux[id].group(2)})
                        else:
                            db.set_field(col_orgtit, {id:id_aux[id].group(2)})

                        #Delete unnecessary line in comment
                        if delete_orgtit:
                            id_aux[id]=re.sub('<p id="tytul_oryginalu">Tytuł oryginału: <em>(.*?)</em></p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunietym tytulem oryginalu: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


    def action_translators(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy translators from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy translators from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colfortranslators = self._get_column_validity(cfg.KEY_TRANSLATORS_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for translators first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_translators = colfortranslators

        overwrite_translators = self._is_overwrite_translators(cfg.OVERWRITE_TRANSLATORS)
        delete_translators = self._is_delete_translators(cfg.DELETE_TRANSLATORS)

        self.dzialaj_translators(book_ids, col_translators, overwrite_translators, delete_translators)


    def dzialaj_translators (self, book_ids, col_translators, overwrite_translators, delete_translators):

        dbA = self.gui.current_db
        db = self.gui.current_db.new_api

        id_aux = {}

        #lis_aux_id = []

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldtranslators = miA.get(col_translators)


            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('(<p id="tlumaczenie">Tłumaczenie: )(.*?)(</p>)',texto)

                if id_aux[id] is None:
                    id_aux[id] = re.search('(<p id="translator">Vertaald door: )(.*?)(</p>)',texto) #obsługa BOL.com

                if id_aux[id] is None:
                    id_aux[id] = re.search('(<p id="translator">)(.*?)(</p>)',texto) #obsługa neutralna

                if id_aux[id] is not None:

                        ###
                        deltemp = self.gui.library_view.model().custom_columns
                        for key, column in deltemp.iteritems():
                            debug_print (key)
                            debug_print (column)
                            #typ = column['datatype']
                            #multi = column['is_multiple']
                            #if typ in column_types and multi:
                            #available_columns[key] = column

                        #library_config = cfg.get_library_config(dbA)
                        #debug_print (library_config)
                        #col = library_config.get(col_translators, '')
                        #debug_print (col)
                        ###

                        debug_print ("Znaleziono tlumaczy w komentarzu: ", id_aux[id].group(2))
                        tlumacze = id_aux[id].group(2).split(",")
                        if oldtranslators is not None:
                            if overwrite_translators:
                                db.set_field(col_translators, {id:tlumacze})
                        else:
                            db.set_field(col_translators, {id:tlumacze})

                        #Delete unnecessary line in comment
                        if delete_translators:
                            id_aux[id]=re.sub('<p id="tlumaczenie">Tłumaczenie: (.*?)</p>','',texto)
                            id_aux[id]=re.sub('<p id="translator">(.*?)</p>','',texto) #usuniecie innych zapisów
                            #debug_print ("BECKY INFO: Komentarz z usunietymi tlumaczami: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


    def action_genre(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy genre from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy genre from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colforgenre = self._get_column_validity(cfg.KEY_GENRE_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for genre first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_genre = colforgenre

        overwrite_genre = self._is_overwrite_genre(cfg.OVERWRITE_GENRE)
        delete_genre = self._is_delete_genre(cfg.DELETE_GENRE)

        self.dzialaj_genre(book_ids, col_genre, overwrite_genre, delete_genre)


    def dzialaj_genre (self, book_ids, col_genre, overwrite_genre, delete_genre):

        dbA = self.gui.current_db
        db = self.gui.current_db.new_api

        id_aux = {}

        #lis_aux_id = []

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldgenre = miA.get(col_genre)


            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('(<p id="kategoria">Kategoria: )(.*?)(</p>)',texto)


                if id_aux[id] is not None:
                    if id_aux[id].group(2):
                        debug_print ("Znaleziono kategorie w komentarzu: ", id_aux[id].group(2))
                        if oldgenre is not None:
                            if overwrite_genre:
                                db.set_field(col_genre, {id:id_aux[id].group(2)})
                        else:
                            db.set_field(col_genre, {id:id_aux[id].group(2)})

                        #Delete unnecessary line in comment
                        if delete_genre:
                            id_aux[id]=re.sub('<p id="kategoria">Kategoria: (.*?)</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunieta kategoria: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


    def action_seria(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy book series from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy book series from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colforseria = self._get_column_validity(cfg.KEY_SERIA_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for book series first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_seria = colforseria

        overwrite_seria = self._is_overwrite_seria(cfg.OVERWRITE_SERIA)
        delete_seria = self._is_delete_seria(cfg.DELETE_SERIA)

        self.dzialaj_seria(book_ids, col_seria, overwrite_seria, delete_seria)


    def dzialaj_seria (self, book_ids, col_seria, overwrite_seria, delete_seria):

        dbA = self.gui.current_db
        db = self.gui.current_db.new_api

        id_aux = {}

        #lis_aux_id = []

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldseria = miA.get(col_seria)

            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('(<p id="seria">Seria: )(.*?)(</p>)',texto)

                if id_aux[id] is not None:
                    if id_aux[id].group(2):
                        debug_print ("Znaleziono serie w komentarzu: ", id_aux[id].group(2))
                        if oldseria is not None:
                            if overwrite_seria:
                                db.set_field(col_seria, {id:id_aux[id].group(2)})
                        else:
                            db.set_field(col_seria, {id:id_aux[id].group(2)})

                        #Delete unnecessary line in comment
                        if delete_seria:
                            id_aux[id]=re.sub('<p id="seria">Seria: (.*?)</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunieta seria: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


    def action_othereditions(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy the list of the other editions from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy the list of the other editions from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colforothereditions = self._get_column_validity(cfg.KEY_OTHEREDITIONS_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for the other editions first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_othereditions = colforothereditions

        overwrite_othereditions = self._is_overwrite_othereditions(cfg.OVERWRITE_OTHEREDITIONS)
        delete_othereditions = self._is_delete_othereditions(cfg.DELETE_OTHEREDITIONS)

        self.dzialaj_othereditions(book_ids, col_othereditions, overwrite_othereditions, delete_othereditions)


    def dzialaj_othereditions(self, book_ids, col_othereditions, overwrite_othereditions, delete_othereditions):

        dbA = self.gui.current_db
        db = self.gui.current_db.new_api

        id_aux = {}

        #lis_aux_id = []

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldothereditions = miA.get(col_othereditions)

            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('(<p id="lista-innych-wydan-short">)(Wszystkie inne wydania mają ten sam tytuł: |Tytuł innego wydania: |Tytuły innych wydań: )(.*?)(</p>)',texto)

                if id_aux[id] is not None:
                    if id_aux[id].group(3):
                        debug_print ("Znaleziono serie w komentarzu: ", id_aux[id].group(3))
                        if oldothereditions is not None:
                            if overwrite_othereditions:
                                db.set_field(col_othereditions, {id:id_aux[id].group(3)})
                        else:
                            db.set_field(col_othereditions, {id:id_aux[id].group(3)})

                        #Delete unnecessary line in comment
                        if delete_othereditions:
                            id_aux[id]=re.sub('<p id="lista-innych-wydan-short">(.*?)</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunieta lista innych wydan: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)

    def action_rating(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy rating from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy rating from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colforrating = self._get_column_validity(cfg.KEY_RATING_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for rating first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_rating = colforrating

        overwrite_rating = self._is_overwrite_rating(cfg.OVERWRITE_RATING)
        delete_rating = self._is_delete_rating(cfg.DELETE_RATING)

        self.dzialaj_rating(book_ids, col_rating, overwrite_rating, delete_rating)


    def dzialaj_rating(self, book_ids, col_rating, overwrite_rating, delete_rating):

        dbA = self.gui.current_db

        db = self.gui.current_db.new_api

        id_aux = {}

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldrating = miA.get(col_rating)
            if texto is not None:
                id_aux[id] = re.search('(<p id="ratingLC">)(Ocena w LC: )(\d+\.*\d*)(</p>)',texto)
                if id_aux[id] is not None:
                    if id_aux[id].group(3):
                        debug_print ("Znaleziono ocene w komentarzu: ", id_aux[id].group(3))
                        if oldrating is not None:
                            if overwrite_rating:
                                db.set_field(col_rating, {id:id_aux[id].group(3)})
                        else:
                            db.set_field(col_rating, {id:id_aux[id].group(3)})

                        #Delete unnecessary line in comment
                        if delete_rating:
                            id_aux[id]=re.sub('<p id="ratingLC">(.*?)</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunieta ocena: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)

    def action_firstpubdate(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy rating from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy first pubdate from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colforfirstpubdate = self._get_column_validity(cfg.KEY_FIRSTPUBDATE_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for first pubdate first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_firstpubdate = colforfirstpubdate

        overwrite_firstpubdate = self._is_overwrite_firstpubdate(cfg.OVERWRITE_FIRSTPUBDATE)
        delete_firstpubdate = self._is_delete_firstpubdate(cfg.DELETE_FIRSTPUBDATE)

        self.dzialaj_firstpubdate(book_ids, col_firstpubdate, overwrite_firstpubdate, delete_firstpubdate)


    def dzialaj_firstpubdate(self, book_ids, col_firstpubdate, overwrite_firstpubdate, delete_firstpubdate):

        dbA = self.gui.current_db

        db = self.gui.current_db.new_api

        id_aux = {}

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldfirstpubdate = miA.get(col_firstpubdate)
            if texto is not None:
                id_aux[id] = re.search('(<p id="pierwsze-wydanie">)(Rok pierwszego wydania: )(\d+)(</p>)',texto)
                if id_aux[id] is not None:
                    if id_aux[id].group(3):
                        debug_print ("Znaleziono rok pierwszego wydania w komentarzu: ", id_aux[id].group(3))
                        if oldfirstpubdate is not None:
                            if overwrite_firstpubdate:
                                db.set_field(col_firstpubdate, {id:id_aux[id].group(3)})
                        else:
                            db.set_field(col_firstpubdate, {id:id_aux[id].group(3)})

                        #Delete unnecessary line in comment
                        if delete_firstpubdate:
                            id_aux[id]=re.sub('<p id="pierwsze-wydanie">(.*?)</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunietym rokiem pierwszego wydania: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


    def action_firstpubdatepl(self,from_menu):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy first pubdate in Poland from comment to custom column'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy first pubdate in Poland from comment to custom column'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        any_valid, colforfirstpubdatepl = self._get_column_validity(cfg.KEY_FIRSTPUBDATEPL_CUSTOM_COLUMN)
        if not any_valid:
            if from_menu == True:
                return
            else:
                if not question_dialog(self.gui, _('Configure plugin'), '<p>'+
                    _('You must specify custom column for first pubdate in Poland first. Do you want to configure this now?'),
                    show_copy_button=False):
                    return
                self.show_configuration()
                return

        col_firstpubdatepl = colforfirstpubdatepl

        overwrite_firstpubdatepl = self._is_overwrite_firstpubdatepl(cfg.OVERWRITE_FIRSTPUBDATEPL)
        delete_firstpubdatepl = self._is_delete_firstpubdatepl(cfg.DELETE_FIRSTPUBDATEPL)

        self.dzialaj_firstpubdatepl(book_ids, col_firstpubdatepl, overwrite_firstpubdatepl, delete_firstpubdatepl)


    def dzialaj_firstpubdatepl(self, book_ids, col_firstpubdatepl, overwrite_firstpubdatepl, delete_firstpubdatepl):

        dbA = self.gui.current_db

        db = self.gui.current_db.new_api

        id_aux = {}

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")
            oldfirstpubdatepl = miA.get(col_firstpubdatepl)
            if texto is not None:
                id_aux[id] = re.search('(<p id="pierwsze-wydanie-pl">)(Rok pierwszego wydania w Polsce: )(\d+)(</p>)',texto)
                if id_aux[id] is not None:
                    if id_aux[id].group(3):
                        debug_print ("Znaleziono w komentarzu rok pierwszego wydania w Polsce: ", id_aux[id].group(3))
                        if oldfirstpubdatepl is not None:
                            if overwrite_firstpubdatepl:
                                db.set_field(col_firstpubdatepl, {id:id_aux[id].group(3)})
                        else:
                            db.set_field(col_firstpubdatepl, {id:id_aux[id].group(3)})

                        #Delete unnecessary line in comment
                        if delete_firstpubdatepl:
                            id_aux[id]=re.sub('<p id="pierwsze-wydanie-pl">(.*?)</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunietym rokiem pierwszego wydania: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


    def action_id(self):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy ID to identifiers'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for copy ID to identifiers'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        overwrite_id = self._is_overwrite_id(cfg.OVERWRITE_ID)
        delete_id = self._is_delete_id(cfg.DELETE_ID)

        self.dzialaj_id(book_ids, overwrite_id, delete_id)


    def dzialaj_id(self, book_ids, overwrite_id, delete_id):

        dbA = self.gui.current_db

        db = self.gui.current_db.new_api

        id_aux = {}

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")

            identifiers = dbA.get_identifiers(id, index_is_id=True)
            oldid = identifiers.get('lubimyczytac', None)

            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('(<p id="lubimyczytac">ID: )(\d+)(</p>)',texto)


                if id_aux[id] is not None:
                    if id_aux[id].group(2):
                        #debug_print ("Znaleziono ID w komentarzu: ", id_aux[id].group(2))
                        if oldid is not None:
                            if overwrite_id:
                                dbA.set_identifier(id,'lubimyczytac', id_aux[id].group(2))
                        else:
                            dbA.set_identifier(id,'lubimyczytac', id_aux[id].group(2))

                        #Delete unnecessary line in comment
                        if delete_id:
                            id_aux[id]=re.sub('<p id="lubimyczytac">ID: (\d+)</p>','',texto)
                            #debug_print ("BECKY INFO: Komentarz z usunietym ID: ", id_aux[id])
                            db.set_field('comments', {id:id_aux[id]})

                        #Refresh metadata
                        self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)

    def action_hr(self):
        if not self.is_library_selected:
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for delete hr'), show=True)
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0 :
            return error_dialog(self.gui, _('No selected book'),
                    _('No book selected for delete hr'), show=True)
        book_ids = self.gui.library_view.get_selected_ids()

        delete_hr = self._is_delete_hr(cfg.DELETE_HR)

        self.dzialaj_hr(book_ids, delete_hr)


    def dzialaj_hr(self, book_ids, delete_hr):

        dbA = self.gui.current_db

        db = self.gui.current_db.new_api

        id_aux = {}

        for id in book_ids:
            miA = dbA.get_metadata (id, index_is_id=True, get_cover=False)
            texto = miA.get ("comments")

            identifiers = dbA.get_identifiers(id, index_is_id=True)
            oldid = identifiers.get('lubimyczytac', None)

            if texto is not None:
                #debug_print ("BECKY INFO: Komentarz: ", texto)
                id_aux[id] = re.search('<hr id="przed-komentarzem">',texto)


                if id_aux[id] is not None:
                    debug_print ("Znaleziono HR w komentarzu: ", id_aux[id])

                    #Delete unnecessary line in comment
                    if delete_hr:
                        id_aux[id]=re.sub('<hr id="przed-komentarzem">','',texto)
                        #debug_print ("BECKY INFO: Komentarz z usunietym HR: ", id_aux[id])
                        db.set_field('comments', {id:id_aux[id]})

                    #Refresh metadata
                    self.gui.iactions['Edit Metadata'].refresh_gui(book_ids, covers_changed=False)


#<hr id="przed-komentarzem">


    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)


    def _get_column_validity(self, conf):
        db = self.gui.current_db
        #all_cols = db.field_metadata.custom_field_metadata()

        library_config = cfg.get_library_config(db)

        badana_col = ''

        any_valid = False

        col = library_config.get(conf, '')
        is_valid = len(col) > 0

        if not is_valid or not col:
            any_valid = False
            badana_col = ''
        else:
            any_valid = True
            badana_col = col

        return any_valid, badana_col


    def _is_overwrite_pages(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_pages = library_config.get(conf, False)

        return overwrite_pages


    def _is_overwrite_originalpages(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_originalpages = library_config.get(conf, False)

        return overwrite_originalpages


    def _is_overwrite_translators(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_translators = library_config.get(conf, False)

        return overwrite_translators


    def _is_overwrite_genre(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_genre = library_config.get(conf, False)

        return overwrite_genre


    def _is_overwrite_seria(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_seria = library_config.get(conf, False)

        return overwrite_seria


    def _is_overwrite_othereditions(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_othereditions = library_config.get(conf, False)

        return overwrite_othereditions


    def _is_overwrite_rating(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_rating = library_config.get(conf, False)

        return overwrite_rating


    def _is_overwrite_firstpubdate(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_firstpubdate = library_config.get(conf, False)

        return overwrite_firstpubdate


    def _is_overwrite_firstpubdatepl(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_firstpubdatepl = library_config.get(conf, False)

        return overwrite_firstpubdatepl


    def _is_overwrite_id(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        overwrite_id = library_config.get(conf, False)

        return overwrite_id


    def _is_delete_pages(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_pages = library_config.get(conf, False)

        return delete_pages


    def _is_delete_originalpages(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_originalpages = library_config.get(conf, False)

        return delete_originalpages


    def _is_delete_translators(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_translators = library_config.get(conf, False)

        return delete_translators


    def _is_delete_genre(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_genre = library_config.get(conf, False)

        return delete_genre


    def _is_delete_seria(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_seria = library_config.get(conf, False)

        return delete_seria


    def _is_delete_othereditions(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_othereditions = library_config.get(conf, False)

        return delete_othereditions


    def _is_delete_rating(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_rating = library_config.get(conf, False)

        return delete_rating


    def _is_delete_firstpubdate(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_firstpubdate = library_config.get(conf, False)

        return delete_firstpubdate


    def _is_delete_firstpubdatepl(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_firstpubdatepl = library_config.get(conf, False)

        return delete_firstpubdatepl


    def _is_delete_id(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_id = library_config.get(conf, False)

        return delete_id


    def _is_delete_hr(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        delete_hr = library_config.get(conf, False)

        return delete_hr


    def _is_confirm(self, conf):
        db = self.gui.current_db
        library_config = cfg.get_library_config(db)

        confirmBox = library_config.get(conf, False)

        return confirmBox


    def konfiguracja_pluginu_lubimyczytac(self):
        self._configure_lubimyczytac()


    def _configure_lubimyczytac(self):
        from calibre.customize.ui import initialized_plugins
        for plugin in initialized_plugins():
            if plugin.name == 'LubimyCzytac':
                if not plugin.is_customizable():
                    return info_dialog(self, _('Plugin not customizable'),
                        _('Plugin: %s does not need customization')%plugin.name, show=True)
                from calibre.customize import InterfaceActionBase
                if isinstance(plugin, InterfaceActionBase) and not getattr(plugin,
                        'actual_iaction_plugin_loaded', False):
                    return error_dialog(self, _('Must restart'),
                            _('You must restart calibre before you can'
                                ' configure the <b>%s</b> plugin')%plugin.name, show=True)
                plugin.do_user_config(self.parent())


    def edycja_metadanych(self):
        from calibre.library import db
        from calibre.gui2.metadata.single import edit_metadata
        db = db()
        row_list = list(range(len(db.data)))
        edit_metadata(db, row_list, 0)
