from calibre.customize.conversion import InputFormatPlugin
from calibre.utils.zipfile import ZipFile
import os.path, posixpath
from lxml import etree
from collections import OrderedDict

def _xml_get_member(zipfile, member):
   for k in zipfile.NameToInfo:
      if k.upper()==member.upper():
         member=k
         break
   if member in zipfile.NameToInfo:
      zi=zipfile.NameToInfo[member]
      fd=zipfile.open(zi)
      res=etree.parse(fd)
      root=res.getroot()
      fd.close()
   else:
      raise Exception("Member not found!")
   return root

RELS="{http://schemas.openxmlformats.org/package/2006/relationships}Relationships"
REL="{http://schemas.openxmlformats.org/package/2006/relationships}Relationship"

REL_BODY="http://www.fictionbook.org/FictionBook3/relationships/body"

def _get_rels(zip, path):
   r=OrderedDict()
   base, fn = posixpath.split(path)
   if fn:
      relfile=posixpath.join(base,'_rels',fn+'.rels')
      root=_xml_get_member(zip, relfile)
      if root is not None:
         rels=root.findall(REL)
         for rel in rels:
            fullpath=posixpath.normpath(posixpath.join(base, rel.get('Target')))
            r[rel.get('Id')]=(fullpath,rel.get('Type'))
   return r

_imgtypes={
   '.jpg': 'image/jpeg',
   '.jpeg': 'image/jpeg',
   '.png': 'image/png',
   '.svg': 'image/svg+xml',
}

NCX="""<?xml version="1.0" encoding="utf-8" ?>
<ncx version="2005-1" xmlns="http://www.daisy.org/z3986/2005/ncx/">
  <head>
    <meta content="1" name="dtb:depth"/>
    <meta content="0" name="dtb:totalPageCount"/>
    <meta content="0" name="dtb:maxPageNumber"/>
  </head>
  <navMap>
"""

def escape(txt):
   return txt.replace('&','&amp;').replace('<','&lt;')

def escapequote(txt):
   return escape(txt).replace('"','&quot;')

class FB3Input(InputFormatPlugin):
   name = "FB3 Input"
   description = _("Converts FB3 files to HTML")
   author = "Sarmat89"
   version = (0, 3, 0)
   file_types = set(['fb3'])

   def __init__(self, *args):
      self.plugin_path=args[0]

   def convert(self, stream, options, ext, log, accel):
      from calibre.ebooks.metadata.opf2 import OPFCreator
      from calibre.ebooks.metadata.opf import get_metadata as opf_metadata
      from calibre.ebooks.metadata.book.base import Metadata

      cwd=os.getcwd()
      log.debug("Dir: "+cwd)

      description=None
      bodypath=None
      html=None
      toc=False

      self._res=self.load_resources(['fb3body.xsl','stylesheet.css'])

      cont=ZipFile(stream)
      rels=_xml_get_member(cont, "_rels/.rels")
      if rels is not None:
         body=rels.find(REL+'[@Type="http://www.fictionbook.org/FictionBook3/relationships/Book"]')
         if body is not None:
            description=body.get('Target')
      if description:
         log.debug("Description:", description)
         rels=_get_rels(cont, description)
         log.debug("Rels:",rels)
         for k in rels:
            if REL_BODY == rels[k][1]:
               bodypath=rels[k][0]
               break
      if bodypath:
         log.debug("Body:", bodypath)
         root=_xml_get_member(cont, bodypath)
         bodyrels=_get_rels(cont, bodypath)
         if root is not None:
            html=self._transform(root)
            log.debug("XHTML root:", html)

      imglist=[]

      if html:
         hr=html.getroot()
         log.debug("Body rels:", repr(bodyrels).replace('), (','\n'))
         base,_=posixpath.split(bodypath)
         imgs={}
         iid=1
         for e in hr.findall(".//{http://www.w3.org/1999/xhtml}img[@src]"):
            k=e.get('src')
            if k in bodyrels:
               url=bodyrels[k][0]
               #if url.startswith('/'):
               #   url=url.lstrip('/')
               #else:
               #   url=posixpath.join(base,url)
               #   log.debug("Creating a full path for",k,":",url)
               _,ext=posixpath.splitext(url)
               if url in imgs:
                  tname=imgs[url]
               else:
                  tname="image%03u%s"%(iid,ext)
                  iid+=1
                  imgs[url]=tname
                  log.debug("Created an image:",k,tname,url)
               e.set('src',tname)
            else:
               log.debug("Stray image:", k)
         f=open('text.html','wb')
         html.write(f, encoding='utf-8')
         f.close()

         for img in imgs:
            if img in cont.NameToInfo:
               fd=open(imgs[img],'wb')
               imgdata=cont.read(cont.NameToInfo[img])
               fd.write(imgdata)
               fd.close()
               _,ext=posixpath.splitext(imgs[img])
               mt=_imgtypes.get(ext.lower(),'image/x-unknown')
               imglist.append((imgs[img],mt))
            else:
               log.debug("Image not in archive:", img)

         self.toc=self.build_toc(html, log)

      cont.close()

      f=open('stylesheet.css','wb')
      f.write(self._res["stylesheet.css"])
      f.close()

      if options.read_metadata_from_opf:
         inf=open(options.read_metadata_from_opf,'rb')
         mi=opf_metadata(inf)[0]
         inf.close()
      else:
         mi=Metadata(u"No title")
      log.debug(mi)
      opf=OPFCreator(cwd,mi)
      filelist=[
         ("text.html","application/xhtml+xml"),
         ("stylesheet.css","text/css"),
         ]
      if self.toc:
         filelist.append(("toc.ncx","application/x-dtbncx+xml"))
      filelist.extend(imglist)
      opf.create_manifest(filelist)
      opf.create_spine(["text.html"])
      f=open('meta.opf','wb')
      opf.render(f)
      f.close()
      return os.path.join(cwd,'meta.opf')

   def _transform(self, fb3):
      ss=self._res.get('fb3body.xsl')
      xsl=etree.fromstring(ss)
      xform=etree.XSLT(xsl)
      res=xform(fb3)
      return res

   def build_toc(self, html, log):
      headers=[]
      root=html.getroot()
      level=0
      hlevel=0
      nodes=root.findall(".//{http://www.w3.org/1999/xhtml}div[@id]")
      for e in nodes:
         ch=e.xpath("h:h2|h:h3|h:h4|h:h5",
               namespaces={"h":"http://www.w3.org/1999/xhtml"})
         if len(ch)>0:
            hdr=ch[0]
            text=' '.join(hdr.itertext()).strip()
            uid=hdr.get('id')
            if len(uid)==0: continue
            if len(text.strip())==0: continue
            el=ord(hdr.tag[-1])-0x32
            if el>hlevel:
               level+=1
            elif el<hlevel:
               level=el
            hlevel=el
            log.debug("Header:", level, el, uid, repr(text))
            headers.append((level,uid,text))
      if len(headers):
         log.debug('Writing TOC')
         f=open('toc.ncx','w',encoding='utf-8',newline='\n')
         f.write(NCX)
         pos=1
         level=-1
         for hl, uid, text in headers:
            while hl<=level:
               f.write('</navPoint>\n')
               level-=1
            e='<navPoint id="pt%d" playOrder="%d">\n<navLabel><text>%s</text></navLabel>\n<content src="text.html#%s"/>\n'%(hl,hl,escape(text),escapequote(uid))
            level=hl
            f.write(e)
            pos+=1
         while level>=0:
            f.write('</navPoint>\n')
            level-=1
         f.write('</navMap></ncx>')
         f.close()
         return True
      return False
