/**
 * Copyright (c) 2005, www.pdfbox.org
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of pdfbox; nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * http://www.pdfbox.org
 *
 */
package org.pdfbox;

import org.pdfbox.exceptions.InvalidPasswordException;

import org.pdfbox.cos.COSArray;
import org.pdfbox.cos.COSDictionary;
import org.pdfbox.cos.COSFloat;
import org.pdfbox.pdmodel.common.PDRectangle;
import org.pdfbox.pdmodel.PDDocument;
import org.pdfbox.pdmodel.PDPage;

//import org.pdfbox.util.DefaultFileFilter;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.Iterator;

/**
 * This object takes a letter sized PDF and rotates each source letter sized page 90 degress
 * and then shallow copies the rotated page. Each page is then cropped to produce a "top" page and
 * a "bottom" page which are then placed into a new desitnation PDF document.
 * This program is intended to produce a more readable PDF document for the iRex iLiad.
 *
 * Based on the PDFReader.java from the PDFBox project.
 * 
 * @author Scott Turner (scotty1024@mac.com)
 * @version $Revision: 1.0 $
 */
public class IliadRotate {

    /**
     * This main method povides a command line argument scanner that feeds all command line
     * arguments into the rotateAndCropPDFFile processor.
     *
     * Each new file will have the name "iLiad." + command line argument.
     *
     * Any Exception for a given command line argument is caught and reported.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {
	// Process each command line argument
	for (int i = 0; i < args.length; i++) {
	    try {
		rotateAndCropPDFFile( args[i] );
	    } catch (Exception e) {
		System.err.println("Error processing '" + args[i] + "': " + e.getMessage());
	    }
	}
    }

    /**
     * This method takes the supplied PDF filename and attempts to open that file. It then opens
     * a new empty PDF destination document.
     * It then pulls the pages from the source PDF document one by one.
     * It then makes a shallow copy of the source page to create a top and bottom page.
     * The top and bottom pages are then rotated 90 degrees and cropped to create a top and bottom
     * from the original.
     * The rotated and cropped pages are then appended to the new destination PDF document.
     *
     * Each new PDF document will have the name "iLiad." + aFileName.
     *
     * @param aFileName a <code>String</code> containing the name of a PDF document to open and process.
     *
     * @throws Exception If anything goes wrong.
     */
    public static void rotateAndCropPDFFile(String aFileName)
	throws Exception
    {
        InputStream input = new FileInputStream(new File( aFileName));

        PDDocument document = PDDocument.load( input );
        if (document.isEncrypted()) {
            try {
                document.decrypt( "" );
            } catch( InvalidPasswordException e ) {
                System.err.println( "Error: The document is encrypted." );
		throw e;
            } catch( org.pdfbox.exceptions.CryptographyException e ) {
                e.printStackTrace();
		throw e;
            }
        }
        
        // Open new destination PDF
	PDDocument doc = new PDDocument();

	// Copy pages from old PDF to new PDF
        Iterator pages = document.getDocumentCatalog().getAllPages().iterator();
	while (pages.hasNext()) {
	    PDPage pageTop = (PDPage)pages.next();

	    // Shallow copy to create "bottom" page
	    PDPage pageBottom = new PDPageWrapper(pageTop);

	    // Get page's width and height
	    PDRectangle bBox = pageTop.findCropBox();
	    if (bBox == null) {
		// If no crop box, use media box
		bBox = pageTop.findMediaBox();
	    }
	    float width = (int)(bBox.getWidth());
	    float height = (int)(bBox.getHeight());

	    // Rotate and Crop top page
	    pageTop.setRotation(90);
	    COSArray rectArray = new COSArray();
	    rectArray.add( new COSFloat( 0.0f));
	    rectArray.add( new COSFloat( (height * 0.5f) - 20.0f));
	    rectArray.add( new COSFloat( (float)width));
	    rectArray.add( new COSFloat( (float)height));
	    pageTop.setCropBox(new PDRectangle(rectArray));

	    // Rotate and Crop bottom page
	    pageBottom.setRotation(90);
	    rectArray = new COSArray();
	    rectArray.add( new COSFloat( 0.0f));
	    rectArray.add( new COSFloat( 0.0f));
	    rectArray.add( new COSFloat( (float)width));
	    rectArray.add( new COSFloat( (height * 0.5f) + 20.0f));
	    pageBottom.setCropBox(new PDRectangle(rectArray));

	    // Add pages to destination PDF
	    doc.addPage(pageTop);
	    doc.addPage(pageBottom);
	}

	// Save new PDF
	File destFile = new File(aFileName);
	doc.save( (destFile.getParent() == null ? "" : (destFile.getParent() + File.separator)) + "iLiad." + destFile.getName());

	// Recover resources
	doc.close();
	document.close();
    }

    /**
     * This class is used since PDFBox doesn't provide a shallow copy constructor for PDPage.
     * Someday it will be added and this class can disappear.
     */
    public static class PDPageWrapper extends PDPage {
	/**
	 * Makes a shallow copy of aPage into a new PDPage.
	 *
	 * @param aPage The existing PDPage
	 */
	public PDPageWrapper( PDPage aPage ) {
	    super( new COSDictionary( (COSDictionary)aPage.getCOSObject()));
	}
    }

 }