#!/usr/bin/perl -w
# 
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

use strict;
use Getopt::Std;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use Env qw(HOME PWD);

#########################
# Global Variables
#########################
my $VERSION='0.2.1';
my %Opts;
my @Imps;


##########################
# Subs
##########################

sub get_imp {
	
    if ( -d ) {
        # for directories, only print path
      	# print "$File::Find::name\n";
        return;
    }

    -f            or return;  # if not a file
    /\.imp$/ or return;  # if not the right extension

    push @Imps,catfile($File::Find::dir, $_);
    @Imps or return;
}

sub get_book_info {
	my ($impfilename) = @_;
	my $start_address = 48;
	my $next_address = 0;
	my @info;
	my $s;

	open IMPFILE, "<$impfilename";
	my $old_rs = $/;
	$/="\0";
	$next_address = $start_address;
	for (my $i = 0; $i < 7; $i++){
		seek IMPFILE, $next_address, 0;
		$s = <IMPFILE>;
		chomp $s;
		push @info,$s;
		$next_address += length($s) + 1;
	}
	close IMPFILE;
	$/ = $old_rs;
	return @info;
}

sub usage {
print "impfind version $VERSION\n";
print <<HELP;
impfind [-hRSflact] directory space seperated list of keywords to search for
impfind -i path to book 

-h help this short help list and the version
-R Recurse through subdirectories
-l list the information for all located books. Category, Title, Author
-S Sensitive to Case Searching
-f full search results the default is just to give a list of paths
-V Verbose gives lots of information very little of which is much use to 
	most people.  Used for a closer look at what is going on for debugging
-i path to a book to show information about
	If the following switches are omitted impfind will search all three
	of fields for any of the keywords.
-a Search only the Author field for the keywords
-t Search only the Title field for the keywords
-c Search only the Category field for the keywords
HELP
exit 0;
}

##########################
# Main program
##########################
my $dir;
my @keywords;
my %book_info_for;
my @fields = ("id","category","sub_category","title","author_last","author_mid","author_first");
my @search_fields = ("category","title","author_first");
my @info;
my $field;
my $keyword;
my $book;
my $tot;

# Get the directory and search keywords from the
# beginning of @ARGV


getopts('hVRSflatci:',\%Opts);
usage if defined $Opts{'h'};
if (defined $Opts{'i'}){
	if (! -f $Opts{'i'} ){
		print $Opts{'i'} . "\ndoes not exist\nexiting\n";
		exit;
	}
	if ( $Opts{'i'} !~ /\.imp$/){
		print $Opts{'i'} . "\ndoes not seem to be an imp file\nexiting\n";
		exit;
	}
	print "Information for\n";
	print $Opts{'i'} . "\n";
	my $fv;
	@info = get_book_info($Opts{'i'});
	foreach $field (@fields){
		$fv = shift @info;
		print $field . ": $fv\n";
	}
	exit;
}
$dir = shift @ARGV;
$dir = curdir() if (! defined $dir);
# Check what we have in $dir to make sure that it is a directory
# if not unshift it back to @ARGV and set $dir to the current directory
if (! -d $dir && $dir ne ''){
	unshift @ARGV,$dir;
	$dir = curdir();
}
# Get what is left in @ARGV and put it in keywords
@keywords = @ARGV;


print "\nSearching $dir\nfor files ending with .imp\n";
if (defined $Opts{'R'}){
	print "Searching through sub directories.\n";
	find( \&get_imp,$dir);
}
else {
	opendir DIR, $dir or die "Cant open $dir for read: $!\n";
	while(my $name = readdir DIR)
	{
		next if $name eq '.' || $name eq '..';
		next if -d $name;
		push @Imps,catfile($dir,$name) if $name =~ /\.imp$/ ;
	}
	close DIR;
}
if (defined $Opts{'V'}){
	foreach (@Imps){
		print "$_\n";
	}
}
$tot = @Imps;
print "Getting information for $tot Books\n";
foreach $book (@Imps) { 
#	print "$book\n";
	@info = get_book_info($book);
	foreach $field (@fields){
		$book_info_for{$book}->{$field} = shift @info ;
	}
}

if (defined $Opts{'l'}) {
	print "Information for all books found\n";
	foreach $book (keys %book_info_for){
		print "\nPath: $book\n";
		print "Category: " . $book_info_for{$book}->{'category'} . "\n";
		print "Title: " . $book_info_for{$book}->{'title'} . "\n";
		print "Author: " . $book_info_for{$book}->{'author_first'} . $book_info_for{$book}->{'author_last'} . "\n"
	}
	exit;
}

# Search fields category, title, and author_first for a list of keywords
# If switches -a -t -c are used they replace the list of search fields
# with a single value author_first title or category
my $n;
my @books_found;
my $added;
my $f;
my $k;
if(defined $Opts{'a'}){
	@search_fields = ("author_first");
	print "\nSearching only the Author Field\n";
}

if(defined $Opts{'c'}){
	@search_fields = ("category");
	print "\nSearching only the Category Field\n";
}

if(defined $Opts{'t'}){
	@search_fields = ("title");
	print "\nSearching only the Title Field\n";
}

foreach $book(keys %book_info_for){
	$added = 0;
	foreach $field (@search_fields){
		foreach $keyword(@keywords){
			if(defined $Opts{'S'}){
				#exact match between keyword and field Case sensitive
				if ($book_info_for{$book}->{$field} =~ /$keyword/){
					push @books_found,$book if ! $added;
					$added = 1;
				}
			}
			else {
				$f = lc($book_info_for{$book}->{$field});
				$k = lc($keyword);
				if ( $f =~ /$k/){
					push @books_found,$book if ! $added;
					$added = 1;
				}
			}
		}
	}
	$added = 0;
}

$tot = @books_found;

print "\nUsing Case Sensitive Searching\n" if defined $Opts{'S'};
print "Found $tot Books\n";
print "Containing @keywords\n";
foreach $book (@books_found){
	if( defined $Opts{'f'}){
		print "Path: $book\n";
		print "Category: " . $book_info_for{$book}->{'category'} . "\n";
		print "Title: " . $book_info_for{$book}->{'title'} . "\n";
		print "Author: " . $book_info_for{$book}->{'author_first'} . "\n";
	}
	else {
		print "$book\n";
	}
}
print "\n";
exit;
