#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy, traceback
from threading import Thread
from Queue import Queue

try:
    from PyQt5.Qt import (Qt, QTabWidget, QVBoxLayout, QTableWidget, QAbstractItemView,
                          QAction, QTableWidgetItem, QSplitter)
except ImportError:
    from PyQt4.Qt import (Qt, QTabWidget, QVBoxLayout, QTableWidget, QAbstractItemView,
                          QAction, QTableWidgetItem, QSplitter)

from calibre.gui2.dialogs.confirm_delete import confirm

import calibre_plugins.import_list.config as cfg
from calibre_plugins.import_list.algorithms import LibraryHashBuilder
from calibre_plugins.import_list.common_utils import get_icon
from calibre_plugins.import_list.page_common import WizardPage
from calibre_plugins.import_list.tab_clipboard import ImportClipboardTab
from calibre_plugins.import_list.tab_csv import ImportCSVTab
from calibre_plugins.import_list.tab_settings import UserSettingsTab, PredefinedSettingsTab
from calibre_plugins.import_list.tab_webpage import ImportWebPageTab

class LoadHashMapsWorker(Thread):
    '''
    Worker thread to populate our hash maps, done on a background thread
    to keep the initial dialog display responsive
    '''
    def __init__(self, db, hash_maps_queue):
        Thread.__init__(self)
        self.db = db
        self.hash_maps_queue = hash_maps_queue

    def run(self):
        try:
            builder = LibraryHashBuilder(self.db)
            self.hash_maps_queue.put(builder.hash_maps)
        except:
            traceback.print_exc()


class PreviewBookTableWidget(QTableWidget):

    def __init__(self, parent, db, columns, column_widths):
        QTableWidget.__init__(self, parent)
        self.db = db
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.verticalHeader().setDefaultSectionSize(24)

        self.populate_table(columns, [])
        if column_widths is not None:
            for c,w in enumerate(column_widths):
                self.setColumnWidth(c, w)
                
    def create_headers(self, columns):
        # Remove any columns which won't be valid as composite munging into field
        if 'series_index' in columns:
            columns.remove('series_index')
        self.columns = columns
        header_labels = []
        for column in self.columns:
            title = column
            if column in self.db.field_metadata.standard_field_keys():
                title = self.db.field_metadata[column]['name']
            elif column.lower().startswith('identifier:'):
                title = 'ID:'+column[11:]
            else:
                # TODO: Is a custom column - not currently supported
                pass
            header_labels.append(title)
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)

    def populate_table(self, columns, books):
        self.create_headers(columns)
        self.books = books
        self.setRowCount(0)
        self.setRowCount(len(books))

        for row, book in enumerate(books):
            self.populate_table_row(row, book)

    def populate_table_row(self, row, book):
        for i in xrange(0, len(self.columns)):
            self.setItem(row, i, QTableWidgetItem(book.get(self.columns[i], '')))

    def get_preview_books(self):
        for row in xrange(0, self.rowCount()):
            for i in xrange(0, len(self.columns)):
                column = self.columns[i]
                self.books[row][column] = unicode(self.item(row, i).text()).strip()
        return self.books


class ImportPage(WizardPage):

    ID = 1

    def init_controls(self):
        self.block_events = True
        self.setTitle('Step 1: Configure a list source')
        l = QVBoxLayout(self)
        self.setLayout(l)

        self.splitter = QSplitter(self)

        self.tw = QTabWidget(self)
        self.splitter.addWidget(self.tw)

        self.user_settings_tab = UserSettingsTab(self, self.library_config[cfg.KEY_SAVED_SETTINGS])
        self.predefined_settings_tab = PredefinedSettingsTab(self)
        self.clipboard_tab = ImportClipboardTab(self)
        self.csv_tab = ImportCSVTab(self)
        self.web_page_tab = ImportWebPageTab(self)
        self.tw.addTab(self.user_settings_tab, 'User Settings')
        self.tw.addTab(self.predefined_settings_tab, 'Predefined')
        self.tw.addTab(self.clipboard_tab, 'Clipboard')
        self.tw.addTab(self.csv_tab, 'CSV File')
        self.tw.addTab(self.web_page_tab, 'Web Page')
        self.tw.currentChanged[int].connect(self._current_tab_changed)

        columns = self.info['state'].get('import_preview_columns', ['title','authors','series','pubdate'])
        column_widths = self.info['state'].get('import_preview_column_widths', None)
        self.preview_table = PreviewBookTableWidget(self, self.db, columns, column_widths)
        self.splitter.addWidget(self.preview_table)
        self.splitter.setStretchFactor(0, 2)
        self.splitter.setStretchFactor(1, 1)
        self.splitter.setChildrenCollapsible(False)
        l.addWidget(self.splitter, 1)

        view_as_type = self.library_config.get(cfg.KEY_LAST_VIEW_TYPE, 'list')
        if view_as_type == 'list':
            self.predefined_settings_tab.view_list_opt.setChecked(True)
        else:
            self.predefined_settings_tab.view_category_opt.setChecked(True)
        last_user = self.library_config.get(cfg.KEY_LAST_USER_SETTING, None)
        self.user_settings_tab.select_setting(last_user)
        last_predefined = self.library_config.get(cfg.KEY_LAST_PREDEFINED_SETTING, None)
        self.predefined_settings_tab.select_setting(last_predefined)

        self.block_events = False
        self._create_context_menu()
        self._rebuild_db_hash_maps()

    def _rebuild_db_hash_maps(self):
        if 'hash_maps' in self.info:
            del self.info['hash_maps']
        self.info['hash_maps_queue'] = Queue()
        worker = LoadHashMapsWorker(self.db, self.info['hash_maps_queue'])
        worker.start()

    def _create_context_menu(self):
        table = self.preview_table
        table.setContextMenuPolicy(Qt.ActionsContextMenu)
        self.remove_book_action = QAction(get_icon('minus.png'), '&Remove book', table)
        self.remove_book_action.setToolTip('Remove selected books from the list')
        self.remove_book_action.triggered.connect(self._remove_book)
        table.addAction(self.remove_book_action)

    def _current_tab_changed(self, idx):
        if hasattr(self.tw.widget(idx), 'preview_button'):
            pb = self.tw.widget(idx).preview_button
            pb.setAutoDefault(True)
            pb.setDefault(True)

    def _remove_book(self):
        message = '<p>Are you sure you want to remove the selected books from the list?</p>'
        if not confirm(message,'import_list_delete_from_list', self):
            return
        rows = sorted(self.preview_table.selectionModel().selectedRows())
        for selrow in reversed(rows):
            self.preview_table.removeRow(selrow.row())
            self.preview_table.books.pop(selrow.row())
        self.completeChanged.emit()

    def clear_preview_books(self):
        self.refresh_preview_books(columns=['title','authors'], books=[])

    def refresh_preview_books(self, columns, books, import_type=None):
        #print('Refresh preview has columns:',columns)
        if not self.block_events:
            self.preview_table.populate_table(columns, books)
        if books:
            self.library_config[cfg.KEY_CURRENT][cfg.KEY_IMPORT_TYPE] = import_type
        self.completeChanged.emit()

    def request_load_settings(self, setting_name, is_predefined, edit_mode):
        if setting_name is not None:
            # Before we do anything further, call save_settings on tabs so that
            # the historical combos have their history stored for when repopulated
            for idx in xrange(0, self.tw.count()):
                if hasattr(self.tw.widget(idx), 'save_settings'):
                    self.tw.widget(idx).save_settings(self.library_config)

            if is_predefined:
                setting = copy.deepcopy(cfg.PREDEFINED_WEB_SETTINGS[setting_name])
                setting[cfg.KEY_IMPORT_TYPE] = cfg.KEY_IMPORT_TYPE_WEB
                setting[cfg.KEY_READING_LIST] = copy.deepcopy(cfg.DEFAULT_READING_LIST_VALUES)
            else:
                setting = self.library_config[cfg.KEY_SAVED_SETTINGS][setting_name]

            import_type = setting[cfg.KEY_IMPORT_TYPE]
            current = self.library_config[cfg.KEY_CURRENT]
            current[cfg.KEY_IMPORT_TYPE] = import_type
            current[import_type] = copy.deepcopy(setting)
            del current[import_type][cfg.KEY_IMPORT_TYPE]
            del current[import_type][cfg.KEY_READING_LIST]
            current[cfg.KEY_READING_LIST] = copy.deepcopy(setting[cfg.KEY_READING_LIST])

            related_tab_page = None
            if import_type == cfg.KEY_IMPORT_TYPE_CLIPBOARD:
                self.library_config[cfg.KEY_LAST_CLIPBOARD_SETTING] = setting_name
                related_tab_page = self.clipboard_tab
            elif import_type == cfg.KEY_IMPORT_TYPE_CSV:
                self.library_config[cfg.KEY_LAST_CSV_SETTING] = setting_name
                related_tab_page = self.csv_tab
            elif import_type == cfg.KEY_IMPORT_TYPE_WEB:
                self.library_config[cfg.KEY_LAST_WEB_SETTING] = setting_name
                related_tab_page = self.web_page_tab

            related_tab_page.restore_settings(self.library_config)

            if edit_mode:
                self.clear_preview_books()
                self.tw.setCurrentWidget(related_tab_page)
            else:
                related_tab_page._preview_rows()
            self.info['current_setting'] = '' if is_predefined else setting_name

    def get_preview_columns(self):
        return self.preview_table.columns

    def get_preview_table_column_widths(self):
        table_column_widths = []
        for c in range(0, self.preview_table.columnCount() - 1):
            table_column_widths.append(self.preview_table.columnWidth(c))
        return table_column_widths

    def isComplete(self):
        '''
        Don't allow the user onto the next wizard page without any rows of data
        or with a row that has no title
        '''
        books = self.preview_table.get_preview_books()
        if not books:
            return False
        for book in books:
            if book['title'] == '':
                return False
        return True

    def initializePage(self):
        self.clipboard_tab.restore_settings(self.library_config)
        self.csv_tab.restore_settings(self.library_config)
        self.web_page_tab.restore_settings(self.library_config)

        tab_idx = self.library_config.get(cfg.KEY_LAST_TAB, 1)
        self.tw.setCurrentIndex(tab_idx)
        self._current_tab_changed(tab_idx)
        splitter_state = self.info['state'].get('import_splitter_state', None)
        if splitter_state is not None:
            self.splitter.restoreState(splitter_state)

    def validatePage(self):
        self.clipboard_tab.save_settings(self.library_config)
        self.csv_tab.save_settings(self.library_config)
        self.web_page_tab.save_settings(self.library_config)
        self.info['books'] = self.preview_table.get_preview_books()
        self.info['book_columns'] = self.get_preview_columns()
        self.info['import_splitter_state'] = bytearray(self.splitter.saveState())
        self.library_config[cfg.KEY_LAST_TAB] = self.tw.currentIndex()
        if self.predefined_settings_tab.view_list_opt.isChecked():
            self.library_config[cfg.KEY_LAST_VIEW_TYPE] = 'list'
        else:
            self.library_config[cfg.KEY_LAST_VIEW_TYPE] = 'category'
        self.library_config[cfg.KEY_LAST_USER_SETTING] = self.user_settings_tab.get_setting_name()
        self.library_config[cfg.KEY_LAST_PREDEFINED_SETTING] = self.predefined_settings_tab.get_setting_name()
        return True
