#!/Python3/python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, division, absolute_import, print_function

__all__=["prettifyXHTMLFile", "convertXMLNotes2HTML", "applyUserOptions", "show_msgbox", "writeFile2Epub", "getUserSelectedFileName", "fileSelector", "copyRefs2SelectedSigilFile", "importMendeleyFile"]


import os, os.path, sys, shutil, inspect, re, stat
import options
from tempfile import mkdtemp 

from lxml import etree
from io import StringIO, BytesIO
import datetime

import tkinter as tk
import tkinter.messagebox as mbox
import tkinter.filedialog as fdbox

try:
    from sigil_bs4 import BeautifulSoup, Tag, NavigableString
except:
    from bs4 import BeautifulSoup, Tag, NavigableString
	
	
def prettifyXHTMLFile(wdir, file):
    """ Reformats, cleans up and prettifies the html file
        by adding appropriate vertical spacing and indents. 
    """
    print('\n>>> In prettifyXHTML...')
    output = os.path.join(wdir, 'prettify.xhtml')
    file = os.path.join(wdir, file)
    outfp = open(output, 'wt', encoding=  'utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
        
            line = line.replace('</body>', '')
            line = line.replace('</html>', '')
            if line.strip == '</body></html>':
                line = ''
        
            if 'xhtml11.dtd' in line or \
                line.strip().startswith('<?xml') or \
                line.strip().startswith('<!DOCTYPE') or \
                line.strip().startswith('<html') or \
                line.strip().startswith('</html>') or \
                line.strip().startswith('<title>') or \
                line.strip().startswith('<meta') or \
                line.strip().startswith('<head>') or \
                line.strip().startswith('</head>') or \
                line.strip().startswith('<body') or \
                line.strip().startswith('</body>'):
                outfp.write(line)
                continue      
            
            if line.strip() == '':
                continue
                
            outfp.write(line + '\n') 
            print('>>> In prettifyXHTML...' + line)
            
    outfp.write('\n\n</body>\n</html>')     
    outfp.close()
    os.remove(file)
    shutil.copy(output, file)
    return(0)	    
   
def convertXMLNotes2HTML(wdir, xml_file):
    print('>>> In convertXMLNotes2HTML...') 
    """ Reformats the exported Medeley xml file:
        * formats all refs to the chosen CMS style
        * formats all entries according to ref source type.
        * converts Mendeley xml format to HTML format.
    """
    # remove '-' for easier handling later using bs4
    output = os.path.join(wdir, 'change.xml')
    xml_file = os.path.join(wdir, xml_file)
    outfp = open(output, 'wt', encoding=  'utf-8')
    with open(xml_file, 'rt', encoding='utf-8') as infp:
        for line in infp:
                line = line.replace('<ref-type name', '<reftype name')
                line = line.replace('</ref-type>', '</reftype>')
                line = line.replace('<pub-location>', '<publocation>')
                line = line.replace('</pub-location>', '</publocation>')
                outfp.write(line)       
    outfp.close()
    os.remove(xml_file)            
    os.rename(output, xml_file)
    
    
    # convert to html and apply CMS styling 
    output = os.path.join(wdir, 'format_xml.html')
    #xml_file = os.path.join(wdir, xml_file)
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(xml_file, 'rt', encoding='utf-8').read()
    
    soup = BeautifulSoup(html, 'lxml')
    
    # formate all notes refs in CMS style for display
    for xml_record in soup.xml.records.find_all('record'):    
        
        note_data = dict()      
        source_type = str()
        
        # get the reference source for 
        # each record - to be used later. 
        reftag = xml_record.find('reftype')
        print('\n>>> REFTAG...' + str(reftag) + '\n')
        if reftag.has_attr('name') != None:
            print('\n>>> NAME value...' + reftag['name'] + '\n')
            if reftag['name'] == 'Journal Article':
                source_type = 'Journal'
            elif reftag['name'] == 'Web Page':
                source_type = 'Web'
            elif reftag['name'] == 'Book':                
                source_type = 'Book'
            else:
                msg = "You can only use the following ref source types with this plugin: 'Journal Article'," + \
                      "'Web Page' or 'Book'. Please try again."
                print('\n>>> Error:' + msg + '\n\n>>> Abort')         
                show_msgbox('Incorrect Ref Source Type', msg, msgtype='error')
                options.SYS_EXIT = True
                shutil.rmtree(wdir, ignore_errors=True)  
                return(-1)                
        
        print('\n>>> XML RECORD...' + str(xml_record))
        print('\n>>> SOURCE_TYPE...' + source_type + '\n')            
            
        # save and store each piece of data for later CSM and source styling.        
        if xml_record.find('author') != None and str(xml_record.find('author').string) != None:
            for author in xml_record.find_all('author'):
                note_data['authors'] =  str(author.string).strip() 
        else:
            note_data['authors'] = 'None'        
        
        if xml_record.find('title') != None and xml_record.title.string != None:
            note_data['title'] = str(xml_record.find('title').string)           
        else:
            note_data['title'] = 'None'        
            
        if xml_record.find('full-title') != None and xml_record.find('full-title').string != None:
            note_data['publication'] = str(xml_record.find('full-title').string)
        else:
            note_data['publication'] = 'None'        
        
        if xml_record.find('year') != None and xml_record.year.string != None:
            note_data['year'] = str(xml_record.find('year').string)
        else:
            note_data['year'] = ' n.d.'        
        
        if xml_record.find('publocation') != None and xml_record.publocation.string != None:
            note_data['pub_loc'] = str(xml_record.find('publocation').string)
        else:
            note_data['pub_loc'] = 'None'
        
        if xml_record.find('publisher') != None and xml_record.publisher.string != None:
            note_data['publisher'] = str(xml_record.find('publisher').string)
        else:
            note_data['publisher'] = 'None'        
        
        if xml_record.find('volume') != None and xml_record.volume.string != None:
            note_data['volume'] = str(xml_record.find('volume').string)
        else:
            note_data['volume'] = ''        
        
        if xml_record.find('issue') != None and xml_record.issue.string != None:
            note_data['issue'] = str(xml_record.find('issue').string)
        else:    
            note_data['issue'] = ''    
           
        if xml_record.find('pages') != None and xml_record.pages.string != None:
            note_data['pages'] = str(xml_record.find('pages').string)
        else:
            note_data['pages'] = ''

        today = datetime.datetime.today()     
        note_data['access_date'] = 'Accessed ' + today.strftime('%B %d, %Y') 
        print('\n>>> ACCESS DATE...' + note_data['access_date'] + '\n')        
        
        if xml_record.find('url') != None and xml_record.url.string != None:
            note_data['url'] = str(xml_record.find('url').string)
            note_data['url'] = '<a href="' + note_data['url'] + '"> ' + note_data['url'] + '</a>'
        else:
            note_data['url'] = ''    
       
        
        # use CMS author-date style
        if options.AUTHOR_DATE_STYLE == True:
            
            if source_type == 'Journal':
                ref_data = ' ' + note_data['authors'] + '.'
                ref_data += ' ' + note_data['year'] + '. '
                ref_data += '"' + note_data['title'] + '."'
                ref_data += ' <i>' + note_data['publication'] + '</i>'
                
                if note_data['volume'] != '':
                    ref_data += ' ' + note_data['volume'] + ','
                else:
                    ref_data += ''                 
                
                if note_data['issue'] != '':
                    ref_data += ' (' + note_data['issue'] +')'     
                else:
                    ref_data += ''
                
                if note_data['pages'] != '':    
                    ref_data += ': ' + note_data['pages'] + '.'
                else:
                    ref_data += ''           
                    
                ref_data += ' ' + note_data['url'] 
            
            elif source_type == 'Web':
                print('\n>>> Begin "Web"...')
                ref_data = ' ' + note_data['authors'] + '.'
                ref_data += ' ' + note_data['year'] + '. '
                ref_data += ' "' + note_data['title'] + '."'
                ref_data += ' <i>' + note_data['publication'] + '.</i>'
                ref_data += ' ' + note_data['access_date'] + '.'
                ref_data += ' ' + str(note_data['url']).strip()
                print('\n>>> End of "Web" - line - ' + ref_data + '\n')

            elif source_type == 'Book':  
                ref_data = ' ' + note_data['authors'] + '.' 
                ref_data += ' ' + note_data['year'] + '.'
                ref_data += ' <i>' + note_data['title'] + '</i>.'
                ref_data += ' ' + note_data['pub_loc'] + ':'
                ref_data += ' ' + note_data['publisher'] + '.'
                ref_data += ' ' + note_data['url'] + '.'   
 
        # use CMS fullnote format    
        if options.FULLNOTE_STYLE == True:
            
            if source_type == 'Journal':
                ref_data = ' ' + note_data['authors'] + ',' 
                ref_data += ' "' + note_data['title'] + '."'
                ref_data += ' <i>' + note_data['publication'] + ',</i>'
                
                if note_data['volume'] != '':
                    ref_data += ' ' + note_data['volume'] + ','
                else:
                    ref_data += ''                 
                
                if note_data['issue'] != '':
                    ref_data += ' no. ' + note_data['issue']     
                else:
                    ref_data += ''
                    
                if note_data['year'] != '':
                    ref_data += ' (' + note_data['year'] + ')'                
                else: 
                    ref_data += ''
                    
                if note_data['pages'] != '':    
                    ref_data += ': ' + note_data['pages'] + '.'
                else:
                    ref_data += ''           
                    
                ref_data += note_data['url']      
            
            elif source_type == 'Web':
                print('\n>>> Begin "Web"...')
                ref_data = ' ' + note_data['authors'] + ','
                ref_data += ' "' + note_data['title'] + '."'
                ref_data += ' <i>' + note_data['publication'] + '.</i>'
                ref_data += ' ' + note_data['access_date'] + '.'
                ref_data += ' ' + str(note_data['url']).strip()
                print('\n>>> End of "Web" - line - ' + ref_data + '\n')             
                
            elif source_type == 'Book':  
                ref_data = ' ' + note_data['authors'] + '.' 
                ref_data += ' <i>' + note_data['title'] + '</i>.'
                ref_data += ' (' + note_data['pub_loc'] + ':'
                ref_data += ' ' + note_data['publisher'] + ','
                ref_data += ' ' + note_data['year'] +').' 
                ref_data += ' ' + note_data['url']                   
                 
                
        # use CMS concise note format    
        if options.NOTE_STYLE == True:
            
            if source_type== 'Journal':
                ref_data = ' ' + note_data['authors'] + '.' 
                ref_data += ' "' + note_data['title'] + '."'             
                ref_data += ' ' + note_data['url']    
            
            elif source_type == 'Web':
                ref_data = ' ' + note_data['authors'] + '.'
                ref_data += ' "' + note_data['title'] + '."'
                ref_data += ' ' + note_data['access_date'] + '.'
                ref_data += ' ' + str(note_data['url']).strip()

            if source_type == 'Book':            
                ref_data = ' ' + note_data['authors'] + '.' 
                ref_data += ' <i>' + note_data['title'] + '</i>.'
                if note_data['pages'] != '':
                    ref_data += ' ' + note_data['pages'] + '.'
                else:
                    ref_data = ''    
                ref_data += ' ' + note_data['url']                   
                 
                
        del note_data 
        print('>>> To html...' + '<p>'+ ref_data + '</p>\n\n')
        # convert all html entities to unicode chars.   
        outfp.write('<p>' + ref_data + '</p>\n\n')        
        
    outfp.close()
    os.remove(xml_file)
    shutil.copy(output, xml_file)
    return(0)    
    
def applyUserOptions(wdir, file):
    print('>>> In applyUserOptions') 
    file = os.path.join(wdir, file)
    output = os.path.join(wdir, 'format_refs.xhtml')
    outfp = open(output, 'w', encoding=  'utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        
        numbers = 0
        for line in infp:
            
            if '<p' in line:
                numbers += 1
                
                soup = BeautifulSoup(line, 'html.parser')
                
                if soup.p != None:
                    print('\n>>> In p tagformatting...') 
                    soup.p['style'] = 'text-indent: 0;margin-left: 1em;margin-top: 0.8333em;'
                    if options.USE_SMALL_FONT == True: 
                        soup.p['style'] = soup.p['style'] + 'font-size: 0.875em;'
                    if options.USE_TEXT_JUSTIFY == True:
                        soup.p['style'] = soup.p['style'] + 'text-align: justify;'                
                    line = str(soup)           
                
                # remove or convert italics
                if '<i>' in line: 
                    if options.REMOVE_ITALICS == False:   # reformat italics using spans       
                        itag = soup.find('i')
                        print('\n>>> Replace <i> with spans...')
                        itag.name = 'span'
                        itag['style'] = 'font-style: italic'
                        line = str(soup)
                    else:
                        print('\n>>> Removing <i> tags...')
                        line = line.replace('<i>', '')
                        line = line.replace('</i>', '')    

                # remove all hyperlink formatting
                if options.REMOVE_HYPERLINK_FORMAT == True:    
                    if '<a' in line and 'href' in line:
                        print('\n>>> Removing link formatting...')
                        atag = soup.find('a')
                        atag.attrs = {}
                        atag.unwrap()
                        line = str(soup)                   
                
                # delete all hyperlinks 
                if options.DELETE_HYPERLINKS == True:        
                    if '<a' in line and 'href' in line:
                        print('\n>>> Delete links...')
                        atag = soup.find('a')
                        atag.clear()
                        atag.decompose()
                        line = str(soup)          
                
                # for anchor tag strings only, convert all  
                # relevant url chars to html entities
                if '<a' in line and 'href=' in line:
                    a_tag = soup.find('a')
                    astring = a_tag.string
                    a_tag.string = astring.replace(r'&', r'&amp;')
                    line = str(soup)
                    
                # add consecutive numbering to all ref/notes    
                if options.ADD_NUMBERING == True:
                    print('\n>>> In Numbering -- line...' + line)
                    pstring = soup.p.get_text()
                    pstring = str(numbers) + '. ' + pstring
                    soup.p.string = pstring
                    line = str(soup)                    
                    
                # add consecutive superscript numbering         
                if options.ADD_SUPER_NUMBERING == True:
                    soup.p.insert(0, soup.new_tag('sup'))
                    soup.p.sup['style'] = 'font-size: 0.75em;'
                    soup.p.sup.string = str(numbers) + ' '
                    line = str(soup)                        
                    
                # add superscript numbering with brackets          
                if options.ADD_SUPER_WITH_BRACKETS == True:
                    soup.p.insert(0, soup.new_tag('sup'))
                    soup.p.sup['style'] = 'font-size: 0.75em;'
                    soup.p.sup.string = '[' + str(numbers) + ']' 
                    line = str(soup)                            
                    
                print('>>> In Mendeley format refs - Write...' + line + ' Numbering:' + str(numbers))           
                outfp.write(line)     
            
    outfp.close()
    os.remove(file)
    shutil.copy(output, file)
    return(0)    
                                
def show_msgbox(title, msg, msgtype='info'):
    """ For general information, warnings and errors
    """
    localRoot = tk.Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Helvetica -12')
    localRoot.quit()
    if msgtype == 'info':
        return(mbox.showinfo(title, msg))
    elif msgtype == 'warning':
        return(mbox.showwarning(title, msg))
    elif msgtype == 'error':
        return(mbox.showerror(title, msg))
    
def writeFile2Epub(bk, wdir, s_id, file): 
    print('\n>>> In write2Epub id, file...' + s_id + ' ' + file)
    file = os.path.join(wdir, file)
    with open(file, 'rt', encoding='utf-8') as fp:
        data = fp.read()           
        bk.writefile(s_id, data)
        
    return(0)    
 
def getUserSelectedFileName(bk, wdir):
    """ Obtains the name of the file selected by the 
        user in Sigil's Book Browser, runs checks and  
        then copies it to the work dir.
    """
    # get user selected file
    files = []
    id = ''
    count= 0
    for (id_type, ident) in bk.selected_iter():
        print('>>> In selected file, id is ' + ident)
        count += 1
        href = bk.id_to_href(ident)
        files.append(bk.href_to_basename(href))
        
    # check if too many files were selected.    
    if len(files) > 1:
       msg = 'Too many files selected!! You must select only one file at a time to add specific endnotes to that particular file. Please try again'
       print('\n>>> Critical Error: ' + msg + '\n\nAbort plugin...')
       show_msgbox('File Selection Error', msg, msgtype='error')
       options.SYS_EXIT = True
       shutil.rmtree(wdir, ignore_errors=True)
       return(-1)               
    
    # check that a file has been selected else error exit
    try:
        file = options.SELECTED_SIGIL_FILE = files[0]
    except:
        msg = "You haven't selected a file from the 'Text' directory of Sigil's Book Browser. Please try again"
        print('>>> Critical Error: ' + msg + "\n\nAbort plugin...\n\n")
        show_msgbox('File Selection Error', msg, msgtype='error')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(-1) 

    # check selected file type    
    if not file.endswith('.xhtml') and not file.endswith('.html'):
        msg = "Your selected Sigil file is the wrong file type. You should select only files that have the '.xhtml' or '.html' extension from the 'Text' directory in Sigil's Book Browser. Please try again"
        print('>>> Critical Error: ' + msg + "\n\nAbort plugin...\n\n")
        show_msgbox('Incorrect File Type', msg, msgtype='error')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(-1)  
    
    # copy selected file to the work dir
    file = os.path.join(wdir, file)
    with open(file, 'wt', encoding='utf-8') as outfp:
        data = bk.readfile(ident)
        outfp.writelines(data)
       
    return(ident, file) 
    
def fileSelector(bk):
    """ Gets the file path of the Zotero html file 
        containing the ref items
    """
    prefs = bk.getPrefs()
    localRoot = tk.Tk()
    localRoot.withdraw()
    
    # gratefully borrowed from Kevin's code
    if sys.platform.startswith('darwin'):
        # localRoot is is an empty topmost root window that is hidden by withdrawing it
        # but localRoot needs to be centred, and lifted and focus_force used
        # so that its child dialog will inherit focus upon launch
        localRoot.overrideredirect(True)
        # center on screen but make size 0 to hide the empty localRoot
        w = localRoot.winfo_screenwidth()
        h = localRoot.winfo_screenheight()
        x = int(w/2)
        y = int(h/2)
        localRoot.geometry('%dx%d+%d+%d' % (0, 0, x, y))
        localRoot.deiconify()
        localRoot.lift()
        localRoot.focus_force()
        
    file_opt = {}
    file_opt['parent'] = localRoot
    file_opt['title']= 'Select Mendeley Export File'
    file_opt['defaultextension'] = '.xml'
    file_opt['initialdir'] = prefs['user_dir_path']
    file_opt['multiple'] = False
    file_opt['filetypes'] = [('Mendeley export files', ('*.xml'))]
    
    filename = fdbox.askopenfilename(**file_opt)
    localRoot.quit()
    return(filename)
    
def copyRefs2SelectedSigilFile(wdir, mend_file, sig_file):
    """ Copies Zotero note/ref items to the selected 
        Sigil file. Also ensures that the copy will work for 
        a sig_file with or without a div section.
    """
        
    print('\n>>> In copyRefs2SelectedSigilFile...')
    mend_file = os.path.join(wdir, mend_file)
    sig_file = os.path.join(wdir, sig_file)

    # append the zotero data to the sig data file
    output = os.path.join(wdir, 'temp.xhtml')  
    outfp = open(output, 'w', encoding='utf-8')
    mend_data = open(mend_file, 'rt', encoding='utf-8').read()
    sig_data = open(sig_file, 'rt', encoding='utf-8').read() 
        
    outfp.writelines(sig_data + '\n\n' + mend_data)
    
    outfp.close() 
    os.remove(mend_file)
    os.remove(sig_file)
    shutil.copy(output, sig_file)
    
    # remove all </body> and </html> tags and re-insert properly at eof
    output = os.path.join(wdir, 'sig_temp.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(sig_file, 'rt', encoding ='utf-8') as infp:
        for line in infp:
            
            if  line.strip().startswith('</body>') or \
                line.strip().startswith('</html>'):
                line = ''   
            outfp.write(line) 

    outfp.write('\n\n</body>\n</html>')
    outfp.close()
    os.remove(sig_file)
    os.rename(output, sig_file)  
    
def importMendeleyFile(bk, wdir):
    """ Copies the Zotero export file to the work 
        dir and save the user dir path to prefs
    """
    prefs = bk.getPrefs()
    mend_file_path = fileSelector(bk)
    mend_file = os.path.basename(mend_file_path)
    new_mend_file = ''
    # if Mendeley file path does not exist - exit
    if mend_file_path == '' or not os.path.exists(mend_file_path):
        print('\n>>> No input file selected!\n\nAbort plugin...')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(0)
    else:#copy Mendeley file to work dir 
        new_mend_file = os.path.join(wdir, 'Mendeley_Refs.html')
        shutil.copy(mend_file_path, new_mend_file)
        folder, _ = os.path.split(mend_file_path)    
        if folder != "":
            prefs['user_dir_path'] = folder
            bk.savePrefs(prefs)
    return(os.path.basename(new_mend_file))            
    