package csv;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public final class CSV {
	public static String toXML(List<String> inputLines, String delim) {
		List<String> header = new ArrayList<>(Arrays.asList(inputLines.get(0).split(delim)));
		String output = "<lines>" + inputLines.stream().skip(1).map(line -> {
			List<String> cells = new ArrayList<String>(Arrays.asList(CSV.returnRow(line, true)));
			return "<line>" + IntStream.range(0, cells.size())
					.mapToObj(i -> "<" + header.get(i) + ">" + cells.get(i) + "</" + header.get(i) + ">")
					.collect(Collectors.joining()) + "</line>" + "\r\n";
		}).collect(Collectors.joining(System.lineSeparator())).replaceAll("&", "&amp;") + "</lines>";
		return "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + System.lineSeparator() + output + System.lineSeparator();
	}

	private final static char QUOTE_CHAR = '"';
	private final static char ESCAPE_CHAR = '\\';
	private final static char SEPARATOR = ',';

	private static char unescape(char c) {
		switch (c) {
		case 'r':
			return '\r';
		case 't':
			return '\t';
		case 'n':
			return '\n';
		default:
			// Handle simple escapes. We could go further and allow arbitrary
			// numeric wchars by
			// testing for numeric sequences here but that is beyond the scope
			// of this app.
			return c;
		}
	}

	//
	// This CSV parser is not a complete parser, but it will parse files
	// exported by older
	// versions. At some stage in the future it would be good to allow full CSV
	// export
	// and import to allow for escape('\') chars so that cr/lf can be preserved.
	//
	public static String[] returnRow(String row, boolean fullEscaping) {
		// Need to handle double quotes etc
		int pos = 0; // Current position
		boolean inQuote = false; // In a quoted string
		boolean inEsc = false; // Found an escape char
		char c; // 'Current' char
		row = row.replaceAll("<", "&lt;");
		row = row.replaceAll(">", "&gt;");
		char next // 'Next' char
				= (row.length() > 0) ? row.charAt(0) : '\0';
		int endPos // Last position in row
				= row.length() - 1;
		ArrayList<String> fields // Array of fields found in row
				= new ArrayList<String>();

		StringBuilder bld // Temp. storage for current field
				= new StringBuilder();

		while (next != '\0') {
			// Get current and next char
			c = next;
			next = (pos < endPos) ? row.charAt(pos + 1) : '\0';

			// If we are 'escaped', just append the char, handling special cases
			if (inEsc) {
				bld.append(unescape(c));
				inEsc = false;
			} else if (inQuote) {
				switch (c) {
				case QUOTE_CHAR:
					if (next == QUOTE_CHAR) {
						// Double-quote: Advance one more and append a single
						// quote
						pos++;
						next = (pos < endPos) ? row.charAt(pos + 1) : '\0';
						bld.append(c);
					} else {
						// Leave the quote
						inQuote = false;
					}
					break;
				case ESCAPE_CHAR:
					if (fullEscaping)
						inEsc = true;
					else
						bld.append(c);
					break;
				default:
					bld.append(c);
					break;
				}
			} else {
				// This is just a raw string; no escape or quote active.
				// Ignore leading space.
				if ((c == ' ' || c == '\t') && bld.length() == 0) {
					// Skip leading white space
				} else {
					switch (c) {
					case QUOTE_CHAR:
						if (bld.length() > 0) {
							// Fields with quotes MUST be quoted...
							throw new IllegalArgumentException();
						} else {
							inQuote = true;
						}
						break;
					case ESCAPE_CHAR:
						if (fullEscaping)
							inEsc = true;
						else
							bld.append(c);
						break;
					case SEPARATOR:
						// Add this field and reset it.
						fields.add(bld.toString());
						bld = new StringBuilder();
						break;
					default:
						// Just append the char
						bld.append(c);
						break;
					}
				}
			}
			pos++;
		}
		;

		// Add the remaining chunk
		fields.add(bld.toString());

		// Return the result as a String[].
		String[] imported = new String[fields.size()];
		fields.toArray(imported);

		return imported;
	}

}