#include "StdAfx.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <windows.h>
#include <setupapi.h>
#include <basetyps.h>
#include <initguid.h>

#include "ebookcomm.h"

DEFINE_GUID (GUID_REB1100,
	     0x3bb1df43, 0xb2d4, 0x11d3,
	     0xbf, 0x85, 0x0, 0x10, 0x5a, 0x0a, 0x47, 0xb3);

char * szFTypes[rebMaxFile] = 
{
	"",
	"RB",
	"INI",
	"---",
	"INF",
	"RI",
	"AN",
	"XML",
	"DB",
	"??",
	"HTM"
};

char * eBookComm::specFileName[REB_TREMAX] = 
{
	NULL,
	"__DIR",
	"__SMDIR"
};

int flds[tokenMax] = 
{
	9, // short name
	4, // extension
	7, // attributes
	9, // file size
	MAX_REB_LONG_FILENAME_LEN,
};

rebFile::rebFile (eBookComm * ebc, rebDirectory * dirP,
				  char * shortNm, char * fTp, char * longNm, int sz, char * attr)
                  :ebcP(ebc),rebProperty (shortNm, attr) 
{
	size = sz;
	myDirP = dirP;
	if (myDirP != NULL)
		myDirP->files->insertHead (this);

	if (longNm)
	   strncpy (longName, longNm, sizeof (longName));
	else
		longName[0] = '\0';
	
	if (fTp)
		strncpy (fType, fTp, sizeof (fType));
	else
		fType[0] = '\0';

	titleP = NULL;

	for (int i=rebRbFile; i < rebMaxFile; i++)
	{
		if (!strcmp (szFTypes[i], fType))
		{
			type = (rebFileT) i;
			break;
		}
	}

	if (i >= rebMaxFile)
		type = rebOtherFile;
}

rebDirectory::rebDirectory (eBookComm * ebc, rebDirectory * dirP,
							char * shortNm, char * longNm, char * attr, int tp)
                            :rebFile (ebc, dirP, shortNm, NULL, longNm, 0, attr)
{
	type    = rebDirFile;
	memType = tp;
	files   = new List<rebFile *> (false);
}

rebDirectory::~rebDirectory ()
{
	if (files)
	{
		removeAll ();
		delete files;
	}
}

// clean all files and directories
int rebDirectory::removeAll(void)
{
	Node<rebFile *> * remove = files;
	rebFile * fP;

	while (remove = remove->getNext ())
	{
		fP = remove->getKey ();
		if (fP->isDirectory())
			delete ((rebDirectory *) fP);
		else
			delete fP;
	}

	files->removeAll ();

	return 0;
}

int rebDirectory::refresh () 
{
	return ebcP->eBookGetDirTree (memType, this); 
}

rebTitle::rebTitle (char * origDescr)
{
	fileP = NULL;
	
	title[0] = '\0';
	filename[0] = '\0';
	size = 0;
	anno = false;
	publisher[0] = '\0';
	author[0] = '\0';
	isbn[0] = '\0';
	url[0] = '\0';
	username[0] = '\0';

	char * paramP,
	     * valP,
	     * quoteP,
	     * nextEqP, 
		 * nextParmP = NULL;

	size_t len;

	// strtok (origDescr, "="); // Let's take out the first token
	
	nextEqP = strchr (origDescr, '=');
	if (nextEqP)
		nextParmP = nextEqP + 1;

	while (nextParmP) 
	{
		paramP = strtok (nextParmP, "=");
		if (paramP)
		{
			len = strlen (paramP);
			valP = &paramP[len+1];
			nextParmP = NULL;
			nextEqP = valP;

retry_param:
			nextEqP = strchr (nextEqP, '=');

			if (nextEqP)
			{
				if (nextEqP[1] != '\"') // This is not a parameter...
					goto retry_param;

				for (quoteP=nextEqP; 
					(quoteP>=valP) && (*quoteP != ' ');
					quoteP--)
					;

				if (*quoteP == ' ')
				{
					*quoteP = '\0';
					nextParmP = quoteP + 1;
				}
			}
			else
			{
				for (quoteP=valP; *quoteP; quoteP++)
					;
			}
			
			//--------------------------------------------------------------
			// The quoteP should point to a quote (the last closing
			// quote of this parameter's value). Unfortunately, the bugs
			// in REB1100 software allowed presence of quotes inside the
			// quoted parameter values, so we need to be looking for the
			// next parameter first, and then go back to find the end
			// of the previous value. I didn't see equal signes inside the
			// quotes yet, so I hope this should do.
			//--------------------------------------------------------------
			quoteP--;
			if (*quoteP != '\"')
				break;

			if (!strcmp (valP, "\"\""))
				valP = "";
			else if (*valP == '\"')
			{
				valP++;
				* quoteP = '\0';
			}
				
			if (*valP)
			{
				if (!stricmp (paramP, "title"))
					strncpy (title, valP, sizeof (title));
				else if (!stricmp (paramP, "filename"))
					strncpy (filename, valP, sizeof (filename));
				else if (!stricmp (paramP, "publisher"))
					strncpy (publisher, valP, sizeof (publisher));
				else if (!stricmp (paramP, "author"))
					strncpy (author, valP, sizeof (author));
				else if (!stricmp (paramP, "isbn"))
					strncpy (isbn, valP, sizeof (isbn));
				else if (!stricmp (paramP, "url"))
					strncpy (url, valP, sizeof (url));
				else if (!stricmp (paramP, "username"))
					strncpy (username, valP, sizeof (username));
				else if (!stricmp (paramP, "size"))
					size = atoi (valP);
				else if (!stricmp (paramP, "ANNO"))
				{
					if (valP[0] == 'n')
						anno = false;
					else
						anno = true;
				}
			}
		}
	}
}

rebProperty::rebProperty (char * nm, char * val)
:valCnt(0) 
{
	setName  (nm);
	setValue (val, 0);
	for (int cnt=1; cnt < MAX_REB_PROP_CNT; cnt++)
		setValue (NULL, cnt);
}

rebProperty::rebProperty (char * origDescr)
{
	char * valP;

	valCnt = 0;
	if (strtok (origDescr, "="))
	{
		valP = strtok (NULL, "\r\n");
		if (valP != NULL)
		{
			valP = strtok (valP, "\"");
			while (valP)
			{
				if (strcmp (valP, " ")) // Not just a space
				{
					setValue (valP, valCnt++);
				}
				valP = strtok (NULL, "\"\r");
				if (valCnt >= MAX_REB_PROP_CNT)
					break;
			}
		}

		setName  (origDescr);
	}
	else
	{
		setName (NULL);
	}

	for (int cnt=valCnt;cnt < MAX_REB_PROP_CNT; cnt++)
		setValue (NULL, cnt);
}

eBookComm::eBookComm(void)
: eBookConnected(false)
{
	hDev   = INVALID_HANDLE_VALUE;

	props  = new List<rebProperty *> (true);
	titles = new List<rebTitle *>    (true);

	for (int i=0; i < REB_TREMAX; i++)
	{
		roots[i] = new rebDirectory (this, NULL, "", "", "-D---", i);
		iFiles[i] = 0;
		iDirs[i]  = 0;
	}

	setDisconnState ();
}

eBookComm::~eBookComm(void)
{
	if (props)
		delete props;

	if (titles)
		delete titles;

	for (int i=0; i < REB_TREMAX; i++)
	{
		if (roots[i])
			delete roots[i];
	}
}

int eBookComm::eBookOpen(void)
{
	HDEVINFO hdi = INVALID_HANDLE_VALUE;
	PSP_INTERFACE_DEVICE_DETAIL_DATA piddd;
	SP_INTERFACE_DEVICE_DATA sidd;
	int len, newlen;

	if (!eBookConnected)
	{
		hdi = SetupDiGetClassDevs (&GUID_REB1100,
						NULL, NULL,
						DIGCF_PRESENT | DIGCF_INTERFACEDEVICE);

		if (hdi == INVALID_HANDLE_VALUE)
		{
#ifdef REB_DEBUG
			printf ("SetupDiGetClassDevs error %d\n",
				GetLastError ());
#endif
			return (FAILURE);
		}

		sidd.cbSize = sizeof (SP_INTERFACE_DEVICE_DATA);
		if (!SetupDiEnumDeviceInterfaces (hdi, 0, &GUID_REB1100, 0, &sidd))
		{
			if (GetLastError () != ERROR_NO_MORE_ITEMS)
			{
#ifdef REB_DEBUG
				printf ("SetupDiEnumDeviceInterfaces error %d\n",
					GetLastError ());
#endif
				return (FAILURE);
			}
#ifdef REB_DEBUG
			fprintf (stderr, "ERROR: UNABLE TO LOCATE REB-1100 UNIT\n");
#endif
			return (FAILURE);
		}

		SetupDiGetInterfaceDeviceDetail (hdi, &sidd, NULL, 0, (PDWORD) &len, NULL);
		if (!(piddd = (PSP_INTERFACE_DEVICE_DETAIL_DATA) malloc (len)))
		{
#ifdef REB_DEBUG
			printf ("malloc error %d\n",
				GetLastError ());
#endif
			return (FAILURE);
		}

		piddd->cbSize = sizeof (SP_INTERFACE_DEVICE_DETAIL_DATA);
		if (!SetupDiGetInterfaceDeviceDetail (hdi, &sidd, piddd,
							len, (PDWORD) &newlen, NULL))
		{
#ifdef REB_DEBUG
			printf ("SetupDiGetInterfaceDeviceDetail error %d\n",
				GetLastError ());
#endif
			free (piddd);
			return (FAILURE);
		}

#ifdef REB_DEBUG
		printf ("LOCATED DEVICE AT: %s\n", piddd->DevicePath);
#endif
		strcpy (eBookName, piddd->DevicePath);
		
		SetupDiDestroyDeviceInfoList (hdi);
		free (piddd);
		
		eBookConnected = true;
	}

	if (hDev == INVALID_HANDLE_VALUE)
	{
		hDev = CreateFile (eBookName,
				GENERIC_READ | GENERIC_WRITE,
				FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0,
				NULL);
	}

	if (hDev == INVALID_HANDLE_VALUE)
	{
#ifdef REB_DEBUG
		printf ("CreateFile error %d\n", GetLastError ());
#endif
		setDisconnState ();
		return (FAILURE);
	}

	if (!DeviceIoControl (hDev, IOCTL_RocketReadSerialNumber, NULL, 0,
				sernum, sizeof (sernum), (LPDWORD) &newlen, NULL))
	{
#ifdef REB_DEBUG
		printf ("DeviceIoControl error %d\n", GetLastError ());
#endif
		setDisconnState ();
		return (FAILURE);
	}

#ifdef REB_DEBUG
	fprintf (stderr, "REB-1100 SERIAL NUMBER IS: %.32s\n", sernum);
#endif

	return (SUCCESS);
}

int eBookComm::eBookClose(bool bForce)
{
	if (bForce)
	{
		if (hDev != INVALID_HANDLE_VALUE)
		{
			CloseHandle (hDev);
			hDev = INVALID_HANDLE_VALUE;
		}
	}

	return (SUCCESS);
}

// // Read the file into the local buffer. return bufSize and data in buffer...
int eBookComm::eReadImmediate(char * szRebName, char * buffer, int * bufSize)
{
	int haveRead = 0;
	long int total = 0;
	int toRead = REB_BLOCK_SZ;

	eBookOpen ();

	if (hDev == INVALID_HANDLE_VALUE)
		return (FAILURE);

	if ((* bufSize) < toRead)
		toRead = * bufSize;

	if (DeviceIoControl (hDev, IOCTL_RocketReadInitiate, 
		szRebName, (int) strlen (szRebName) + 1,
		buffer, toRead, (LPDWORD) &haveRead, NULL) < 0) 
	{
#ifdef REB_DEBUG
		fprintf (stderr, "UNEXPECTED IOCTL ERROR #%d\n", GetLastError ());
#endif
		// there was an error, chances are the REB went offline
		setDisconnState ();
		return (FAILURE);
	}

	if (!haveRead) 
	{
#ifdef REB_DEBUG
		fprintf (stderr, "IOCTL RETURNED %d bytes \n", haveRead);
#endif
		eBookClose ();
		return (FAILURE);
	}

	total = haveRead;

	while ((haveRead == REB_BLOCK_SZ) && (total < * bufSize))
	{
		if ((total + REB_BLOCK_SZ) > (* bufSize))
			toRead = total - (* bufSize);
		else
			toRead = REB_BLOCK_SZ;

#ifdef REB_DEBUG
		printf ("%d OCTETS READ.\n", total);
#endif
		if (DeviceIoControl (hDev, IOCTL_RocketReadContinue, 
			NULL, 0, buffer+total, toRead, (LPDWORD) &haveRead, NULL) < 0)
		{
#ifdef REB_DEBUG
			fprintf (stderr, "UNEXPECTED IOCTL ERROR #%d\n", GetLastError ());
#endif
			// there was an error, chances are the REB went offline
			setDisconnState ();
			return (FAILURE);
		}

		total += haveRead;
	}
	
	eBookClose ();
	* bufSize = total;

	return (SUCCESS);
}

int eBookComm::eReadFile(char * szRebName, char * szPcName, int sz, PctDoneFnc * fnc)
{
	DWORD haveRead, haveWritten;
	long int total = 0;
	int pctRead = 0;
	HANDLE hPcFile;

	eBookOpen ();

	if (hDev == INVALID_HANDLE_VALUE)
		return (FAILURE);

    hPcFile       = CreateFile (szPcName,
                                GENERIC_WRITE,
                                FILE_SHARE_READ,
                                NULL,
                                CREATE_ALWAYS,
                                FILE_ATTRIBUTE_NORMAL,
                                NULL);

    if (hPcFile == INVALID_HANDLE_VALUE)
    {
#ifdef REB_DEBUG
		fprintf (stderr, "Failure opening file %s\n", szPcName);
#endif
		eBookClose ();
		return (FAILURE);
    }

	if (DeviceIoControl (hDev, IOCTL_RocketReadInitiate, 
		szRebName, (int) strlen (szRebName) + 1,
		internalBuffer, REB_BLOCK_SZ, (LPDWORD) &haveRead, NULL) < 0) 
	{
#ifdef REB_DEBUG
		fprintf (stderr, "UNEXPECTED IOCTL ERROR #%d\n", GetLastError ());
#endif
		// there was an error, chances are the REB went offline
		setDisconnState ();
		CloseHandle (hPcFile);
		// DeleteFile (szPcName);
		return (FAILURE);
	}

	if (!haveRead) 
	{
#ifdef REB_DEBUG
		fprintf (stderr, "IOCTL RETURNED %d bytes \n", haveRead);
#endif
		eBookClose ();
		CloseHandle (hPcFile);
		// DeleteFile (szPcName);
		return (FAILURE);
	}

	if (WriteFile (hPcFile, internalBuffer, 
		haveRead, &haveWritten, NULL) == false)
	{
#ifdef REB_DEBUG
		fprintf (stderr, "UNEXPECTED WRITE ERROR #%d\n", GetLastError ());
#endif
		eBookClose ();
		CloseHandle (hPcFile);
		return (FAILURE);
	}

	total = haveRead;
	if (sz)
		pctRead = total * 100 / sz;
	if (fnc)
		fnc (pctRead);

	while (haveRead == REB_BLOCK_SZ)
	{
#ifdef REB_DEBUG
		printf ("%d OCTETS READ.\n", total);
#endif
		if (DeviceIoControl (hDev, IOCTL_RocketReadContinue, 
			NULL, 0, internalBuffer, REB_BLOCK_SZ, 
			(LPDWORD) &haveRead, NULL) < 0)
		{
#ifdef REB_DEBUG
			fprintf (stderr, "UNEXPECTED IOCTL ERROR #%d\n", GetLastError ());
#endif
			// there was an error, chances are the REB went offline
			setDisconnState ();
			CloseHandle (hPcFile);
			DeleteFile (szPcName);
			return (FAILURE);
		}

		if (WriteFile (hPcFile, internalBuffer, 
			haveRead, &haveWritten, NULL) == false)
		{
#ifdef REB_DEBUG
			fprintf (stderr, "UNEXPECTED WRITE ERROR #%d\n", GetLastError ());
#endif
			eBookClose ();
			CloseHandle (hPcFile);
			return (FAILURE);
		}

		total += haveRead;
		if (sz)
			pctRead = total * 100 / sz;
		if (fnc)
			fnc (pctRead);
	}
	
	eBookClose ();
	CloseHandle (hPcFile);
	return (SUCCESS);
}

int eBookComm::eWriteFileFromBuffer (char * buffer, int sendSize, int fnameSz, PctDoneFnc * fnc)
{
	int octets;
	DWORD status;
	char *p;
	int origSize;

	eBookOpen ();

	if (hDev == INVALID_HANDLE_VALUE)
		return (FAILURE);

	p = buffer;
	octets = sendSize;
	sendSize += fnameSz;
	origSize = sendSize;

	if (DeviceIoControl (hDev, IOCTL_RocketWriteInitiate, p, 
		sendSize, NULL, 0, &status, NULL) < 0) 
	{
#ifdef REB_DEBUG
		fprintf (stderr, "UNEXPECTED IOCTL ERROR #%d\n", GetLastError ());
#endif
		setDisconnState ();
		return (FAILURE);
	}

	if (octets > REB_BLOCK_SZ)
		octets = REB_BLOCK_SZ;

	sendSize -= (octets + fnameSz);
	p += (octets + fnameSz);
	
	if (fnc)
		fnc ((origSize - sendSize) * 100 / origSize);

	while (sendSize > 0) 
	{
		octets = sendSize;
		if (octets > REB_BLOCK_SZ)
			octets = REB_BLOCK_SZ;

		if (DeviceIoControl (hDev, IOCTL_RocketWriteContinue, 
			p, octets, NULL, 0, &status, NULL) < 0) 
		{
			setDisconnState ();
			return (FAILURE);
		}
		sendSize -= octets;
		p += octets;
		if (fnc)
			fnc ((origSize - sendSize) * 100 / origSize);

#ifdef REB_DEBUG
		printf ("%d OCTETS REMAINING.\n", sendSize);
#endif
	}

	eBookClose ();
	return (SUCCESS);
}

int eBookComm::eWriteImmediate(char * szRebName, char * buffer, int bufSize)
{
	long int fnameSz;
	char *send_ptr;
	int retVal;

	fnameSz = (long) strlen (szRebName) + 1;

	send_ptr = (char *) malloc (bufSize + fnameSz);
	if (send_ptr == NULL) {
#ifdef REB_DEBUG
		fprintf (stderr, "Failed to allocate %d bytes for send buffer.\n",
			bufSize + fnameSz);
#endif
		return (FAILURE);
	}

	memcpy (send_ptr, szRebName, fnameSz);
	memcpy (send_ptr + fnameSz, buffer, bufSize);

	retVal = eWriteFileFromBuffer (send_ptr, bufSize, fnameSz);

#ifdef REB_DEBUG
	printf ("FILE SEND COMPLETED... \n");
#endif
	free (send_ptr);
	return (retVal);
}

int eBookComm::eWriteFile(char * szRebName, char * szPcName, PctDoneFnc * fnc)
{
	DWORD fileSize, fnameSz, octets;
	HANDLE hPcFile;
	BOOL bResult;
	int  retVal;
	char *send_ptr;

    hPcFile       = CreateFile (szPcName,
                                GENERIC_READ,
                                FILE_SHARE_READ,
                                NULL,
                                OPEN_EXISTING,
                                FILE_ATTRIBUTE_NORMAL,
                                NULL);

    if (hPcFile == INVALID_HANDLE_VALUE)
    {
#ifdef REB_DEBUG
		fprintf (stderr, "Failure opening file %s\n", szPcName);
#endif
		return (FAILURE);
    }

	fileSize = GetFileSize (hPcFile, NULL);
#ifdef REB_DEBUG
	fprintf (stderr, "Preparing to write %d bytes\n", fileSize);
#endif
	if (fileSize <= 0)
	{
		CloseHandle (hPcFile);
		return (FAILURE);
    }

	fnameSz = (long) strlen (szRebName) + 1;

	send_ptr = (char *) malloc (fileSize + fnameSz);
	if (send_ptr == NULL) {
#ifdef REB_DEBUG
		fprintf (stderr, "Failed to allocate %d bytes for send buffer.\n",
			fileSize + fnameSz);
#endif
		CloseHandle (hPcFile);
		return (FAILURE);
	}

	memcpy (send_ptr, szRebName, fnameSz);

	// Attempt a read operation.
	bResult = ReadFile(hPcFile, send_ptr + fnameSz, 
		fileSize, &octets, NULL) ; 
	if ((bResult == FALSE) ||  (octets != fileSize)) 
	{ 
		// read operation failed 
#ifdef REB_DEBUG
		fprintf (stderr, "UNEXPECTED WRITE ERROR #%d\n", GetLastError ());
		fprintf (stderr, "ERROR - Read %d bytes out of %d bytes\n", octets,
			fileSize);
#endif
		free (send_ptr);
		CloseHandle (hPcFile);
		return (FAILURE);
	} 

	retVal = eWriteFileFromBuffer (send_ptr, fileSize, fnameSz, fnc);

#ifdef REB_DEBUG
	printf ("\nFILE SEND COMPLETED... \n");
#endif
	free (send_ptr);
	CloseHandle (hPcFile);
	return (retVal);
}

int eBookComm::eBookGetParameters(void)
{
	int bufSize = sizeof (internalBuffer);
	char * currP = internalBuffer;
	char * newLP;
	rebProperty       * propP;

	props->removeAll ();

	if (eReadImmediate ("__STATUS", internalBuffer, &bufSize)
		== SUCCESS)
	{
		internalBuffer[bufSize] = '\0'; // Let's make sure it's a string
		while (* currP)
		{
			newLP = strchr (currP, '\n');
			if (newLP)
			{
				* newLP = '\0';
				newLP++;
			}
			else
			{
				newLP = currP + strlen (currP);
			}
			if (strchr (currP, '='))
			{
				propP = new rebProperty (currP);
				if (propP)
				{
					props->insertHead (propP);
					checkKnownProps (propP, true);
				}
			}
			currP = newLP;
		}
	}
	else
		return (FAILURE);

	if (devId == NULL)
		resetF = true;

	return (SUCCESS);
}

int eBookComm::eBookGetTitles(void)
{
	int bufSize = sizeof (internalBuffer);
	char * currP = internalBuffer;
	char * newLP, * tmpP;
	rebTitle    * titleP;
	rebProperty * propP;

	titles->removeAll ();

	if (eReadImmediate ("__TITLES", internalBuffer, &bufSize)
		== SUCCESS)
	{
		internalBuffer[bufSize] = '\0'; // Let's make sure it's a string
		while (* currP)
		{
			newLP = strchr (currP, '\n');
			if (newLP)
			{
				* newLP = '\0';
				newLP++;
			}
			else
			{
				newLP = currP + strlen (currP);
			}
			if (tmpP = strchr (currP, '='))
			{
				if (tmpP != strrchr (currP, '='))
				{
					titleP = new rebTitle (currP);
					if (titleP)
					{
						titles->insertHead (titleP);
					}
				}
				else
				{
					propP = new rebProperty (currP);
					checkKnownProps (propP, false);
					delete propP;
				}
			}
			currP = newLP;
		}
	}
	else
		return (FAILURE);

	if (devId == NULL)
		resetF = true;

	return (SUCCESS);
}

int eBookComm::eBookGetDirTree(int memType, rebDirectory * dirP)
{
	int bufSize = sizeof (internalBuffer);
	char * currP = internalBuffer;
	char * newLP;
	char propLine[256];
	rebFile       * fileP;
	int  lC, i, nTokens = 0, offset, fSz;
	char * tokens[tokenMax];
	bool inSyncF = false;
	bool refrF   = true;

	if (dirP == NULL)
	{
		dirP = roots[memType];
	}

	if (dirP == roots[memType]) // We have to rebuild the entire subtree
	{
		refrF = false;
		inSyncF = true;
	}

	dirP->removeAll ();
	iFiles[memType] = iDirs[memType] = 0;

	if (eReadImmediate (specFileName[memType], internalBuffer, &bufSize)
		== SUCCESS)
	{
		internalBuffer[bufSize] = '\0'; // Let's make sure it's a string
		for (lC=0; * currP; lC++)
		{
			newLP = strchr (currP, '\n');
			if (newLP)
			{
				* newLP = '\0';
				if (newLP[- 1] == '\r')
					newLP[- 1] = '\0';
				newLP++;
			}
			else
			{
				newLP = currP + strlen (currP);
			}

			memset (propLine, '\0', sizeof (propLine));
			strncpy (propLine, currP, sizeof (propLine));
			currP = newLP;

			if (lC >= 2) // Skip the first two lines
			{
				if (propLine[0] == '+')
				{
					offset = 3;
				}
				else
				{
					offset = 0;
					if (refrF == false)
						dirP = roots[memType];
					else if (inSyncF)
						return (SUCCESS);
				}

				nTokens = 0;

				for (i=0; i < tokenMax; i++)
				{
					tokens[i] = propLine + offset;
					offset += flds[i];
					propLine[offset - 1] = '\0';
					if (i == tokenLgName)
						tokens[i] = strtok (tokens[i], "\n");
					else
						tokens[i] = strtok (tokens[i], " \t");
					if (tokens[i])
						nTokens++;
				}

				if (nTokens <= 2) // Skip the . and .. directories.
					continue;

				if (tokens[tokenAttr] && (tokens[tokenAttr][1] == 'D'))
				{
					iDirs[memType]++;

					if (refrF) // If we are just refreshing one directory
					{
						if (!strcmp (dirP->getLnName (), tokens[tokenLgName]))
							inSyncF = true;   // We are in THE directory
						else if (inSyncF)
							return (SUCCESS); // We got out, don't need to keep going
						
						continue;
					}


					dirP = new rebDirectory (this, dirP, 
						tokens[tokenShName], 
						tokens[tokenLgName], 
						tokens[tokenAttr], 
						memType);

					if (dirP == NULL)
						return (FAILURE);
				}
				else if (inSyncF)
				{
					iFiles[memType]++;

					if (tokens[tokenSz])
						fSz = atoi (tokens[tokenSz]);
					else
						fSz = 0;

					fileP = new rebFile (this, dirP,
						tokens[tokenShName],
						tokens[tokenType],
						tokens[tokenLgName],
						fSz,
						tokens[tokenAttr]);

					if (fileP == NULL)
						return (FAILURE);
				}
			}
		}
	}
	else
		return (FAILURE);

	if (inSyncF)
		return (SUCCESS);

	return (FAILURE);
}

void eBookComm::setDisconnState(void)
{
	eBookConnected = false;

	eBookClose (true);

	freeIntMem = 0;
	freeSMMem = 0;
	totalIntMem = 0;
	totalSMMem = 0;
	batCap = NULL;
	smPresent = false;
	devId = NULL;
	resetF = false;
	osVer = NULL;
	userId = NULL;
	eDate[0] = '\0';
	eTime[0] = '\0';
	
	props->removeAll ();
	titles->removeAll ();
	for (int i=0; i < REB_TREMAX; i++)
	{
		iDirs[i] = iFiles[i] = 0;
		roots[i]->removeAll ();
	}
}

void eBookComm::checkKnownProps(rebProperty * prop, bool bStatFile)
{
	char * valP = prop->getValue ();
	char * nameP = prop->getName ();

	if (!strcmp (nameP, "SM_PRESENT"))
	{
		if (!strcmp (valP, "Yes"))
			smPresent = true;
		else
			smPresent = false;
		return;
	}
	else if (!strcmp (nameP, "FLASHFREE"))
	{
		freeIntMem = atoi (valP);
		return;
	}
	else if (!strcmp (nameP, "SM_FLASHFREE"))
	{
		freeSMMem = atoi (valP);
		return;
	}
	else if (!strcmp (nameP, "FLASHID"))
	{
		totalIntMem = atoi (prop->getValue (1)) * 0x400 * 0x400;
		return;
	}
	else if (!strcmp (nameP, "OFLASHID01"))
	{
		totalSMMem = atoi (prop->getValue (1)) * 0x400 * 0x400;
		return;
	}
	else if (!strcmp (nameP, "EXTID"))
	{
		devId = valP;
		return;
	}
	else if (!strcmp (nameP, "BATTERY"))
	{
		batCap = valP;
		return;
	}
	else if (!strcmp (nameP, "USERID"))
	{
		userId = valP;
		return;
	}
	else if (!strcmp (nameP, "TIME"))
	{
		int hh, mm, ss;
		sscanf (valP, "%d%d%d", &hh, &mm, &ss);
		sprintf (eTime, "%02d:%02d:%02d",
			hh, mm, ss);
		return;
	}
	else if (!strcmp (nameP, "DATE"))
	{
		for (int i=0; valP[i]; i++)
		{
			if (valP[i] == ' ')
				eDate[i] = '/';
			else
				eDate[i] = valP[i];
		}
		eDate[i] = '\0';
		return;
	}
	else if (!strcmp (nameP, "VERSION"))
	{
		if (bStatFile)
			osVer = valP;
	}
	else if (!strcmp (nameP, "TITLECOUNT"))
	{
		if (!bStatFile)
			titleCnt = atoi (valP);
	}
	return;
}

bool eBookComm::eBookCheck(void)
{
	eBookOpen ();
	eBookClose ();

	return eBookConnected;
}

// Find the title entries corresponding to the eBook files
int eBookComm::eBookFixTitles(int memType)
{
	Node <rebTitle *> * rtN;
	Node <rebFile  *> * rdN = roots[memType]->files;
	Node <rebFile  *> * rfN;
	rebFile * fP, * f1P;
	rebTitle * tP;

	while (rdN = rdN->getNext ())
	{
		fP = rdN->getKey();
		if (fP->isDirectory())
		{
			rfN = ((rebDirectory *) fP)->files;
			while (rfN = rfN->getNext())
			{
				f1P = rfN->getKey ();
				if (f1P->getType () == rebRbFile)
				{
					f1P->makePath (internalBuffer);

					rtN = titles;
					while (rtN = rtN->getNext())
					{
						tP = rtN->getKey ();
						char * pathP = tP->filename;
						if (pathP[0] == '\\')
							pathP++;

						if (!strcmp (internalBuffer, pathP))
						{
							tP->fileP = f1P;
							f1P->titleP = tP;
							break;
						}
					}
				}
			}
		}
	}
	return SUCCESS;
}

// make the full path of the file
char * rebFile::makePath (char * buf)
{
	if (buf == NULL)
		buf = (char *) malloc (MAX_REB_LONG_FILENAME_LEN);

	rebDirectory * rdP = getDir();
	if (rdP == NULL)
		rdP = (rebDirectory *) this;

	if (rdP->getMemType () == TRE_SMARTMED)
		strcpy (buf, "SM\\");
	else
		buf[0] = '\0';

	strcat (buf, rdP->getLnName ());
	strcat (buf, "\\");
	strcat (buf, getLnName ());

	return buf;
}

int rebFile::readFile(char * szPcName, PctDoneFnc * fnc)
{
	char pathName[MAX_REB_LONG_FILENAME_LEN];

	if (size <= 0)
		return (FAILURE);

	return (ebcP->eReadFile (makePath (pathName), szPcName, size, fnc));
}

int rebFile::deleteTitle ()
{
	char pathName[MAX_REB_LONG_FILENAME_LEN];
	char szScript[MAX_REB_LONG_FILENAME_LEN * 2];
	int  iLen;

	iLen = sprintf (szScript, "CMD!\nDELTITLE %s\n#/* Erase title */\n",
		makePath (pathName));

	return ebcP->eWriteImmediate ("__WHATYOUWANT", szScript, iLen);
}

char * niceFormatInt (DWORD number)
{
	static char lclStr[64];
	DWORD i, divProd;
	int   currLen;

	for (i=1000000000; i > 0; i /= 1000)
	{
		divProd = number / i;
		if (divProd >= 1000)
			currLen += sprintf (lclStr + currLen, ",%03d", divProd % 1000);
		else if (divProd > 0)
			currLen = sprintf (lclStr, "%d", divProd);
	}

	return (lclStr);
}


