#!/usr/bin/env python
from __future__ import unicode_literals, division, absolute_import, print_function

import sys, platform, uuid, re, shutil, tempfile
import xml.etree.ElementTree as ET
import os
import os.path
from os.path import expanduser, basename
import subprocess, codecs
from subprocess import Popen, PIPE
from PIL import Image
import lxml
from io import BytesIO

PY2 = sys.version_info[0] == 2
if PY2:
    from Tkinter import Tk, BOTH, StringVar, IntVar, BooleanVar, PhotoImage 
    from ttk import Frame, Button, Style, Label, Entry, Checkbutton, Combobox, Separator
    import tkFileDialog as tkinter_filedialog
    import tkMessageBox as messagebox
else:
    from tkinter import Tk, BOTH, StringVar, IntVar, BooleanVar, PhotoImage,messagebox
    from tkinter.ttk import Frame, Button, Style, Label, Entry, Checkbutton, Combobox, Separator
    import tkinter.filedialog as tkinter_filedialog

# auxiliary KindleUnpack libraries for azw3/mobi splitting
from dualmetafix_mmap import DualMobiMetaFix, pathof, iswindows
from mobi_split import mobi_split

# for metadata parsing
try:
    from sigil_bs4 import BeautifulSoup
except:
    from bs4 import BeautifulSoup

# detect OS
isosx = sys.platform.startswith('darwin')
islinux = sys.platform.startswith('linux')

# display kindlegen file selection dialog
def GetFileName(title):
    file_path = tkinter_filedialog.askopenfilename(title=title)
    return file_path

# display kindlegen file selection dialog
def GetDir(title):
    folder = tkinter_filedialog.askdirectory(title=title)
    return folder

# display message box
def AskYesNo(title, message):
    root = Tk()
    root.withdraw()
    answer = messagebox.askquestion(title, message)
    return answer

# get 'C:\Users\<User>\AppData\Local\' folder location    
def GetLocalAppData():
    # check for Windows 7 or higher
    if sys.getwindowsversion().major >= 6:
        return(os.path.join(os.getenv('USERPROFILE'), 'AppData', 'Local'))
    else:
        return(os.path.join(os.getenv('USERPROFILE'), 'Local Settings', 'Application Data'))

def GetDesktop():
    # output directory
    home = expanduser('~')
    desktop = os.path.join(home, 'Desktop')
    if os.path.isdir(desktop):
        return desktop
    else:
        return home
        
# find kindlegen binary
def findKindleGen():
    kg_path = None

    if iswindows:
        # C:\Users\<User>\AppData\Local\Amazon\Kindle Previewer\lib\kindlegen.exe
        default_windows_path = os.path.join(GetLocalAppData(), 'Amazon', 'Kindle Previewer','lib', 'kindlegen.exe')
        if os.path.isfile(default_windows_path):
            kg_path = default_windows_path
        # C:\Users\<User>\AppData\Local\Amazon\Kindle Previewer\lib\\fc\bin\kindlegen.exe
        default_windows_path2 = os.path.join(GetLocalAppData(), 'Amazon', 'Kindle Previewer 3','lib', 'fc', 'bin', 'kindlegen.exe')
        if os.path.isfile(default_windows_path2):
            kg_path = default_windows_path2

    if islinux:
        # /usr/local/bin/kindlegen
        default_linux_path = os.path.join('/usr', 'local', 'bin', 'kindlegen')
        if os.path.isfile(default_linux_path):
            kg_path = default_linux_path
        # ~/bin/kindlegen
        default_linux_path2 = os.path.join(expanduser('~'), 'bin', 'kindlegen')
        if os.path.isfile(default_linux_path2):
            kg_path = default_linux_path2
           
    if isosx:
        # /Applications/Kindle Previewer.app/Contents/MacOS/lib/kindlegen
        default_osx_path = os.path.join('/Applications', 'Kindle Previewer.app', 'Contents', 'MacOS', 'lib', 'kindlegen')
        if os.path.isfile(default_osx_path):
            kg_path = default_osx_path
        # /Applications/Kindle Previewer 3.app/Contents/MacOS/lib/kindlegen/fc/bin/kindlegen
        default_osx_path2 = os.path.join('/Applications', 'Kindle Previewer 3.app', 'Contents', 'MacOS', 'lib', 'fc', 'bin', 'kindlegen')
        if os.path.isfile(default_osx_path2):
            kg_path = default_osx_path2

    # display select file dialog box
    if not kg_path:
        kg_path = GetFileName('Select kindlegen executable')
        if not kg_path or not os.path.basename(kg_path).startswith('kindlegen'):
            kg_path = None
    return kg_path

# simple kindlegen wrapper
def kgWrapper(*args):
    process = Popen(list(args), stdout=PIPE, stderr=PIPE)
    ret = process.communicate()
    return ret

class Dialog(Frame):
    global Cancel
    Cancel = True

    def __init__(self, parent, bk):
        # display the dialog box
        Frame.__init__(self, parent)   
        self.parent = parent
        self.bk = bk
        self.initUI()

    def savevalues(self):
        global Cancel
        Cancel = False

        # save dialog box values in dictionary
        prefs = self.bk.getPrefs()
        prefs['donotaddsource'] = self.donotaddsource.get()
        prefs['compression'] = self.compression.get()
        prefs['verbose'] = self.verbose.get()
        prefs['western'] = self.western.get()
        prefs['gif'] = self.gif.get()
        prefs['locale'] = self.locale.get()
        prefs['add_asin'] = self.add_asin.get()
        prefs['azw3_only'] = self.azw3_only.get()
        prefs['mobi7'] = self.mobi7.get()
        prefs['thumbnail'] = self.thumbnail.get()
        prefs['mobi_dir'] = self.mobi_dir.get()
        self.bk.savePrefs(prefs)
        #self.master.destroy()
        self.master.quit()
       
    def cancel(self):
        #self.master.destroy()
        self.master.quit()

    def getdir(self):
        mobi_dir = GetDir('Select output folder.')
        if mobi_dir != '':
            self.mobi_dir.set(mobi_dir)
        else:
            self.mobi_dir.set(GetDesktop())
       
    def initUI(self):
        prefs = self.bk.getPrefs()

        # look for kindlegen binary
        if not 'kg_path' in prefs:
            kg_path = findKindleGen()
            if kg_path and os.path.basename(kg_path).startswith('kindlegen'):
                prefs['kg_path'] = kg_path
        
        # define dialog box properties
        self.parent.title("KindleGen")
        self.pack(fill=BOTH, expand=1)

        # start reading location
        if srl_def: 
            srlLabel = Label(self, text="SRL: " + srl_def)
        else:
            srlLabel = Label(self, foreground="red", text="SRL: NOT FOUND")
        srlLabel.place(x=10, y=10)

        # HTML TOC
        if toc_def:
            tocLabel = Label(self, text="TOC: " + toc_def)
        else:
            tocLabel = Label(self, foreground="red", text="TOC: NOT FOUND")
        tocLabel.place(x=10, y=30)

        # Cover
        if cover_def:
            coverLabel = Label(self, text="Cover: " + cover_def)
        else:
            coverLabel = Label(self, foreground="red", text="Cover: NOT FOUND")
        coverLabel.place(x=10, y=50)

        # Don't add source check button
        self.donotaddsource = BooleanVar(None)
        if 'donotaddsource' in prefs:
            self.donotaddsource.set(prefs['donotaddsource'])
        donotaddsourceCheckbutton = Checkbutton(self, text="Don't add source files", variable=self.donotaddsource)
        donotaddsourceCheckbutton.place(x=10, y=70)

        # compression label
        options = ['0', '1', '2']
        compressionLabel = Label(self, text="Compression: ")
        compressionLabel.place(x=10, y=90)
        # compression list box
        self.compression = StringVar()
        compression = Combobox(self, textvariable=self.compression)
        compression['values'] = options
        if 'compression' in prefs:
            compression.current(int(prefs['compression']))
        else:
            compression.current(0)
        compression.place(x=100, y=90, width=40, height=18)

        # Verbose output check button
        self.verbose = BooleanVar(None)
        if 'verbose' in prefs:
            self.verbose.set(prefs['verbose'])
        verboseCheckbutton = Checkbutton(self, text="Verbose output", variable=self.verbose)
        verboseCheckbutton.place(x=10, y=110)

        # Western check button check button
        self.western = BooleanVar(None)
        if 'western' in prefs:
            self.western.set(prefs['western'])
        westernCheckbutton = Checkbutton(self, text="Western (Windows-1252)", variable=self.western)
        westernCheckbutton.place(x=10, y=130)

        # Gif to jpeg check button
        self.gif = BooleanVar(None)
        if 'gif' in prefs:
            self.gif.set(prefs['gif'])
        gifCheckbutton = Checkbutton(self, text="Convert JPEG to GIF", variable=self.gif)
        gifCheckbutton.place(x=10, y=150)

        # Select locale list box
        locales = ['en', 'de', 'fr', 'it', 'es', 'zh', 'ja', 'pt', 'ru']
        localeLabel = Label(self, text="Language: ")
        localeLabel.place(x=10, y=170)
        # locale list box
        self.locale = StringVar()
        locale = Combobox(self, textvariable=self.locale)
        locale['values'] = locales
        if 'locale' in prefs:
            index = [i for i,x in enumerate(locales) if x == prefs['locale']]
            locale.current(int(index[0]))
        else:
            locale.current(0)
        locale.place(x=100, y=170, width=40, height=18)

        # Add ASIN check button
        self.add_asin = BooleanVar(None)
        if 'add_asin' in prefs:
            self.add_asin.set(prefs['add_asin'])
        add_asinCheckbutton = Checkbutton(self, text="Add fake ASIN", variable=self.add_asin)
        add_asinCheckbutton.place(x=10, y=190)

        # Generate azw3 check button
        self.azw3_only = BooleanVar(None)
        if 'azw3_only' in prefs:
            self.azw3_only.set(prefs['azw3_only'])
        azw3_onlyCheckbutton = Checkbutton(self, text="Generate AZW3", variable=self.azw3_only)
        azw3_onlyCheckbutton.place(x=10, y=210)

        # Generate mobi7 check button
        self.mobi7 = BooleanVar(None)
        if 'mobi7' in prefs:
            self.mobi7.set(prefs['mobi7'])
        mobi7Checkbutton = Checkbutton(self, text="Generate Mobi7", variable=self.mobi7)
        mobi7Checkbutton.place(x=10, y=230)

        # Generate thumbnails check button
        self.thumbnail = BooleanVar(None)
        if 'thumbnail' in prefs:
            self.thumbnail.set(prefs['thumbnail'])
        thumbnailCheckbutton = Checkbutton(self, text="Generate thumbnail", variable=self.thumbnail)
        thumbnailCheckbutton.place(x=10, y=250)

        # output dir label
        mobi_dirLabel = Label(self, text="Output dir: ")
        mobi_dirLabel.place(x=10, y=270)
        # output dir text box
        self.mobi_dir=StringVar(None)
        if 'mobi_dir' in prefs:
            self.mobi_dir.set(prefs['mobi_dir'])
        else:
            self.mobi_dir.set(GetDesktop())
        mobi_dirEntry=Entry(self, textvariable=self.mobi_dir)
        mobi_dirEntry.place(x=80, y=270, width=90)
        browseButton = Button(self, text="...", command=self.getdir)
        browseButton.place(x=180, y=270, width=30, height=18)

        # OK and Cancel buttons
        cancelButton = Button(self, text="Cancel", command=self.cancel)
        cancelButton.place(x=130, y=300)
        okButton = Button(self, text="OK", command=self.savevalues)
        okButton.place(x=10, y=300)

# main plugin routine
def run(bk):
    plugin_warnings = ''
    global toc_def
    toc_def = None
    global srl_def
    srl_def = None
    global cover_def
    cover_def = None
    
    #--------------
    # run checks
    #--------------
        
    # get epub version number
    if bk.launcher_version() >= 20160102:
        epubversion = bk.epub_version()
    else:
        opf_contents = bk.readotherfile('OEBPS/content.opf')
        epubversion = BeautifulSoup(opf_contents, 'lxml').find('package')['version']

    # get metadata soup
    metadata_soup = BeautifulSoup(bk.getmetadataxml(), 'lxml')

    #-----------------------------------------------
    # get required metadata items (title & language
    #-----------------------------------------------

    # get title
    if metadata_soup.find('dc:title'):
        dc_title = metadata_soup.find('dc:title').string
    else:
        print('\nError: Missing title metadata!\n\nPlease click OK to close the Plugin Runner window.')
        return -1

    # get language
    dc_language = 'en'
    if metadata_soup.find('dc:language'):
        dc_language = metadata_soup.find('dc:language').string
    else:
        print('\nError: Missing language metadata!\n\nPlease click OK to close the Plugin Runner window.')
        return -1

    # get/set asin
    asin = str(uuid.uuid4())[24:82]
    dc_identifier = metadata_soup.find('dc:identifier', {'opf:scheme' : re.compile('(MOBI-ASIN|AMAZON)')})
    if dc_identifier:
        asin = dc_identifier.string

    #---------------------------------------
    # check guides/landmarks cover metadata
    #---------------------------------------
    if epubversion.startswith("3"):

        # look for nav.xhtml
        opf_soup = BeautifulSoup(bk.get_opf(), 'lxml')
        nav_item = opf_soup.find('item', {'properties' : 'nav'})
        if nav_item:
            nav_href = nav_item['href']
            nav_id = bk.href_to_id(nav_href)

            # get landmarks from nav document
            landmarks = {}
            nav_soup = BeautifulSoup(bk.readfile(nav_id), 'html.parser')
            for landmark in nav_soup.find_all('a', {'epub:type' : re.compile('.*?')}):
                epub_type = landmark['epub:type']
                href = landmark['href']
                landmarks[epub_type] = href
                
            # check for required landmarks items
            if not 'toc' in landmarks:
                plugin_warnings += '\nWarning: Missing TOC epub3 landmark: Use Add Semantics > Table of Contents to mark the TOC.'
            else: 
                toc_def = os.path.basename(landmarks['toc'])
            if not 'bodymatter' in landmarks:
                plugin_warnings += '\nWarning: Missing SRL epub3 landmark: Use Add Semantics > Bodymatter to mark the SRL.'
            else: 
                srl_def = os.path.basename(landmarks['bodymatter'])
        else:
            plugin_warnings += '\nError: nav document not found!'
        
        # look for cover image
        cover_id = None
        cover_item = opf_soup.find('item', {'properties' : 'cover-image'})
        if cover_item:
            cover_href = cover_item['href']
            cover_id = bk.href_to_id(cover_href)
            cover_def = os.path.basename(cover_href)
        else:
            plugin_warnings += '\nWarning: Cover not specified (cover-image property missing).'
            
            # look for cover page image references
            if 'cover' in landmarks:
                cover_page_id = bk.href_to_id(landmarks['cover'].replace('../', '')) 
                cover_page_html = bk.readfile(cover_page_id)
                cover_image_href = re.search('(href|src)="(\.\.\/Images\/[^"]+)"', cover_page_html)
                if cover_image_href:
                    cover_id = bk.href_to_id(cover_image_href.group(2).replace('../', ''))
                    plugin_warnings += '\n"' + cover_id + '" should have a cover-image property.'
                
    # get epub2 data
    else:
        # get guide items
        opf_guide_items = {}
        for ref_type, title, href in bk.getguide():
            opf_guide_items[ref_type] = href

        # check for required guide items
        if not 'toc' in opf_guide_items:
            plugin_warnings += '\nWarning: Missing TOC guide item. Use Add Semantics > Table of Contents to mark the TOC.'
        else: 
            toc_def = os.path.basename(opf_guide_items['toc'])
            
        if not 'text' in opf_guide_items:
            plugin_warnings += '\nWarning: Missing SRL guide item. Use Add Semantics > Text to mark the SRL.'
        else: 
            srl_def = os.path.basename(opf_guide_items['text'])

        # get cover image path
        cover_id = None
        cover_image = metadata_soup.find('meta', {'name' : 'cover'})
        if cover_image:
            cover_id = cover_image['content']
            cover_href = bk.id_to_href(cover_id)
            cover_def = os.path.basename(cover_href)
        else:
            plugin_warnings += '\nWarning: Cover not specified (cover metadata missing).'

            # look for cover page image references
            if 'cover' in opf_guide_items:
                cover_page_id = bk.href_to_id(opf_guide_items['cover'].replace('../', '')) 
                cover_page_html = bk.readfile(cover_page_id)
                cover_image_href = re.search('(href|src)="(\.\.\/Images\/[^"]+)"', cover_page_html)
                if cover_image_href:
                    cover_id = bk.href_to_id(cover_image_href.group(2).replace('../', ''))
                    plugin_warnings += '\n<meta name="cover" content="' + cover_id + '" />'
            
    # check minimum cover width
    img = None
    if cover_id:
        imgdata = bk.readfile(cover_id)
        img = Image.open(BytesIO(imgdata)).convert('L')
        width, height = img.size
        if width < 500:
            plugin_warnings += '\nWarning: The cover is too small: ' + str(width) + ' x ' + str(height)

    # set Tk parameters for dialog box
    root = Tk()
    root.geometry("240x340+300+300")
    app = Dialog(root, bk)
    if not PY2 and not isosx:
        icon_img = PhotoImage(file=os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'sigil.png'))
        root.tk.call('wm', 'iconphoto', root._w, icon_img)
    root.mainloop()
    prefs = bk.getPrefs()

    #--------------
    # main routine
    #--------------
    if 'kg_path' in prefs and not Cancel == True:

        kg_path = prefs['kg_path']

        #----------------------------------
        # display confirmation message box
        #----------------------------------
        if plugin_warnings != '':
            plugin_warnings = '\n*************************************************************' + plugin_warnings + '\n*************************************************************\n'
            print(plugin_warnings)
            answer = AskYesNo('Ignore warnings?', 'Do you want to ignore these warnings?')
            if answer == 'no':
                print('\nPlugin terminated by user.\n\nPlease click OK to close the Plugin Runner window.')
                return -1

     
        # create temp folder
        temp_dir = tempfile.mkdtemp()

        # copy book files
        bk.copy_book_contents_to(temp_dir)
        
        # get content.opf
        opf_path = os.path.join(temp_dir, 'OEBPS','content.opf')
        
        #------------------------------------------
        # define kindlegen command line parameters
        #------------------------------------------

        # define temporary mobi file name
        mobi_path = os.path.join(temp_dir, 'OEBPS', 'sigil.mobi')
        args = [kg_path, opf_path]
        
        if 'compression' in prefs and prefs['compression'] in ['0', '1', '2'] or [0, 1, 2]:
            args.append('-c' + str(prefs['compression']))

        if 'donotaddsource' in prefs and prefs['donotaddsource'] == True:
            args.append('-dont_append_source')

        if 'verbose' in prefs and prefs['verbose'] == True:
            args.append('-verbose')

        if 'western' in prefs and prefs['western'] == True:
            args.append('-western')

        if 'gif' in prefs and prefs['gif'] == True:
            args.append('-gif')

        if 'locale' in prefs and prefs['locale'] in ['en', 'de', 'fr', 'it', 'es', 'zh', 'ja', 'pt', 'ru']:
            args.append('-locale')
            args.append(prefs['locale'])

        args.append('-o')
        args.append('sigil.mobi')
        
        # run kindlegen
        print("Running KindleGen ... please wait")
        result = kgWrapper(*args)

        # print kindlegen messages
        kg_messages = result[0].decode('utf-8', 'ignore').replace(temp_dir, '').replace('\r', '')
        print(kg_messages)

        # display kindlegen warning messages
        if re.search('W\d+:', kg_messages):
            print('\n*************************************************************\nkindlegen warnings:\n*************************************************************')
            kg_warnings = kg_messages.splitlines()
            for line in kg_warnings:
                if re.search('W\d+:', line):
                    print(line) 

        #--------------------------------------
        # define output directory and filenames
        #--------------------------------------

        # output directory
        home = expanduser('~')
        desktop = os.path.join(home, 'Desktop')
        if 'mobi_dir' in prefs and os.path.isdir(prefs['mobi_dir']) == True:
            dst_folder = prefs['mobi_dir']
        else:
            if os.path.isdir(desktop):
                dst_folder = desktop
            else:
                dst_folder = home
            prefs['mobi_dir'] = dst_folder 
            bk.savePrefs(prefs)

        # make sure the output file name is safe
        title = re.sub('[/|\?|<|>|\\\\|:|\*|\||"|\^| ]+', '_', dc_title)
        if 'add_asin' in prefs and prefs['add_asin'] == True:
            dst_path = os.path.join(dst_folder, title  + '_' + asin + '.mobi')
            azw_path = os.path.join(dst_folder, title  + '_' + asin + '.azw3')
        else:
            dst_path = os.path.join(dst_folder, title  + '.mobi')
            azw_path = os.path.join(dst_folder, title + '.azw3')
            
        #------------------------------------------
        # split mobi file into azw3 and mobi7 parts
        #------------------------------------------
        
        # if kindlegen didn't fail, there should be a .mobi file
        if os.path.isfile(mobi_path):
            
            # add ASIN and set book type to EBOK using KevinH's dualmetafix_mmap.py
            if 'add_asin' in prefs and prefs['add_asin'] == True:
                dmf = DualMobiMetaFix(mobi_path, asin)
                open(pathof(mobi_path + '.tmp'),'wb').write(dmf.getresult())
                
                # if DualMobiMetaFix didn't fail, there should be a .temp file
                if os.path.isfile(str(mobi_path) + '.tmp'):
                    print('\nASIN: ' + asin + ' added')

                    # delete original file and rename temp file
                    os.remove(str(mobi_path))
                    os.rename(str(mobi_path) + '.tmp', str(mobi_path))
                else:
                    print('\nASIN couldn\'t be added.')

            #--------------------------------------
            # generate mobi thumbnail image for PW
            #--------------------------------------
            if img and 'thumbnail' in prefs and prefs['thumbnail'] == True:
                img.thumbnail((330, 330), Image.ANTIALIAS)
                if 'add_asin' in prefs and prefs['add_asin'] == True:
                    img_dest_path = os.path.join(dst_folder, 'thumbnail_' + asin + '_EBOK_portrait.jpg')
                else:
                    img_dest_path = os.path.join(dst_folder, 'thumbnail_EBOK_portrait.jpg')
                img.save(img_dest_path)
                if os.path.isfile(img_dest_path):
                    print('\nThumbnail copied to: ' + img_dest_path)
                else:
                    print('\nPlugin Error: Thumbnail creation failed.')

            # copy output files
            if 'azw3_only' in prefs and prefs['azw3_only'] == True or 'mobi7' in prefs and prefs['mobi7'] == True:
                mobisplit = mobi_split(pathof(mobi_path))
                
                if mobisplit.combo:
                    outmobi8 = pathof(azw_path)
                    outmobi7 = outmobi8.replace('.azw3', '.mobi')
                    if 'azw3_only' in prefs and prefs['azw3_only'] == True:
                        open(outmobi8, 'wb').write(mobisplit.getResult8())
                        print('AZW3 file copied to ' + azw_path)
                    if 'mobi7' in prefs and prefs['mobi7'] == True:
                        open(outmobi7, 'wb').write(mobisplit.getResult7())
                        print('MOBI7 file copied to ' + azw_path.replace('.azw3', '.mobi'))
                else:
                    print('\nPlugin Error: Invalid mobi file format.')
                    
            else:
                shutil.copyfile(mobi_path, dst_path)
                print('\nMobi file copied to ' + dst_path)

        else:
            # display error messages
            if re.search('E\d+:', kg_messages):
                print('\n*************************************************************\nkindlegen errors:\n*************************************************************')
                kg_errors = kg_messages.splitlines()
                for line in kg_errors:
                    if re.search('E\d+:', line):
                        print(line) 

        # delete temp folder
        shutil.rmtree(temp_dir)
        
    else:
        if Cancel == True:
            print('\nPlugin terminated by user.')
        else:
            print('\nPlugin Warning: Kindlegen path not selected or invalid.\nPlease re-run the plugin and select the Kindlegen binary.')
    
    print('\nPlease click OK to close the Plugin Runner window.')    
    return 0


def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
