#!/usr/bin/env python
# -*- coding: utf-8 -*-

#  Integration with Irex Iliad
import os, shutil, xml.dom.minidom, datetime, sys, tempfile, subprocess, time
import pyPdf
import gtk, gtk.glade
import referencer
from referencer import _

document_types = ["Newspapers", "Books", "Documents", "Notes"]
memory_types = ["Internal", "CF", "MMC", "USB"]

#Set some defaults
def get_sync_directory():
    return referencer.pref_get("iliad_wireless_dir") or os.path.expanduser('~')
def get_memory_type():
    return referencer.pref_get("iliad_wireless_memory_type") or "Internal"
def get_document_type():
    return referencer.pref_get("iliad_wireless_document_type") or "Documents"
def get_margin(side):
    return referencer.pref_get("iliad_margin_%s" %side) or "0"


referencer_plugin_info = {
	"longname": _("Send to Iliad"),
	"author": "Ian Helgesen",
	"ui":   """
		<ui>
			<menubar name='MenuBar'>
				<menu action='DocMenu'>
                    <placeholder name='PluginDocMenuActions'>
                        <menuitem action='_plugin_iliad_send'/>
                    </placeholder>
				</menu>
			</menubar>
			<toolbar name='ToolBar'>
                <placeholder name='PluginToolBarActions'>
                    <toolitem action='_plugin_iliad_send'/>
                </placeholder>
			</toolbar>
			<popup name='DocPopup'>
                <placeholder name='PluginDocPopupActions'>
                    <menuitem action='_plugin_iliad_send'/>
                </placeholder>
			</popup>
		</ui>
		"""
}

referencer_plugin_actions = [
	{
	"name":        "_plugin_iliad_send",
	"label":     _("Send to Iliad"),
	"tooltip":   _("Send the selected documents to Iliad"),
	"icon":        "iliad.png",
	"callback":    "do_send",
	"accelerator": "<control><shift>i"
	}
]

class document_information():
    "document metadata"
    def __init__(self, document):
        "Assign all properties."
        #self.cover = get_cover(self.file)
        self.filename = document.get_filename()[7:]
        self.title = document.get_field('title')
        self.author = document.get_field("author")
        self.year = document.get_field('year')
        self.key = document.get_key()
        self.cover = None
        self.temp_dir = None
        self.outdir = None

        #PS files
        if os.path.splitext(self.filename)[1].lower() == '.ps':
            self.temp_dir = tempfile.mkdtemp()
            new_file = os.path.join(self.temp_dir, \
                os.path.splitext(os.path.split(self.filename)[1])[0] + '.pdf')
            subprocess.Popen(['ps2pdf', self.filename, new_file],  \
            stdout=subprocess.PIPE, shell=False)
            time.sleep(10)
            self.filename = new_file

        doc_file = pyPdf.PdfFileReader(file(self.filename, "rb"))
        self.pages = doc_file.getNumPages()


def write_manifest(output):
    """Creates a manifest.xml file using the metadata from output object"""
    manifest = xml.dom.minidom.Document()
    package = manifest.createElement("package")
    manifest.appendChild(package)
    metadata = manifest.createElement("metadata")
    package.appendChild(metadata)
    dcmetadata = manifest.createElement("dc-metadata")
    metadata.appendChild(dcmetadata)

    #Title
    title = manifest.createElement("Title")
    title.appendChild(manifest.createTextNode(output.title))
    dcmetadata.appendChild(title)

    #Description
    description = manifest.createElement("Description")
    description.appendChild(manifest.createTextNode( \
        "%s -- %s       (%s pages)" \
        % (output.author, output.year, output.pages)))
    dcmetadata.appendChild(description)

    #Date
    date = manifest.createElement("Date")
    date.appendChild(manifest.createTextNode( \
    datetime.date.today().strftime("%Y-%m-%dT00:00:00")))
    dcmetadata.appendChild(date)

    #Unused attributes
    dcmetadata.appendChild(manifest.createElement("Format"))
    dcmetadata.appendChild(manifest.createElement("Identifier"))
    dcmetadata.appendChild(manifest.createElement("Language"))
    dcmetadata.appendChild(manifest.createElement("Type"))

    #Y-Metadata section
    ymetadata = manifest.createElement("y-metadata")
    metadata.appendChild(ymetadata)

    startpage = manifest.createElement("startpage")
    startpage.appendChild(manifest.createTextNode( \
        os.path.split(output.filename)[1]))
    ymetadata.appendChild(startpage)

    if output.cover:
        cover = manifest.createElement("image")
        cover.appendChild(manifest.createTextNode("cover.png"))
        ymetadata.appendChild(cover)

    version = manifest.createElement("version")
    version.appendChild(manifest.createTextNode("000"))
    ymetadata.appendChild(version)

    if referencer.pref_get("iliad_fullscreen") == 'True':
        view_settings = manifest.createElement("viewer-settings")
        fullscreen = manifest.createElement("modefullscreen")
        fullscreen.appendChild(manifest.createTextNode("1"))
        view_settings.appendChild(fullscreen)
        package.appendChild(view_settings)

    manifest_file = open(os.path.join(output.outdir, "manifest.xml"),"w")
    manifest.writexml(manifest_file, "", "", "", "UTF-8")
    manifest_file.close()

def crop(output):
    margins = (get_margin("left"), get_margin("top"), get_margin("right"), \
        get_margin("bottom"))
    cropped_file = os.path.join(output.outdir, \
        os.path.splitext(os.path.split(output.filename)[1])[0] + "_cropped.pdf")
    subprocess.Popen(['pdfcrop', "--margins", '%s %s %s %s' % margins, \
        output.filename, cropped_file])
    output.filename = cropped_file

def do_send (library, documents):
    "Send selected documents to output folder."
    for document in documents:
        output = document_information(document)
        output.outdir = \
            os.path.join(get_sync_directory(), 'outbox', get_memory_type(), \
            get_document_type().lower(), output.key)

        #Copy to output
        if not os.path.exists(output.outdir):

            os.makedirs(output.outdir)
            if referencer.pref_get("iliad_crop") == 'True':
                crop(output)
            else:
                shutil.copy(output.filename, output.outdir)
            write_manifest(output)
            if output.temp_dir:
                shutil.rmtree(output.temp_dir)
        else:
            print "%s already exists." % output.outdir


    return True


class pref_window:
    """GTK Settings window."""
    def __init__(self):
        "Create preferences window."
        for path in sys.path:
            if os.path.exists(os.path.join(path, 'iliad.glade')):
                self.gladefile = os.path.join(path, 'iliad.glade')
                break
        self.wTree = gtk.glade.XML(self.gladefile)
        self.prefwindow = self.wTree.get_widget("prefwindow")
        dic = {
            "on_doc_type_changed" : self.set_doctype,
            "on_memory_type_changed" : self.set_mem_type,
            "on_sync_dir_current_folder_changed" : self.set_sync_dir,
            "on_fullscreen_toggled" : self.set_fullscreen,
            "on_crop_toggled" : self.set_crop,
            "on_top_margin_changed" : self.set_top_margin,
            "on_bottom_margin_changed" : self.set_bottom_margin,
            "on_left_margin_changed" : self.set_left_margin,
            "on_right_margin_changed" : self.set_right_margin,
            "on_close_clicked" : self.closebutton_clicked
            }
        self.wTree.signal_autoconnect (dic)

        #Iliad Sync Directory
        self.wTree.get_widget('sync_dir').set_current_folder( \
            get_sync_directory())

        #Memory type
        self.wTree.get_widget('memory_type').set_active( \
            memory_types.index(get_memory_type()))

        #Document type
        self.wTree.get_widget('doc_type').set_active( \
            document_types.index(get_document_type()))

        #Fullscreen
        if referencer.pref_get("iliad_fullscreen") == 'True':
            self.wTree.get_widget('fullscreen').set_active(True)

        #Cropping
        if referencer.pref_get("iliad_crop") == 'True':
            self.wTree.get_widget('crop').set_active(True)

        margin_types = ['top', 'bottom', 'left', 'right']
        for margin_type in margin_types:
            if referencer.pref_get("iliad_margin_%s" % margin_type):
                self.wTree.get_widget("%s_margin" % margin_type).set_value( \
                int(referencer.pref_get("iliad_margin_%s" % margin_type)))


    def set_sync_dir(self, widget):
        referencer.pref_set("iliad_wireless_dir", widget.get_current_folder())

    def set_mem_type(self, widget):
        active = widget.get_active()
        referencer.pref_set("iliad_wireless_memory_type", memory_types[active])

    def set_doctype(self, widget):
        active = widget.get_active()
        referencer.pref_set("iliad_wireless_document_type", \
            document_types[active])

    def set_fullscreen(self, widget):
        referencer.pref_set("iliad_fullscreen", repr(widget.get_active()))

    def set_crop(self, widget):
        referencer.pref_set("iliad_crop", repr(widget.get_active()))
        self.wTree.get_widget('margins').set_sensitive(widget.get_active())

    def set_top_margin(self, widget):
        referencer.pref_set("iliad_margin_top", repr(widget.get_value_as_int()))
    def set_bottom_margin(self, widget):
        referencer.pref_set("iliad_margin_bottom", repr(widget.get_value_as_int()))
    def set_left_margin(self, widget):
        referencer.pref_set("iliad_margin_left", repr(widget.get_value_as_int()))
    def set_right_margin(self, widget):
        referencer.pref_set("iliad_margin_right", repr(widget.get_value_as_int()))

    def closebutton_clicked(self, widget):
        self.prefwindow.hide_all()

def referencer_config():
    ui = pref_window()
    ui.prefwindow.show_all()