// I don't see any point to have a Unicode-incapable program today.  I
// support hard coding to Unicode (or WCHAR.)  Use of TCHAR/TSTR
// families of data types and TEXT macro is purely a matter of coding
// style and it doesn't mean you can make an ANSI build from the same
// source file.  If you undef UNICODE, the program may not run, or
// even compile, properly.  (I actually don't know what happens!)

#define UNICODE 1
#define _UNICODE 1

// We want visual style of the property page enabled on Windows XP.
// Note however this doesn't work with MinGW for the moment...

#define ISOLATION_AWARE_ENABLED 1

#include <algorithm>
#include <new>
#include <string>
#include <vector>

#include <windows.h>
#include <shlwapi.h>
#include <shlobj.h>
#include <objidl.h>
#include <shobjidl.h>
#include <thumbcache.h>
#include <propsys.h>
#include <propkey.h>
#include <propvarutil.h>
#include <wincodec.h>
#include <strsafe.h>

#include "version.h"
#include "resource.h"

////////////////////////////////////////////////////////////////
// The CLSIDs
////////////////////////////////////////////////////////////////

CLSID const CLSID_MobiHandler = {0xba91bbaf,0x9243,0x49bd,{0xb0,0xda,0x97,0xbf,0xc2,0x5e,0xb9,0x76}}; // {BA91BBAF-9243-49BD-B0DA-97BFC25EB976}

////////////////////////////////////////////////////////////////
// Some fundamentals...
////////////////////////////////////////////////////////////////

typedef std::basic_string<TCHAR> TString;
static const TString EMPTY;

typedef std::vector<TString> TStrVec;

#undef min
#undef max

inline size_t min(size_t x, size_t y) { return x < y ? x : y; }
inline size_t max(size_t x, size_t y) { return x > y ? x : y; }
inline size_t clip(size_t x, size_t y, size_t z) { return min(max(x, y), z); }

////////////////////////////////////////////////////////////////
// Debugging aids
////////////////////////////////////////////////////////////////

#ifndef TESTING
#define TESTING 0
#endif

#if TESTING
#define protected public
#define private public
#endif

#ifndef ALLOW_DEBUG
#define ALLOW_DEBUG 1
#endif

#if ALLOW_DEBUG

#define ERRLOG(ARGS)		(g_globals.m_debug     && log ARGS)
#define TRACE(ARGS)		(g_globals.m_debug > 1 && log ARGS)
#define TRACE_QUERY(R,I)	(g_globals.m_debug > 1 && trace_query_interface(this->serial(), (R), (I)))
#define TRACE_METHOD(M)		(g_globals.m_debug > 1 && log("%04lu:%hs", this->serial(), (M)))
#define TRACE_METHOD_ARG(M,A)	(g_globals.m_debug > 1 && log("%04lu:%hs:%ls", this->serial(), (M), (A)))

static bool log(LPCSTR fmt, ...) throw();

static bool trace_query_interface(ULONG serial, REFIID riid, char const *name)
{
  LPOLESTR guid = NULL;
  StringFromIID(riid, &guid);
  log("%04lu:%hs:%ls:%hs", serial, "QueryInterface", guid ? guid : L"{?}", name ? name : "(unsupported)");
  CoTaskMemFree(guid);
  return false;
}

#else

#define ERRLOG(ARGS) (0)
#define TRACE(ARGS) (0)
#define TRACE_QUERY(R,I) (0)
#define TRACE_METHOD(M)	(0)
#define TRACE_METHOD_ARG(M,A) (0)

#endif

#define ENSURELOG(HR) (ERRLOG(("ENSURE failed:%hs:%d:0x%08x", __FILE__, __LINE__, HR)))
#define ENSURE(E) do { HRESULT _hr = (E); if (FAILED(_hr)) { ENSURELOG(_hr); return _hr; } } while (0)



////////////////////////////////////////////////////////////////
// A lot of familiar workarounds for C++'s absense of final clause in
// try block (and somrthing more.)
////////////////////////////////////////////////////////////////

// Yet another COM smart pointer.  T should be a COM interface type,
// and riid should be the corresponding IID.  All methods are
// exception-free.  Use COMPTR macro defined below to instanciate a
// comptr type.
template <class T, REFIID riid>
class comptr
{
public:
  comptr(T *p = NULL) throw() : m_ptr(p) {}
  ~comptr() throw() { safeRelease(); }
  void replaceWith(T *p) throw() { safeRelease(); m_ptr = p; safeAddRef(); }
  void safeAddRef()  throw() { if (m_ptr) m_ptr->AddRef(); }
  void safeRelease() throw() { if (m_ptr) m_ptr->Release(); }
  operator T *() throw() { return  m_ptr; }
  operator bool() throw() { return  m_ptr != NULL; }
  T **operator &() throw() { return &m_ptr; }
  T *operator->() throw() { return  m_ptr; }
public:
  REFIID getIID() throw() { return riid; }
  void **getPPV() throw() { return reinterpret_cast<void **>(&m_ptr); }
private:
  comptr(comptr<T, riid> &); // Don't use it.
  comptr<T, riid> &operator=(comptr<T, riid> const &val); // Don't use it.
private:
  T *m_ptr;
};
#define COMPTR(TYPE) comptr<TYPE, IID_##TYPE>

template <class T>
class array
{
public:
  typedef T *iterator;
  typedef T const *const_iterator;
public:
  array(size_t n = 0) : m_ptr(new T[n]), m_size(n) {}
  ~array() { delete[] m_ptr; }
  void resize(size_t n) { delete[] m_ptr; m_ptr = NULL; m_ptr = new T[n]; m_size = n; }
  operator T *() throw() { return m_ptr; }
  size_t size() throw() { return m_size; }
public:
  iterator begin() throw() { return m_ptr; }
  const_iterator begin() const throw() { return m_ptr; }
  iterator end() throw() { return m_ptr + m_size; }
  const_iterator end() const throw() { return m_ptr + m_size; }
private:
  T *m_ptr;
  size_t m_size;
};

template <class T>
class auto_CoTaskMem
{
public:
  auto_CoTaskMem() : m_ptr(NULL) {}
  ~auto_CoTaskMem() { if (m_ptr) CoTaskMemFree(m_ptr); }
  operator T *()  { return  m_ptr; }
  T **operator &(){ return &m_ptr; }
private:
  T *m_ptr;
};

template <class T>
class auto_HGLOBAL
{
public:
  auto_HGLOBAL(HGLOBAL hMem) : m_hmem(hMem), m_ptr((T)GlobalLock(hMem)) { if (m_ptr == NULL) throw std::bad_alloc(); }
  ~auto_HGLOBAL() { GlobalUnlock(m_hmem); }
  operator T() throw() { return m_ptr; }
private:
  HGLOBAL const m_hmem;
  T const m_ptr;
};

struct auto_STGMEDIUM : public STGMEDIUM
{
  auto_STGMEDIUM()  { ZeroMemory(this, sizeof(STGMEDIUM)); }
  ~auto_STGMEDIUM() { if (tymed != TYMED_NULL) ReleaseStgMedium(this); }
};

struct auto_PROPVARIANT : public PROPVARIANT
{
public:
  auto_PROPVARIANT()  { PropVariantInit(this); }
  ~auto_PROPVARIANT() { PropVariantClear(this); }
};  

////////////////////////////////////////////////////////////////
// More IStream helper functions than shlwapi
////////////////////////////////////////////////////////////////

static inline HRESULT IStream_seek(IStream *stream, UINT pos)
{
  LARGE_INTEGER seek_pos;
  seek_pos.QuadPart = pos;
  return stream->Seek(seek_pos, STREAM_SEEK_SET, NULL);
}

static inline HRESULT IStream_copy(IStream *from, IStream *to, UINT bytes)
{
  ULARGE_INTEGER copy_bytes;
  copy_bytes.QuadPart = bytes;
  return from->CopyTo(to, copy_bytes, NULL, NULL);
}

////////////////////////////////////////////////////////////////
// Globals
////////////////////////////////////////////////////////////////

struct ModuleGlobals
{
  ModuleGlobals() : m_hLog(INVALID_HANDLE_VALUE) {}

  void initializeOnce() throw ();
  void initialize() throw ();
  void finalize() throw ();

  static HANDLE prepareLogFile();
  static void getRegValue(LPCTSTR subkey, LPCTSTR value, LPDWORD data) throw();

  DWORD m_debug;
  DWORD m_useMobiThumbnail;
  HANDLE m_hLog;

  HMODULE m_propsys_dll;
  HRESULT (WINAPI *m_PSCreateMemoryPropertyStore)(REFIID riid, void **ppv) throw();
  HRESULT (WINAPI *m_PSCreateAdapterFromPropertyStore)(IPropertyStore *pps, REFIID riid, void **ppv) throw();
  HRESULT (WINAPI *m_PSStringFromPropertyKey)(REFPROPERTYKEY propkey, LPWSTR psz, UINT cch) throw();
  HRESULT (WINAPI *m_PSGetNameFromPropertyKey)(REFPROPERTYKEY propkey, PWSTR *ppszCanonicalName) throw();
  HRESULT (WINAPI *m_InitPropVariantFromStringVector)(PCWSTR *prgsz, ULONG cElems, PROPVARIANT *ppropvar) throw();
  


  HINSTANCE s_hInstDll;
  CRITICAL_SECTION s_critical_section;
  volatile bool s_initialized;
};
static struct ModuleGlobals g_globals;

void ModuleGlobals::initializeOnce() throw ()
{
  if (!s_initialized) {
    initialize();
  }
}

void ModuleGlobals::initialize() throw ()
{
  // Fetch options from registry.  We need to do this first, since
  // following codes depend on m_debug.
  m_debug = 0;
  m_useMobiThumbnail = 0;
  LPOLESTR clsid;
  if (SUCCEEDED(StringFromCLSID(CLSID_MobiHandler, &clsid))) {
    TString subkey = TEXT("CLSID\\") + TString(clsid) + TEXT("\\Options");
    CoTaskMemFree(clsid); clsid = NULL;
    getRegValue(subkey.c_str(), TEXT("Debug"), &m_debug);
    getRegValue(subkey.c_str(), TEXT("UseMobiThumbnail"), &m_useMobiThumbnail);
  }

  m_hLog = INVALID_HANDLE_VALUE;
#if ALLOW_DEBUG
  if (m_debug) {
    m_hLog = prepareLogFile();
    if (INVALID_HANDLE_VALUE == m_hLog) {
      m_debug = 0;
    } else {
      log("MobiHandler %hs", STRING_VERSION);
    }
  }
#endif

  // Our property code needs functions and COM objects in propsys.dll,
  // that are not available on vanila Windows XP, (although it is a
  // part of Windows Search distribution for Windows XP.)  I don't
  // want entire MobiHandler to fail when propsys.dll is unavailable,
  // because the DLL is needed only by property hadlers, and other
  // handlers, i.e., property sheet extension, thumbnail handler, and
  // image extract handler can work fine without it.
  // 
  // I tried delayed-loading feature, but it didn't work well.  (I'm
  // not sure why; it might be my mistake.)  So, I decided to use the
  // good old technique that keep working since 16 bit Windows era.

  m_propsys_dll = LoadLibraryA("propsys.dll");
  if (!m_propsys_dll) {
    ERRLOG(("%hs:%hs", "Missing DLL", "propsys.dll"));
  }
  if (m_propsys_dll) {

    bool missing = false;

#define INIT_PROPSYS_FUNC(FUNC, TYPE) \
    do {										\
      if (!(m_##FUNC = reinterpret_cast<TYPE>(GetProcAddress(m_propsys_dll, #FUNC)))) {	\
	missing = true;									\
	ERRLOG(("%hs:%hs:%hs", "Missing Function", "propsys.dll", #FUNC));		\
      }											\
    } while (0)

    INIT_PROPSYS_FUNC(PSCreateMemoryPropertyStore,      HRESULT (WINAPI *)(REFIID, void **));
    INIT_PROPSYS_FUNC(PSCreateAdapterFromPropertyStore, HRESULT (WINAPI *)(IPropertyStore *, REFIID, void **));
    INIT_PROPSYS_FUNC(PSStringFromPropertyKey,		HRESULT (WINAPI *)(REFPROPERTYKEY propkey, LPWSTR psz, UINT cch));
    INIT_PROPSYS_FUNC(PSGetNameFromPropertyKey,		HRESULT (WINAPI *)(REFPROPERTYKEY propkey, PWSTR *ppszCanonicalName));
    INIT_PROPSYS_FUNC(InitPropVariantFromStringVector,  HRESULT (WINAPI *)(PCWSTR *, ULONG, PROPVARIANT *));

    if (missing) {
      // If any of the functions we need is missing, we can't use propsys.dll.
      FreeLibrary(m_propsys_dll);
      m_propsys_dll = NULL;
    }

  }

  s_initialized = true;
}

// Revert all globals, releasing resources gently.  finalize() may be
// called multiple times.
void ModuleGlobals::finalize() throw ()
{
  TRACE(("Finalizing module"));

  s_initialized = false;

  m_PSCreateMemoryPropertyStore = NULL;
  m_PSCreateAdapterFromPropertyStore = NULL;
  m_PSStringFromPropertyKey = NULL;
  m_PSGetNameFromPropertyKey = NULL;
  m_InitPropVariantFromStringVector = NULL;

  if (m_propsys_dll) {
    FreeLibrary(m_propsys_dll);
    m_propsys_dll = NULL;
  }

  if (INVALID_HANDLE_VALUE != m_hLog) {
    CloseHandle(m_hLog);
    m_hLog = INVALID_HANDLE_VALUE;
  }

  m_debug = 0;
}

#if ALLOW_DEBUG

static bool log(LPCSTR fmt, ...) throw()
{
  if (INVALID_HANDLE_VALUE == g_globals.m_hLog) return false;

  TCHAR tfmt[1024];
  StringCbPrintf(tfmt, sizeof(tfmt), TEXT("%hs"), fmt);

  TCHAR text[1024];
  StringCbPrintf(text, sizeof(text), TEXT("%04x:%04x:"), GetCurrentProcessId(), GetCurrentThreadId());
  int const n = lstrlen(text);

  va_list ap;
  va_start(ap, fmt);
  StringCchVPrintf(text + n, sizeof(text) / sizeof(TCHAR) - n, tfmt, ap);
  va_end(ap);

  StringCbCat(text, sizeof(text), TEXT("\r\n"));

  LPSTR outbytes;
#if UNICODE
  CHAR bytes[sizeof(text) + sizeof(text) / 2];
  int result = WideCharToMultiByte(CP_UTF8, 0, text, -1, bytes, sizeof(bytes), NULL, NULL);
  if (result) {
    outbytes = bytes;
  } else {
    outbytes = "(WideCharToMultiByte failed)\r\n";
  }
#else
  outbytes = text;
#endif

  DWORD written;
  WriteFile(g_globals.m_hLog, outbytes, lstrlenA(outbytes), &written, NULL);
  return true;
}

// Possibly create and initialize, then open (again) the log file and
// returns a handle to it.  Note that any unexpected errors in this
// method are fatal, but we can't log the events because the log file
// is not yet ready (and will never be in this session.)
/*static*/
HANDLE ModuleGlobals::prepareLogFile() throw ()
{
  TCHAR log_path[MAX_PATH + 1];
  ZeroMemory(log_path, sizeof(log_path));
#if 1
  // As you will see, the log file is in the user's My Pictures folder
  // in this version.  I don't think it is best, but it's definitely
  // easiest than putting in other more reasonable locations...
  if (FAILED(SHGetFolderPath(NULL, CSIDL_MYPICTURES, NULL, SHGFP_TYPE_CURRENT, log_path))) return INVALID_HANDLE_VALUE;
#else
  // Well, the log is now in the TEMP folder.  Do you think it is
  // better than My Picutres?  The problem is that an average user
  // doesn't know where is TEMP folder, and the actual location of the
  // TEMP folder greately varies between systems, so it's almost
  // impossible to ask a complaining user to grab and send the log
  // file.
  if (0 == GetTempPath(MAX_PATH, log_path)) return INVALID_HANDLE_VALUE;
#endif
  if (FAILED(PathAppend(log_path, TEXT("mobihandlerlog.txt")))) return INVALID_HANDLE_VALUE;

#if UNICODE

  // FILE_SHARE_DELETE is excluded here so that the second CreateFile
  // is guaranteed to open the same object.
  HANDLE hFile1 = CreateFile(log_path, GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
  if (INVALID_HANDLE_VALUE == hFile1) return INVALID_HANDLE_VALUE;

  // The file pointer of hFile1 is at the beginning of the file, so
  // the follwoing UTF-8 BOM is always written there, even when we are
  // appending our log to an existing file.
  static CHAR const BOMCRLF[] = { '\xEF', '\xBB', '\xBF', '\x0D', '\x0A' };
  DWORD written;
  WriteFile(hFile1, BOMCRLF, sizeof(BOMCRLF), &written, NULL);

#endif

  HANDLE hFile2 = CreateFile(log_path, FILE_APPEND_DATA, FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

#if UNICODE
  CloseHandle(hFile1);
#endif

  return hFile2;
}

#endif

// Get an integral registry value data in HKCR tree.  The value may be
// REG_DWORD or REG_SZ.  Returns 0 upon any error.
/*static*/ 
void ModuleGlobals::getRegValue(LPCTSTR subkey, LPCTSTR value, LPDWORD data) throw ()
{
  BYTE blob[16];
  DWORD type = 0, size = sizeof(blob);
  //if (ERROR_SUCCESS == SHGetValue(HKEY_CLASSES_ROOT, subkey, value, &type, blob, &size)) {
  LSTATUS s =  SHGetValue(HKEY_CLASSES_ROOT, subkey, value, &type, blob, &size);
  if (ERROR_SUCCESS == s) {
    if (type == REG_DWORD && size == sizeof(DWORD)) {
      *data = *reinterpret_cast<LPDWORD>(blob);
      return;
    }
    if (type == REG_SZ && size < sizeof(blob)) {
      *data = StrToInt(reinterpret_cast<LPCTSTR>(blob));
      return;
    }
  }
  *data = 0;
}

////////////////////////////////////////////////////////////////
// WIC factory helper
////////////////////////////////////////////////////////////////

static inline HRESULT get_WIC_factory(IWICImagingFactory **factory)
{
  return CoCreateInstance(CLSID_WICImagingFactory,
			  NULL,
			  CLSCTX_INPROC_SERVER,
			  IID_IWICImagingFactory,
			  reinterpret_cast<void **>(factory));
}



////////////////////////////////////////////////////////////////
// The base class for all my COM implementations.  It primarily does
// reference counting.
////////////////////////////////////////////////////////////////

class CUnknown;

#define HANDLE_QUERY(RIID, PPVOBJ, INTERFACE) \
  if (IsEqualIID((RIID), IID_##INTERFACE)) do {		\
    TRACE_QUERY(RIID, #INTERFACE);		\
    *(PPVOBJ) = static_cast<INTERFACE *>(this);		\
    this->AddRef();					\
    return S_OK;					\
  } while (0)

class CUnknown : public IUnknown
{
public:
  CUnknown();
  virtual ~CUnknown();

  // IUnknown
  STDMETHODIMP QueryInterface(REFIID riid, void **ppvObject) throw();
  STDMETHODIMP_(ULONG) AddRef() throw();
  STDMETHODIMP_(ULONG) Release() throw();

public:
  static HRESULT lockServer(BOOL fLock) throw();
  static bool moduleBusy() throw();
  ULONG serial() const throw() { return m_serial; }

private:
  volatile ULONG m_references;
  static ULONG s_module_references;
  ULONG const m_serial;
  static ULONG s_master_serial;
};

ULONG CUnknown::s_module_references = 0;
ULONG CUnknown::s_master_serial = 0;

// The COM document mandates that the reference count maintained by
// AddRef and Release (i.e., CUnknown::m_reference, in this case) must
// be a 32 bit unsigned value, although Win32 provides no unsigned
// versions of InterlockedIncrement and InterlockedDecrement.  Am I
// too square?  FIXME.

static inline ULONG interlockedIncrement(ULONG volatile *p)
{
  return static_cast<ULONG>(InterlockedIncrement(reinterpret_cast<LONG volatile *>(p)));
}

static inline ULONG interlockedDecrement(ULONG volatile *p)
{
  return static_cast<ULONG>(InterlockedDecrement(reinterpret_cast<LONG volatile *>(p)));
}

inline CUnknown::CUnknown() :
  m_references(1),
  m_serial(interlockedIncrement(&s_master_serial))
{
  interlockedIncrement(&s_module_references);
}

inline CUnknown::~CUnknown()
{
  interlockedDecrement(&s_module_references);
}

inline STDMETHODIMP CUnknown::QueryInterface(REFIID riid, void **ppvObject) throw()
{
  HANDLE_QUERY(riid, ppvObject, IUnknown);

  TRACE_QUERY(riid, NULL);

  *ppvObject = NULL;
  return E_NOINTERFACE;
}

ULONG CUnknown::AddRef() throw()
{
  return interlockedIncrement(&m_references);
}

ULONG CUnknown::Release() throw()
{
  ULONG n = interlockedDecrement(&m_references);
  if (n == 0) delete this;
  return n;
}

/*static*/
HRESULT CUnknown::lockServer(BOOL fLock) throw()
{
  if (fLock) {
    interlockedIncrement(&s_module_references);
  } else {
    interlockedDecrement(&s_module_references);
  }
  return S_OK;
}
  
/*static*/
bool CUnknown::moduleBusy() throw()
{
  return s_module_references > 0;
}

////////////////////////////////////////////////////////////////
// IClassFactory implementation
////////////////////////////////////////////////////////////////

template <class T>
static inline HRESULT createInstance(REFIID riid, void **ppvObject) throw()
{
  try {
    *ppvObject = NULL;
    T *obj = new T;
    HRESULT hr = obj->QueryInterface(riid, ppvObject);
    obj->Release();
    return hr;
  } catch (std::bad_alloc) {
    return E_OUTOFMEMORY;
  } catch (...) {
    return E_UNEXPECTED;
  }
}

class CClassFactoryBase : public CUnknown, public IClassFactory
{
public:
  CClassFactoryBase();
  ~CClassFactoryBase();

  // IUnknown
  STDMETHODIMP QueryInterface(REFIID riid, void **ppvObject) throw();
  STDMETHODIMP_(ULONG) AddRef()  throw() { return CUnknown::AddRef(); }
  STDMETHODIMP_(ULONG) Release() throw() { return CUnknown::Release(); }
	
  // IClassFactory
  STDMETHODIMP CreateInstance(IUnknown *pUnkOuter, REFIID riid, void **ppvObject) = 0;
  STDMETHODIMP LockServer(BOOL fLock) throw() { return CUnknown::lockServer(fLock); }
};

CClassFactoryBase::CClassFactoryBase()
{
  TRACE_METHOD("CClassFactoryBase");
}

CClassFactoryBase::~CClassFactoryBase()
{
  TRACE_METHOD("~CClassFactoryBase");
}

STDMETHODIMP CClassFactoryBase::QueryInterface(REFIID riid, void **ppvObject) throw()
{
  HANDLE_QUERY(riid, ppvObject, IClassFactory);
  return CUnknown::QueryInterface(riid, ppvObject);
}

template<class T>
class CClassFactory : public CClassFactoryBase
{
public:
  STDMETHODIMP CreateInstance(IUnknown *pUnkOuter, REFIID riid, void **ppvObject) throw()
  {
    TRACE_METHOD("CreateInstance");

    if (pUnkOuter) {
      *ppvObject = NULL;
      return CLASS_E_NOAGGREGATION;
    }

    return createInstance<T>(riid, ppvObject);
  }
};

////////////////////////////////////////////////////////////////
// MobiBook: a data object that holds metadata of a .mobi file.  Note
// that MobiBook is not a COM object, but inherits CUnknown (hence
// IUnknown) for our own purposes.  We will never expose MobiBook for
// other modules.
////////////////////////////////////////////////////////////////

class MobiBook : public CUnknown
{
protected:
  class CopyStream;

public:
  explicit MobiBook(TString const &filepath);
  explicit MobiBook(IStream *stream);
  ~MobiBook();

private:
  void init();

public:
  bool isSample()   throw() { return m_sample; }
  bool isPersonal() throw() { return m_personal; }
  bool hasDRM()     throw() { return m_drm; }
  TString const &getFilepath()  throw() { return m_filepath; }
  TString const &getName()      throw() { return m_name; }
  TString const &getTitle()     throw() { return m_title; }
  TString const &getAuthor()    throw() { return m_author; }
  TString const &getPublisher() throw() { return m_publisher; }
  TString const &getCreated()   throw() { return m_created; }
  TString const &getPublished() throw() { return m_published; }
  TString const &getLanguage()  throw() { return m_language; }
  TStrVec const &getCreators()  throw() { return m_creators; }
  TStrVec const &getContributors() throw() { return m_contributors; }

public:
  bool isMobi() throw();
  bool hasSources() throw() { return m_source_len > 0; }
  bool hasCover() throw() { return m_cover_len > 0; }
  HRESULT saveSources(IStream *save_stream);
  HRESULT getCover(IWICBitmapSource **);

protected:
  // HRESULT values on analyze() and m_status indicates as follows:
  // S_OK:	this is a valid MOBI file.
  // S_FALSE:	this is *not* a valid MOBI file.
  // E_*:	Unexpected errors on processing; try again later.
  HRESULT analyze() throw();
  HRESULT m_status;

protected:
  // m_filepath and m_stream are exclusive.
  HRESULT getMobiStream(IStream **);
  TString const m_filepath;
  COMPTR(IStream) m_stream;

protected:
  bool m_sample;
  bool m_personal;
  bool m_drm;
  TString m_name;
  TString m_title;
  TString m_author;
  TString m_publisher;
  TString m_created;
  TString m_published;
  TString m_language;
  TStrVec m_creators;
  TStrVec m_contributors;
  UINT32 m_source_off;
  UINT32 m_source_len;
  UINT32 m_cover_off;
  UINT32 m_cover_len;
};

MobiBook::MobiBook(TString const &filepath) :
  m_filepath(filepath)
{
  TRACE_METHOD("MobiBook(filepath)");
  m_name = PathFindFileName(filepath.c_str());
  init();
}

MobiBook::MobiBook(IStream *stream) :
  m_filepath()
{
  TRACE_METHOD("MobiBook(stream)");
  m_stream.replaceWith(stream);
  // We can use IStream::Stat with STATFLAG_DEFAULT to *try* to get
  // the name of stream, if needed.
  init();
}

MobiBook::~MobiBook()
{
  TRACE_METHOD("~MobiBook");
}

void MobiBook::init()
{
  m_status = E_PENDING;
  m_sample = false;
  m_personal = false;
  m_drm = false;
  m_source_off = 0;
  m_source_len = 0;
  m_cover_off = 0;
  m_cover_len = 0;
}

static inline UINT get2(BYTE const *data)
{
  return (data[0] << 8) + data[1];
}

static inline UINT get4(BYTE const *data)
{
  return (data[0] << 24) + (data[1] << 16) + (data[2] << 8) + data[3];
}

static TString load_text(BYTE const *text, int length, UINT codepage)
{
  if (length == 0) return EMPTY;
  int count = MultiByteToWideChar(codepage, 0, reinterpret_cast<LPCSTR>(text), length, NULL, 0);
  if (count == 0) return EMPTY;
  array<WCHAR> buffer(count);
  int again = MultiByteToWideChar(codepage, 0, reinterpret_cast<LPCSTR>(text), length, buffer, count);
  if (again != count) return EMPTY;

  // If use were in ACP build, i.e., UNICODE macro was not defined, we
  // need to use WideCharToMultiByte to convert back the content of
  // the buffer into CP_ACP...
  return TString(buffer, count);
}

static TString get_langname(UINT lang)
{
  if (lang == 0) return EMPTY; // Or, shouldn't we return something like "(unknown)"?  FIXME.
  int count = GetLocaleInfo(static_cast<LCID>(lang), LOCALE_SLANGUAGE, NULL, 0);
  if (count == 0) return EMPTY;
  array<TCHAR> buffer(count);
  int again = GetLocaleInfo(static_cast<LCID>(lang), LOCALE_SLANGUAGE, buffer, count);
  if (count != again) return EMPTY;
  return TString(buffer, count);
}

static TString get_trans_mobi_date(BYTE const *data, UINT length)
{
  WORD ymd[3];
  ZeroMemory(ymd, sizeof(ymd));
  UINT s = 0, i = 0, v = 0;
  TString t = load_text(data, length, CP_UTF8) + TEXT(' ');
  for (TString::const_iterator p = t.begin(); p != t.end() && i < 3; p++) {
    UINT n = *p - TEXT('0');
    if (n > 9) s |= 8;
    switch (s) {
    case 3:
      v = v * 10 + n;
    case 11:
      ymd[2] = ymd[1]; ymd[1] = ymd[0]; ymd[0] = v; v = 0; i++; s = 4;
      break;
    case 5:
      v = v * 10 + n;
    case 9: case 10: case 13:
      ymd[i++] = v; v = 0;
    case 8: case 12:
      s &= 4;
      break;
    default:
      v = v * 10 + n; s++;
      break;
    }
  }

  // I'm not very sure the following special handling is a good thing.  FIXME.
  if (ymd[1] > 12 && ymd[2] <= 12) std::swap(ymd[1], ymd[2]);

  // We should switch to Year-Month format if ymd[2] == 0, and
  // Year-only format if ymd[1] == 0.  Unfortunately, Year-Month is
  // available only on Vista, and Year-only is not available on all
  // Windows...

  SYSTEMTIME st;
  ZeroMemory(&st, sizeof(st));
  st.wYear =  ymd[0];
  st.wMonth = ymd[1];
  st.wDay =   ymd[2];
  int n = GetDateFormat(LOCALE_USER_DEFAULT, DATE_LONGDATE, &st, NULL, NULL, 0);
  if (n <= 0) return t;
  array<TCHAR> buffer(n);
  int m = GetDateFormat(LOCALE_USER_DEFAULT, DATE_LONGDATE, &st, NULL, buffer, buffer.size());
  if (m != n) return t;

  return TString(buffer);
}

bool MobiBook::isMobi() throw()
{
  if (FAILED(m_status)) {
    m_status = analyze();
  }
  return S_OK == m_status;
}

// Returns an IStream to read the MOBI file of this MobiBook.  The
// seek pointer of the returned IStream is not guaranteed; the caller
// should use IStream::Seek before calling IStream::Read to get
// predictable results.
HRESULT MobiBook::getMobiStream(IStream **stream)
{
  if (m_stream) {
    m_stream->AddRef();
    *stream = m_stream;
  } else {
    ENSURE(SHCreateStreamOnFile(m_filepath.c_str(), STGM_READ, stream));
  }
  return S_OK;
}

// Try to analyze the file (stream) given to the constructor as a MOBI
// format and to load various metadata into this object.  Returns S_OK
// if everything went fine.  Returns S_FALSE if the file was
// considered as no MOBI format.  Returns some error result if
// something unexpected happen during the analysis, e.g., no enough
// memory or I/O errors.  As you will see, this is the very heart of
// this application.
HRESULT MobiBook::analyze() throw()
{

#define PREMISE(E) do { if (!(E)) { TRACE(("analyze:Not a MOBI:%d:(%hs)", __LINE__, #E)); return S_FALSE; } } while (0)

  // This limit is just arbitrary and you can adjust it if necessary,
  // but we need some limit anyway, since otherwise the shell (i.e.,
  // Windows Explorer) will freeze on a maliciously crafted fake MOBI
  // file.  The limit is only for the first record despite the name.
  static UINT const MAX_RECORD_SIZE = 64000;

  try {
    array<BYTE> data;

    COMPTR(IStream) stream;
    ENSURE(getMobiStream(&stream));
    
    // We need to know the size of the file in bytes.
    STATSTG statstg;
    ENSURE(stream->Stat(&statstg, STATFLAG_NONAME));

    // a MIBI file can't be larget than 4GB (a restriction of the Palm
    // DB format), so the higher 32 bits of the file size should be
    // zero, and we need to take care of lower 32 bits only.
    // ... Well, you are right.  To be very precise, a MOBI file *can*
    // be larget then 4GB; the largest possible length appears to be
    // (4G + 3) bytes.  Should we take care of the case?  I don't
    // think so...
    UINT const file_size = statstg.cbSize.LowPart;
    PREMISE(statstg.cbSize.HighPart == 0);

    // Make briefly sure this is a MOBI BOOK file.  First, we test
    // against the Palm DB format.  Read the PalmDB header (including
    // the record count.)
    PREMISE(file_size >= 78);
    data.resize(78);
    ENSURE(IStream_seek(stream, 0));
    ENSURE(IStream_Read(stream, data, data.size()));

    // We don't use the name stored at the very beginning of the file
    // (aka, PalmOS name.)  We use the Windows file name instead.  In
    // my experience, the first 32 bytes of many MOBI files don't
    // appear to be used for human readable names.

    // Extract the "creation date" in the Palm DB header.
    UINT creation_date = get4(data + 36);

    // Make sure the type/creator signature is of MOBI.
    PREMISE(0 == memcmp(data + 60, "BOOKMOBI", 8));

    // The size of the record directory must be >= 2.
    UINT record_count = get2(data + 76);
    PREMISE(record_count >= 2);

    // Read the record directory and extract offsets.  We add an extra
    // entry at the end to keep the file length, so that the calculation
    // of the record size is easier.
    PREMISE(file_size >= 78 + record_count * 8);
    data.resize(record_count * 8);
    ENSURE(IStream_Read(stream, data, data.size()));
    array<UINT> records(record_count + 1);
    for (UINT n = 0, pos = 0; n < record_count; n++, pos += 8) {
      records[n] = get4(data + pos);
    }
    records[record_count] = file_size;

    // Make sure that (1) the first record offset points to somewhere
    // after the record directory, (2) the record offsets in the
    // directory are increasing.  (The case of equal offsets is fine,
    // since there may be zero-length records.)
    PREMISE(records[0] >= 78 + record_count * 8);
    for (UINT n = 1; n < records.size(); n++) {
      // a record of size zero is allowed.
      PREMISE(records[n - 1] <= records[n]);
    }

    // OK.  It looks like a Palm-DB file containing a "BOOKMOBI" data.
    
    // Before going further, convert the creation date got from the
    // Palm DB header into our metadata format.  It is more
    // complicated than you might think (although most of the codes
    // just look silly.)
    m_created.clear();
    if (creation_date) {
      ULARGE_INTEGER ut;
      ut.QuadPart = creation_date * 10000000LL
	+ (creation_date & 0x80000000
	   ? 0x153b281e0fb4000LL	// 1904-1-1T00:00Z
	   : 0x19db1ded53e8000LL);	// 1970-1-1T00:00Z
      FILETIME ft;
      ft.dwLowDateTime = ut.LowPart;
      ft.dwHighDateTime = ut.HighPart;
      SYSTEMTIME st;
      if (FileTimeToSystemTime(&ft, &st)) {
	int n = GetDateFormat(LOCALE_USER_DEFAULT, DATE_LONGDATE, &st, NULL, NULL, 0);
	if (n > 0) {
	  array<TCHAR> buffer(n);
	  if (GetDateFormat(LOCALE_USER_DEFAULT, DATE_LONGDATE, &st, NULL, buffer, buffer.size())) {
	    m_created.assign(buffer);
	  }
	}
      }
    }

    // Load and analyze the first record.
    UINT record_size = records[1] - records[0];
    PREMISE(record_size >= 256 && record_size <= MAX_RECORD_SIZE);
    data.resize(record_size);
    ENSURE(IStream_seek(stream, records[0]));
    ENSURE(IStream_Read(stream, data, data.size()));

    // Analyze PDOC header.
    UINT compression = get2(data);
    // PREMISE(compression == 1 || compression == 2 || compression == 0x4448);
    PREMISE(get2(data + 2) == 0);
    // UINT text_length = get4(data + 4);
    m_drm = get2(data + 12) != 0;

    // Analyze MOBI header.
    PREMISE(memcmp(data + 16, "MOBI", 4) == 0);
    UINT mobi_header_length = get4(data + 20);
    PREMISE(mobi_header_length >= 128 && mobi_header_length <= data.size() - 16);

    UINT codepage = get4(data + 28);

    UINT nbi = get4(data + 80);
    UINT title_off = get4(data + 84);
    UINT title_len = get4(data + 88);
    // The title data is in the record 0 and after the MOBI header block.
    PREMISE(title_off >= mobi_header_length + 16 && title_off + title_len <= data.size());
    m_title = load_text(data + title_off, title_len, codepage);

    m_language = get_langname(get4(data + 92));
    
    UINT image_record_base = get4(data + 108);
    PREMISE(image_record_base <= record_count);

    UINT cover_image = 0;

    // Analyze EXTH header.  Note that this header is believed to be
    // optional.  Be prepared.
    BYTE const *const exth = data + 16 + mobi_header_length;
    if (exth + 12 < data + data.size() && memcmp(exth, "EXTH", 4) == 0) {

      UINT const exth_header_length = get4(exth + 4);
      PREMISE(exth_header_length >= 8 && exth_header_length <= data.size() - (exth - data));

      // creator (100) and contributor (108) require special handling,
      // because they can occur multiple times, and we need to
      // concatenate them into a single property "author(s)".
      m_creators.clear();
      m_contributors.clear();

      BYTE const *p = exth + 12; 
      for (UINT count = get4(exth + 8); count > 0; --count) {
	PREMISE(p + 8 <= exth + exth_header_length);
	UINT type = get4(p);
	UINT length = get4(p + 4);
	PREMISE(length >= 8 && p + length <= exth + exth_header_length);
	switch (type) {
	case 100: // author/creator
	  m_creators.push_back(load_text(p + 8, length - 8, codepage));
	  break;
	case 101: // publisher
	  m_publisher = load_text(p + 8, length - 8, codepage);
	  break;
	  //	case 103: // description
	  //	  m_description = load_text(p + 8, length - 8, codepage);
	  //	  break;
	  //	case 104: // ISBN
	  //	  m_isbn = load_text(p + 8, length - 8, codepage);
	  //	  break;
	case 106: // publishing date
	  m_published = get_trans_mobi_date(p + 8, length - 8);
	  break;
	case 108: // contributor
	  m_contributors.push_back(load_text(p + 8, length - 8, codepage));
	  break;
	  //	case 109: // rights
	  //	  m_rights = load_text(p + 8, length - 8, codepage);
	  //	  break;
	  //	case 113: // ASIN
	  //	  m_asin = load_text(p + 8, length - 8, codepage);
	  //	  break;
	case 115: // sample
	  m_sample = true;
	  break;
	case 201: //cover image offset
	case 202: //preview image offset
	  // A MOBI file usually contains two cover images; a
	  // full-size image and a preview image.  The preview image
	  // is smaller and we can process it faster (low overhead),
	  // although it's quality is usually worse than the full-size
	  // image.  Apparently there is a trade-off.  I introduced an
	  // option (UseMobiThumbnail) to switch between them.  A user
	  // with less powerful PC can turn it on to prefer the
	  // preview image over full-size one so that the image comes
	  // up quickly.
	  if (length == 12 &&
	      (cover_image == 0 ||
	       type == (g_globals.m_useMobiThumbnail ? 202 : 201))) {
	    cover_image = image_record_base + get4(p + 8);
	    PREMISE(cover_image < record_count);
	  }
	  break;
	  //	case 404: // text-to-speach disabled
	  //	  m_tts_disabled = (data[p + 8] != 0);
	  //	  break;
	case 501: // cdetype
	  m_personal = (length == 12 && memcmp(p + 8, "PDOC", 4) == 0);
	  break;
	}
	p += length;
      }

      // Bind all creators and contributors to form a single author string.
      static TString const SEPARATOR(TEXT("; "));
      m_author.clear();
      for (TStrVec::const_iterator i = m_creators.begin(); i != m_creators.end(); i++) {
	m_author += *i + SEPARATOR;
      }
      for (TStrVec::const_iterator i = m_contributors.begin(); i != m_contributors.end(); i++) {
	m_author += *i + SEPARATOR;
      }
      if (!m_author.empty()) m_author.resize(m_author.size() - SEPARATOR.size());
    }

    // Take care of a cover image for preview.
    m_cover_off = 0;
    m_cover_len = 0;
    if (!cover_image) {
      // EXTH header contained no cover image info.  Use the content
      // of a record at _first_image_ in place of a cover if it
      // appears to be in a known image file format.  I believe it
      // matches the Kindle's own behaviour.
      // ... However, be careful, if the MOBI file contained no images
      // at all, image_record_base points to a wrong record, or even
      // worse, at the end of the file.  We need to identify the case
      // carefully.  Probably the data we call "image record base"
      // (i.e., offset 108 in the MOBI header) is *not* really an
      // "image record base".
      if (image_record_base > 0 && image_record_base < record_count) {
	data.resize(4);
	ENSURE(IStream_seek(stream, records[image_record_base]));
	ENSURE(IStream_Read(stream, data, data.size()));
	if (memcmp(data, "GIF8", 4) == 0 ||
	    memcmp(data, "\x89PNG", 4) == 0 ||
	    memcmp(data, "\xFF\xD8\xFF\xE0", 4) == 0) {
	  cover_image = image_record_base;
	}
      }
    }
    if (cover_image) {
      m_cover_off = records[cover_image];
      m_cover_len = records[cover_image + 1] - records[cover_image];
    }

    // Look for a SRCS record.  It is usually a second-from-the-last
    // record, so we start examining from there.
    m_source_off = m_source_len = 0;
    data.resize(20);
    for (UINT n = record_count - 2; n >= nbi; --n) {
      // I'm not sure the minimum size of a valid and complete zip
      // archive, but it should be around 140 bytes.  Hence, a valid
      // SRCS record can't be smaller than 128 bytes.
      if (records[n + 1] - records[n] < data.size()) continue;
      ENSURE(IStream_seek(stream, records[n]));
      ENSURE(IStream_Read(stream, data, data.size()));
      if (memcmp(data, "SRCS", 4) == 0 && memcmp(data + 16, "PK\3\4", 4) == 0) {
	m_source_off = records[n] + 16;
	m_source_len = records[n + 1] - m_source_off;
	break;
      }
    }

    // Done.
    return S_OK;

  } catch (std::bad_alloc) {
    return E_OUTOFMEMORY;
  } catch (...) {
    return E_UNEXPECTED;
  }
}

HRESULT MobiBook::saveSources(IStream *save_stream)
{
  if (save_stream == NULL) return E_POINTER;
  if (m_source_len == 0) return E_INVALIDARG;

  COMPTR(IStream) stream;
  ENSURE(getMobiStream(&stream));
  ENSURE(IStream_seek(stream, m_source_off));
  ENSURE(IStream_copy(stream, save_stream, m_source_len));

  return S_OK;
}

HRESULT MobiBook::getCover(IWICBitmapSource **out)
{
  *out = NULL;
  if (m_cover_len == 0) return E_FAIL;

  COMPTR(IWICImagingFactory) factory;
  ENSURE(get_WIC_factory(&factory));

  COMPTR(IStream) stream;
  ENSURE(getMobiStream(&stream));
  
  COMPTR(IWICStream) cover;
  ENSURE(factory->CreateStream(&cover));

  ULARGE_INTEGER off, len;
  off.QuadPart = m_cover_off;
  len.QuadPart = m_cover_len;
  ENSURE(cover->InitializeFromIStreamRegion(stream, off, len));
  
  COMPTR(IWICBitmapDecoder) decoder;
  ENSURE(factory->CreateDecoderFromStream(cover, NULL, WICDecodeMetadataCacheOnDemand, &decoder));

  COMPTR(IWICBitmapFrameDecode) frame;
  ENSURE(decoder->GetFrame(0, &frame));

  ENSURE(frame->QueryInterface(IID_IWICBitmapSource, reinterpret_cast<void **>(out)));
  return S_OK;
}

////////////////////////////////////////////////////////////////
// The main COM object implementation.  It is an all-in-one
// implementation for a property sheet extension, a thumbnail
// extension, and a property system extension.
////////////////////////////////////////////////////////////////

class MobiHandler :
  CUnknown,
  IShellExtInit, IInitializeWithStream, IInitializeWithFile, IPersistFile,
  IShellPropSheetExt, IExtractImage, IThumbnailProvider, IPropertySetStorage, IPropertyStore, IPropertyStoreCapabilities
{
public:
  MobiHandler();
  virtual ~MobiHandler();

  // IUnknwon
  STDMETHODIMP QueryInterface(REFIID riid, void **ppvObject) throw();
  STDMETHODIMP_(ULONG) AddRef()  throw() { return CUnknown::AddRef(); }
  STDMETHODIMP_(ULONG) Release() throw() { return CUnknown::Release(); }

  // IShellExtInit
  STDMETHODIMP Initialize(LPCITEMIDLIST pidlFolder, IDataObject *pdtobj, HKEY hkeyProgID) throw();

  // IInitializeWithSteram
  STDMETHODIMP Initialize(IStream *pstream, DWORD grfMode) throw();

  // IInitializeWithFile
  STDMETHODIMP Initialize(LPCWSTR pszFilePath, DWORD grfMode) throw();

  // IPersist (through IpersistFile)
  STDMETHODIMP GetClassID(CLSID *pClassID) throw();

  // IPersistFile
  STDMETHODIMP Load(LPCOLESTR pszFileName, DWORD dwMode) throw();
  STDMETHODIMP GetCurFile(LPOLESTR *ppszFileName) throw() { ppszFileName = NULL; return E_NOTIMPL; }
  STDMETHODIMP IsDirty() throw() { return E_NOTIMPL; }
  STDMETHODIMP Save(LPCOLESTR pszFileName, BOOL fRemember) throw() { return E_NOTIMPL; }
  STDMETHODIMP SaveCompleted(LPCOLESTR pszFileName) throw() { return E_NOTIMPL; }

  // IShellPropSheetExt
  STDMETHODIMP AddPages(LPFNADDPROPSHEETPAGE pfnAddPage, LPARAM lParam) throw();
  STDMETHODIMP ReplacePage(UINT uPageID, LPFNADDPROPSHEETPAGE pfnReplacePage, LPARAM lParam) throw() { return E_NOTIMPL; }

  // IExtractImage
  STDMETHODIMP GetLocation(LPWSTR pszPathBuffer, DWORD cchMax, DWORD *pdwPriority, const SIZE *prgSize, DWORD dwRecClrDepth, DWORD *pdwFlags) throw();
  STDMETHODIMP Extract(HBITMAP *phBmpImage) throw();

  // IThumbnailProvider
  STDMETHODIMP GetThumbnail(UINT cx, HBITMAP *phbmp, WTS_ALPHATYPE *pdwAlpha) throw();

  // IPropertySetStorage
  STDMETHODIMP Create(REFFMTID fmtid, const CLSID *pclsid, DWORD grfFlags, DWORD grfMode, IPropertyStorage **ppPropStg) throw();
  STDMETHODIMP Delete(REFFMTID fmtid) throw();
  STDMETHODIMP Enum(IEnumSTATPROPSETSTG **ppenum) throw();
  STDMETHODIMP Open(REFFMTID fmtid, DWORD grfMode, IPropertyStorage **ppPropStg) throw();

  // IPropertyStore
  STDMETHODIMP Commit() throw();
  STDMETHODIMP GetAt(DWORD iProp, PROPERTYKEY *pkey) throw();
  STDMETHODIMP GetCount(DWORD *cProps) throw();
  STDMETHODIMP GetValue(REFPROPERTYKEY key, PROPVARIANT *pv) throw();
  STDMETHODIMP SetValue(REFPROPERTYKEY key, REFPROPVARIANT propvar) throw();

  // IPropertyStoreCapabilities
  STDMETHODIMP IsPropertyWritable(REFPROPERTYKEY key) throw();

protected:
  void initDialog();
  void exitDialog();
  void moveBook(int);
  void updateBook();
  void updateLayout(bool isMobi);
  void assignCoverBitmap(HBITMAP hBitmap);
  HRESULT saveSources() throw();
  HRESULT updateCoverBitmap(MobiBook *book);
  HRESULT prepareProperties(bool withStorage);

protected:
  void setHWND(HWND hwnd) { m_hwnd = hwnd; }
  INT_PTR dialogProc(UINT uMsg, WPARAM wParam, LPARAM lParam);
  static INT_PTR CALLBACK DialogProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam) throw();
  static UINT CALLBACK PropSheetPageProc(HWND hwnd, UINT uMsg, LPPROPSHEETPAGE ppsp);

protected:
  MobiBook *currentBook() { return m_books[m_current_book]; }
  void releaseBooks();

protected:
  std::vector<MobiBook *> m_books;
  UINT m_current_book;
  HWND m_hwnd;
  HBITMAP m_nocover;
  HBITMAP m_pending_bitmap;
  COMPTR(IPropertyStore) m_prop_store;
  COMPTR(IPropertySetStorage) m_prop_storage;
};

MobiHandler::MobiHandler()
{
  TRACE_METHOD("MobiHandler");
  m_current_book = 0;
  m_nocover = NULL;
  m_pending_bitmap = NULL;
}

MobiHandler::~MobiHandler()
{
  releaseBooks();
  if (m_pending_bitmap) {
    DeleteObject(m_pending_bitmap);
  }
  // We should never see a non-null m_nocover here.
  TRACE_METHOD("~MobiHandler");
}

void MobiHandler::releaseBooks()
{
  for (std::vector<MobiBook *>::reverse_iterator i = m_books.rbegin(); i != m_books.rend(); i++) {
    delete *i;
  }
  m_books.clear();
}

STDMETHODIMP MobiHandler::QueryInterface(REFIID riid, void **ppvObject)
{
  HANDLE_QUERY(riid, ppvObject, IShellPropSheetExt);
  HANDLE_QUERY(riid, ppvObject, IInitializeWithStream);
  HANDLE_QUERY(riid, ppvObject, IInitializeWithFile);
  HANDLE_QUERY(riid, ppvObject, IShellExtInit);
  HANDLE_QUERY(riid, ppvObject, IPersist);
  HANDLE_QUERY(riid, ppvObject, IPersistFile);
  HANDLE_QUERY(riid, ppvObject, IExtractImage);
  HANDLE_QUERY(riid, ppvObject, IThumbnailProvider);
  HANDLE_QUERY(riid, ppvObject, IPropertyStore);
  HANDLE_QUERY(riid, ppvObject, IPropertyStoreCapabilities);

  // We rely on the propsys.dll to support IPropertySetStorage.
  if (g_globals.m_propsys_dll) {
    HANDLE_QUERY(riid, ppvObject, IPropertySetStorage);
  }

  return CUnknown::QueryInterface(riid, ppvObject);
}

STDMETHODIMP MobiHandler::Initialize(LPCITEMIDLIST pidlFolder, IDataObject *pdtobj, HKEY hkeyProgID) throw()
{
  TRACE_METHOD("Initialize(LPCITEMIDLIST, IDataObject *, HKEY)");

  if (!m_books.empty()) return HRESULT_FROM_WIN32(ERROR_ALREADY_INITIALIZED);

  try {

    FORMATETC formatetc;
    ZeroMemory(&formatetc, sizeof(formatetc));
    formatetc.cfFormat = CF_HDROP;
    formatetc.ptd = NULL;
    formatetc.dwAspect = DVASPECT_CONTENT;
    formatetc.lindex = -1;
    formatetc.tymed = TYMED_HGLOBAL;
    
    auto_STGMEDIUM medium;
    ENSURE(pdtobj->GetData(&formatetc, &medium));
    if (TYMED_HGLOBAL != medium.tymed) return E_FAIL;

    auto_HGLOBAL<HDROP> hDrop(medium.hGlobal);
    if (NULL == hDrop) return E_FAIL;

    UINT nFiles = DragQueryFile(hDrop, (UINT)-1, NULL, 0);
    m_books.reserve(nFiles);
    
    for (UINT n = 0; n < nFiles; n++) {
      UINT length = DragQueryFile(hDrop, n, NULL, 0);
      if (length == 0) continue;
      array<TCHAR> filename(length + 2);
      UINT copied = DragQueryFile(hDrop, n, filename, length + 1);
      if (copied != length) continue;
      m_books.push_back(new MobiBook(TString(filename, filename.size())));
    }

    return m_books.empty() ? E_FAIL : S_OK;

  } catch (std::bad_alloc) {
    return E_OUTOFMEMORY;
  } catch (...) {
    return E_UNEXPECTED;
  }
}

STDMETHODIMP MobiHandler::Initialize(IStream *pstream, DWORD grfMode) throw()
{
  TRACE_METHOD("Initialize(IStream *, DWORD)");

  if (!m_books.empty()) return HRESULT_FROM_WIN32(ERROR_ALREADY_INITIALIZED);
  if (grfMode == STGM_READWRITE) return STG_E_ACCESSDENIED;

  try {

    m_books.push_back(new MobiBook(pstream));
    return S_OK;

  } catch (std::bad_alloc) {
    return E_OUTOFMEMORY;
  } catch (...) {
    return E_FAIL;
  }
}

STDMETHODIMP MobiHandler::Initialize(LPCWSTR pszFilePath, DWORD grfMode) throw()
{
  TRACE_METHOD("Initialize(LPCWSTR, DWORD)");

  if (!m_books.empty()) return HRESULT_FROM_WIN32(ERROR_ALREADY_INITIALIZED);
  if (grfMode == STGM_READWRITE) return STG_E_ACCESSDENIED;

  try {

    m_books.push_back(new MobiBook(pszFilePath));
    return S_OK;

  } catch (std::bad_alloc) {
    return E_OUTOFMEMORY;
  } catch (...) {
    return E_FAIL;
  }
}

STDMETHODIMP MobiHandler::GetClassID(CLSID *pClassID) throw()
{
  TRACE_METHOD("GetClassID");

  *pClassID = CLSID_MobiHandler;
  return S_OK;
}

STDMETHODIMP MobiHandler::Load(LPCOLESTR pszFileName, DWORD dwMode) throw()
{
  TRACE_METHOD("Load");

  // The MSDN document says dwMode is just a suggestion, so I decided
  // to ignore it.  BTW, I'm not sure what we should do if this method
  // is called *after* initialization.  Just following
  // InitializeWithFIle seems safe.  This probably is completely wrong
  // for the original idea of IPersist* interfaces...
  if (!m_books.empty()) return HRESULT_FROM_WIN32(ERROR_ALREADY_INITIALIZED);
  try {

    m_books.push_back(new MobiBook(pszFileName));
    return S_OK;

  } catch (std::bad_alloc) {
    return E_OUTOFMEMORY;
  } catch (...) {
    return E_FAIL;
  }
}

STDMETHODIMP MobiHandler::AddPages(LPFNADDPROPSHEETPAGE pfnAddPage, LPARAM lParam) throw()
{
  TRACE_METHOD("AddPages");

  // Unless we have at least one valid mobi book, we don't add MOBI
  // property page.  I consider this is a successful result.
  if (m_books.empty()) return S_FALSE;

  INITCOMMONCONTROLSEX iccx;
  ZeroMemory(&iccx, sizeof(iccx));
  iccx.dwSize = sizeof(iccx);
  iccx.dwICC = ICC_STANDARD_CLASSES | ICC_UPDOWN_CLASS;
  InitCommonControlsEx(&iccx);

  PROPSHEETPAGE psp;
  ZeroMemory(&psp, sizeof(psp));
  psp.dwSize = sizeof(PROPSHEETPAGE);
  psp.dwFlags = PSP_USECALLBACK;
  psp.hInstance = g_globals.s_hInstDll;
  psp.pszTemplate = MAKEINTRESOURCE(IDD_MOBIPAGE);
  psp.pfnDlgProc = DialogProc;
  psp.lParam = reinterpret_cast<LPARAM>(this);
  psp.pfnCallback = PropSheetPageProc;

  HPROPSHEETPAGE hPage = CreatePropertySheetPage(&psp);
  if (hPage == NULL) {
    return E_OUTOFMEMORY;
  }

  if (!pfnAddPage(hPage, lParam)) {
    DestroyPropertySheetPage(hPage);
    return E_FAIL;
  }

  // psp.lParam now owns *this, so invoke AddRef().  The corresponding
  // Release() will be invoked in PropSheetPageProc().
  AddRef();

  return S_OK;
}

void MobiHandler::initDialog()
{
  m_current_book = 0;

  // Keep the "NO COVER" image handy.
  m_nocover = reinterpret_cast<HBITMAP>(SendDlgItemMessage(m_hwnd, IDC_STATIC_COVER, STM_GETIMAGE, IMAGE_BITMAP, 0));

  // Adjust the apperance of the up-down control, based on whether we
  // have multiple files.
  if (m_books.size() < 2) {
    // Hide the up-down control; we don't allow switching of files.
    ShowWindow(GetDlgItem(m_hwnd, IDC_UPDOWN_NAME), SW_HIDE);
  } else {
    // Initialize the up-down control appropriately.
    SendDlgItemMessage(m_hwnd, IDC_UPDOWN_NAME, UDM_SETBUDDY, reinterpret_cast<WPARAM>(GetDlgItem(m_hwnd, IDC_EDIT_NAME)), 0);
    SendDlgItemMessage(m_hwnd, IDC_UPDOWN_NAME, UDM_SETRANGE32, m_books.size() - 1, 0);
    SendDlgItemMessage(m_hwnd, IDC_UPDOWN_NAME, UDM_SETPOS32, 0, 0);
  }

  updateBook();
}

void MobiHandler::exitDialog()
{
  assignCoverBitmap(m_nocover);
  m_nocover = NULL;
}

void MobiHandler::moveBook(int pos)
{
  m_current_book = clip(static_cast<UINT>(pos), 0, m_books.size() - 1);
  updateBook();
}

void MobiHandler::updateBook()
{
  MobiBook *book = currentBook();
  updateLayout(book->isMobi());
  SetDlgItemText(m_hwnd, IDC_EDIT_NAME, book->getName().c_str());
  if (book->isMobi()) {
    SetDlgItemText(m_hwnd, IDC_EDIT_TITLE, book->getTitle().c_str());
    SetDlgItemText(m_hwnd, IDC_EDIT_AUTHOR, book->getAuthor().c_str());
    SetDlgItemText(m_hwnd, IDC_EDIT_PUBLISHER, book->getPublisher().c_str());
    SetDlgItemText(m_hwnd, IDC_EDIT_CREATED, book->getCreated().c_str());
    SetDlgItemText(m_hwnd, IDC_EDIT_PUBLISHED, book->getPublished().c_str());
    SetDlgItemText(m_hwnd, IDC_EDIT_LANGUAGE, book->getLanguage().c_str());
    CheckDlgButton(m_hwnd, IDC_CHECK_SAMPLE, book->isSample() ? BST_CHECKED : BST_UNCHECKED);
    CheckDlgButton(m_hwnd, IDC_CHECK_PERSONAL, book->isPersonal() ? BST_CHECKED : BST_UNCHECKED);
    CheckDlgButton(m_hwnd, IDC_CHECK_DRM, book->hasDRM() ? BST_CHECKED : BST_UNCHECKED);
    EnableWindow(GetDlgItem(m_hwnd, IDC_BUTTON_SOURCES), book->hasSources());
    updateCoverBitmap(book);
  }
}

void MobiHandler::updateLayout(bool isMobi)
{
  for (int item = IDC_GROUP_MOBI_BEGIN; item < IDC_GROUP_MOBI_END; item++) {
    HWND hwnd = GetDlgItem(m_hwnd, item);
    if (hwnd) ShowWindow(hwnd, isMobi ? SW_SHOW : SW_HIDE);
  }
  for (int item = IDC_GROUP_NOTMOBI_BEGIN; item < IDC_GROUP_NOTMOBI_END; item++) {
    HWND hwnd = GetDlgItem(m_hwnd, item);
    if (hwnd) ShowWindow(hwnd, isMobi ? SW_HIDE : SW_SHOW);
  }
}

void MobiHandler::assignCoverBitmap(HBITMAP hBitmap)
{
  HBITMAP old_bitmap = reinterpret_cast<HBITMAP>(SendDlgItemMessage(m_hwnd, IDC_STATIC_COVER, STM_SETIMAGE, IMAGE_BITMAP, reinterpret_cast<LPARAM>(hBitmap)));
  if (old_bitmap != NULL && old_bitmap != m_nocover && old_bitmap != hBitmap) {
    DeleteObject(old_bitmap);
  }
  HBITMAP current_bitmap = reinterpret_cast<HBITMAP>(SendDlgItemMessage(m_hwnd, IDC_STATIC_COVER, STM_GETIMAGE, IMAGE_BITMAP, 0));
  if (hBitmap != NULL  && hBitmap != m_nocover && hBitmap != current_bitmap) {
    DeleteObject(hBitmap);
  }
}

HRESULT MobiHandler::updateCoverBitmap(MobiBook *book)
{
  // The COVER_BORDER_WIDTH should match with the baked border in
  // nocover.bmp to provide a consistent UI.
  UINT const COVER_BORDER_WIDTH = 1;

  if (!book || !book->isMobi() || !book->hasCover()) {
    assignCoverBitmap(m_nocover);
    return S_OK;
  }

  // Scale the bitmap so that it just fits in the SS_BITMAP control
  // from its inside.

  WICPixelFormatGUID format;

  COMPTR(IWICImagingFactory) factory;
  ENSURE(get_WIC_factory(&factory));

  COMPTR(IWICBitmapSource) cover;
  ENSURE(book->getCover(&cover));

  ENSURE(cover->GetPixelFormat(&format));
  if (!IsEqualGUID(format, GUID_WICPixelFormat32bppBGR)) {
    COMPTR(IWICFormatConverter) converter;
    ENSURE(factory->CreateFormatConverter(&converter));
    ENSURE(converter->Initialize(cover, GUID_WICPixelFormat32bppBGR, WICBitmapDitherTypeNone, NULL, 0.0, WICBitmapPaletteTypeCustom));
    cover.replaceWith(converter);
  }

  UINT width, height;
  ENSURE(cover->GetSize(&width, &height));
  if (width == 0 || height == 0) return E_FAIL;

  RECT rect;
  ZeroMemory(&rect, sizeof(rect));
  if (!GetWindowRect(GetDlgItem(m_hwnd, IDC_STATIC_COVER), &rect)) return E_FAIL;
  UINT x = rect.right - rect.left - COVER_BORDER_WIDTH * 2;
  UINT y = rect.bottom - rect.top - COVER_BORDER_WIDTH * 2;

  if (width > x || height > y || (width < x && height < y)) {

    if (width * y > height * x) {
      y = clip((x * height + width / 2) / width, 1, y);
    } else {
      x = clip((y * width + height / 2) / height, 1, x);
    }

    COMPTR(IWICBitmapScaler) scaler;
    ENSURE(factory->CreateBitmapScaler(&scaler));
    ENSURE(scaler->Initialize(cover, x, y, WICBitmapInterpolationModeFant));
    cover.replaceWith(scaler);
  }

#if 0
  ENSURE(source->GetPixelFormat(&format));
  if (!IsEqualGUID(format, GUID_WICPixelFormat32bppBGR)) {
    COMPTR(IWICFormatConverter) converter;
    ENSURE(factory->CreateFormatConverter(&converter));
    ENSURE(converter->Initialize(source, GUID_WICPixelFormat32bppBGR, WICBitmapDitherTypeNone, NULL, 0.0, WICBitmapPaletteTypeCustom));
    cover.replaceWith(converter);
  }
#endif

  x += COVER_BORDER_WIDTH * 2;
  y += COVER_BORDER_WIDTH * 2;

  array<UINT32> bits(x * y);
  std::fill(bits.begin(), bits.end(), 0xFF000000);
  UINT const offset = x * COVER_BORDER_WIDTH + COVER_BORDER_WIDTH;
  ENSURE(cover->CopyPixels(NULL, x * sizeof(UINT32), (bits.size() - offset) * sizeof(UINT32), reinterpret_cast<LPBYTE>(bits + offset)));

  BITMAPINFO bmi;
  ZeroMemory(&bmi, sizeof(bmi));
  bmi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  bmi.bmiHeader.biWidth = x;
  bmi.bmiHeader.biHeight = -(INT)y;
  bmi.bmiHeader.biPlanes = 1;
  bmi.bmiHeader.biBitCount = 32;
  bmi.bmiHeader.biCompression = BI_RGB;

  HDC hDC = GetDC(m_hwnd);
  HBITMAP hBitmap = CreateDIBitmap(hDC, &bmi.bmiHeader, CBM_INIT, bits, &bmi, 0);
  ReleaseDC(m_hwnd, hDC);

  assignCoverBitmap(hBitmap);
  return S_OK;
}

HRESULT MobiHandler::saveSources()
{
  HRESULT hr = S_OK;
  MobiBook *book = currentBook();

  try {

    // Allocate a buffer to receive the user's response, and fill it
    // with the default.  Yes, thefollowing size of the buffer is too
    // long.
    TString const &filepath = book->getFilepath();
    array<TCHAR> savename(filepath.size() + MAX_PATH);
    lstrcpy(savename, filepath.c_str());
    lstrcpy(PathFindExtension(savename), TEXT(".zip"));

    // Open a dialog box and receive the path to save the sources into.
    OPENFILENAME ofn;
    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = m_hwnd;
    ofn.lpstrFilter = TEXT("ZIP Files\0*.zip\0All Files\0*.*\0\0");
    ofn.lpstrFile = savename;
    ofn.nMaxFile = savename.size();
    ofn.Flags = OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST | OFN_NOCHANGEDIR;
    ofn.lpstrDefExt = TEXT("zip");
    if (!GetSaveFileName(&ofn)) {
      // It is an expected condition that the user cancels the
      // operation, so we should not show a "Save failed" message,
      // although we return a failure HRESULT code (as
      // IFileSaveDialog::Show does.)
      return HRESULT_FROM_WIN32(ERROR_CANCELLED);
    }

    // Save sources into the specified file.  We don't follow ENSURE
    // convention here, since we need to show a friendly message upon
    // any failure.
    COMPTR(IStream) dst;
    hr = SHCreateStreamOnFile(savename, STGM_WRITE | STGM_CREATE, &dst);
    if (SUCCEEDED(hr)) {
      hr = book->saveSources(dst);
    }

  } catch (std::bad_alloc) {
    hr = E_OUTOFMEMORY;
  } catch (...) {
    hr = E_UNEXPECTED;
  }

  if (FAILED(hr)) {
    MessageBox(m_hwnd, TEXT("Save Failed."), book->getName().c_str(), MB_OK);
  }
  return hr;
}

INT_PTR MobiHandler::dialogProc(UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  switch (uMsg) {

  case WM_INITDIALOG:
    {
      initDialog();
      return TRUE;
    }
    break;

  case WM_COMMAND:
    {
      switch (wParam) {
      case MAKEWPARAM(IDC_BUTTON_SOURCES, BN_CLICKED):
	saveSources();
	break;
      }
    }
    break;

  case WM_NOTIFY:
    {
      LPNMHDR lpnmhdr = reinterpret_cast<LPNMHDR>(lParam);
      switch (lpnmhdr->idFrom) {
      case IDC_UPDOWN_NAME:
	LPNMUPDOWN lpnmupdown = reinterpret_cast<LPNMUPDOWN>(lParam);
	moveBook(lpnmupdown->iPos + lpnmupdown->iDelta);
	break;
      }
    }
    break;

  }

  return FALSE;
}

/* static */
INT_PTR CALLBACK MobiHandler::DialogProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  if (uMsg == WM_INITDIALOG) {
    // This is the first time we see the hwnd for our property page
    // and the pointer to its corresponding MobiHandler object at a
    // same time.  The window long ptr at GWLP_USERDATA or
    // MobiHandler::m_hwnd are not set yet.  Tie them now, so that we
    // can access from from each side freely.
    LPPROPSHEETPAGE ppsp = reinterpret_cast<LPPROPSHEETPAGE>(lParam);		
    reinterpret_cast<MobiHandler *>(ppsp->lParam)->setHWND(hwnd);
    SetWindowLongPtr(hwnd, GWLP_USERDATA, ppsp->lParam);
  }

  MobiHandler *handler = reinterpret_cast<MobiHandler *>(GetWindowLongPtr(hwnd, GWLP_USERDATA));
  if (handler) {
    try {
      return handler->dialogProc(uMsg, wParam, lParam);
    } catch (...) {
    }
  }

  return FALSE;
}

/* static */
UINT CALLBACK MobiHandler::PropSheetPageProc(HWND hwnd, UINT uMsg, LPPROPSHEETPAGE ppsp)
{
  if (uMsg == PSPCB_RELEASE) {
    MobiHandler *handler = reinterpret_cast<MobiHandler *>(ppsp->lParam);
    if (handler) {
      try {
	handler->exitDialog();
	// Untie the page and the MobiHandler object, then release the latter.
	SetWindowLongPtr(hwnd, GWLP_USERDATA, 0);
	handler->Release();
	ppsp->lParam = 0;
      } catch (...) {
      }
    }
  }
  return 1;
}

STDMETHODIMP MobiHandler::GetLocation(LPWSTR pszPathBuffer, DWORD cchMax, DWORD *pdwPriority, const SIZE *prgSize, DWORD dwRecClrDepth, DWORD *pdwFlags)
{
  TRACE_METHOD("GetLocation");

  try {

    if (m_books.empty()) return E_FAIL;

    // If the book has no cover image in it, Explorer appears to
    // expect IExtractImage::GetLocation to complete successfully and
    // the succeeding IExtractImage::Extract to return E_NOTIMPL.  So,
    // when the book has no cover, we return S_OK with
    // m_pending_bitmap set to NULL.  The following steps (i.e.,
    // IExtractImage::Extract or the latter half of
    // IThumbnailProvider::GetThumbnail) should take care of it and
    // should return E_NOTIMPL to indicate the case.
    // 
    // The "not mobi" case is handled in a same way, since K4PC puts
    // some no mobi file (e.g., Topaz) under the same class as mobi,
    // and we want to show the ordinary .tpz icon for them.

    m_pending_bitmap = NULL;
    if (!m_books.front()->isMobi() || !m_books.front()->hasCover()) return S_OK;

    if (pszPathBuffer) {
      TString const &filepath = m_books.front()->getFilepath();
      // if (filepath.empty()) return E_FAIL;
      if (filepath.size() >= cchMax) return E_OUTOFMEMORY;
      lstrcpy(pszPathBuffer, filepath.c_str());
    }

    if (pdwPriority) *pdwPriority = IEIT_PRIORITY_NORMAL;
    *pdwFlags |= IEIFLAG_REFRESH;

    COMPTR(IWICBitmapSource) cover;
    ENSURE(m_books.front()->getCover(&cover));

    COMPTR(IWICImagingFactory) factory;
    ENSURE(get_WIC_factory(&factory));

    // We ignore dwRecClrDepth and go with our own (always 32bpp BGR),
    // because MSDN says dwRecClrDepth is just a recommendation,
    // because we anyway need 32bpp as an intermediate format so that
    // the scaling works well, and because our code depends on the
    // pixel depth and we need additional codes if we support other
    // formats.
    GUID format;
    ENSURE(cover->GetPixelFormat(&format));
    if (!IsEqualGUID(format, GUID_WICPixelFormat32bppBGR)) {
      COMPTR(IWICFormatConverter) converter;
      ENSURE(factory->CreateFormatConverter(&converter));
      ENSURE(converter->Initialize(cover, GUID_WICPixelFormat32bppBGR, WICBitmapDitherTypeNone, NULL, 0.0, WICBitmapPaletteTypeCustom));
      cover.replaceWith(converter);
    }

    UINT width, height;
    ENSURE(cover->GetSize(&width, &height));
    if (width == 0 || height == 0) return E_FAIL;

    UINT cx = prgSize->cx;
    UINT cy = prgSize->cy;
  
    if (*pdwFlags & (IEIFLAG_ORIGSIZE | IEIFLAG_SCREEN)) {
      // Scale down the image as needed, keeping the original aspect
      // ratio, to fit in an area of the specified size.  I believe
      // this is what Microsoft explains as "the approximate size
      // passed in prgSize, but crop it if necessary" or "render as if
      // for the screen."
      if (width <= cx && height <= cy) {
	// The original image is smaller than (or same as) the
	// specified area.  Simply use the original image size.
	cx = width;
	cy = height;
      } else if (width * cy > height * cx) {
	// Scale the image to touch the specified area horizontally.
	cy = clip((cx * height + width / 2) / width, 1, cy);
      } else {
	// Scale the image to touch the specified area vertically.
	cx = clip((cy * width + height / 2) / height, 1, cx);
      }
    }
      
    if (width != cx || height != cy) {
      COMPTR(IWICBitmapScaler) scaler;
      ENSURE(factory->CreateBitmapScaler(&scaler));
      ENSURE(scaler->Initialize(cover, cx, cy, WICBitmapInterpolationModeFant));
      cover.replaceWith(scaler);
      width = cx;
      height = cy;
    }

    ENSURE(cover->GetPixelFormat(&format));
    if (!IsEqualGUID(format, GUID_WICPixelFormat32bppBGR)) {
      COMPTR(IWICFormatConverter) converter;
      ENSURE(factory->CreateFormatConverter(&converter));
      ENSURE(converter->Initialize(cover, GUID_WICPixelFormat32bppBGR, WICBitmapDitherTypeNone, NULL, 0.0, WICBitmapPaletteTypeCustom));
      cover.replaceWith(converter);
    }

    array<BYTE> bits(width * height * sizeof(UINT32));
    ENSURE(cover->CopyPixels(NULL, width * sizeof(UINT32), bits.size(), bits));

    BITMAPINFO bmi;
    ZeroMemory(&bmi, sizeof(bmi));
    bmi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
    bmi.bmiHeader.biWidth = width;
    bmi.bmiHeader.biHeight = -(INT)height;
    bmi.bmiHeader.biPlanes = 1;
    bmi.bmiHeader.biBitCount = 32;
    bmi.bmiHeader.biCompression = BI_RGB;

    HDC hDC = GetDC(NULL);
    // The MSDN document explicitly states we need a DIB _section_.
    LPVOID ptr;
    HBITMAP hBitmap = CreateDIBSection(hDC, &bmi, DIB_RGB_COLORS, &ptr, NULL, NULL);
    ReleaseDC(NULL, hDC);
    if (!hBitmap) return HRESULT_FROM_WIN32(GetLastError());

    memcpy(ptr, bits, bits.size());
    m_pending_bitmap = hBitmap;
    return S_OK;

  } catch (std::bad_alloc) {
    return E_OUTOFMEMORY;
  } catch (...) {
    return E_UNEXPECTED;
  }
}

STDMETHODIMP MobiHandler::Extract(HBITMAP *phBmpImage)
{
  TRACE_METHOD("Extract");

  *phBmpImage = m_pending_bitmap;
  if (!m_pending_bitmap) return E_NOTIMPL;
  m_pending_bitmap = NULL;
  return S_OK;
}

STDMETHODIMP MobiHandler::GetThumbnail(UINT cx, HBITMAP *phbmp, WTS_ALPHATYPE *pdwAlpha)
{
  TRACE_METHOD("GetThumbnail");

  *pdwAlpha = WTSAT_UNKNOWN;
  *phbmp = NULL;

  SIZE size = { cx, cx };
  DWORD flags = IEIFLAG_ORIGSIZE | IEIFLAG_QUALITY;
  ENSURE(GetLocation(NULL, 0, NULL, &size, 0, &flags));

  if (!m_pending_bitmap) return E_NOTIMPL;

  *phbmp = m_pending_bitmap;
  m_pending_bitmap = NULL;
  *pdwAlpha = WTSAT_RGB;
  return S_OK;
}

static HRESULT set_string(IPropertyStore *store, REFPROPERTYKEY key, TString const &value)
{
  if (value.empty()) return S_FALSE;
  auto_PROPVARIANT propvariant;
  ENSURE(InitPropVariantFromString(value.c_str(), &propvariant));
  ENSURE(store->SetValue(key, propvariant));
  return S_OK;
}

static HRESULT set_strvec(IPropertyStore *store, REFPROPERTYKEY key, TStrVec const &vect)
{
  if (vect.empty()) return S_FALSE;

  array<PCWSTR> array(vect.size());
  for (size_t i = 0; i < vect.size(); i++) {
    array[i] = vect[i].c_str();
  }

  auto_PROPVARIANT propvariant;
  ENSURE(g_globals.m_InitPropVariantFromStringVector(array, array.size(), &propvariant));
  ENSURE(store->SetValue(key, propvariant));

  return S_OK;
}

HRESULT MobiHandler::prepareProperties(bool withStorage = false)
{
  if (!g_globals.m_propsys_dll) return E_FAIL;
  if (m_books.empty() || !m_books.front()->isMobi()) return E_FAIL;

  // Create and initialize a property store if we have not prepared one yet.
  if (!m_prop_store) {

    COMPTR(IPropertyStore) store;
    ENSURE(g_globals.m_PSCreateMemoryPropertyStore(store.getIID(), store.getPPV()));

    // Feed properties in the memory property store.
    ENSURE(set_string(store, PKEY_Title, m_books.front()->getTitle()));
    ENSURE(set_string(store, PKEY_Language, m_books.front()->getLanguage()));

    // I'm not sure the following switching is necessary/reasonable.  FIXME.
    if (withStorage) {
      ENSURE(set_string(store, PKEY_Author, m_books.front()->getAuthor()));
    } else {
      ENSURE(set_strvec(store, PKEY_Author, m_books.front()->getCreators()));
    }

    m_prop_store.replaceWith(store);
  }

  if (!m_prop_storage && withStorage) {
    ENSURE(g_globals.m_PSCreateAdapterFromPropertyStore(m_prop_store, m_prop_storage.getIID(), m_prop_storage.getPPV()));
  }

  return S_OK;
}

STDMETHODIMP MobiHandler::Create(REFFMTID fmtid, const CLSID *pclsid, DWORD grfFlags, DWORD grfMode, IPropertyStorage **ppPropStg)
{
  TRACE_METHOD("Create");

  *ppPropStg = NULL;
  return STG_E_ACCESSDENIED;
}

STDMETHODIMP MobiHandler::Delete(REFFMTID fmtid)
{
  TRACE_METHOD("Delete");

  return STG_E_ACCESSDENIED;
}

STDMETHODIMP MobiHandler::Enum(IEnumSTATPROPSETSTG **ppenum)
{
  TRACE_METHOD("Enum");

  *ppenum = NULL;
  ENSURE(prepareProperties(true));
  return m_prop_storage->Enum(ppenum);
}

STDMETHODIMP MobiHandler::Open(REFFMTID fmtid, DWORD grfMode, IPropertyStorage **ppPropStg)
{
  // TRACE_METHOD("Open"); 

  *ppPropStg = NULL;
  if (grfMode & (STGM_WRITE | STGM_READWRITE)) return STG_E_ACCESSDENIED;
  ENSURE(prepareProperties(true));
  return m_prop_storage->Open(fmtid, grfMode, ppPropStg);
}

STDMETHODIMP MobiHandler::Commit()
{
  TRACE_METHOD("Commit");

  // I feel something like it's silly to prepare a property store just
  // to commit it (that in turn clears its content)...
  ENSURE(prepareProperties());
  releaseBooks();
  return m_prop_store->Commit();
}

STDMETHODIMP MobiHandler::GetAt(DWORD iProp, PROPERTYKEY *pkey)
{
  TRACE_METHOD("GetAt");

  *pkey = PKEY_Null;
  ENSURE(prepareProperties());
  return m_prop_store->GetAt(iProp, pkey);
}

STDMETHODIMP MobiHandler::GetCount(DWORD *cProps)
{
  TRACE_METHOD("GetCount");

  *cProps = 0;
  ENSURE(prepareProperties());
  return m_prop_store->GetCount(cProps);
}

static WCHAR const *propertykey(WCHAR *keystr, REFPROPERTYKEY key)
{
  HRESULT hr;

  hr = g_globals.m_PSStringFromPropertyKey(key, keystr, PKEYSTR_MAX);
  if (FAILED(hr)) {
    lstrcpyW(keystr, L"{???} ?");
  }
  
  WCHAR *name = NULL;
  hr = g_globals.m_PSGetNameFromPropertyKey(key, &name);
  if (SUCCEEDED(hr)) {
    lstrcatW(keystr, L" (");
    lstrcatW(keystr, name);
    lstrcatW(keystr, L")");
  }

  return keystr;
}

STDMETHODIMP MobiHandler::GetValue(REFPROPERTYKEY key, PROPVARIANT *pv)
{
  WCHAR keystr[1024]; // XXX
  TRACE_METHOD_ARG("GetValue", propertykey(keystr, key));

  PropVariantInit(pv);
  ENSURE(prepareProperties());
  return m_prop_store->GetValue(key, pv);
}

STDMETHODIMP MobiHandler::SetValue(REFPROPERTYKEY key, REFPROPVARIANT propvar)
{
  TRACE_METHOD("SetValue");
  return STG_E_ACCESSDENIED;
}

STDMETHODIMP MobiHandler::IsPropertyWritable(REFPROPERTYKEY key)
{
  TRACE_METHOD("IsPropertyWritable");
  return S_FALSE;
}

////////////////////////////////////////////////////////////////
// DLL entry points
////////////////////////////////////////////////////////////////

// I'm not sure what SHOULD happen if DllCanUnloadNow and
// DllGetClassObject were called at the same time by separate threads.
// As you can see, we simply ignore the case for the moment...

STDAPI DllCanUnloadNow() throw()
{
  HRESULT result = S_FALSE;

  EnterCriticalSection(&g_globals.s_critical_section);

  if (!CUnknown::moduleBusy()) {
    g_globals.finalize();
    result = S_OK;
  }

  LeaveCriticalSection(&g_globals.s_critical_section);

  return result;
}

STDAPI DllGetClassObject(REFCLSID rclsid, REFIID riid, LPVOID *ppv) throw()
{
  HRESULT result = E_FAIL;
  *ppv = NULL;

  EnterCriticalSection(&g_globals.s_critical_section);
  g_globals.initializeOnce();

  if (g_globals.s_initialized) {

    result = CLASS_E_CLASSNOTAVAILABLE;

    if (IsEqualCLSID(rclsid, CLSID_MobiHandler)) {
      result = createInstance< CClassFactory<MobiHandler> >(riid, ppv);
    }

  }

  LeaveCriticalSection(&g_globals.s_critical_section);

  return result;
}

EXTERN_C BOOL WINAPI DllMain(HINSTANCE hInstDll, DWORD dwReason, LPVOID lpReserved) throw()
{
  switch (dwReason) {

  case DLL_PROCESS_ATTACH:
    DisableThreadLibraryCalls(hInstDll);
    InitializeCriticalSection(&g_globals.s_critical_section);
    g_globals.s_hInstDll = hInstDll;
    break;

  case DLL_PROCESS_DETACH:
    // Do we need this?  Are we ALLOWED this?
    DeleteCriticalSection(&g_globals.s_critical_section);
    break;

  }

  return TRUE;
}

////////////////////////////////////////////////////////////////
// Register/Unregister entry points.  I'm not sure I need these.
// Microsoft, please tell me the truth.  FIXME.
////////////////////////////////////////////////////////////////

STDAPI DllUnregisterServer(void) throw()
{
  return S_OK;
}

STDAPI DllRegisterServer(void) throw()
{
  return S_OK;
}
