#define UNICODE 1
#define _UNICODE 1

#include <windows.h>
#include <shlobj.h>
#include <msi.h>
#include <msiquery.h>

static void reglog(MSIHANDLE hInstall, LPCSTR func, LPCWSTR key, LONG err)
{
  if (err) {
    MSIHANDLE hRecord;
    hRecord = MsiCreateRecord(3);
    if (hRecord) {
      MsiRecordSetStringA(hRecord, 0, "Error accessing registry, func = [1], key/value = [2], error = [3]");
      MsiRecordSetStringA(hRecord, 1, func);
      MsiRecordSetStringW(hRecord, 2, key);
      MsiRecordSetInteger(hRecord, 3, err);
      MsiProcessMessage(hInstall, INSTALLMESSAGE_INFO, hRecord);
      MsiCloseHandle(hRecord);
    }
  }
}

static INT_PTR advapi_func(MSIHANDLE hInstall, LPCSTR func)
{
  MSIHANDLE hRecord;
  HMODULE hModule = GetModuleHandleA("advapi32");
  if (hModule) {
    INT_PTR ret = (INT_PTR)GetProcAddress(hModule, func);
    if (ret) return ret;
  }
  hRecord = MsiCreateRecord(1);
  if (hRecord) {
    MsiRecordSetStringA(hRecord, 0, "advapi function [1] is not available");
    MsiRecordSetStringA(hRecord, 1, func);
    MsiProcessMessage(hInstall, INSTALLMESSAGE_INFO, hRecord);
    MsiCloseHandle(hRecord);
  }
  return 0;
}

static LONG disable_reflection(MSIHANDLE hInstall, HKEY hKey)
{
  typedef LONG (WINAPI *RegDisableReflectionKey_type)(HKEY);
  static RegDisableReflectionKey_type pRegDisableReflectionKey;
  if (!pRegDisableReflectionKey) {
    pRegDisableReflectionKey = (RegDisableReflectionKey_type)advapi_func(hInstall, "RegDisableReflectionKey");
    if (!pRegDisableReflectionKey) {
      /* Consider the case as success, since unavailability of this
	 API indicates that the underlying platform does not support
	 registry reflection hence it is already disabled.  */
      return ERROR_SUCCESS;
    }
  }
  return pRegDisableReflectionKey(hKey);
}

static LONG open_key(MSIHANDLE hInstall, HKEY hKey, LPCWSTR subkey, PHKEY phNewKey)
{
  LONG err = RegOpenKeyExW(hKey, subkey, 0, KEY_ALL_ACCESS | KEY_WOW64_64KEY, phNewKey);
  reglog(hInstall, "RegOpenKeyExW", subkey, err);
  if (ERROR_SUCCESS == err) disable_reflection(hInstall, *phNewKey);
  return err;
}

static LONG create_key(MSIHANDLE hInstall, HKEY hKey, LPCWSTR subkey, PHKEY phNewKey, LPDWORD pdisp)
{
  LONG err = RegCreateKeyExW(hKey, subkey, 0, 0, 0, KEY_ALL_ACCESS | KEY_WOW64_64KEY, NULL, phNewKey, pdisp);
  reglog(hInstall, "RegCreateKeyEx", subkey, err);
  if (ERROR_SUCCESS == err) disable_reflection(hInstall, *phNewKey);
  return err;
}

static LONG close_key(MSIHANDLE hInstall, HKEY hKey)
{
  LONG err = RegCloseKey(hKey);
  reglog(hInstall, "RegCloseKey", NULL, err);
  return err;
}


static LONG delete_key(MSIHANDLE hInstall, HKEY hKey, LPCWSTR subkey)
{
  LONG err;
  typedef LONG (WINAPI *RegDeleteKeyExW_type)(HKEY, LPCWSTR, REGSAM, DWORD);
  static RegDeleteKeyExW_type pRegDeleteKeyExW;
  if (!pRegDeleteKeyExW) {
    pRegDeleteKeyExW = (RegDeleteKeyExW_type)advapi_func(hInstall, "RegDeleteKeyExW");
  }
  if (pRegDeleteKeyExW) {
    err = pRegDeleteKeyExW(hKey, subkey, KEY_WOW64_64KEY, 0);
  } else {
    err = RegDeleteKeyW(hKey, subkey);
  }
  reglog(hInstall, "RegDeleteKeyExW", subkey, err);
  return err;
}

/* When the specified registry value exists, the function behaves
   differently based on the pset setting.  If pset is NULL, the new
   value is set.  If pset is not NULL, the existing value is preserved
   and *pset is set to false.  When the specified registry value
   doesn't exist, the function set a new value, and *pset is set to
   true (if pset is not NULL.)  */
static LONG set_value(MSIHANDLE hInstall, HKEY hKey, LPCWSTR value, LPCWSTR data, LPBOOL pset)
{
  LONG err;

  if (pset) {
    err = RegQueryValueExW(hKey, value, 0, NULL, NULL, NULL);
    if (ERROR_FILE_NOT_FOUND != err) {
      *pset = FALSE;
      return ERROR_SUCCESS;
    }
    *pset = TRUE;
  }

  err = RegSetValueExW(hKey, value, 0, REG_SZ, (BYTE const *)data, (lstrlenW(data) + 1) * sizeof(WCHAR));
  reglog(hInstall, "RegSetValueExW", value, err);
  return err;
}

static LONG delete_value(MSIHANDLE hInstall, HKEY hKey, LPCWSTR value)
{
  LONG err = RegDeleteValueW(hKey, value);
  reglog(hInstall, "RegDeleteValueW", value, err);
  return err;
}

static LONG older(LONG last, LONG next)
{
  return last ? last : next;
}

static LONG do_register(MSIHANDLE hInstall, LPCWSTR clsid, LPCWSTR dll, LPCWSTR desc)
{
  LONG err = 0;
  HKEY hKey1 = NULL, hKey2 = NULL, hKey3 = NULL, hKey4 = NULL, hKey5 = NULL, hKey6 = NULL, hKey7 = NULL;
  DWORD disp2 = 0, disp3 = 0, disp5 = 0, disp6 = 0, disp7 = 0;
  BOOL done5 = FALSE, done6 = FALSE, done7 = FALSE;

  if (!err) err = open_key(hInstall, HKEY_CLASSES_ROOT, L"CLSID", &hKey1);
  if (!err) err = create_key(hInstall, hKey1, clsid, &hKey2, &disp2);
  if (!err) err = set_value(hInstall, hKey2, NULL, desc, NULL);
  if (!err) err = create_key(hInstall, hKey2, L"InprocServer32", &hKey3, &disp3);
  if (!err) err = set_value(hInstall, hKey3, NULL, dll, NULL);
  if (!err) err = set_value(hInstall, hKey3, L"ThreadingModel", L"Apartment", NULL);

  if (!err) err = open_key(hInstall, HKEY_LOCAL_MACHINE, L"SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertyHandlers", &hKey4);
  if (!err) err = create_key(hInstall, hKey4, L".azw",  &hKey5, &disp5);
  if (!err) err = set_value(hInstall, hKey5, NULL, clsid, &done5);
  if (!err) err = create_key(hInstall, hKey4, L".mobi", &hKey6, &disp6);
  if (!err) err = set_value(hInstall, hKey6, NULL, clsid, &done6);
  if (!err) err = create_key(hInstall, hKey4, L".prc",  &hKey7, &disp7);
  if (!err) err = set_value(hInstall, hKey7, NULL, clsid, &done7);

  if (hKey7 && err && done7) delete_value(hInstall, hKey7, NULL);
  if (hKey7) err = older(err, close_key(hInstall, hKey7));
  if (hKey6 && err && done6) delete_value(hInstall, hKey6, NULL);
  if (hKey6) err = older(err, close_key(hInstall, hKey6));
  if (hKey5 && err && done5) delete_value(hInstall, hKey5, NULL);
  if (hKey5) err = older(err, close_key(hInstall, hKey5));
  if (hKey4 && err && (disp7 & REG_CREATED_NEW_KEY)) delete_key(hInstall, hKey4, L".prc");
  if (hKey4 && err && (disp6 & REG_CREATED_NEW_KEY)) delete_key(hInstall, hKey4, L".mobi");
  if (hKey4 && err && (disp5 & REG_CREATED_NEW_KEY)) delete_key(hInstall, hKey4, L".azw");
  if (hKey4) err = older(err, close_key(hInstall, hKey4));

  if (hKey3 && err) delete_value(hInstall, hKey3, NULL);
  if (hKey3) err = older(err, close_key(hInstall, hKey3));
  if (hKey2 && err && (disp3 & REG_CREATED_NEW_KEY)) delete_key(hInstall, hKey2, L"InprocServer32");
  if (hKey2 && err) delete_value(hInstall, hKey2, NULL);
  if (hKey2) err = older(err, close_key(hInstall, hKey2));
  if (hKey1 && err && (disp2 & REG_CREATED_NEW_KEY)) delete_key(hInstall, hKey1, clsid);
  if (hKey1) err = older(err, close_key(hInstall, hKey1));

  return err;
}

static LONG do_unregister(MSIHANDLE hInstall, LPCWSTR clsid)
{
  HKEY hKey1, hKey2, hKey3;

  hKey1 = hKey2 = hKey3 = NULL;
  open_key(hInstall, HKEY_CLASSES_ROOT, L"CLSID", &hKey1);
  if (hKey1) open_key(hInstall, hKey1, clsid, &hKey2);
  if (hKey2) open_key(hInstall, hKey2, L"InprocServer32", &hKey3);
  if (hKey3) delete_value(hInstall, hKey3, NULL);
  if (hKey3) delete_value(hInstall, hKey3, L"ThreadingModel");
  if (hKey3) close_key(hInstall, hKey3);
  if (hKey2) delete_key(hInstall, hKey2, L"InprocServer32");
  if (hKey2) delete_value(hInstall, hKey2, NULL);
  if (hKey2) close_key(hInstall, hKey2);
  if (hKey1) delete_key(hInstall, hKey1, clsid);
  if (hKey1) close_key(hInstall, hKey1);

  hKey1 = NULL;
  open_key(hInstall, HKEY_LOCAL_MACHINE, L"SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertyHandlers", &hKey1);
  hKey2 = NULL;
  if (hKey1) open_key(hInstall, hKey1, L".azw",  &hKey2);
  if (hKey2) delete_value(hInstall, hKey2, NULL);
  if (hKey2) close_key(hInstall, hKey2);
  hKey2 = NULL;
  if (hKey1) open_key(hInstall, hKey1, L".mobi", &hKey2);
  if (hKey2) delete_value(hInstall, hKey2, NULL);
  if (hKey2) close_key(hInstall, hKey2);
  hKey2 = NULL;
  if (hKey1) open_key(hInstall, hKey1, L".prc",  &hKey2);
  if (hKey2) delete_value(hInstall, hKey2, NULL);
  if (hKey2) close_key(hInstall, hKey2);
  hKey2 = NULL;
  if (hKey1) close_key(hInstall, hKey1);

  return ERROR_SUCCESS; /* always indicate a success so that uninstall is not disturbed. */
}

static LONG find_in_32(LPCWSTR clsid)
{
  LONG err;
  HKEY hKey1 = NULL, hKey2 = NULL;
  err = RegOpenKeyExW(HKEY_CLASSES_ROOT, L"CLSID", 0, KEY_READ | KEY_WOW64_32KEY, &hKey1);
  if (err == ERROR_SUCCESS) {
    err = RegOpenKeyExW(hKey1, clsid, 0, KEY_READ | KEY_WOW64_32KEY, &hKey2);
    if (err == ERROR_SUCCESS) {
      RegCloseKey(hKey2);
    }
    RegCloseKey(hKey1);
  }
  return err;
}

UINT WINAPI Reg64(MSIHANDLE hInstall)
{
  LPTSTR cmd = NULL;
  DWORD cmd_len = 0;
  int argc = 0;
  LPWSTR *argv;
  LONG err = -1;
  UINT ret = ERROR_INSTALL_FAILURE;

  if (ERROR_MORE_DATA == MsiGetProperty(hInstall, TEXT("CustomActionData"), TEXT(""), &cmd_len)) {
    cmd_len += 1;
    cmd = (LPTSTR)LocalAlloc(LPTR, cmd_len * sizeof(TCHAR));
    if (cmd) {
      if (ERROR_SUCCESS == MsiGetProperty(hInstall, TEXT("CustomActionData"), cmd, &cmd_len)) {
	argv = CommandLineToArgvW(cmd, &argc);
	if (argv) {
	  if (argc == 3) {
	    LPCWSTR clsid = argv[0];
	    if (ERROR_SUCCESS == find_in_32(clsid)) {
	      err = do_register(hInstall, clsid, argv[1], argv[2]);
	    } else {
	      err = do_unregister(hInstall, clsid);
	    }
	    if (ERROR_SUCCESS == err) ret = ERROR_SUCCESS;
	  }
	  LocalFree((HLOCAL)argv);
	}
      }
      LocalFree((HLOCAL)cmd);
    }
  }
  
  if (ERROR_SUCCESS != ret) {
    MSIHANDLE hRecord;
    hRecord = MsiCreateRecord(4);
    if (hRecord) {
      MsiRecordSetInteger(hRecord, 1, 1723);
      MsiRecordSetStringA(hRecord, 2, "Alissa.Reg64 (64 bit registry access)");
      MsiRecordSetStringA(hRecord, 3, "Reg64");
      MsiRecordSetStringA(hRecord, 4, "custom.dll");
      MsiProcessMessage(hInstall, INSTALLMESSAGE_ERROR, hRecord);
      MsiCloseHandle(hRecord);
    }
  }

  return ret;
}

UINT WINAPI ShNotify(MSIHANDLE hInstall)
{
  SHChangeNotify(SHCNE_ASSOCCHANGED, SHCNF_IDLIST, NULL, NULL);
  return ERROR_SUCCESS;
}

static UINT process_exit_message(MSIHANDLE hInstall, int err, int icon)
{
  MSIHANDLE hRecord;
  hRecord = MsiCreateRecord(1);
  if (hRecord) {
    MsiRecordSetInteger(hRecord, 1, err);
    MsiProcessMessage(hInstall, (INSTALLMESSAGE)(INSTALLMESSAGE_USER | MB_OK | icon), hRecord);
    MsiCloseHandle(hRecord);
  }
  return ERROR_SUCCESS;
}

UINT WINAPI Success(MSIHANDLE hInstall)
{
  return process_exit_message(hInstall, 32, 0);
}

UINT WINAPI Failure(MSIHANDLE hInstall)
{
  return process_exit_message(hInstall, 33, MB_ICONWARNING);
}
