#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Ian Stott <I_P_S@hotmail.com>'
__docformat__ = 'restructuredtext en'


import os, shutil
from PyQt5.Qt import QMenu, QToolButton, QApplication, Qt, QMessageBox,  QUrl
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import error_dialog, question_dialog, Dispatcher, info_dialog, open_url
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.dialogs.message_box import ProceedNotification, ErrorNotification
from calibre.ptempfile import PersistentTemporaryDirectory, remove_dir

import calibre_plugins.similar_stories.config as cfg
from calibre_plugins.similar_stories.common_utils import set_plugin_icon_resources, get_icon, WriteDebugLog
from calibre_plugins.similar_stories.dialogs import QueueProgressDialog
from calibre_plugins.similar_stories.statistics import get_Term_Frequency

from calibre_plugins.similar_stories.common_utils import set_plugin_icon_resources, get_icon, \
                                                     create_menu_action_unique

PLUGIN_ICONS = ['images/similar_stories.png']

class CalculateSimilarityAction(InterfaceAction):

    name = 'Similar Stories'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Similar Stories', None, 'Find stories similar to the currently selected one\n'
                                        'and store similarity score in in custom column', ())
    action_type = 'current'

    def genesis(self):
        self.menu = QMenu(self.gui)
        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
#        self.qaction.triggered.connect(self.similar_stories)
        # Assign our menu to this action and an icon
        self.rebuild_menus()
        self.qaction.setMenu(self.menu)
        self.qaction.triggered.connect(self.similar_stories)


    def rebuild_menus(self):
        m = self.menu
        m.clear()
        create_menu_action_unique(self, m, _('&Find Similar Books...'), image=PLUGIN_ICONS[0],
                         triggered=self.similar_stories)
        m.addSeparator()
        create_menu_action_unique(self, m, _('&About plugin'), None,
                                  shortcut=False, triggered=self.about)

        create_menu_action_unique(self, m, _('&Help about plugin'), None,
                                  shortcut=False, triggered=self.show_help)

        m.addSeparator()
        create_menu_action_unique(self, m, _('&Customize plugin')+'...', 'config.png',
                                  shortcut=False, triggered=self.show_configuration)
        self.gui.keyboard.finalize()

    def about(self):
        # Get the about text from a file inside the plugin zip file
        # The get_resources function is a builtin function defined for all your
        # plugin code. It loads files from the plugin zip file. It returns
        # the bytes from the specified file.
        #
        # Note that if you are loading more than one file, for performance, you
        # should pass a list of names to get_resources. In this case,
        # get_resources will return a dictionary mapping names to bytes. Names that
        # are not found in the zip file will not be in the returned dictionary.
        text = get_resources('about.txt')
        QMessageBox.about(self.gui, 'About the Similar Stories Plugin',
                text.decode('utf-8'))

    def show_help(self, *args):
        # copy help file from resources to temp directory
        tdir = PersistentTemporaryDirectory('_help_dir', prefix='')
        helpfilename = 'similar_stories.html'
        text = get_resources(helpfilename)
        dest_file = os.path.join(tdir, helpfilename )
        f = open(dest_file, 'w')
        f.write(text)
        f.close()
        # set url for help file 
        url = 'file:///' + dest_file
        # display help file in a browser
        open_url(QUrl(url))

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def similar_stories(self):
        # Verify we have a custom column configured to store the similarity score in
        db = self.gui.library_view.model().db
        similarity_col = cfg.get_custom_columns_for_library(db)
        all_cols = db.field_metadata.custom_field_metadata()
        is_valid_similarity_col = len(similarity_col) > 0 and similarity_col in all_cols
        if not is_valid_similarity_col:
            if not question_dialog(self.gui, 'Configure plugin', '<p>'+
                'You must specify a custom column to contain the similarity score first. Do you want to configure this now?',
                show_copy_button=False):
                return
            self.show_configuration()
            return

        # have custom column selected
        
        # identify initial book. This will be be first one selected
        # hopefully this is the first in the rows
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        
        # set target from the first book selected
        book_ids = self.gui.library_view.get_selected_ids()
        target_id = book_ids[0]

        # would like to include option of searching across currently visible books
        # but can't find function that returns the IDs of visible books
        
        title = db.title(target_id, index_is_id=True)
#        root = db.library_path
#        path = db.path(target_id, index_is_id=True)
#        author   = db.authors(target_id, index_is_id=True)
#        book_path = root + os.sep + path + os.sep + title + ' - ' + author
#        
        WriteDebugLog( 'Target title: ' + title + ' target_id: ' + str(target_id) )
    
        c = cfg.plugin_prefs[cfg.STORE_NAME]
        format_order = c.get(cfg.KEY_FORMAT_ORDER, cfg.DEFAULT_STORE_VALUES[cfg.KEY_FORMAT_ORDER])

#        found_format = False
#        for format in format_order:
#            if db.has_format(target_id, format, index_is_id=True):
#                target_format=format
#                found_format = True

        similarity_algorithm = None
        if similarity_col:
            similarity_algorithm = c.get(cfg.KEY_SIMILARITY_ALGORITHM,
                                    cfg.DEFAULT_STORE_VALUES[cfg.KEY_SIMILARITY_ALGORITHM])
        indexing_method = c.get(cfg.KEY_INDEXING_METHOD,
                                    cfg.DEFAULT_STORE_VALUES[cfg.KEY_INDEXING_METHOD])    
            
#        target_dict={}
#        if found_format:
#            target_dict = get_book_dictionary(target_format, book_path, similarity_algorithm)
#            WriteDebugLog('Target Dict #2: len?: %d'%len(target_dict))
#        else:
#            return
        
        # now identify books to find similarity score 

#        # Create a temporary directory to copy all the ePubs to while scanning
 #       tdir = PersistentTemporaryDirectory('_similar_stories', prefix='')

        # Queue all the books and kick off the job
        WriteDebugLog('Queueing jobs')
        QueueProgressDialog(self.gui, book_ids, target_id, format_order, similarity_algorithm, indexing_method, similarity_col,
                            self._queue_job, db)

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def _queue_job(self, target_book_id, books_to_scan, similarity_algorithm, indexing_method, similarity_custom_column):
        WriteDebugLog('starting _queue_job')
        if not books_to_scan:
            # All failed so cleanup our temp directory
#            remove_dir(tdir)
            return

        func = 'arbitrary_n'
        cpus = self.gui.job_manager.server.pool_size
        args = ['calibre_plugins.similar_stories.jobs', 'do_count_statistics',
                (books_to_scan, similarity_algorithm, indexing_method, target_book_id, cpus)]
        desc = 'Calculate similarity score'
        job = self.gui.job_manager.run_job(
                self.Dispatcher(self._get_statistics_completed), func, args=args,
                    description=desc)
        job.target_book_id = target_book_id
        job.similarity_custom_column = similarity_custom_column
        self.gui.status_bar.show_message('Calculating similarity score for %d books'%len(books_to_scan))
        WriteDebugLog('finished _queue_job')

    def _get_statistics_completed(self, job):
        WriteDebugLog('starting _get_statistics_completed')
#        remove_dir(job.tdir)
        if job.failed:
            return self.gui.job_exception(job, dialog_title='Failed to calculate similarity score')
        self.gui.status_bar.show_message('Calculating simialrity score completed', 3000)
        book_similarity_map = job.result

        if len(book_similarity_map)  == 0:
            # Must have been some sort of error in processing this book
            msg = 'Failed to generate any statistics. <b>View Log</b> for details'
            p = ErrorNotification(job.details, 'Count log', 'Similar Stories failed', msg,
                    show_copy_button=False, parent=self.gui)
        else:
            payload = (book_similarity_map, job.similarity_custom_column)
            all_ids = set(book_similarity_map.keys())
            msg = '<p>Similar Stories plugin found <b>%d similarities(s)</b>. ' % len(all_ids) + \
                  'Proceed with updating your library?'
            p = ProceedNotification(self._update_database_columns,
                    payload, job.details,
                    'Count log', 'Count complete', msg,
                    show_copy_button=False, parent=self.gui)
        p.show()
        WriteDebugLog('finished _get_statistics_completed')

    def _update_database_columns(self, payload):
        WriteDebugLog('starting _update_database_columns')
        (book_similarity_map, similarity_custom_column) = payload

        modified = set()
        db = self.gui.current_db
        custom_cols = db.field_metadata.custom_field_metadata()
        if similarity_custom_column:
            similarity_col = custom_cols[similarity_custom_column]

        # At this point we want to re-use code in edit_metadata to go ahead and
        # apply the changes. So we will create empty Metadata objects so only
        # the custom column field gets updated
        id_map = {}
        all_ids = set(book_similarity_map.keys())
        for id in all_ids:
            mi = Metadata(_('Unknown'))
            if id in book_similarity_map:
                similarity_col['#value#'] = book_similarity_map[id]
                mi.set_user_metadata(similarity_custom_column, similarity_col)
            id_map[id] = mi
        edit_metadata_action = self.gui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map)
