#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Ian Stott <I_P_S@hotmail.com>'
__docformat__ = 'restructuredtext en'

from PyQt5.Qt import (QWidget, QGridLayout, QLabel, QLineEdit, QPushButton,
                      QGroupBox, QComboBox, QRadioButton, QVBoxLayout)
from calibre.utils.config import JSONConfig

from calibre_plugins.similar_stories.common_utils import get_library_uuid, CustomColumnComboBox
from calibre_plugins.similar_stories.common_utils import KeyboardConfigDialog

STORE_NAME = 'Options'
KEY_FORMAT_ORDER = 'formatOrder'
KEY_SIMILARITY_ALGORITHM = 'algorithm'
STORE_LIBRARIES = 'libraries'
KEY_SIMILARITY_CUSTOM_COLUMN = 'customColumn'

SIMILARITY_ALGORITHMS = ['Tanimoto', 'Euclid', 'Cosine','Tanimoto (binary)','PMRA']

KEY_INDEXING_METHOD = 'indexing_method'
INDEXING_METHODS = ['memory','file']

DEFAULT_STORE_VALUES = {
                        KEY_FORMAT_ORDER: ['EPUB','MOBI'],
                        KEY_SIMILARITY_ALGORITHM: 0,
                        KEY_INDEXING_METHOD:0
                        }
DEFAULT_LIBRARY_VALUES = { KEY_SIMILARITY_CUSTOM_COLUMN: '' }

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Similar Stories')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES
plugin_prefs.defaults[STORE_LIBRARIES] = {}


def get_custom_columns_for_library(db):
    library_id = get_library_uuid(db)
    libraries = plugin_prefs[STORE_LIBRARIES]
    library = libraries.get(library_id, DEFAULT_LIBRARY_VALUES)
    similarity_custom_column = library.get(KEY_SIMILARITY_CUSTOM_COLUMN, DEFAULT_LIBRARY_VALUES[KEY_SIMILARITY_CUSTOM_COLUMN])
    return similarity_custom_column


class AlgorithmComboBox(QComboBox):

    def __init__(self, parent, algorithms, selected_algorithm):
        QComboBox.__init__(self, parent)
        self.populate_combo(algorithms, selected_algorithm)

    def populate_combo(self, algorithms, selected_algorithm):
        self.clear()
        for item in algorithms:
            self.addItem(item)
        self.setCurrentIndex(selected_algorithm)


class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        c = plugin_prefs[STORE_NAME]
        avail_columns = self.get_custom_columns()
        similarity_col = get_custom_columns_for_library(self.plugin_action.gui.current_db)
        format_order = c.get(KEY_FORMAT_ORDER, DEFAULT_STORE_VALUES[KEY_FORMAT_ORDER])
        similarity_algorithm = c.get(KEY_SIMILARITY_ALGORITHM, DEFAULT_STORE_VALUES[KEY_SIMILARITY_ALGORITHM])
        indexing_method  = c.get(KEY_INDEXING_METHOD, DEFAULT_STORE_VALUES[KEY_INDEXING_METHOD])

        # --- Similarity Options ---
        similarity_group_box = QGroupBox('Similarity score options:', self)
        layout.addWidget(similarity_group_box)
        similarity_group_box_layout = QGridLayout()
        similarity_group_box.setLayout(similarity_group_box_layout)

        similarity_column_label = QLabel('&Custom column:', self)
        similarity_column_label.setToolTip('Select the preferred similarity score algorithm')
        self.similarity_column_combo = CustomColumnComboBox(self, avail_columns, similarity_col)
        similarity_column_label.setBuddy(self.similarity_column_combo)
        similarity_group_box_layout.addWidget(similarity_column_label, 0, 0, 1, 1)
        similarity_group_box_layout.addWidget(self.similarity_column_combo, 0, 1, 1, 2)

        similarity_algorithm_label = QLabel('&Algorithm:', self)
        self.similarity_algorithm_combo = AlgorithmComboBox(self, SIMILARITY_ALGORITHMS, similarity_algorithm)
        similarity_algorithm_label.setBuddy(self.similarity_algorithm_combo)
        similarity_group_box_layout.addWidget(similarity_algorithm_label, 1, 0, 1, 1)
        similarity_group_box_layout.addWidget(self.similarity_algorithm_combo, 1, 1, 1, 2)

        indexing_method_label = QLabel('&Indexing Method:', self)
        self.indexing_method_combo = AlgorithmComboBox(self, INDEXING_METHODS, indexing_method)
        indexing_method_label.setBuddy(self.indexing_method_combo)
        similarity_group_box_layout.addWidget(indexing_method_label, 2, 0, 1, 1)
        similarity_group_box_layout.addWidget(self.indexing_method_combo, 2, 1, 1, 2)


        # --- Other options ---
        layout.addSpacing(5)
        other_group_box = QGroupBox('Other options:', self)
        layout.addWidget(other_group_box)
        other_group_box_layout = QGridLayout()
        other_group_box.setLayout(other_group_box_layout)
        format_label = QLabel('Preferred format:', self)
        format_label.setToolTip('When both an ePub and a Mobi format exist for a book\n'+
                                'which should be used for the statistics?\n\n'+
                                'If only one of these formats exist, the setting is ignored.')
        self.opt_epub = QRadioButton('&ePub', self)
        self.opt_mobi = QRadioButton('&Mobi', self)
        other_group_box_layout.addWidget(format_label, 0, 0, 1, 1)
        other_group_box_layout.addWidget(self.opt_epub, 0, 1, 1, 1)
        other_group_box_layout.addWidget(self.opt_mobi, 0, 2, 1, 1)
        if format_order[0] == 'EPUB':
            self.opt_epub.setChecked(True)
        else:
            self.opt_mobi.setChecked(True)

        keyboard_shortcuts_button = QPushButton('Keyboard shortcuts...', self)
        keyboard_shortcuts_button.setToolTip(_(
                    'Edit the keyboard shortcuts associated with this plugin'))
        keyboard_shortcuts_button.clicked.connect(self.edit_shortcuts)
        layout.addWidget(keyboard_shortcuts_button)

    def save_settings(self):
        new_prefs = {}
        if self.opt_epub.isChecked():
            new_prefs[KEY_FORMAT_ORDER] = ['EPUB','MOBI']
        else:
            new_prefs[KEY_FORMAT_ORDER] = ['MOBI','EPUB']
        new_prefs[KEY_SIMILARITY_ALGORITHM] = self.similarity_algorithm_combo.currentIndex()
        new_prefs[KEY_INDEXING_METHOD] = self.indexing_method_combo.currentIndex()
        plugin_prefs[STORE_NAME] = new_prefs

        libraries = plugin_prefs[STORE_LIBRARIES]
        library_id = get_library_uuid(self.plugin_action.gui.current_db)
        library = libraries.get(library_id, DEFAULT_LIBRARY_VALUES)
        library[KEY_SIMILARITY_CUSTOM_COLUMN] = self.similarity_column_combo.get_selected_column()
        libraries[library_id] = library
        plugin_prefs[STORE_LIBRARIES] = libraries

    def get_custom_columns(self):
        column_types = ['float']
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ in column_types:
                available_columns[key] = column
        return available_columns

    def edit_shortcuts(self):
        d = KeyboardConfigDialog(self.plugin_action.gui, self.plugin_action.action_spec[0])
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()
