/**
 *
 * Copyright (c) 2007 Mitch Reierson.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "dev1150.h"
#include "exception.h"

using namespace std;

Dev1150::Dev1150() : _hDevice(0)
{}

Dev1150::~Dev1150()
{
  finalize();
}

void Dev1150::initialize()
{
  struct usb_bus *bus;
  struct usb_device *dev;

  bool bFound = false;

  usb_init();

  usb_find_busses();
  usb_find_devices();

  // find the device on the usb bus
  for (bus = usb_busses; bus && !bFound; bus = bus->next) {
    for (dev = bus->devices; dev && !bFound; dev = dev->next) {
      if (dev->descriptor.idVendor == 0x993 && dev->descriptor.idProduct == 0x2) {
	printf("Found eBook 1150, attempting to open ...");
	_hDevice = usb_open(dev);
	if (_hDevice) {
	  printf("Success\n");
	  bFound = true;
	} else {
	  throw DevOpenError();
	}
      }
    }
  }

  if (!bFound) {
    throw DevNotFoundError();
  }

  int ret;
  ret = usb_set_configuration(_hDevice, 0x1);
  if (ret != 0) {
    throw DevOpenError();
  }

  ret = usb_claim_interface(_hDevice, 0x0);
  if (ret != 0) {
    throw DevOpenError();
  }
}

void Dev1150::finalize()
{
  if (_hDevice) {
    usb_release_interface(_hDevice, 0x0);
    usb_close(_hDevice);
  }
}

void Dev1150::read(string &data)
{
  char buffer[4092] = "";

  int ret;
  ret = usb_bulk_read(_hDevice, 0x2, buffer, 4092, 5000);
  if (ret < 0) {
    throw DevReadError();
  }

  data.assign(buffer, ret);
}

void Dev1150::write(string data)
{
  int ret;
  ret = usb_bulk_write(_hDevice, 0x2, const_cast<char*>(data.c_str()), data.length(), 5000);
  if (ret < 0) {
    throw DevWriteError();
  }
}
