/* makepatch.c -- Make a 'Patcher' style patch from differences between two binary files.

   Version 0.01, created 29 Sep 2016 by Geoffrey Reynolds.
   Version 0.02, updated 27 Feb 2017 by Geoffrey Reynolds.
   This file is hereby placed into the public domain.

   Compile: gcc -Wall -O -o makepatch makepatch.c
   Usage: makepatch [-b] OLDFILE NEWFILE

   Prints the byte differences between OLDFILE and NEWFILE on standard output
   in the form of a patch suitable for use as input to patch32lsb.

   If the -b switch is used then addresses will be relative to a base address
   which makes a patch easier to update if only its location changes.

   OLDFILE and NEWFILE must be the same length.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define VERSION 0.02
#define CHUNKSIZE 8

int main(int argc, char **argv)
{
  FILE *infile[2];
  unsigned char bytes[2][CHUNKSIZE];
  unsigned long addr, base;
  int i, base_flag = 0;
  const char *fmt_str;

  if (argc > 1 && strcmp(argv[1],"-b") == 0) {
    argc--;
    argv++;
    base_flag = 1;
  }

  if (argc != 3) {
    fprintf(stderr, "Usage: makepatch [-b] OLDFILE NEWFILE\n");
    exit(EXIT_FAILURE);
  }

  for (i = 0; i < 2; i++) {
    if ((infile[i] = fopen(argv[i+1],"rb")) == NULL) {
      fprintf(stderr, "Failed to open input file: %s\n", argv[i+1]);
      exit(EXIT_FAILURE);
    }
  }

  printf("<Patch>\n");
  printf("patch_name = `Unknown`\n");
  printf("patch_enable = `no`\n");

  if (base_flag)
    fmt_str = "replace_bytes = %.4lX,";
  else
    fmt_str = "replace_bytes = %.6lX,";

  for (addr = 0, base = 0; !feof(infile[0]); addr += CHUNKSIZE) {
    int len = fread(bytes[0],1,CHUNKSIZE,infile[0]);
    if (len != fread(bytes[1],1,CHUNKSIZE,infile[1]))
      break;
    if (memcmp(bytes[0],bytes[1],len)) {
      if (base_flag) {
	base_flag = 0;
	base = addr;
	printf("base_address = %.6lX\n",base);
      }
      printf(fmt_str,addr-base);
      for (i = 0; i < len; i++)
	printf(" %.2X",bytes[0][i]);
      printf(",");
      for (i = 0; i < len; i++)
	printf(" %.2X",bytes[1][i]);
      printf("\n");
    }
  }

  for (i = 0; i < 2; i++) {
    if (!feof(infile[i])) {
      fprintf(stderr, "Files are not the same length\n");
      exit(EXIT_FAILURE);
    }
    fclose(infile[i]);
  }

  printf("</Patch>\n");

  exit(EXIT_SUCCESS);
}
