﻿#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
						print_function)

__license__   = 'GPL v3'
__copyright__ = '2017, Wrangly <andreatos@gmail.com>, Kloon <kloon@techgeek.co.in>'
__docformat__ = 'restructuredtext en'

import re
import socket, re, datetime
from threading import Thread
from lxml.html import fromstring
from calibre.ebooks.metadata.book.base import Metadata
import lxml, sys
import lxml.html as lh
from calibre.utils.date import utcnow
from datetime import datetime
from dateutil import parser
from calibre.ebooks.metadata import MetaInformation
from calibre.utils.cleantext import clean_ascii_chars
from calibre import browser

class Worker(Thread):

	def __init__(self, url, result_queue, browser, log, relevance, plugin, timeout=30):
		Thread.__init__(self)
		self.daemon = True
		self.url, self.result_queue = url, result_queue
		self.log, self.timeout = log, timeout
		self.relevance, self.plugin = relevance, plugin
		self.browser = browser.clone_browser()
		self.cover_url = self.biblionet_id = self.isbn = None

	def run(self):
		try:
			self.get_details()
		except:
			self.log.exception('get_details failed for url: %r'%self.url)

	def get_details(self):
		try:
			raw = self.browser.open_novisit(self.url, timeout=self.timeout).read().strip()
			raw = raw.decode('utf-8', errors='replace')
			if not raw:
				log.error('Failed to get raw result for query: %r'%query)
				return
		except Exception as e:
			if callable(getattr(e, 'getcode', None)) and e.getcode() == 404:
				self.log.error('URL malformed: %r'%self.url)
				return
			attr = getattr(e, 'args', [None])
			attr = attr if attr else [None]
			if isinstance(attr[0], socket.timeout):
				msg = 'Biblionet.gr timed out. Try again later.'
				self.log.error(msg)
			else:
				msg = 'Failed to make details query: %r'%self.url
				self.log.exception(msg)
			return

		root = fromstring(clean_ascii_chars(raw))
		self.parse_details(root)

	def parse_details(self, root):
		try:
			biblionet_id = self.parse_biblionet_id(self.url)
			self.log.info('Parsed biblionet.gr identifier: %s'%biblionet_id)
		except:
			self.log.exception('Error parsing biblionet.gr id for url: %r'%self.url)
			biblionet_id = None

		try:
			title = self.parse_title(root)
			self.log.info('Parsed title: %s'%title)
		except:
			self.log.exception('Error parsing title for url: %r'%self.url)
			title = None
		
		try:
			authors = self.parse_authors(root)
			self.log.info('Parsed authors: %s'%authors)
		except:
			self.log.exception('Error parsing authors for url: %r'%self.url)
			authors = []

		if not title or not authors or not biblionet_id:
			self.log.error('Could not find title/authors/biblionet.gr id for %r'%self.url)
			self.log.error('Biblionet.gr id: %r Title: %r Authors: %r'%(biblionet_id, title, authors))
			return

		mi = Metadata(title, authors)
		mi.set_identifier('biblionet', biblionet_id)
		self.biblionet_id = biblionet_id

		try:
			isbn = self.parse_isbn(root)
			self.log.info('Parsed ISBN: %s'%isbn)
			if isbn:
				self.isbn = mi.isbn = isbn
		except:
			self.log.exception('Error parsing ISBN for url: %r'%self.url)
					
		try:
			mi.comments = self.parse_comments(root)
			self.log.info('Parsed comments: %s'%mi.comments)
		except:
			self.log.exception('Error parsing comments for url: %r'%self.url)

		try:
			self.cover_url = self.parse_covers(root)
			self.log.info('Parsed URL for cover: %r'%self.cover_url)
			self.plugin.cache_identifier_to_cover_url(self.biblionet_id, self.cover_url)
			mi.has_cover = bool(self.cover_url)
		except:
			self.log.exception('Error parsing cover for url: %r'%self.url)

		try:
			mi.tags = self.parse_tags(root)
			self.log.info('Parsed tags: %s'%mi.tags)
		except:
			self.log.exception('Error parsing tags for url: %r'%self.url)
					
		try:
			mi.publisher = self.parse_publisher(root)
			self.log.info('Parsed publisher: %s'%mi.publisher)
		except:
			self.log.exception('Error parsing publisher for url: %r'%self.url)	
			
		try:
			mi.pubdate = self.parse_published_date(root)
			self.log.info('Parsed publication date: %s'%mi.pubdate)
		except:
			self.log.exception('Error parsing published date for url: %r'%self.url)
			
		mi.source_relevance = self.relevance

		if self.biblionet_id and self.isbn:
			self.plugin.cache_isbn_to_identifier(self.isbn, self.biblionet_id)

		self.plugin.clean_downloaded_metadata(mi)

		self.result_queue.put(mi)

	def parse_biblionet_id(self, url):
		try:
			m = re.search('/book/(.*)', url)
			if m:
				return m.group(1)
		except:
			return None
		
	def parse_isbn(self, root):
		isbn = None
		isbn_node = root.xpath('//*[@class="book_details"]/text()')
		for isbn_value in isbn_node:
			m = re.search('(\d{3}-\d{3}-\d{5}-\d{1}-\d{1}|\d{3}-\d{3}-\d{3}-\d{3}-\d{1}|\d{3}-\d{3}-\d{4}-\d{2}-\d{1}|\d{3}-\d{3}-\d{2}-\d{4}-\d{1})', isbn_value)
			if m:
				isbn = m.group(1)
				break	
		return isbn				
	
	def parse_title(self, root):
		title_node = root.xpath('//h1/strong/text()')
		self.log.info('Title: %s'%title_node)
		if title_node:
			return title_node[0].strip()
		
	def parse_authors(self, root): 
		otherpeople_node = root.xpath('//h1/..//text()[substring(.,string-length(.) - string-length(": ") +1)= ": "]')
		name_nodes = root.xpath('//a[@class="booklink" and contains(@href, "/author/")]/text()')
		if name_nodes:
			if otherpeople_node:			
				previous_nodes = root.xpath('//h1/..//text()[substring(.,string-length(.) - string-length(": ") +1)= ": "][1]/preceding-sibling::a[@class="booklink" and contains(@href, "/author/")]/text()')
				if previous_nodes:
					author_nodes = root.xpath('//h1/..//text()[substring(.,string-length(.) - string-length(": ") +1)= ": "][1]/preceding-sibling::a[@class="booklink" and contains(@href, "/author/")]/text()')
				else:
					author_nodes = root.xpath('//a[@class="booklink" and contains(@href, "/author/")]/text()[1]')
					self.log.info('Authors: %r'%author_nodes)
			else:
				author_nodes = root.xpath('//a[@class="booklink" and contains(@href, "/author/")]/text()')
		else:	
				author_nodes = root.xpath('//h1/../text()[position()=last()-1]')
		if author_nodes:
			return [unicode(author) for author in author_nodes]
			
			
	def parse_tags(self, root):
		import calibre_plugins.biblionet_gr.config as cfg	
        default_get_all_tags = cfg.DEFAULT_STORE_VALUES[cfg.KEY_GET_ALL_TAGS]
        get_all_tags = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.KEY_GET_ALL_TAGS, default_get_all_tags)	
		
		if get_all_tags:
			newtags = []
			tags_node = root.xpath('//*[@class="subjectlink"]/text()')
			tags_node = [unicode(text) for text in tags_node if text]
			for tag in tags_node:
				tagId = self._convertTags(tag)
				if tagId is not None:
					newtags.append(tagId)
				else:
					tag = re.sub("[\(\[].*?[\)\]]", "",unicode(text).replace(',', ''))
					newtags.append(tag)
			if not newtags:
				tags_node = [re.sub("[\(\[].*?[\)\]]", "",unicode(text).replace(',', '')) for text in tags_node if text.strip()]
				if tags_node:
					return tags_node
			tags_node = [re.sub("[\(\[].*?[\)\]]", "",unicode(text).replace(',', ' -').replace(".","-")) for text in tags_node if text.strip()]
			return newtags + tags_node
		else:
			tags_node = root.xpath('//*[@class="subjectlink"]/text()')
			tags_node = [re.sub("[\(\[].*?[\)\]]", "",unicode(text).replace(',', '')) for text in tags_node if text.strip()]
			if tags_node:
				return tags_node
    
	def _convertTags(self, displaytag):
		displaytag = displaytag.strip().split("DDC:",1)[1].replace(" ","").replace("]", "").split(".",1)[0]  if displaytag else None
		displaytag = displaytag[:-1] + '0'
		tagTbl = { None: 'None', 
					u'000': 'Γενικά βιβλία',
					u'010': 'Βιβλιογραφία',
					u'020': 'Βιβλιοθηκονομία',
					u'030': 'Εγκυκλοπαίδειες-Λεξικά', 
					u'050': 'Περιοδικές εκδόσεις',
					u'060': 'Οργανισμοί',
					u'070': 'Δημοσιογραφία',
					u'080': 'Δοκίμια-Ομιλίες-Αποφθέγματα-Σύμμικτα',
					u'090': 'Σπάνια βιβλία-Χειρόγραφα-Αρχεία',
					u'100': 'Φιλοσοφία',
					u'110': 'Μεταφυσική',
					u'120': 'Γνώση',
					u'130': 'Παραφυσικά φαινόμενα', 
					u'140': 'Φιλοσοφικά συστήματα και διδασκαλίες',
					u'150': 'Ψυχολογία',
					u'160': 'Λογική',
					u'170': 'Ηθική',
					u'180': 'Αρχαία Φιλοσοφία',
					u'190': 'Νεότερη Φιλοσοφία',
					u'200': 'Θρησκεία',
					u'210': 'Φυσική θεολογία',
					u'220': 'Βίβλος',
					u'230': 'Χριστιανισμός', 
					u'240': 'Χριστιανική ηθική και ευλάβεια',
					u'250': 'Χριστιανική εκκλησία και τάγματα',
					u'260': 'Χριστιανική κοινωνική και εκκλησιαστική θεολογία',
					u'270': 'Εκκλησιαστική ιστορία',
					u'280': 'Χριστιανικά δόγματα',	
					u'290': 'Άλλες θρησκείες',
					u'300': 'Κοινωνικές επιστήμες',
					u'310': 'Στατιστική',
					u'320': 'Πολιτικές επιστήμες', 
					u'330': 'Οικονομία',
					u'340': 'Δίκαιο',
					u'350': 'Δημόσια διοίκηση',
					u'360': 'Κοινωνικά προβλήματα και υπηρεσίες',
					u'370': 'Εκπαίδευση-Παιδαγωγική',
					u'380': 'Εμπόριο-Επικοινωνίες-Μεταφορές',
					u'390': 'Λαογραφία', 
					u'400': 'Γλώσσα',
					u'410': 'Γλωσσολογία',
					u'420': 'Αγγλική γλώσσα',
					u'430': 'Γερμανική γλώσσα',
					u'440': 'Γαλλική γλώσσα',
					u'450': 'Ιταλική γλώσσα',
					u'459': 'Ρουμανική γλώσσα',					
					u'460': 'Ισπανική και πορτογαλική γλώσσα',
					u'470': 'Λατινική γλώσσα',
					u'480': 'Ελληνική γλώσσα', 
					u'490': 'Άλλες γλώσσες',
					u'500': 'Φυσικές και θετικές επιστήμες',
					u'510': 'Μαθηματικά',
					u'520': 'Αστρονομία',
					u'530': 'Φυσική',
					u'540': 'Χημεία',
					u'550': 'Γεωλογία',
					u'560': 'Παλαιοντολογία',
					u'570': 'Βιολογικές επιστήμες',
					u'580': 'Βοτανικές επιστήμες', 
					u'590': 'Ζωολογία',
					u'600': 'Τεχνολογία',
					u'610': 'Ιατρική ',
					u'620': 'Μηχανική',
					u'630': 'Γεωργία και συναφείς τεχνολογίες',
					u'640': 'Οικιακή οικονομία',
					u'650': 'Διοίκηση και οργάνωση',
					u'660': 'Χημική τεχνολογία',
					u'670': 'Βιομηχανία',
					u'680': 'Βιομηχανία ειδικών προϊόντων', 
					u'690': 'Οικοδομική',
					u'700': 'Καλές τέχνες',
					u'710': 'Αστική και τέχνη τοπίου',
					u'720': 'Αρχιτεκτονική',
					u'730': 'Γλυπτική',
					u'740': 'Σχέδιο',
					u'750': 'Ζωγραφική',
					u'760': 'Γραφικές τέχνες',
					u'770': 'Φωτογραφία ',
					u'780': 'Μουσική', 
					u'790': 'Ψυχαγωγικές τέχνες',
					u'800': 'Λογοτεχνία',
					u'810': 'Αμερικανική λογοτεχνία',
					u'820': 'Αγγλική και αγγλόφωνη λογοτεχνία',
					u'830': 'Γερμανική και γερμανόφωνη λογοτεχνία',
					u'840': 'Γαλλική και γαλλόφωνη λογοτεχνία',
					u'850': 'Ιταλική λογοτεχνία',
					u'860': 'Ισπανική-ισπανόφωνη και πορτογαλική λογοτεχνία',
					u'870': 'Λατινική γραμματεία',
					u'880': 'Ελληνική γραμματεία', 
					u'881': 'Ελληνική γραμματεία Βυζαντινή', 
					u'889': 'Νεοελληνική λογοτεχνία', 					
					u'890': 'Λογοτεχνία άλλων γλωσσών',
					u'900': 'Ιστορία',
					u'910': 'Γεωγραφία',
					u'920': 'Βιογραφίες ',
					u'930': 'Αρχαιότητα',
					u'940': 'Ευρώπη-Ιστορία',
					u'950': 'Ασία-Ιστορία',
					u'960': 'Αφρική-Ιστορία',
					u'970': 'Βόρεια Αμερική-Ιστορία',
					u'980': 'Νότια Αμερική-Ιστορία', 
					u'990': 'Ιστορία άλλων περιοχών'}
		return tagTbl.get(displaytag, None)				
							
			
	def parse_comments(self, root):
		description_node = root.xpath('//p[@align="justify"]/text()')
		if description_node:
			return '\n'.join(description_node)
			
	def parse_publisher(self, root):
		publisher_node = root.xpath('//h1/..//*[@class="booklink"][contains(@href,"/com/")]/text()')
		if publisher_node:
			return publisher_node[0]
		
	def parse_published_date(self, root):
		pub_year = None
		publication_node = root.xpath('//*[@class="book_details"]/text()')
		for publication_value in publication_node:
			m = re.search('(\d{4})', publication_value)
			if m:
				pub_year = m.group(1)
				break
		
		if not pub_year:
			return None
		default = datetime.utcnow()
		from calibre.utils.date import utc_tz
		default = datetime(default.year, default.month, default.day, tzinfo=utc_tz)
		pub_date = parser.parse(pub_year, default=default)
		if pub_date:
			return pub_date

	def parse_covers(self, root):
		from calibre_plugins.biblionet_gr import Biblionet_gr
		return [Biblionet_gr.COVER_URL + self.biblionet_id +'.jpg']
