#include <math.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fts.h>
#include <dlfcn.h>
#include <libgen.h>
#include <errno.h>
#include <img2ibitmap.h>
#include "main.h"
#include "translations.h"
#include "version.h"

iconfigedit pbivce[] = {
	{ CFG_INFO, &ci_about, GetLangText("@Conf_def_info"), NULL, NULL, NULL, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Zoom"), GetLangText("@Default_zoom_hint"), "layout_mode", "0", layout_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Fine_update"), GetLangText("@Fine_update_hint"), "do_fine_update", "0", off_on_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Show_status_bar"), GetLangText("@Status_bar_hint"), "do_draw_panel", "1", off_on_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Crop_margins"), NULL, "do_crop_margins", "0", off_on_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Crop_margin_size"), NULL, "margin_size_idx", "1", margin_size_variants, NULL },
	{ CFG_INFO, &ci_about, GetLangText("@Conf_sys_info"), NULL, NULL, NULL, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Unpack_tmp"), GetLangText("@Unpack_tmp_hint"), "unpack_in_tmp", "0", off_on_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Check_rar_images"), GetLangText("@Check_rar_images_hint"), "check_rar_images", "0", off_on_variants, NULL },
	{ CFG_TEXT, &ci_empty, GetLangText("@Copy_to_folder"), GetLangText("@Scrn_saved"), "save_dir", DEFIMGSAVEPATH, NULL, NULL },
	{ CFG_NUMBER, &ci_empty, GetLangText("@Step_percent"), GetLangText("@Step_percent_hint"), "step_percentage", "75", NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Reset_page"), GetLangText("@Reset_page_hint"), "do_reset_last_page", "0", off_on_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Start_next"), GetLangText("@Start_next_hint"), "do_start_next_archive", "0", off_on_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Open_orient"), GetLangText("@Open_orient_hint"), "open_orientation", "0", rot_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Autorotate"), GetLangText("@Autorotate_hint"), "autorot", "0", autorot_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Read"), NULL, "readdir_idx", "0", readdir_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@pbiv_log_dir"), NULL, "log_dir_idx", "0", logdir_variants, NULL },
	{ 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};

iconfigedit dispce[] = {
	{ CFG_INDEX, &ci_empty, GetLangText("@Brightness"), NULL, "brightness", "0", bc_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Contrast"), NULL, "contrast", "0", bc_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Gamma"), NULL, "gamma", "1.0", gamma_variants, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Depth"), NULL, "depth", "8", depth_variants, NULL },
	{ 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};

/* This is the new ifconfigedit structure used on Touch devices.  It requires
 * an extra NULL on the end.  The hint* should also be NULL. */
iconfigedit_new pbivce_new[] = {
	{ CFG_INFO, &ci_about, GetLangText("@Conf_def_info"), NULL, NULL, NULL, NULL, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Zoom"), NULL, "layout_mode", "0", layout_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Fine_update"), NULL, "do_fine_update", "0", off_on_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Show_status_bar"), NULL, "do_draw_panel", "1", off_on_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Crop_margins"), NULL, "do_crop_margins", "0", off_on_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Crop_margin_size"), NULL, "margin_size_idx", "1", margin_size_variants, NULL, NULL },
	{ CFG_INFO, &ci_about, GetLangText("@Conf_sys_info"), NULL, NULL, NULL, NULL, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Unpack_tmp"), NULL, "unpack_in_tmp", "0", off_on_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Check_rar_images"), NULL, "check_rar_images", "0", off_on_variants, NULL, NULL },
	{ CFG_TEXT, &ci_empty, GetLangText("@Copy_to_folder"), NULL, "save_dir", DEFIMGSAVEPATH, NULL, NULL, NULL },
	{ CFG_NUMBER, &ci_empty, GetLangText("@Step_percent"), NULL, "step_percentage", "75", NULL, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Reset_page"), NULL, "do_reset_last_page", "0", off_on_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Start_next"), NULL, "do_start_next_archive", "0", off_on_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Open_orient"), NULL, "open_orientation", "0", rot_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Autorotate"), NULL, "autorot", "0", autorot_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Read"), NULL, "readdir_idx", "0", readdir_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@pbiv_log_dir"), NULL, "log_dir_idx", "0", logdir_variants, NULL, NULL },
	{ 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};

iconfigedit_new dispce_new[] = {
	{ CFG_INDEX, &ci_empty, GetLangText("@Brightness"), NULL, "brightness", "0", bc_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Contrast"), NULL, "contrast", "0", bc_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Gamma"), NULL, "gamma", "1.0", gamma_variants, NULL, NULL },
	{ CFG_INDEX, &ci_empty, GetLangText("@Depth"), NULL, "depth", "8", depth_variants, NULL, NULL },
	{ 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};

int check_dir(char* dir) {
	/* Return 0 if dir is usable. */
	struct stat sb;
	int ret = -1;
	int buflen;
	char *buf, cmd_pref[] = "mkdir -p ";

	if (dir && *dir) {
		ret = stat(dir, &sb);
		if (ret && (errno == ENOENT)) {
			buflen = sizeof(cmd_pref) + strlen(dir);
			if ((buf = (char*)malloc(buflen))) {
				strcpy(buf, cmd_pref);
				strcat(buf, dir);
				ret = system(buf);
				free(buf);
			}
		} else if (!ret && !S_ISDIR(sb.st_mode))
			ret = -1;
	}

	return ret;
}

int check_image_save_dir(void) {
	return check_dir(image_save_dir);
}

int ensure_path(char *filepath) {
	/* Create the directory path for the file if necessary */
	int ret = 0;
	char *copy, *dir;

	if (filepath && *filepath) {
		if ((copy = strdup(filepath))) {
			if ((dir = dirname(copy)))
				ret = check_dir(dir);
			free(copy);
		}
	}
	return ret;
}

void after_error_redraw(int i) {
	draw();
}

void conf_handler() {
	v.SetMoveStep((double)ReadInt(pbivcfg, "step_percentage", (int)(v.GetMoveStep()*100+0.5))/100.0);
	do_start_next_archive = ReadInt(pbivcfg, "do_start_next_archive", 1) != 0;
	Log::set_dir(logdirs[ReadInt(pbivcfg, "log_dir_idx", 0)]);
	image_save_dir = (char*)ReadString(pbivcfg, "save_dir", DEFIMGSAVEPATH);
	v.SetCroppedMargins(ReadInt(pbivcfg, "do_crop_margins", 0) != 0);
	v.SetMarginSize(2<<(ReadInt(pbivcfg, "margin_size_idx", 1)));
	autorot_idx = ReadInt(pbivcfg, "autorot", 0);
	if (check_image_save_dir()) {
		image_save_dir = NULL;
		Dialog(ICON_ERROR, GetLangText("@Program_name"), GetLangText("@Cant_create_dir"), GetLangText("@Close"), NULL, after_error_redraw);
	} else {
		SaveConfig(pbivcfg);
		show_image(mng.Current());
	}
}

void read_config() {
	iconfig *gcfg = GetGlobalConfig();
	lang = (char*)ReadString(gcfg, "language", "en");

	pbivcfg = OpenConfig(PBIVCFGFILE, caps.has_new_config_struct ? (iconfigedit*)pbivce_new : pbivce);
	layout_mode = ReadInt(pbivcfg, "layout_mode", LAYOUT_DEFAULT);
	do_fine_update = ReadInt(pbivcfg, "do_fine_update", 0) != 0;
	do_draw_panel = ReadInt(pbivcfg, "do_draw_panel", 1) != 0;
	SetPanelType(do_draw_panel ? 1 : 0);
	v.SetCroppedMargins(ReadInt(pbivcfg, "do_crop_margins", 0) != 0);
	v.SetMarginSize(2<<(ReadInt(pbivcfg, "margin_size_idx", 1)));
	do_reset_last_page = ReadInt(pbivcfg, "do_reset_last_page", 1) != 0;
	do_start_next_archive = ReadInt(pbivcfg, "do_start_next_archive", 1) != 0;
	mng.SetUnpackInTmp(ReadInt(pbivcfg, "unpack_in_tmp", 0) != 0);
	ImagesMng::SetCheckRarMagic(ReadInt(pbivcfg, "check_rar_images", 0) != 0);
	v.SetMoveStep((double)ReadInt(pbivcfg, "step_percentage", (int)(v.GetMoveStep()*100+0.5))/100.0);
	image_save_dir = (char*)ReadString(pbivcfg, "save_dir", DEFIMGSAVEPATH);
	if (check_image_save_dir())
		image_save_dir = NULL;
	if (!QueryGSensor() || GetGlobalOrientation() != -1) {
		SetOrientation(ReadInt(pbivcfg, "open_orientation", 0));
		autorot_idx = ReadInt(pbivcfg, "autorot", 0);
	}
	v.SetReadingDirection((ReadInt(pbivcfg, "readdir_idx", 0) != 0) ?
	                      DIRECTION_LEFT : DIRECTION_RIGHT);
	Log::set_dir(logdirs[ReadInt(pbivcfg, "log_dir_idx", 0)]);
}

int cliprect_height() {
	return ScreenHeight() - (do_draw_panel ? 1 : 0) * PanelHeight();
}

imenu custom_menu[] = {
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_CUSTOM+10, GetLangText("@Custom_scale"), NULL },
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_CUSTOM+30, GetLangText("@Custom_layout"), NULL },
	{ 0, 0, NULL, NULL }
};

imenu layout_menu[] = {
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_DEFAULT, GetLangText("@Fit_default"), NULL },
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_DOUBLE, GetLangText("@Fit_double"), NULL },
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_FIT_PAGE, GetLangText("@Fit_page"), NULL },
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_FIT_HEIGHT, GetLangText("@Fit_height"), NULL },
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_FIT_WIDTH, GetLangText("@Fit_width"), NULL },
	{ ITEM_ACTIVE,  LAYOUT+LAYOUT_FULL_SIZE, GetLangText("@Full_size"), NULL },
	{ ITEM_SUBMENU, 0, GetLangText("@Custom_zoom"), custom_menu },
	{ 0, 0, NULL, NULL }
};

imenu panel_bkgd_menu[] = {
	{ ITEM_ACTIVE,  PANEL+2, GetLangText("@White"), NULL },
	{ ITEM_ACTIVE,  PANEL+3, GetLangText("@Bkgd_lgrey"), NULL },
	{ ITEM_ACTIVE,  PANEL+4, GetLangText("@Gray"), NULL },
	{ 0, 0, NULL, NULL }
};

imenu panel_menu[] = {
	{ ITEM_ACTIVE,  PANEL+0, GetLangText("@Toggle_auto_panels"), NULL },
	{ ITEM_ACTIVE,  PANEL+1, GetLangText("@Toggle_scale_panels"), NULL },
	{ ITEM_SUBMENU, 0, GetLangText("@Select_bkgd_shade"), panel_bkgd_menu },
	{ 0, 0, NULL, NULL }
};

imenu rotation_menu[] = {
	{ ITEM_ACTIVE,  TURN+0, GetLangText("@Rotate_0"), NULL},
	{ ITEM_ACTIVE,  TURN+1, GetLangText("@Rotate_90"), NULL},
	{ ITEM_ACTIVE,  TURN+3, GetLangText("@Rotate_180"), NULL},
	{ ITEM_ACTIVE,  TURN+2, GetLangText("@Rotate_270"), NULL},
	{ 0, 0, NULL, NULL }
};

imenu margin_size_menu[] = {
	{ ITEM_ACTIVE,  CROP+10+MARGIN_SMALL, GetLangText("@border_small"), NULL},
	{ ITEM_ACTIVE,  CROP+10+MARGIN_MEDIUM, GetLangText("@border_medium"), NULL},
	{ ITEM_ACTIVE,  CROP+10+MARGIN_LARGE, GetLangText("@border_large"), NULL},
	{ 0, 0, NULL, NULL }
};

imenu margin_thresh_menu[] = {
	{ ITEM_ACTIVE,  CROP+20+MARGIN_THRESH_LOW, "2", NULL},
	{ ITEM_ACTIVE,  CROP+20+MARGIN_THRESH_MED, "4", NULL},
	{ ITEM_ACTIVE,  CROP+20+MARGIN_THRESH_HIGH, "8", NULL},
	{ 0, 0, NULL, NULL }
};

imenu crop_menu[] = {
	{ ITEM_ACTIVE,  CROP+0, GetLangText("@Crop_toggle"), NULL},
	{ ITEM_SUBMENU,      0, GetLangText("@Crop_margin_size"), margin_size_menu},
	{ ITEM_SUBMENU,      0, GetLangText("@Crop_margin_thresh"), margin_thresh_menu},
	{ 0, 0, NULL, NULL }
};

/* On newer devices, the full menu usually doesn't fit on the screen,
 * so we break out the items that usually only get chosen once into
 * their own separate menu. */
imenu menu_more[] = {
	{ ITEM_SUBMENU, 0, GetLangText("@Zoom"), layout_menu },
	{ ITEM_ACTIVE,  FINE, GetLangText("@Fine_updates"), NULL },
	{ ITEM_ACTIVE,  STATUS, GetLangText("@Toggle_status_bar"), NULL },
	{ ITEM_SUBMENU, 0, GetLangText("@Crop_margins"), crop_menu },
	{ ITEM_SUBMENU, 0, GetLangText("@Panel_detection"), panel_menu },
	{ ITEM_ACTIVE,  CONF, GetLangText("@Configuration"), NULL },
	{ ITEM_ACTIVE,  DISP, GetLangText("@Display_params"), NULL },
	{ ITEM_ACTIVE,  HELP, GetLangText("@Help"), NULL },
	{ ITEM_ACTIVE,  ABOUT, GetLangText("@About"), NULL },
	{ 0, 0, NULL, NULL }
};

imenu menu[20];

void set_menu_item(unsigned int i, short type, short index, char *text, struct imenu_s *submenu) {
	if (i < (sizeof(menu)/sizeof(imenu))) {
		menu[i].type = type;
		menu[i].index = index;
		menu[i].text = text;
		menu[i].submenu = submenu;
	} else
		Log::msg("Error! Too many items for main menu");
}

void create_main_menu(void) {
	unsigned int i = 0;
	bool long_menu = (caps.fw_major < 4);
	bool rotation = !(QueryGSensor() && GetGlobalOrientation() == -1);
	bool contents = !caps.has_menu_button;

	set_menu_item(i++, ITEM_HEADER, 0, GetLangText("@Menu"), NULL);
	set_menu_item(i++, ITEM_ACTIVE,  GOTO, GetLangText("@Goto_page"), NULL);
	if (long_menu) {
		set_menu_item(i++, ITEM_SUBMENU, 0, GetLangText("@Zoom"), layout_menu);
		set_menu_item(i++, ITEM_ACTIVE,  FINE, GetLangText("@Fine_updates"), NULL);
		set_menu_item(i++, ITEM_ACTIVE,  STATUS, GetLangText("@Toggle_status_bar"), NULL);
		set_menu_item(i++, ITEM_SUBMENU, 0, GetLangText("@Crop_margins"), crop_menu);
		set_menu_item(i++, ITEM_SUBMENU, 0, GetLangText("@Panel_detection"), panel_menu);
		set_menu_item(i++, ITEM_ACTIVE,  CONF, GetLangText("@Configuration"), NULL);
		set_menu_item(i++, ITEM_ACTIVE,  DISP, GetLangText("@Display_params"), NULL);
	}
	if (rotation)
		set_menu_item(i++, ITEM_SUBMENU, 0, GetLangText("@Rotate_dir"), rotation_menu);
	if (contents)
		set_menu_item(i++, ITEM_ACTIVE, TOC, GetLangText("@Contents"), NULL);
	set_menu_item(i++, ITEM_ACTIVE,  DICT, GetLangText("@Dictionary"), NULL);
	set_menu_item(i++, ITEM_ACTIVE,  SAVE, GetLangText("@Copy_to_folder"), NULL);
	if (long_menu) {
		set_menu_item(i++, ITEM_ACTIVE,  HELP, GetLangText("@Help"), NULL);
		set_menu_item(i++, ITEM_ACTIVE,  ABOUT, GetLangText("@About"), NULL);
	} else {
		set_menu_item(i++, ITEM_SUBMENU, 0, GetLangText("@About_more"), menu_more);
	}
	set_menu_item(i++, ITEM_ACTIVE,  EXIT, GetLangText("@Exit"), NULL);
	set_menu_item(i++, 0, 0, NULL, NULL);
}

void show_image(ibitmap *img) {
	if (img) {
		/* Auto-rotate new images if desired. */
		if (autorot_idx && v.CheckForNewImage(img) &&
		    !(QueryGSensor() && GetGlobalOrientation() == -1))
			rotate_screen((img->width > img->height) ? autorot_idx : 0);
		v.SetImage(img);
		draw();
		if (!mng.IsBuffered()) {
			mng.Buffering();
			if (do_draw_panel) {
				draw_panel(pagemap);
				PartialUpdate(0, cliprect_height(), ScreenWidth(), PanelHeight());
			}
		}
	}
}

boolean use_new_conf() {
	/* The newer 4.0 API found on the 62X uses a different layout for the 
	 * iconfigedit struct.  So, we have to call OpenConfigEditor() with
	 * the correct struct layout, depending on which API is being used.
	 */
	return caps.has_new_config_struct;
}

void open_conf(void) {
	if (use_new_conf())
		/* This is the newer API */
		OpenConfigEditor(GetLangText("@Conf_title"), pbivcfg, (iconfigedit*)pbivce_new, conf_handler, NULL);
	else 
		OpenConfigEditor(GetLangText("@Conf_title"), pbivcfg, pbivce, conf_handler, NULL);
}

void set_lut(void) {
	int brightness, contrast;
	double gamma;

	brightness = atoi(bc_variants[brightness_idx]);
	contrast = atoi(bc_variants[contrast_idx]);
	gamma = atof(gamma_variants[gamma_idx]);
	mng.RecomputeLUT(brightness, contrast, gamma);
}

void disp_conf_handler(void) {
	brightness_idx = ReadInt(dispcfg, "brightness", 4);
	contrast_idx = ReadInt(dispcfg, "contrast", 4);
	gamma_idx = ReadInt(dispcfg, "gamma", 5);
	depth_idx = ReadInt(dispcfg, "depth", DEPTH8);

	set_lut();
	set_max_ibitmap_depth(atoi(depth_variants[depth_idx]));
	show_image(mng.Current());
}

void open_display_conf(void) {
	if (use_new_conf())
		OpenConfigEditor(GetLangText("@Display_params"), dispcfg, (iconfigedit*)dispce_new, disp_conf_handler, NULL);
	else
		OpenConfigEditor(GetLangText("@Display_params"), dispcfg, dispce, disp_conf_handler, NULL);
}

void pageselect_handler(int pagenum) {
	if (pagenum < 1)
	 pagenum = 1;
	else if (pagenum > (int)mng.GetFilesCount())
		pagenum = mng.GetFilesCount();
	show_image(mng.GoTo(pagenum-1));
}

void kpage_handler(char *s) {
	if (s && *s)
		pageselect_handler(atoi(s));
}

void contents_handler(long long pos) {
	pageselect_handler((int)pos+1);
}

void update_fitmode() {
	switch (layout_mode) {
		case LAYOUT_DEFAULT:
			if (GetOrientation() == 0 || GetOrientation() == 3)  //Portrait
				v.SetFitMode(FIT_BOTH);
			else                         //Landscape
				v.SetFitMode(FIT_WIDTH);
			break;
		case LAYOUT_DOUBLE:
			if (GetOrientation() == 0 || GetOrientation() == 3)  //Portrait
				v.SetFitMode(FIT_HEIGHT);
			else                         //Landscape
				v.SetFitMode(FIT_WIDTH);
			break;
		case LAYOUT_FIT_PAGE:
			v.SetFitMode(FIT_BOTH);
			break;
		case LAYOUT_FIT_HEIGHT:
			v.SetFitMode(FIT_HEIGHT);
			break;
		case LAYOUT_FIT_WIDTH:
			v.SetFitMode(FIT_WIDTH);
			break;
		case LAYOUT_FULL_SIZE:
			v.SetFitMode(FIT_NOT);
			break;
		case LAYOUT_CUSTOM:
			v.SetFitMode(FIT_CUSTOM);
			break;
	}
}

void scale_handler(char *nscale) {
	if (nscale && *nscale && strcmp(nscale, "0")) {
		custom_scale = atoi(nscale);
		layout_mode = LAYOUT_CUSTOM;
		update_fitmode();
		v.SetFitScale(custom_scale/100.0);
		show_image(mng.Current());
	}
}

void menu_handler(int index) {
	lastmenuindex = index;

	switch (index) {
		case GOTO:
			//PageSelector is broken on newer devices.
			if (caps.fw_major <= 2)
				OpenPageSelector(pageselect_handler);
			else {
				sprintf(kbuf, "%d", mng.GetCurrentFileIndex()+1);
				OpenKeyboard(GetLangText("@Goto_page"), kbuf, BUF_SIZE, KBD_NUMERIC, kpage_handler);
			}
			break;
		case LAYOUT+LAYOUT_DEFAULT:
		case LAYOUT+LAYOUT_DOUBLE:
		case LAYOUT+LAYOUT_FIT_PAGE:
		case LAYOUT+LAYOUT_FIT_HEIGHT:
		case LAYOUT+LAYOUT_FIT_WIDTH:
		case LAYOUT+LAYOUT_FULL_SIZE:
			if (index-LAYOUT != layout_mode) {
				layout_mode = index - LAYOUT;
				update_fitmode();
				show_image(mng.Current());
			}
			break;
		case LAYOUT+LAYOUT_CUSTOM+10:
			sprintf(kbuf, "%d", custom_scale);
			OpenKeyboard(GetLangText("@Enter_scale"), kbuf, BUF_SIZE, KBD_NUMERIC, scale_handler);
			break;
		case LAYOUT+LAYOUT_CUSTOM+30:
			setting_custom_layout = true;
			Dialog(ICON_INFORMATION, GetLangText("@Program_name"), GetLangText("@Set_scale_ptr"), GetLangText("@Close"), NULL, NULL);
			break;
		case FINE:
			do_fine_update = !do_fine_update;
			break;
		case STATUS:
			do_draw_panel = !do_draw_panel;
			SetPanelType(do_draw_panel ? 1 : 0);
			v.SetClipRect(0, 0, ScreenWidth(), cliprect_height());
			show_image(mng.Current());
			break;
		case PANEL+0:
			v.ToggleAutoPanelMode();
			show_image(mng.Current());
			break;
		case PANEL+1:
			v.ToggleScalePanelMode();
			show_image(mng.Current());
			break;
		case PANEL+2:
			v.SetPanelBkgdThreshold(220);
			break;
		case PANEL+3:
			v.SetPanelBkgdThreshold(168);
			break;
		case PANEL+4:
			v.SetPanelBkgdThreshold(127);
			break;
		case CONF:
			open_conf();
			break;
		case DISP:
			open_display_conf();
			break;
		case DICT:
			OpenDictionaryView(NULL, NULL);
			break;
		case TURN+0:
		case TURN+1:
		case TURN+2:
		case TURN+3:
			rotate_screen(index-TURN);
			show_image(mng.Current());
			break;
		case CROP:
			v.ToggleCroppedMargins();
			show_image(mng.Current());
			break;
		case CROP+10+MARGIN_SMALL:
		case CROP+10+MARGIN_MEDIUM:
		case CROP+10+MARGIN_LARGE:
			v.SetMarginSize(2<<(index-CROP-10));
			show_image(mng.Current());
			break;
		case CROP+20+MARGIN_THRESH_LOW:
		case CROP+20+MARGIN_THRESH_MED:
		case CROP+20+MARGIN_THRESH_HIGH:
			v.SetMarginThreshold(2<<(index-CROP-20));
			show_image(mng.Current());
			break;
		case SAVE:
			if (image_save_dir && !mng.SaveCurrentImage(image_save_dir)) {
				std::string msg = GetLangText("@Scrn_saved");
				msg += " ";
				msg += image_save_dir;
				msg += "/";
				char *filenamec = strdup(mng.GetCurrentFileName().c_str());
				msg += basename(filenamec);
				Message(ICON_INFORMATION, GetLangText("@Program_name"), msg.c_str(), 3000);
				free(filenamec);
			}
			break;
		case TOC:
			OpenContents(toc, mng.GetFilesCount(), mng.GetCurrentFileIndex(), contents_handler);
			break;
		case HELP:
			show_help();
			break;
		case ABOUT:
			show_about();
			break;
		case EXIT:
			CloseApp();
			break;
		default:
			break;
	}
}

void password_handler(char *password) {
	if (password) {
		can_ask_for_password = true;
		mng.SetPassword(password);
		show_image(mng.Current());
	} else
		CloseApp();
}

void need_password() {
	if (can_ask_for_password) {
		can_ask_for_password = false;
		OpenKeyboard("@Password", kbuf, BUF_SIZE, KBD_PASSWORD, password_handler);
	}
}

void apply_settings() {
	Settings settings;
	uint32_t tag;
	int readdir;
	char *settings_file = GetAssociatedFile(have_archive ? infile.c_str() : dirbuff, 0);	
	Log::msg("settings file = %s", settings_file);
	FILE *fp = fopen(settings_file, "rb");
	if (fp &&
	    (fread(&settings, 1, sizeof(settings), fp) == sizeof(settings)) &&
	    settings.magic == 0x9751) {
		layout_mode = settings.layout_mode;
		custom_scale = settings.custom_scale & 0xffff;
		v.SetFitScale(custom_scale/100.0);
		update_fitmode();
		do_fine_update = settings.do_fine_update != 0;
		if (do_reset_last_page && (settings.page == ((int)mng.GetFilesCount()-1)))
			settings.page = 0;
		/* The following were all added after the Settings struct was finalized.
		 * So, each new setting gets added with a tag to identify it, so that
		 * we don't have to worry about backwards compatibility anymore. */
		while ((fread(&tag, 1, sizeof(tag), fp) == sizeof(tag))) {
			switch (tag) {
				case BRIGHTNESS:
					fread(&brightness_idx, 1, sizeof(brightness_idx), fp);
					break;
				case CONTRAST:
					fread(&contrast_idx, 1, sizeof(contrast_idx), fp);
					break;
				case GAMMA:
					fread(&gamma_idx, 1, sizeof(gamma_idx), fp);
					break;
				case DEPTH:
					fread(&depth_idx, 1, sizeof(depth_idx), fp);
					break;
				case READDIR:
					fread(&readdir, 1, sizeof(readdir), fp);
					v.SetReadingDirection(readdir);
					break;
				default:
					Log::msg("unknown settings tag: %d", tag);
					break;
			}
		}
		set_lut();
		set_max_ibitmap_depth(atoi(depth_variants[depth_idx]));
		// Go to the saved last page if the input isn't an image file
		if (!ImagesMng::IsSupportedImage(infile.c_str()))
			v.SetImage(mng.GoTo(settings.page));
		fclose(fp);
	} else
		v.SetImage(mng.Current());

	/* This is a total hack to use the configuration editor for
	 * modifying the display characteristics.  There really isn't
	 * a configuration file to open. */
	dispcfg = OpenConfig("/tmp/dispcfg", dispce);
	WriteInt(dispcfg, "brightness", brightness_idx);
	WriteInt(dispcfg, "contrast", contrast_idx);
	WriteInt(dispcfg, "gamma", gamma_idx);
	WriteInt(dispcfg, "depth", depth_idx);
}

void save_settings() {
	Settings settings;
	uint32_t tag;
	int readdir = v.GetReadingDirection();
	char *settings_file = GetAssociatedFile(have_archive ? infile.c_str() : dirbuff, 0);	
	Log::msg("using settings file %s", settings_file);
	ensure_path(settings_file);
	settings.magic = 0x9751;
	settings.page = mng.GetCurrentFileIndex();
	settings.layout_mode = layout_mode;
	settings.custom_scale = (custom_scale & 0xffff);
	settings.do_fine_update = do_fine_update ? 1 : 0;
	FILE *fp = fopen(settings_file, "wb");
	if (fp) {
		fwrite(&settings, 1, sizeof(settings), fp);
		/* The following are the extra settings. */
		tag = BRIGHTNESS;
		fwrite(&tag, 1, sizeof(tag), fp);
		fwrite(&brightness_idx, 1, sizeof(brightness_idx), fp);
		tag = CONTRAST;
		fwrite(&tag, 1, sizeof(tag), fp);
		fwrite(&contrast_idx, 1, sizeof(contrast_idx), fp);
		tag = GAMMA;
		fwrite(&tag, 1, sizeof(tag), fp);
		fwrite(&gamma_idx, 1, sizeof(gamma_idx), fp);
		tag = DEPTH;
		fwrite(&tag, 1, sizeof(tag), fp);
		fwrite(&depth_idx, 1, sizeof(depth_idx), fp);
		tag = READDIR;
		fwrite(&tag, 1, sizeof(tag), fp);
		fwrite(&readdir, 1, sizeof(readdir), fp);
		fclose(fp);
		Log::msg("successfully save to settings file");
	} else {
		Log::msg("error opening file_settings: %d", errno);
	}
}

void create_toc() {
	int numpages = mng.GetFilesCount();
	if ((toc = (tocentry*)malloc(numpages * sizeof(tocentry)))) {
		for (int i = 0; i < numpages; i++) {
			toc[i].level = 1;
			toc[i].page = i+1;
			toc[i].position = i;
			toc[i].text = (char*)mng.GetFileName(i).c_str();
		}
	}
	Log::msg("numpages = %d", numpages);
}

void dir_selected(char *path) {
	std::string dir(path);
	if(!dir.empty())
	{
		strcpy(dirbuff,path);
		mng.Open(dir,"");
		apply_settings();
		create_toc();
		show_image(mng.Current());
	}
}

bool open_images() {
	have_archive = ImagesMng::IsSupportedArchive(infile);
	bool can_open = have_archive || ImagesMng::IsSupportedImage(infile.c_str());

	if (can_open) {
		ShowHourglass();
		mng.Open(infile);
		apply_settings();
		HideHourglass();
		create_toc();
		show_image(mng.Current());
	}

	return can_open;
}

void draw_pagemap() {
	unsigned int margin = caps.pagemap_margin;  //Add this so icon moves away from right edge.
	unsigned short width = caps.pagemap_width;
	unsigned short height = caps.pagemap_height;
	unsigned short depth = 8;
	unsigned short scanline = (width + margin) * depth/8;

	int x, y, X, Y;
	double scaleX, scaleY;
	
	if (!pagemap) {
		pagemap = (ibitmap*)malloc(sizeof(ibitmap) + scanline*height);
		pagemap->width = width + margin;
		pagemap->height = height;
		pagemap->depth = depth;
		pagemap->scanline = scanline;
		memset(pagemap->data, caps.pagemap_clear, scanline*height);
	}

	irect imgr = v.GetImageRect();
	irect clipr = v.GetClipRect();

	scaleX = imgr.w / width;
	scaleY = imgr.h / height;
	for (y = 0; y < height; y++) {
		Y = (int)((y + 0.5) * scaleY + 0.5) + imgr.y;
		for (x = 0; x < width; x++) {
			X = (int)((x + 0.5) * scaleX + 0.5) + imgr.x;
			if (X<clipr.x || X>=(clipr.x+clipr.w) || Y<clipr.y || Y>=(clipr.y+clipr.h))
				pagemap->data[y*scanline+x+margin] = caps.pagemap_back;
		  else
				pagemap->data[y*scanline+x+margin] = caps.pagemap_fore;
		}
	}
}

void get_new_translations() {
	trans_t *t = NULL;

	// Add new translations here after creating the trans_XX
	// array in translations.h
	if      (!strcmp(lang, "en")) t = trans_en;
	else if (!strcmp(lang, "fr")) t = trans_fr;
	else if (!strcmp(lang, "pg")) t = trans_pt;
	else if (!strcmp(lang, "pt")) t = trans_pt;
	else if (!strcmp(lang, "de")) t = trans_de;
	else if (!strcmp(lang, "nl")) t = trans_nl;
	else if (!strcmp(lang, "ja")) t = trans_ja;
	else if (!strcmp(lang, "hu")) t = trans_hu;
	else if (!strcmp(lang, "pl")) t = trans_pl;
	else if (!strcmp(lang, "bg")) t = trans_bg;
	else                   t = trans_en;

	while (t->label) {
		AddTranslation(t->label, t->trans);
		t++;
	}
}

void open_next_archive(int i) {
	if ((i == 1) && !next_archive.empty()) {
		OpenBook(next_archive.c_str(), NULL, OB_ADDTOLAST);
		CloseApp();
	}
}

void ask_for_next_archive() {
	/* Look to see if there is a following archive file in the same
	 * directory, and if so, ask the reader if they want to open it. */
	std::string nextmsg;
	size_t slash_pos = infile.find_last_of("/");
	std::string parent_dir = infile.substr(0, slash_pos);
	const char *dirs[] = { parent_dir.c_str(), NULL };
	FTS *tree = fts_open((char * const *)dirs,
	                     FTS_PHYSICAL | FTS_NOCHDIR | FTS_NOSTAT, NULL);

	next_archive = "";

	if (tree) {
		FTSENT *node;
		std::vector<std::string> archives;
		std::string name;

		while ((node = fts_read(tree))) {
			if ((node->fts_info & FTS_F) && (node->fts_level == 1)) {
				name = node->fts_path;
			  if (ImagesMng::IsSupportedArchive(name))
					archives.push_back(name);
			}
		}
		fts_close(tree);

		if (!archives.empty()) {
			ImagesMng::FileNameSort(archives);

			unsigned int i = 0;
			while (i<archives.size() && archives[i] != infile) i++;
			if (i < archives.size()-1) {
				next_archive += archives[i+1];
				nextmsg = GetLangText("@Open_next");
				nextmsg += "\n" + next_archive;
				Dialog(ICON_QUESTION, GetLangText("@Program_name"), nextmsg.c_str(), GetLangText("@Yes"), GetLangText("@No"), open_next_archive);
			}

			archives.clear();
		}
	}
}

void handle_next_page() {
	if (mng.HasNext())
		show_image(mng.Next());
	else if (have_archive && do_start_next_archive)
		ask_for_next_archive();
}

void get_gti_pointer() {
	/* This gets the pointer to the GetTouchInfo() function if it is available. */
	void *handle;

	if ((handle = dlopen("libinkview.so", RTLD_LAZY))) {
		*(void **) (&gti) = dlsym(handle, "GetTouchInfo");
		dlclose(handle);
	} else
		gti = NULL;
}

void add_def(std::string &msg, const char *key_label, const char *def_label) {
	msg += "\n";
	msg += GetLangText(key_label);
	msg += ": ";
	msg += GetLangText(def_label);
}

void add_def2(std::string &msg, const char *key_label, const char *def_label1,
              const char *def_label2) {
	msg += "\n";
	msg += GetLangText(key_label);
	msg += ": ";
	msg += GetLangText(def_label1);
	msg += " ";
	msg += GetLangText(def_label2);
}

void show_help() {
	std::string msg = "===== ";
	msg += GetLangText("@Button_help_title");
	msg += " =====";
	if (caps.has_dpad_buttons) {
		add_def(msg, "@Key_ok", "@KA_mmnu");
		add_def(msg, "@Key_ok_h", "@KA_dict");
	}
	if (caps.has_menu_button)
		add_def(msg, "@Key_menu", "@Contents");
	if (!use_next_prev_for_stepping)
		add_def(msg, "@Key_next", "@KA_pgdn");
	else
		add_def(msg, "@Key_next", "@Help_next");
	if (caps.has_volume_buttons)
		add_def(msg, "@Key_plus", "@KA_pgdn");
	if (!use_next_prev_for_stepping)
		add_def(msg, "@Key_prev", "@KA_pgup");
	else
		add_def(msg, "@Key_prev", "@Help_prev");
	if (caps.has_volume_buttons)
		add_def(msg, "@Key_minus", "@KA_pgup");
	if (caps.has_back_button)
		add_def(msg, "@Key_back", "@Help_back");
	if (caps.has_dpad_buttons) {
		add_def(msg, "@Key_left", "@Help_left");
		add_def2(msg, "@Key_left_h", "@Read", "@RightToLeft");
		add_def(msg, "@Key_right", "@Help_right");
		add_def2(msg, "@Key_right_h", "@Read", "@LeftToRight");
		add_def(msg, "@Key_up", "@Help_up");
		add_def(msg, "@Key_down", "@Help_down");
	}
	if (use_next_prev_for_stepping) {
		add_def2(msg, "@Key_next_h", "@Read", "@LeftToRight");
		add_def2(msg, "@Key_prev_h", "@Read", "@RightToLeft");
	}
	if (caps.has_mag_buttons)
		add_def(msg, "@Key_zoomin", "@KA_zmin");
	if (caps.has_dpad_buttons)
		add_def(msg, "@Key_up_h", "@KA_zmin");
	if (caps.has_mag_buttons)
		add_def(msg, "@Key_zoomout", "@KA_zout");
	if (caps.has_dpad_buttons)
		add_def(msg, "@Key_down_h", "@KA_zout");
	if (caps.has_touchscreen && gti) {
		msg += "\n===== ";
		msg += GetLangText("@Gesture_help_title");
		msg += " =====";
		add_def(msg, "@Two-finger_tap", "@Help_unzoom");
		add_def(msg, "@Two-finger_pinch", "@Zoom");
		add_def(msg, "@Two-finger_swipe", "@Help_swipe");
	}
	msg += "\n\n";
	msg += GetLangText("@Translation_by");
	Dialog(ICON_INFORMATION, GetLangText("@Program_name"), msg.c_str(), GetLangText("@Close"), NULL, NULL);
}

void show_about() {
	std::string msg = GetLangText("@Program_name");
	msg += " ";
	msg += PBIV_VERSION;
	msg += "\n";
	msg += "is a program for viewing images collected within directories\n";
	msg += "or within archives such as CBR(RAR), CBZ(ZIP) and TAR files.\n";
	msg += "This program was originally written by Michail Polubisok, and\n";
	msg += "later augmented by Rob Komar.\n";
	msg += "The cartoon panel detection code is based on a flood fill algorithm\n";
	msg += "by Lode Vandevenne (http://lodev.org/cgtutor/floodfill.html).\n";
	Dialog(ICON_INFORMATION, GetLangText("@Program_name"), msg.c_str(), GetLangText("@Close"), NULL, NULL);
}

void show_page_info() {
	char buff[64];
	sprintf(buff, "%s: %d/%d\n%s: %.0f%%\n", GetLangText("@Page"), mng.GetCurrentFileIndex() + 1, mng.GetFilesCount(), GetLangText("@Zoom"), v.GetScale() * 100);
	if (v.IsAutoPanelModeEnabled()) {
		if (v.IsScalePanelModeEnabled())
			strcat(buff, " <[ ]>  ");
		else
			strcat(buff, " [ ]  ");
	}
	if (v.IsCroppedMargins())
		strcat(buff, " >||<");
	Message(ICON_INFORMATION, GetLangText("@Program_name"), buff, 3000);
}

void draw_panel(ibitmap *panelicon) {
	char buff[64];
	sprintf(buff, "%d/%d - %.0f%%", mng.GetCurrentFileIndex() + 1, mng.GetFilesCount(), v.GetScale() * 100);
	if (v.IsAutoPanelModeEnabled()) {
		if (v.IsScalePanelModeEnabled())
			strcat(buff, " <[]>");
		else
			strcat(buff, " []");
	}
	if (v.IsCroppedMargins())
		strcat(buff, " >||<");
	if (mng.GetFilesCount() == 0)
		DrawPanel(PANELICON_LOAD, buff, "---", -1);
	else {
		if (panelicon == pagemap) {
			draw_pagemap();
			/* Update panelicon in case the pointer to pagemap changed. */
			panelicon = pagemap;
		}
		DrawPanel(panelicon, buff, (char*) mng.GetCurrentFileName().c_str(), -1);
	}
}

void draw()
{
	ClearScreen();
	v.Draw();
	Log::msg("Index: %d, size: %d, file: %s", mng.GetCurrentFileIndex(),
	         mng.GetFilesCount(), mng.GetCurrentFileFullname().c_str());
	if (do_draw_panel)
		draw_panel(mng.IsBuffered() ? pagemap : (ibitmap*)PANELICON_LOAD);
	FullUpdate();
	if (do_fine_update)
		FineUpdate();
}

void rotate_screen(int new_or) {
	if (GetOrientation() != new_or) {
		SetOrientation(new_or);
		Log::msg("Setting orientation to %d", new_or);
		update_fitmode();
		v.SetClipRect(0, 0, ScreenWidth(), cliprect_height());
	}
}

void set_custom_layout(int x0, int x1) {
	irect clip_r = v.GetClipRect();
	irect img_r = v.GetImageRect();
	double scale = v.GetScale(), rescale;
	int new_img_x, new_img_x_aligned;

	layout_mode = LAYOUT_CUSTOM;
	update_fitmode();
	if (x1 > x0) {
		/* Zoom in so the line fits the screen. */
		rescale = (double)clip_r.w/(x1-x0+1);
		new_img_x = (int)nearbyint((img_r.x-x0)*rescale);
	} else {
		/* Zoom current view out into line. */
		rescale = (double)(x0-x1+1)/clip_r.w;
		new_img_x = x1 - (int)nearbyint(-img_r.x*rescale);
	}
	scale *= rescale;
	new_img_x_aligned = clip_r.x +
	                    (clip_r.w - (int)(mng.Current()->width*scale))/2;
	v.SetFitScale(scale);
	custom_scale = (int)nearbyint(100*scale);
}

void step(int direction) {
	int plane = 0;

	if (direction == DIRECTION_UP)
		plane = DIRECTION_VERTICAL;
	else if (direction == DIRECTION_DOWN)
		plane = DIRECTION_VERTICAL;
	else if (direction == DIRECTION_LEFT)
		plane = v.IsPanelModeEnabled() ? DIRECTION_LEFT : DIRECTION_HORIZONTAL;
	else if (direction == DIRECTION_RIGHT)
		plane = v.IsPanelModeEnabled() ? DIRECTION_RIGHT : DIRECTION_HORIZONTAL;

	if (step_divisor != 1)
		v.SetMoveStep(v.GetMoveStep()/step_divisor);

	if (v.CanMoveDirection(plane)) {
		// It is possible to move in this plane, so stay on this page.
		if (v.CanMoveDirection(direction)) {
			v.MoveDirection(direction);
			draw();
		}
	} else {
		// It is not possible to move in this plane, so change pages.
		if (direction == DIRECTION_UP) {
			if (mng.HasPrev())
				show_image(mng.Prev());
		} else if (direction == DIRECTION_DOWN) {
			handle_next_page();
		} else if (direction == DIRECTION_LEFT) {
			if (v.IsPanelModeEnabled() && v.GetReadingDirection() == DIRECTION_LEFT) {
				//Jump forward to next page when reading panels right to left.
				handle_next_page();
			} else {
				if (mng.HasPrev())
					show_image(mng.Prev());
			}
		} else if (direction == DIRECTION_RIGHT) {
			if (v.IsPanelModeEnabled() && v.GetReadingDirection() == DIRECTION_LEFT) {
				//Jump back to previous page when reading panels right to left.
				if (mng.HasPrev())
					show_image(mng.Prev());
			} else
				handle_next_page();
		}
	}

	if (step_divisor != 1) {
		v.SetMoveStep(v.GetMoveStep()*step_divisor);
		step_divisor = 1;
	}
}

void set_drag_rotation(int dx, int dy) {
	/* Double drag, rotate screen so top-to-bottom is in drag direction. */
	int old_or, new_or, dtmp;

	old_or = new_or = GetOrientation();

	/* Rotate the directions to the Portrait coordinate system. */
	if (old_or == 1) {
		dtmp = dy; dy = dx; dx = -dtmp;
	} else if (old_or == 2) {
		dtmp = dy; dy = -dx; dx = dtmp;
	} else if (old_or == 3) {
		dy = -dy; dx = -dx;
	}

	if (abs(dx) > 2*abs(dy))
		new_or = (dx > 0) ? 2 : 1;
	else if (abs(dy) > 2*abs(dx))
		new_or = (dy > 0) ? 0 : 3;

	rotate_screen(new_or);
	show_image(mng.Current());
}

void handle_prev_event(void) {
	int step_dir = (v.GetReadingDirection() == DIRECTION_RIGHT) ? DIRECTION_LEFT :
	                                                              DIRECTION_RIGHT;
	if (use_next_prev_for_stepping && v.IsPanelModeEnabled())
		step(step_dir);
	else if (use_next_prev_for_stepping && v.CanMoveDirection(step_dir))
		step(step_dir);
	else if (use_next_prev_for_stepping && v.CanMoveDirection(DIRECTION_UP)) {
		if (v.CanMoveDirection(DIRECTION_HORIZONTAL))
			v.MoveAlign((step_dir == DIRECTION_RIGHT) ? ALIGN_LEFT : ALIGN_RIGHT);
		step(DIRECTION_UP);
	} else
		if (mng.HasPrev())
			show_image(mng.Prev());
}

void handle_next_event(void) {
	int step_dir = v.GetReadingDirection();
	if (use_next_prev_for_stepping && v.IsPanelModeEnabled())
		step(step_dir);
	else if (use_next_prev_for_stepping && v.CanMoveDirection(step_dir))
		step(step_dir);
	else if (use_next_prev_for_stepping && v.CanMoveDirection(DIRECTION_DOWN)) {
		if (v.CanMoveDirection(DIRECTION_HORIZONTAL))
			v.MoveAlign((step_dir == DIRECTION_RIGHT) ? ALIGN_LEFT : ALIGN_RIGHT);
		step(DIRECTION_DOWN);
	} else
		handle_next_page();
}

void handle_tap(int x, int y) {
	/* Launch event depending on where the screen was tapped.
	 * Use approximately the same zones as are used in the reading apps. */
	int sw = ScreenWidth(), sh = cliprect_height();
	int l10 = (sw < sh) ? sw/10 : sh/10, l5 = l10*2;

	if (y < l5) { /* Top zone */
		if (x < l5) { /* Top left; Exit */
			CloseApp();
		} else if (x > (sw - l5)) { /* Top right; More... menu */
			if (caps.fw_major >= 4)
				OpenMenu(menu_more, lastmenuindex, sw/2-MENU_HW, sh/2-MENU_HH, menu_handler);
		}
	} else if (y > (sh - l5)) { /* Bottom zone */
		if ((x < l5) || (x > (sw - l5)))	/* Bottom corners; Prev */
			handle_prev_event();
		else if ((x > 3*l10) && (x < (sw - 3*l10))) /* Bottom middle; page info */
			show_page_info();
	} else if ((y > 3*l10) && (y < (sh - 3*l10))) { /* Middle zone */
		if ((x < l5) || (x > (sw - l5)))	/* Middle sides; Next */
			handle_next_event();
		else if ((x > 3*l10) && (x < (sw - 3*l10))) /* Middle; main menu */
			OpenMenu(menu, lastmenuindex, sw/2-MENU_HW, sh/2-MENU_HH, menu_handler);
	}
}

void set_pinched_layout(int x00, int y00, int x01, int y01,
                        int x10, int y10, int x11, int y11) {
	/* 00/01 are the start positions of the pinch gesture,
	 * and 10/11 are the end positions. */
	int sw = ScreenWidth(), sh = cliprect_height();
	int startw, endw, startxc, startyc;
	int xdisplayc, ydisplayc;
	irect cliprect = v.GetClipRect(), imagerect = v.GetImageRect();
	double scale, rescale;

	/* Compute the center of the starting pinch, and the "width" at the start and end.
	 * The "width" is normalized to the screen's aspect ratio. */
	//startw = (abs(x00-x01) > abs(y00-y01)*sw/sh) ? abs(x00-x01) : abs(y00-y01)*sw/sh;
	//endw = (abs(x10-x11) > abs(y10-y11)*sw/sh) ? abs(x10-x11) : abs(y10-y11)*sw/sh;
	startxc = (x00+x01)/2 + cliprect.x - imagerect.x;
	startyc = (y00+y01)/2 + cliprect.y - imagerect.y;
	if (abs(x00-x01) > abs(y00-y01)*sw/sh) {
		startw = abs(x00-x01);
		endw = abs(x10-x11);
	} else {
		startw = abs(y00-y01)*sw/sh;
		endw = abs(y10-y11)*sw/sh;
	}

	rescale = double(endw)/startw;
	scale = v.GetScale() * rescale;

	v.SetScale(scale);

	//Move the center of the pinch to the center of the display.
	imagerect = v.GetImageRect();
	xdisplayc = cliprect.x + cliprect.w/2 - imagerect.x;
	ydisplayc = cliprect.y + cliprect.h/2 - imagerect.y;
	v.Move(int(startxc * rescale) - xdisplayc, int(startyc * rescale) - ydisplayc);
}

void pointer_evt_handler(int type, int par1, int par2) {
	int dx, dy, dx1, dy1, dp;
	iv_mtinfo *mti;
	iv_mtinfo_54 *mti54;	/* iv_mtinfo changed starting with firmware 5.4 */
	bool mtinfo_new = (caps.fw_major > 5) ||
	                  ((caps.fw_major == 5) && (caps.fw_minor >= 4));

	if ((type == EVT_POINTERDOWN) || (type == EVT_TOUCHDOWN)) {
		X00 = par1;
		Y00 = par2;
		pointing = 1;
	} else if ((type == EVT_MTSYNC) && pointing && (par2 >= 2)) {
		if (gti && (mti = (*gti)())) {
			mti54 = (iv_mtinfo_54*)mti;
			if (pointing == 1) {
				pointing = 2;
				/* On the Inkpad, the original pointer down location (X00, Y00)
				 * may end up associated with the mti[1] struct, so we reset it
				 * here, and stop trusting par1 and par2 for MTSYNC events. */
				X00 = (mtinfo_new) ? mti54[0].x : mti[0].x;
				Y00 = (mtinfo_new) ? mti54[0].y : mti[0].y;
				X01 = (mtinfo_new) ? mti54[1].x : mti[1].x;
				Y01 = (mtinfo_new) ? mti54[1].y : mti[1].y;
			} else {
				/* Inkpad MTSYNC events are buggy, and sometimes show the same
				 * coordinates for the two points, so we screen those out. */
				int X0 = (mtinfo_new) ? mti54[0].x : mti[0].x;
				int Y0 = (mtinfo_new) ? mti54[0].y : mti[0].y;
				int X1 = (mtinfo_new) ? mti54[1].x : mti[1].x;
				int Y1 = (mtinfo_new) ? mti54[1].y : mti[1].y;
				if ((abs(X0 - X1) > ptr_eps) || (abs(Y0 - Y1) > ptr_eps)) {
					X10 = X0;
					Y10 = Y0;
					X11 = X1;
					Y11 = Y1;
				}
			}
		}
	} else if (((type == EVT_POINTERUP) || (type == EVT_TOUCHUP)) && pointing) {
		if (pointing == 1) {
			/* The end of a single touch event. */
			dx = X00 - par1;
			dy = Y00 - par2;

			if (abs(dx) <= ptr_eps && abs(dy) <= ptr_eps) {
				/* Treat this as a tap for the menu. */
				handle_tap(X00, Y00);
			} else if (setting_custom_layout) {
				/* Set the Custom Layout based on the end points of the line. */
				setting_custom_layout = false;
				set_custom_layout(X00, par1);
				show_image(mng.Current());
			} else {
				if (!v.CanMoveDirection(DIRECTION_HORIZONTAL) &&
				    abs(dx) > 2*abs(dy)) {
					/* Page flip */
					if (dx > 0)
						handle_next_page();
					else if (dx < 0 && mng.HasPrev())
						show_image(mng.Prev());
				} else if (!v.CanMoveDirection(DIRECTION_VERTICAL) &&
				           abs(dy) > 2*abs(dx)) {
					/* Page flip */
					if (dy > 0)
						handle_next_page();
					else if (dy < 0 && mng.HasPrev())
						show_image(mng.Prev());
				} else {
					/* Pan image. */
					if ((dx < 0 && !v.CanMoveDirection(DIRECTION_LEFT)) ||
					    (dx > 0 && !v.CanMoveDirection(DIRECTION_RIGHT)))
						dx = 0;
					if ((dy < 0 && !v.CanMoveDirection(DIRECTION_UP)) ||
					    (dy > 0 && !v.CanMoveDirection(DIRECTION_DOWN)))
						dy = 0;
					if (dx != 0 || dy != 0) {
						v.Move(dx, dy);
						draw();
					}
				}
			}
		} else {
			/* The end of a multi-touch event. */
			dx = X10 - X00;
			dy = Y10 - Y00;
			dx1 = X11 - X01;
			dy1 = Y11 - Y01;
			dp = dx*dx1 + dy*dy1;	/* dot product for testing parallel/anti-parallel */
			if (abs(dx) <= ptr_eps && abs(dy) <= ptr_eps &&
			    abs(dx1) <= ptr_eps && abs(dy1) <= ptr_eps) {
				/* Double tap, unzoom. */
				v.ZoomFit();
				draw();
			} else if ((abs(dx) > ptr_eps || abs(dy) > ptr_eps) &&
			           (abs(dx1) > ptr_eps || abs(dy1) > ptr_eps) &&
			           (dp > 0)) {
				/* Double drag, rotate screen. */
				set_drag_rotation(dx, dy);
			} else {
				/* Pinch to zoom. */
				set_pinched_layout(X00, Y00, X01, Y01, X10, Y10, X11, Y11);
				draw();
			}
		}
		pointing = 0;
	}
}

int main_handler(int type, int par1, int par2)
{
	Log::msg("[%i %i %i]", type, par1, par2);

	if (type == EVT_INIT)
	{
		start_orientation = GetOrientation();
		v.SetMoveStep(.75);
		pointing = 0;
		setting_custom_layout = false;
		can_ask_for_password = true;
		get_gti_pointer();

		caps.InitCaps();
		caps.LogCaps();

		read_config();
		create_main_menu();
		brightness_idx = contrast_idx = 4;
		gamma_idx = 5;
		depth_idx = (caps.max_screen_depth == 4) ?
		            DEPTH4 :
								(caps.max_screen_depth == 24) ? DEPTH24 : DEPTH8;
		set_max_ibitmap_depth(atoi(depth_variants[depth_idx]));
		get_new_translations();
		update_fitmode();
		v.SetClipRect(0, 0, ScreenWidth(), cliprect_height());
		ptr_eps = ((ScreenWidth() > cliprect_height()) ? ScreenWidth() : cliprect_height())/50;
		if (!caps.has_dpad_buttons) 	//No D-Pad; use Next/Prev to step.
			use_next_prev_for_stepping = true;

		draw();

		if (infile.empty() || !open_images())
			OpenDirectorySelector(GetLangText("@Open_dir"), dirbuff, BUF_SIZE, dir_selected);
		dont_redraw = true;
	}

	if (type == EVT_SHOW)
	{
		if (dont_redraw){	/* Avoid extra redrawing after EVT_INIT does the job. */
			dont_redraw = false;
			if (!infile.empty())
				BookReady(infile.c_str());	//Need this for opening on bootup
		} else if (mng.Current())
			show_image(mng.Current());
	}

	if (type == EVT_KEYRELEASE && par2 == 0)
	{
		switch (par1)
		{
		case KEY_OK:
			Log::msg("Key OK click");
			OpenMenu(menu, lastmenuindex, ScreenWidth()/2-MENU_HW, ScreenHeight()/2-MENU_HH, menu_handler);
			break;

		case KEY_BACK:
			Log::msg("Key Back click");
			if (v.IsPanelModeEnabled()) {
				v.DisablePanelMode();
				draw();
			} else
				step_divisor <<= 1;
			break;

		case KEY_LEFT:
			Log::msg("Key Left click");
			if (v.CanMoveDirection(DIRECTION_LEFT))
				step(DIRECTION_LEFT);
			else {
				int vertical_dir = (v.GetReadingDirection() == DIRECTION_LEFT) ?
				                   DIRECTION_DOWN :
				                   DIRECTION_UP;
				if (v.CanMoveDirection(vertical_dir)) {
					if (v.CanMoveDirection(DIRECTION_HORIZONTAL))
						v.MoveAlign(ALIGN_RIGHT);
					step(vertical_dir);
				}
			}
			break;

		case KEY_RIGHT:
			Log::msg("Key Right click");
			if (v.CanMoveDirection(DIRECTION_RIGHT))
				step(DIRECTION_RIGHT);
			else {
				int vertical_dir = (v.GetReadingDirection() == DIRECTION_RIGHT) ?
				                   DIRECTION_DOWN :
				                   DIRECTION_UP;
				if (v.CanMoveDirection(vertical_dir)) {
					if (v.CanMoveDirection(DIRECTION_HORIZONTAL))
						v.MoveAlign(ALIGN_LEFT);
					step(vertical_dir);
				}
			}
			break;

		case KEY_UP:
			Log::msg("Key Up click");
			step(DIRECTION_UP);
			break;

		case KEY_DOWN:
			Log::msg("Key Down click");
			step(DIRECTION_DOWN);
			break;

		case KEY_MINUS:
		case KEY_PREV:
			Log::msg("Key Prev (Minus) click");
			handle_prev_event();
			break;

		case KEY_PLUS:
		case KEY_NEXT:
			Log::msg("Key Next (Plus) click");
			handle_next_event();
			break;

#ifdef KEY_ZOOMIN
		case KEY_ZOOMIN:
			Log::msg("Key Zoomin click");
			if (v.CanZoomIn())
			{
				v.ZoomIn();
				draw();
			}
			break;
#endif

#ifdef KEY_ZOOMOUT
		case KEY_ZOOMOUT:
			Log::msg("Key Zoomout click");
			if (v.CanZoomOut())
			{
				v.ZoomOut();
				draw();
			}
			break;
#endif

		case KEY_MENU:
			Log::msg("Key Menu click");
			OpenContents(toc, mng.GetFilesCount(), mng.GetCurrentFileIndex(), contents_handler);
			break;
		}
	}
	if (type == EVT_KEYREPEAT && par2 == 1)
	{
		switch (par1)
		{
		case KEY_OK:
			Log::msg("Key OK hold");
			OpenDictionaryView(NULL, NULL);
			break;

		case KEY_LEFT:
			Log::msg("Key Left hold");
			if (v.SetReadingDirection(DIRECTION_LEFT))
				draw();
			break;

		case KEY_RIGHT:
			Log::msg("Key Right hold");
			if (v.SetReadingDirection(DIRECTION_RIGHT))
				draw();
			break;

		case KEY_UP:
			Log::msg("Key Up hold");
			if (v.CanZoomIn())
			{
				v.ZoomIn();
				draw();
			}
			break;

		case KEY_DOWN:
			Log::msg("Key Down hold");
			if (v.CanZoomOut())
			{
				v.ZoomOut();
				draw();
			}
			break;

		case KEY_MINUS:
			Log::msg("Key Minus hold");
			break;

		case KEY_PREV:
			Log::msg("Key Prev hold");
			/* This is for navigating panels on the Touch */
			if (use_next_prev_for_stepping && v.SetReadingDirection(DIRECTION_LEFT))
				draw();
			break;

		case KEY_PLUS:
			Log::msg("Key Plus hold");
			break;

		case KEY_NEXT:
			Log::msg("Key Next hold");
			/* This is for navigating panels on the Touch */
			if (use_next_prev_for_stepping && v.SetReadingDirection(DIRECTION_RIGHT))
				draw();
			break;

		case KEY_MENU:
			Log::msg("Key Menu hold");
			break;

		case KEY_BACK:
			CloseApp();
			break;
		}
	}

	if (ISPOINTEREVENT(type) || ISTOUCHEVENT(type))
		pointer_evt_handler(type, par1, par2);
	else if (pointing) {
		/* Cancel action if something else happened between pointer up and down. */
		FILE *fq = fopen("/tmp/pbimageviewer.out", "a");
		if (fq) {
			fprintf(fq, "Pointer action canceled by event type: %d\n", type);
			fclose(fq);
		}
		pointing = 0;
	}

	if (type == EVT_ORIENTATION) {
		rotate_screen(par1);
		show_image(mng.Current());
	}

/* This is so the code compiles under the older ABI.  It won't actually
 * see any EVT_BACKGROUND events. */
#ifndef EVT_BACKGROUND
#define EVT_BACKGROUND 152
#endif

	if (type == EVT_BACKGROUND) {
		// Save the settings when the multi-task menu comes up because killing
		// the program there doesn't generate EVT_EXIT events.
		SaveConfig(pbivcfg);
		save_settings();
	}

	if (type == EVT_SNAPSHOT) {
		// This provides the startup screen while booting.
		DrawPanel((ibitmap*)PANELICON_LOAD, "@snapshot_info", NULL, -1);
		PageSnapshot();
	}

	if (type == EVT_EXIT)
	{
		// Occurs in main handler when exiting or when SIGINT received.
		CloseConfig(pbivcfg);
		save_settings();
		CloseConfig(dispcfg);
		unlink("/tmp/dispcfg");
		if (have_archive)
			mng.CleanupArchive();
		free(toc);
		free(pagemap);
		if (!QueryGSensor() || GetGlobalOrientation() != -1)
			SetOrientation(start_orientation);
	}


	return 0;

}

int main(int argc, char **argv)
{
	if (argc == 2)
		infile = argv[1];
	else
		infile.erase();

	InkViewMain(main_handler);
	return 0;
}
