#include <stdio.h>	/* For sprintf() */
#include <string.h>	/* For memcpy() memset() */
#include <unistd.h>	/* For unlink() */
#include <dll.h>  /* for libunrar */
#include <inkview.h>	/* For Message() */
#include "RarHandler.h"
#include "ImagesMng.h"
#include "Log.h"

bool RarHandler::_check_rar_magic = false;

RarHandler::RarHandler(std::string& outdir) : ArchiveHandler(outdir) {
}

RarHandler::~RarHandler() {
	_rarfile.clear();
}

static int GetRarFileMagic(unsigned int msg, long UserData, long P1, long P2) {
	/* This is a callback function that gets executed by RarProcessFile()
	 * for checking the magic numbers of files. It copies the first 8 bytes
	 * of the file into the location pointed to by UserData (file_magic),
	 * and returns -1 to cancel further callbacks on that file. */
	char *file_magic = (char*)UserData;
	char *file_data = (char*)P1;
	int len = (int)((P2 > 8) ? 8 : P2);
	if (msg == UCM_PROCESSDATA) {
		memcpy(file_magic, file_data, len);
	} else {
		memset(file_magic, 0, 8);
	}
	return -1;
}

static void TransNameWToName(RARHeaderDataEx *file) {
	int i;
	int n;
	char *p = file->FileName;
	for (i = 0; i < wcslen(file->FileNameW); i++) {
		if ((n = wctomb(p, file->FileNameW[i])) != -1)
			p += n;
		if ((p - file->FileName + MB_CUR_MAX) > sizeof(file->FileName))
			break;
	}
}

int RarHandler::Open(std::string& path, std::vector<std::string>& files) {
	int buflen = 1024;
	char errmsg[buflen];
	char file_magic[8];
	HANDLE rar;
	struct RAROpenArchiveData archive;
	struct RARHeaderDataEx file;
	int ret, failed = 0;

	_rarfile = path;
	memset(&archive, 0, sizeof(archive));
	archive.ArcName = (char*)_rarfile.c_str();
	archive.OpenMode = _check_rar_magic ? RAR_OM_EXTRACT : RAR_OM_LIST;

	if ((rar = RAROpenArchive(&archive))) {
		/* Use callback on extraction to get magic header from each file. */
		if (_check_rar_magic)
			RARSetCallback(rar, GetRarFileMagic, (long)file_magic);

		memset(&file, 0, sizeof(file));
		while (true) {
			if((ret = RARReadHeaderEx(rar, &file)))
				break;	// End of file
			TransNameWToName(&file);
			if (!_check_rar_magic || file.Flags & 0x04) {
				/* Password encrypted file, so guess if it's supported based on name. */
				if (ImagesMng::IsSupportedImage(file.FileName))
					files.push_back(file.FileName);
			} else if (file.FileAttr & 0x20) {	//Ordinary file
				RARProcessFile(rar, RAR_TEST, NULL, NULL);	// Updates file_magic[]
				if (ImagesMng::GetMagicType(file_magic) != ImagesMng::UNSUPPORTED_IMAGE)
					files.push_back(file.FileName);
			}
			if ((ret = RARProcessFile(rar, RAR_SKIP, NULL, NULL)))
				break;	// End of file
		}
		RARCloseArchive(rar);
		failed = 0;
	} else {
		if (snprintf(errmsg, buflen,
		             "OpenRar: couldn't open archive %s: error = %d\n",
		             _rarfile.c_str(), archive.OpenResult) >= buflen)
			errmsg[buflen-1] = '\0';	/* Force Null termination */
		Message(ICON_ERROR, GetLangText("@Program_name"), errmsg, 3000);
		failed = 1;
	}

	return failed;
}

int RarHandler::UnpackFile(std::string& name, std::string &outname) {
	int ret, need_passwd = 0;
	int buflen = 1024;
	char errmsg[buflen];
	HANDLE rar;
	struct RAROpenArchiveData archive;
	struct RARHeaderDataEx file;

	memset(&archive, 0, sizeof(archive));
	archive.ArcName = (char*)_rarfile.c_str();
	archive.OpenMode = RAR_OM_EXTRACT;

	if ((rar = RAROpenArchive(&archive))) {
		if (_passwd.size())
			RARSetPassword(rar, (char*)_passwd.c_str());
		outname = _dir;
		size_t slash_pos = name.find_last_of("/");
		if (slash_pos != std::string::npos)
			outname += name.substr(slash_pos + 1);
		else
			outname += name;
		unlink(outname.c_str());  /* Get rid of any old copies. */
		memset(&file, 0, sizeof(file));
		while (true) {
			if(RARReadHeaderEx(rar, &file))
				break;		// End of file
			TransNameWToName(&file);
			if (name != file.FileName)
				RARProcessFile(rar, RAR_SKIP, NULL, NULL);	// Skip to next file
			else {
				if ((file.Flags & 0x04)  && !_passwd.size()) {
					outname = "";		// Need a password, but don't have one
					need_passwd = 1;
				} else {
					ret = RARProcessFile(rar, RAR_EXTRACT, NULL, (char*)outname.c_str());
					if (ret) {
						/* If there are errors in decoding, then we probably have the wrong password. */
						if ((ret == ERAR_BAD_DATA) || (ret == ERAR_NO_MEMORY)) {
							Message(ICON_ERROR, GetLangText("@Program_name"), GetLangText("@PasswordNotMatch"), 3000);
							need_passwd = 1;
						}
						unlink(outname.c_str());
						outname = "";
					}
				}
				break;
			}
		}
		RARCloseArchive(rar);
	} else {
		if (snprintf(errmsg, buflen,
		            "UnpackRarFile: couldn't open archive %s: error = %d\n",
		            _rarfile.c_str(), archive.OpenResult) >= buflen)
			errmsg[buflen-1] = '\0';	/* Force NULL termination */
		Message(ICON_ERROR, GetLangText("@Program_name"), errmsg, 3000);
	}

	return need_passwd;
}
