/*
 * ImagesMng.cpp
 *
 *  Created on: 28.01.2010
 *      Author: Admin
 */

#include <errno.h>
#include <stdio.h>
#include <math.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fts.h>
#define PNG_DEBUG 3
#include <png.h>
#include <img2ibitmap.h>
#include "ImagesMng.h"
#include "ZipHandler.h"
#include "RarHandler.h"
#include "C7zHandler.h"

void need_password();		/* This is in main.cpp */

/**************************** Public methods *********************************/

ImagesMng::ImagesMng()
{

	_current_index = -1;
	_prev_img = NULL;
	_current_img = NULL;
	_next_img = NULL;
	_archive_handler = NULL;
	RecomputeLUT(0, 0, 1.0);
}

ImagesMng::~ImagesMng()
{
	ClearBuffers();
	_files.clear();
	_dir.clear();
	_passwd.clear();
	if (_archive_handler)
		delete _archive_handler;
}

void ImagesMng::SetPassword(char *password) {
	if (_archive_handler)
		_archive_handler->SetPassword(password);
}

void ImagesMng::SetCheckRarMagic(bool _b) {
	RarHandler::SetCheckMagic(_b);
}

void ImagesMng::Open(std::string dir, std::string filename)
{
	if (dir[dir.size() - 1] != '/')
		dir += "/";

	_archive_type = GetArchiveType(dir+filename);
	if ((_archive_type == ZIP) || (_archive_type == RAR) ||
	    (_archive_type == C7Z))
		OpenArchive(dir, filename);
	else
		OpenDir(dir, filename);
}

void ImagesMng::Open(std::string fullname)
{
	size_t slash_pos = fullname.find_last_of("/");
	if (slash_pos != std::string::npos)
		Open(fullname.substr(0, slash_pos), fullname.substr(slash_pos + 1));
}

ibitmap* ImagesMng::Current()
{
	if (!_current_img)	/* Refresh in case of previously missing password. */
		_current_img = GetImage(_current_index);
	return _current_img;
}

ibitmap* ImagesMng::Prev()
{
	if (HasPrev())
	{
		if (_next_img != NULL)
			delete _next_img;
		_next_img = _current_img;
		if (_prev_img != NULL)
		{
			_current_img = _prev_img;
			_prev_img = NULL;
			_current_index--;
		}
		else
		{
			_current_img = GetImage(--_current_index);
		}
		//		return Current();
	}
	return Current();
}

ibitmap* ImagesMng::Next()
{
	if (HasNext())
	{
		if (_prev_img != NULL)
			delete _prev_img;
		_prev_img = _current_img;
		if (_next_img != NULL)
		{
			_current_img = _next_img;
			_next_img = NULL;
			_current_index++;
		}
		else
		{
			_current_img = GetImage(++_current_index);
		}
		//		return Current();
	}
	return Current();
}

ibitmap* ImagesMng::GoTo(int index)
{
	if (index >= 0 && index < (int)_files.size())
	{
		if (index == _current_index)
		{
			return Current();
		}
		if (index == _current_index - 1)
		{
			return Prev();
		}
		if (index == _current_index + 1)
		{
			return Next();
		}
		ClearBuffers();
		_current_index = index;
		_current_img = GetImage(_current_index);
		//		return Current();
	}
	return Current();
}

bool ImagesMng::IsBuffered()
{
	if (_prev_img == NULL && HasPrev())
		return false;
	if (_next_img == NULL && HasNext())
		return false;
	return true;
}

void ImagesMng::Buffering()
{
	if (_next_img == NULL && HasNext())
		_next_img = GetImage(_current_index + 1);
	if (_prev_img == NULL && HasPrev())
		_prev_img = GetImage(_current_index - 1);
}

std::string ImagesMng::GetFileName(size_t index)
{
	if (index >= 0 && index < _files.size())
	{
		return _files[index];
	}
	std::string empty;
	return empty;
}

std::string ImagesMng::GetFileFullame(size_t index)
{
	if (index >= 0 && index < _files.size())
	{
		return _dir + _files[index];
	}
	std::string empty;
	return empty;
}

ibitmap* ImagesMng::GetImage(size_t index)
{
	ibitmap *bm = NULL;
	std::string filename;
	PrepareImageFile(index, filename);
	if (filename.size()) {
		bm = OpenFile(filename);
		if ((_archive_type == ZIP) || (_archive_type == RAR) ||
		    (_archive_type == C7Z))
			unlink(filename.c_str());
	}
	return bm;
}

bool ImagesMng::IsSupportedArchive(std::string path) {
	return GetArchiveType(path) != UNSUPPORTED_ARCHIVE;
}

int ImagesMng::CleanupArchive() {
	int ret = 0;
  if (_dir != "/tmp/") {
		std::string cmd = "rm -rf \"" + _dir + "\"";
		ret = system(cmd.c_str());
		if (ret && WIFEXITED(ret))
			ret = WEXITSTATUS(ret);
	}
	return ret;
}

int ImagesMng::SaveCurrentImage(char *dir) {
	/* Saves a copy of the current image file to the given directory. */
	int ret = -1;
	std::string filename;

	PrepareImageFile(_current_index, filename);
	if (filename.size()) {
		std::string cmd = "cp \"" + filename + "\" \"" + dir + "/\"";
		ret = system(cmd.c_str());
		if ((_archive_type == ZIP) || (_archive_type == RAR) ||
		    (_archive_type == C7Z))
			unlink(filename.c_str());
	}
	return ret;
}

void ImagesMng::RecomputeLUT(int b, int c, double g)
{
	if (b < -255 || b > 255) b = 0;
	if (c < -255 || c > 255) c = 0;
	if (g < 0.0) g = 1.0;

	_applyLUT = (b || c || (g != 1.0));

	boolean LUT_changed = false;
	for (int i = 0; i < 256; i++) {
		int p = i;
		p += b;
		p = 128 + ((p-128)*259*(c+255))/(255*(259-c));
		if (p < 0) p = 0;
		p = (int)(255.0*pow((p/255.0), 1.0/g)+0.5);
		if (p > 255) p = 255;
		if (p != _lut[i]) {
			LUT_changed = true;
			_lut[i] = (unsigned char)p;
		}
	}

	if (LUT_changed) {
		ClearBuffers();
		if (_current_index >= 0)
			_current_img = GetImage(_current_index);
		//Let the buffered images be loaded after the current is displayed.
	}
}

struct File_name {
	/* We read entire numbers in and compare them rather than
	 * digit by digit, so that, for example, "2" comes before
	 * "10" or "03". */
	bool operator()(const std::string& a, const std::string& b) const {
		for (unsigned int ia = 0, ib = 0; ia < a.length(); ia++, ib++) {
			if (ia >= b.length()) return false;
			if (isdigit(a[ia]) && isdigit(b[ib])) {
				/* Compare numeric sequences. */
				unsigned int na = 0, nb = 0;
				bool notdone;
				do {
					na = 10*na + (a[ia] - '0');
					if ((notdone = isdigit(a[ia+1]))) ia++;
				} while (notdone);
				do {
					nb = 10*nb + (b[ib] - '0');
					if ((notdone = isdigit(b[ib+1]))) ib++;
				} while (notdone);
				if (na < nb) return true;
				if (nb < na) return false;
			} else {
				/* Compare alphanumerically */
				char al = tolower(a[ia]);
				char bl = tolower(b[ib]);
				if (al < bl) return true;
				if (bl < al) return false;
			}
		}
		return (a.length() >= b.length());	/* Favour longer "identical" names */
	}
};

void ImagesMng::FileNameSort(std::vector<std::string> &names) {
	sort(names.begin(), names.end(), File_name());
}

/**************************** Private methods ********************************/

void ImagesMng::OpenArchive(std::string dir, std::string filename) {
	std::string zipfile = dir + filename;
	std::string outdir = "/tmp/";

	if (_archive_handler) {
		delete _archive_handler;
		_archive_handler = NULL;
	}

	if (_archive_type == ZIP)
		_archive_handler = (ArchiveHandler*) new ZipHandler(outdir);
	else if (_archive_type == RAR)
		_archive_handler = (ArchiveHandler*) new RarHandler(outdir);
	else if (_archive_type == C7Z)
		_archive_handler = (ArchiveHandler*) new C7zHandler(outdir);

	if ((_archive_handler) && !(_archive_handler->Open(zipfile, _files))) {
		ClearBuffers();
		_dir = outdir;
		_current_index = -1;

		if (!_files.empty()) {
			FileNameSort(_files);
			_current_index = 0;
		}
	} 
}

void ImagesMng::OpenDir(std::string dir, std::string filename)
{
	if (IsSupportedArchive(filename))
		UnpackArchiveToDir(dir, filename);

	const char *dirs[] = { dir.c_str(), NULL };
	FTS *tree = fts_open((char * const *)dirs,
	                     FTS_PHYSICAL | FTS_NOCHDIR | FTS_NOSTAT, NULL);
	if (tree)
	{
		ClearBuffers();
		_files.clear();
		_dir = dir;
		_current_index = -1;

		FTSENT *node;
		std::string name;
		int i = 0;

		while ((node = fts_read(tree)))
		{
			if (node->fts_info & FTS_F) {
				name = node->fts_path;
				if (GetFileType(name) != UNSUPPORTED_IMAGE) {
					// Remove the directory path from the name so that we can check
					// it against the input filename parameter.
					name = node->fts_path + _dir.size();
					_files.push_back(name);
					i++;
				}
			}
		}
		fts_close(tree);

		if (!_files.empty())
		{
			FileNameSort(_files);
			if (!filename.empty()) {
				for (int i = 0; i < _files.size(); i++) {
					if (filename == _files[i]) {
						_current_index = i;
						break;
					}
				}
			}
			if(_current_index == -1)
				_current_index = 0;
			name = _dir + _files[_current_index];
		}
		else
		{
			//_current_index = -1;
			Log::msg("В данном каталоге нет файлов для отображения");
		}
	}
	else
		Log::msg("Не удалось открыть выбранный каталог");
}

void ImagesMng::ModifyImage(ibitmap *bm)
{
	/* Apply brightness, contrast and gamma settings to image.
	 * N.B. this won't work for indexed images. */
	int numcomps = (bm->depth < 8) ? bm->depth : bm->depth / 8;
	unsigned char r, g, b;

	Log::msg("ModifyImage: depth = %d, bytes = %d", bm->depth, numcomps);

	if (_applyLUT) {
		for (int j = 0; j < bm->height; j++) {
			for (int i = 0; i < bm->width; i++) {
				if (getRGB(bm, i, j, &r, &g, &b, 0) != -1) {
					r = _lut[r]; g = _lut[g]; b = _lut[b];
					setRGB(bm, i, j, r, g, b, 1);
				}
			}
		}
	}
}

ibitmap* ImagesMng::OpenPNG(const char *filename)
{
	png_byte header[8]; // 8 is the maximum size that can be checked

	/* open file and test for it being a png */
	FILE *fp = fopen(filename, "rb");
	if (!fp)
	{
		Log::msg("[read_png_file] File %s could not be opened for reading", filename);
		return NULL;
	}
	fread(header, 1, 8, fp);
	if (png_sig_cmp(header, 0, 8))
	{
		Log::msg("[read_png_file] File %s is not recognized as a PNG file", filename);
		return NULL;
	}

	/* initialize stuff */
	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

	if (!png_ptr)
	{
		Log::msg("[read_png_file] png_create_read_struct failed");
		return NULL;
	}

	png_infop info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr, NULL, NULL);
		Log::msg("[read_png_file] png_create_info_struct failed");
		return NULL;
	}

	if (setjmp(png_jmpbuf(png_ptr)))
	{
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		Log::msg("[read_png_file] Error during init_io");
		return NULL;
	}

	png_init_io(png_ptr, fp);
	png_set_sig_bytes(png_ptr, 8);

	// changed by zola so we don't need to have public FILE pointers
	//	png_set_read_fn(png_ptr, file, user_read_data_fcn);

	png_read_info(png_ptr, info_ptr); // Read the info section of the png file

	// Convert palette color to true color
	if (png_ptr->color_type == PNG_COLOR_TYPE_PALETTE)
		png_set_palette_to_rgb(png_ptr);

	// Convert low bit colors to 8 bit colors
	if (png_ptr->bit_depth < 8)
	{
		if (!(png_ptr->color_type & PNG_COLOR_MASK_COLOR))
			png_set_gray_1_2_4_to_8(png_ptr);
		else
			png_set_packing(png_ptr);
	}

	// Convert high bit colors to 8 bit colors
	if (png_ptr->bit_depth == 16)
		png_set_strip_16(png_ptr);

	if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
	{
		png_set_tRNS_to_alpha(png_ptr);
		png_set_strip_alpha(png_ptr);
	}

	if (png_ptr->color_type & PNG_COLOR_MASK_ALPHA)
		png_set_strip_alpha(png_ptr);

	if (get_max_ibitmap_depth() < 24) {
		// Convert true color to gray color
		if (png_ptr->color_type & PNG_COLOR_MASK_COLOR)
			png_set_rgb_to_gray_fixed(png_ptr, 1, -1, -1);
	}

	png_set_interlace_handling(png_ptr);
	png_read_update_info(png_ptr, info_ptr);

	/* read file */
	if (setjmp(png_jmpbuf(png_ptr)))
	{
		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
		Log::msg("[read_png_file] Error during read_image");
		return NULL;
	}

	ibitmap* bit = new_ibitmap(info_ptr->width, info_ptr->height,
	                           info_ptr->bit_depth * info_ptr->channels);

	png_bytep * row_pointers = (png_bytep*) malloc(sizeof(png_bytep) * info_ptr->height);
	for (unsigned int y = 0; y < info_ptr->height; y++)
		row_pointers[y] = &(bit->data[y * bit->scanline]);

	png_read_image(png_ptr, row_pointers);

	png_read_end(png_ptr, NULL);
	free(row_pointers);
	png_destroy_read_struct(&png_ptr, &info_ptr, 0); // Clean up memory
	fclose(fp);

	return bit;
}

ibitmap* ImagesMng::OpenJPEG(const char *filename)
{
#if 1
	struct jpeg_decompress_struct cinfo;
	int depth;
	FILE *pFile;
	if ((pFile = fopen(filename, "rb")) == NULL)
	{
		Log::msg("[read_jpeg_file] Error opening file: %s", filename);
		return NULL;
	}
	struct jpeg_error_mgr jerr;
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);
	jpeg_stdio_src(&cinfo, pFile);

	// Decode

	jpeg_read_header(&cinfo, TRUE);
	depth = get_opt_ibitmap_depth(8*cinfo.num_components);
	if (depth < 8) depth = 8;	/* Can't use 4-bit here. */
	ibitmap* bit = new_ibitmap(cinfo.image_width, cinfo.image_height, depth);
	cinfo.out_color_space = (depth >= 24) ? JCS_RGB : JCS_GRAYSCALE;
	jpeg_start_decompress(&cinfo);

	int scanline = bit->scanline;

	unsigned char** rowPtr = new unsigned char*[(&cinfo)->image_height];
	for (unsigned int i = 0; i < (&cinfo)->image_height; i++)
		rowPtr[i] = &(bit->data[i * scanline]);

	int rowsRead = 0;
	while ((&cinfo)->output_scanline < (&cinfo)->output_height)
	{
		rowsRead += jpeg_read_scanlines(&cinfo, &rowPtr[rowsRead], (&cinfo)->output_height - rowsRead);
	}

	delete[] rowPtr;
	jpeg_finish_decompress(&cinfo);

	// End decode

	jpeg_destroy_decompress(&cinfo);
	fclose(pFile);

	return bit;
#else
	return LoadJPEG(filename,1000,1000,100,100,1);
#endif
}

ibitmap* ImagesMng::OpenBMP(const char *filename)
{
	return bmp2ibitmap(filename);
}

ibitmap* ImagesMng::OpenGIF(const char *filename)
{
	return gif2ibitmap(filename);
}

ibitmap* ImagesMng::OpenTIFF(const char *filename)
{
	return tiff2ibitmap(filename);
}

ibitmap* ImagesMng::OpenFile(std::string filename)
{
	ibitmap *bm = NULL;
	int type = GetFileType(filename);
	switch (type) {
		case BMP:
			bm = OpenBMP(filename.c_str());
			break;
		case JPG:
			bm = OpenJPEG((char*) filename.c_str());
			break;
		case PNG:
			bm = OpenPNG((char*) filename.c_str());
			break;
		case GIF:
			bm = OpenGIF((char*) filename.c_str());
			break;
		case TIF:
			bm = OpenTIFF((char*) filename.c_str());
			break;
		default:
			bm = NewBitmap(10,10);	/* Provide a blank image. */
			break;
	}
	
	ModifyImage(bm);

	return bm;
}

int ImagesMng::GetMagicType(char buf[8]) {
	/* Return the file type based on the magic number in the buffer. */
	int type;
	char gif87a_magic[] = "GIF87a";
	char gif89a_magic[] = "GIF89a";
	char tiffle_magic[] = { 'I', 'I', 0x2a, 0x00 };
	char tiffbe_magic[] = { 'M', 'M', 0x00, 0x2a };
	char jpeg_magic[] = { 0xff, 0xd8, 0xff };
	char png_magic[] = { 0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a };
	char bmp_magic[] = { 0x42, 0x4d };

	if (!memcmp(buf, gif87a_magic, 6) ||
	    !memcmp(buf, gif89a_magic, 6))
		type = GIF;
	else if (!memcmp(buf, tiffle_magic, 4) ||
	         !memcmp(buf, tiffbe_magic, 4))
		type = TIF;
	else if (!memcmp(buf, jpeg_magic, 3))
		type = JPG;
	else if (!memcmp(buf, png_magic, 8))
		type = PNG;
	else if (!memcmp(buf, bmp_magic, 2))
		type = BMP;
	else
		type = UNSUPPORTED_IMAGE;

	return type;
}

int ImagesMng::GetFileType(std::string path)
{
	/* Return the file type based on the magic number in the header. */
	int type = UNSUPPORTED_IMAGE;
	int fd;
	char file_magic[8];

	if (!path.empty()) {
		if ((fd = open(path.c_str(), O_RDONLY)) != -1) {
			if ((read(fd, file_magic, 8) == 8))
				type = GetMagicType(file_magic);
			close(fd);
		}
	}
	return type;
}

void ImagesMng::ClearBuffers()
{
	if (_prev_img != NULL)
	{
		delete _prev_img;
		_prev_img = NULL;
	}
	if (_current_img != NULL)
	{
		delete _current_img;
		_current_img = NULL;
	}
	if (_next_img != NULL)
	{
		delete _next_img;
		_next_img = NULL;
	}
}

bool ImagesMng::HasSuffix(const char* path, const char *suf) {
	return (strcasecmp(path+strlen(path)-strlen(suf), suf) == 0);
}

bool ImagesMng::IsSupportedImage(const char *path) {
	return HasSuffix(path, "bmp") || HasSuffix(path, "png") ||
	       HasSuffix(path, "jpg") || HasSuffix(path, "jpeg") ||
	       HasSuffix(path, "tif") || HasSuffix(path, "tiff") ||
				 HasSuffix(path, "gif");
}

int ImagesMng::GetArchiveType(std::string path) {
	int type = UNSUPPORTED_ARCHIVE, fd;
	char file_magic[4];
	char rar_magic[] = "Rar!";
	char zip_magic[] = { 'P', 'K', 0x03, 0x04 };
	char c7z_magic[] = { 0x37, 0x7a, 0xbc, 0xaf, 0x27, 0x1c };

	if (path.empty())
		return type;

	const char* f = path.c_str();
	
	/* Check magic number first, in case the suffix is wrong. */
	if ((fd = open(f, O_RDONLY)) != -1) {
		if ((read(fd, file_magic, 4) == 4)) {
			if (!memcmp(file_magic, rar_magic, 4))
				type = RAR;
			else if (!memcmp(file_magic, zip_magic, 4))
				type = ZIP;
			else if (!memcmp(file_magic, c7z_magic, 6))
				type = C7Z;
		}
		close(fd);
		if (type != UNSUPPORTED_ARCHIVE)
			return type;
	}

	if (HasSuffix(f, ".zip") || HasSuffix(f, ".cbz"))
		type = ZIP;
	else if (HasSuffix(f, ".rar") || HasSuffix(f, ".cbr"))
		type = RAR;
	else if (HasSuffix(f, ".7z") || HasSuffix(f, ".cb7"))
		type = C7Z;
	else if (HasSuffix(f, ".tar.gz") || HasSuffix(f, ".tgz"))
		type = TGZ;
	else if (HasSuffix(f, ".tar.bz2") || HasSuffix(f, ".tbz2"))
		type = TBZ2;
	else if (HasSuffix(f, ".tar") || HasSuffix(f, ".cbt"))
		type = C7Z;		//Use lib7zip to handle it.
	else
		type = UNSUPPORTED_ARCHIVE;

	return type;
}

void ImagesMng::UnpackArchiveToDir(std::string& dir, std::string filename) {
	int ret;
	std::string error;
	std::string archive = dir + filename;
	if (unpack_in_tmp)
		dir = "/tmp/" + filename + ".dir/";
	else
		dir += filename + ".dir/";
	if ((ret = mkdir(dir.c_str(), 0755)) && (errno != EEXIST)) {
		error = GetLangText("@Cant_create_dir");
		error += ": " + dir;
		Message(ICON_ERROR, GetLangText("@Program_name"), error.c_str(), 3000);
		CloseApp();
	}

	std::string cmd;
	switch (_archive_type) {
		case TAR:
			cmd = "tar -C \""+dir+"\" -xf \""+archive+"\"";
			break;
		case TGZ:
			cmd = "tar -C \""+dir+"\" -zxf \""+archive+"\"";
			break;
		case TBZ2:
			cmd = "tar -C \""+dir+"\" -jxf \""+archive+"\"";
			break;
		case UNSUPPORTED_ARCHIVE:
		default:
			error = GetLangText("@Unk_book_type");
			error += ": "+filename;
			Message(ICON_ERROR, GetLangText("@Program_name"), error.c_str(), 3000);
			break;
	}

	if (cmd.empty() || (ret = system(cmd.c_str()))) {
		if (WIFEXITED(ret))
			ret = WEXITSTATUS(ret);
		if (_archive_type != ZIP || ret != 1) {
			if (cmd.size()) {
				error = GetLangText("@Unpack_fail");
				error += ": "+cmd;
				Message(ICON_ERROR, GetLangText("@Program_name"), error.c_str(), 10000);
			}
			cmd = "rm -rf \""+dir+"\"";
			system(cmd.c_str());
			CloseApp();
		}
	}
}

void ImagesMng::PrepareImageFile(size_t index, std::string &filename) {
	/* Unpacks the file first if necessary, and returns the path to the file. */
	if (index >= 0 && index < _files.size()) {
		if ((_archive_type == ZIP) || (_archive_type == RAR) ||
		    (_archive_type == C7Z)) {
			if (_archive_handler && _archive_handler->UnpackFile(_files[index], filename))
				need_password();
		} else
			filename = _dir + _files[index];
	} else
		filename = "";
}
