#include <stdio.h>
#include <inkview.h>
#include "DevCaps.h"
#include "Log.h"

DevCaps::DevCaps() {
	/* Baseline capabilities. */
	has_dpad_buttons = true;
	has_mag_buttons = false;
	has_volume_buttons = false;
	has_back_button = true;
	has_menu_button = true;
	has_touchscreen = true;
	has_new_config_struct = false;
	max_screen_depth = 8;
	pagemap_margin = 4;
	pagemap_width = 12;
	pagemap_height = 18;
	pagemap_clear = 0x55;	/* Roughly the colour of the toolbar */
	pagemap_back = 0xff;	/* The colour denoting the entire virtual page. */
	pagemap_fore = 0x0;		/* The colour denoting what part of the page is in view. */
	model_num = 0;
	fw_major = 0;
	fw_minor = 0;
	device_model.clear();
	software_version.clear();
}

DevCaps::~DevCaps() {
}

void DevCaps::InitCaps() {
	char *dm = GetDeviceModel();
	char *swv = GetSoftwareVersion();
	int ret;

	device_model = dm;
	software_version = swv;

	/* Software version looks someting like "W902.2.1.3" */
	ret = sscanf(swv, "%*[^0-9]%u.%u.%u.%*u", &model_num, &fw_major, &fw_minor);
	if (ret == 3) {
		if (fw_major >= 4)
			has_new_config_struct = true;
		switch (model_num) {
			case 603:
			case 612:
				/* These have the baseline capabilities. */
				break;
			case 903:
			case 912:
				/* These have everything. */
				has_mag_buttons = true;
				has_volume_buttons = true;
				break;
			case 602:
				has_touchscreen = false;
				break;
			case 902:
				has_mag_buttons = true;
				has_volume_buttons = true;
				has_touchscreen = false;
				break;
			case 360:
			case 515:
			case 611:
			case 613:
				has_back_button = false;
				has_touchscreen = false;
				break;
			case 801:
				max_screen_depth = 24;
				/* Fall through to options for 623 */
			case 622:
			case 623:
				has_dpad_buttons = false;
				has_back_button = false;
				pagemap_margin = 0;
				pagemap_width = 24;
				pagemap_height = 32;
				pagemap_clear = 0xff;
				pagemap_back = 0x55;
				pagemap_fore = 0x0;
				break;
			default:
				Log::msg("DevCaps: Unhandled model number %d", model_num);
				break;
		}
	} else {
		/* The software version had an unexpected format, so just assume
		 * that everything is available. */
		Log::msg("DevCaps: Unhandled software version %s", swv);
	}		

	ReadCaps();
}

void DevCaps::ReadCaps() {
	/* Look for a device capabilities file and override any settings
	 * with the values in it. */
	char cfgpath[1024];
	int val;

	strcpy(cfgpath, USERDATA);
	strcat(cfgpath, "/share/pbiv/");
	sprintf(cfgpath+strlen(cfgpath), "%d", model_num);
	strcat(cfgpath, ".cfg");

	iconfig *devcapcfg = OpenConfig(cfgpath, NULL);

	val = ReadInt(devcapcfg, "has_dpad_buttons", -1);
	if (val != -1) has_dpad_buttons = (val != 0);
	val = ReadInt(devcapcfg, "has_mag_buttons", -1);
	if (val != -1) has_mag_buttons = (val != 0);
	val = ReadInt(devcapcfg, "has_volume_buttons", -1);
	if (val != -1) has_volume_buttons = (val != 0);
	val = ReadInt(devcapcfg, "has_back_button", -1);
	if (val != -1) has_back_button = (val != 0);
	val = ReadInt(devcapcfg, "has_menu_button", -1);
	if (val != -1) has_menu_button = (val != 0);
	val = ReadInt(devcapcfg, "has_touchscreen", -1);
	if (val != -1) has_touchscreen = (val != 0);
	val = ReadInt(devcapcfg, "has_new_config_struct", -1);
	if (val != -1) has_new_config_struct = (val != 0);
	val = ReadInt(devcapcfg, "max_screen_depth", -1);
	if (val != -1) max_screen_depth = (unsigned int)val;
	val = ReadInt(devcapcfg, "pagemap_margin", -1);
	if (val != -1) pagemap_margin = (unsigned int)val;
	val = ReadInt(devcapcfg, "pagemap_width", -1);
	if (val != -1) pagemap_width = (unsigned int)val;
	val = ReadInt(devcapcfg, "pagemap_height", -1);
	if (val != -1) pagemap_height = (unsigned int)val;
	val = ReadInt(devcapcfg, "pagemap_clear", -1);
	if (val != -1) pagemap_clear = (unsigned char)val;
	val = ReadInt(devcapcfg, "pagemap_back", -1);
	if (val != -1) pagemap_back = (unsigned char)val;
	val = ReadInt(devcapcfg, "pagemap_fore", -1);
	if (val != -1) pagemap_fore = (unsigned char)val;

	CloseConfig(devcapcfg);
}

void DevCaps::LogCaps() {
	Log::msg("Model number = %u", model_num);
	Log::msg("Firmware version = %u.%u", fw_major, fw_minor);
	Log::msg("has_dpad_buttons = %s", has_dpad_buttons ? "yes" : "no");
	Log::msg("has_mag_buttons = %s", has_mag_buttons ? "yes" : "no");
	Log::msg("has_volume_buttons = %s", has_volume_buttons ? "yes" : "no");
	Log::msg("has_back_button = %s", has_back_button ? "yes" : "no");
	Log::msg("has_menu_button = %s", has_menu_button ? "yes" : "no");
	Log::msg("has_touchscreen = %s", has_touchscreen ? "yes" : "no");
	Log::msg("has_new_config_struct = %s", has_new_config_struct ? "yes" : "no");
	Log::msg("max_screen_depth = %u", max_screen_depth);
	Log::msg("pagemap_margin = %u", pagemap_margin);
	Log::msg("pagemap_width = %u", pagemap_width);
	Log::msg("pagemap_height = %u", pagemap_height);
	Log::msg("pagemap_clear = %u", pagemap_clear);
	Log::msg("pagemap_back = %u", pagemap_back);
	Log::msg("pagemap_fore = %u", pagemap_fore);
}
