﻿#!/usr/bin/env python
#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2016, John Howell <jhowell@acm.org>'
__docformat__ = 'restructuredtext en'

import re
import cookielib
import urllib
import json

from calibre.ebooks.BeautifulSoup import BeautifulSoup
from calibre.utils.date import parse_only_date

from calibre_plugins.overdrive_link.book import LibraryBook
from calibre_plugins.overdrive_link.formats import (FORMAT_ADOBE_EPUB, FORMAT_OCD)
from calibre_plugins.overdrive_link.library import SearchableLibrary
from calibre_plugins.overdrive_link.net import (open_url, hostname_from_url)
from calibre_plugins.overdrive_link.match import (normalize_author, normalize_title)
from calibre_plugins.overdrive_link.parseweb import (LibraryError, must_find, text_only)



# from http://api.oneclickdigital.com/v1/facets/language
LANGUAGE_CODE = {
    'Bulgarian': 'bulgarian',
    'Chinese': 'chinese',
    'Dutch': 'dutch-flemish',
    'Flemish': 'dutch-flemish',
    'English': 'english',
    'French': 'french',
    'German': 'german',
    'Italian': 'italian',
    'Portuguese': 'portuguese',
    'Spanish': 'spanish',
    'Swahili': 'swahili',
    }
    

  
def fix_author(a):
    if ',' in a:
        name_parts = a.rpartition(',')
        a = name_parts[2] + ' ' + name_parts[0]

    return [normalize_author(aa, unreverse=False) for aa in a.split(';')]
    
 
def ocd_hostname(library_id):
    return library_id if '.' in library_id else library_id + '.oneclickdigital.com'
    
   
   
class OneClickDigital(SearchableLibrary):
    id = 'ocd'
    name = 'OneClickdigital'
    formats_supported = {FORMAT_ADOBE_EPUB, FORMAT_OCD}

    
    @staticmethod    
    def validate_library_id(library_id, migrate=True, config=None):
        if (':' in library_id) or ('/' in library_id):
            library_id = hostname_from_url(library_id)
            
        if library_id.lower().endswith('.oneclickdigital.com'): library_id = library_id[:-len('.oneclickdigital.com')]  # strip suffix
        
        if not re.match(r'^([0-9a-zA-Z.]+)$', library_id):
            raise ValueError('OneClickdigital library id must be alphanumeric: "%s"' % library_id)
                
        return library_id.lower()
          
        
    @staticmethod    
    def validate_book_id(book_id, library_id):
        if not re.match(r'^([0-9]+)$', book_id):
            raise ValueError('OneClickdigital book id must be numeric: "%s"' % book_id)
            
        if len(book_id) <= 7:
            raise ValueError('OneClickdigital book id is too short: "%s"' % book_id)  # old ids were sequential, new are isbn
           
        return book_id
        
        
    @staticmethod    
    def book_url(library_id, book_id):
        return 'http://%s/#titles/%s' % (ocd_hostname(library_id), book_id)
    
        
    @staticmethod    
    def book_key_library_id(library_id):
        return library_id   # has same book ids at all libraries, but different available formats

         
    def __init__(self):
        self.signed_in = False
        self.cookiejar = cookielib.CookieJar()
        
        
    def sign_in(self, force=False):
        self.hostname = ocd_hostname(self.library_id)
        self.library_url = 'http://%s' % self.hostname
        
        response = open_url(self.log, self.library_url, None, cookiejar=self.cookiejar)
        soup = BeautifulSoup(response.data, convertEntities=BeautifulSoup.HTML_ENTITIES)
        
        if '<h1>Error:' in response.data:
            raise LibraryError(text_only(soup))
        
        pass_key_input = must_find(soup, 'input', attrs={'id': 'passkey'}) 
        pass_key = pass_key_input['value']
        
        response = open_url(self.log, 'http://%s/spa/config?passKey=%s' % (self.hostname, pass_key),
                None, cookiejar=self.cookiejar, referer=self.library_url, origin=self.library_url)
        
        json_data = json.loads(response.data)
        self.api_root_url = json_data['ApiRootUrl']
        self.bearer_token = json_data['BearerToken']
        self.signed_in = True
        
        
    def find_books(self, books, search_author, search_title, keyword_search, find_recommendable):
        if not self.signed_in:
            self.log.info('Cannot perform search at %s due to sign in failure' % self.name)
            return False
            
        PAGE_SIZE = 60
        
        page_num = 1
        total_pages = 1
        results_processed = 0
            
        while (page_num <= total_pages):
            
            data = {}
            data['search-source'] = 'advanced'
            
            if search_author:
                data['author'] = search_author
                data['sort-by'] = 'author'
                
            if search_title:
                data['title'] = search_title
                data['sort-by'] = 'title'
                
            if self.config.search_language in LANGUAGE_CODE:
                data['language'] = LANGUAGE_CODE[self.config.search_language]
                
            if (FORMAT_OCD in self.config.search_formats) and (FORMAT_ADOBE_EPUB in self.config.search_formats):
                pass    # default if not present
            elif FORMAT_OCD in self.config.search_formats:
                data['mediatype'] = 'eaudio'
            else:
                data['mediatype'] = 'ebook'
                
            data['page-size'] = unicode(PAGE_SIZE)
            data['page-index'] = unicode((page_num - 1))
            data['sort-order'] = 'asc'
            
            addheaders = [("Authorization", "bearer %s" % self.bearer_token)]
            
            response = open_url(self.log, '%sv1/search?%s' % (self.api_root_url, urllib.urlencode(data)),
                    None, cookiejar=self.cookiejar, referer=self.library_url, origin=self.library_url, addheaders=addheaders)
                    
            #self.log.info('result: %s' % response.data)

            json_data = json.loads(response.data)
            
            if 'message' in json_data:
                raise LibraryError(json_data['message'])
                
            accept_media_type = json_data.get('acceptMediaType', '')
            if accept_media_type != 'complete':
                self.log.warn('acceptMediaType %s' % accept_media_type)
                
            total_pages = json_data.get('pageCount', 0)
            self.log.info('Page %d of %d' % (page_num, total_pages))
            
            for item_entry in json_data.get('items', []):
                item = item_entry['item']
                isbn = item['isbn']     # book id
                
                authors = []
                for author_entry in item['authors']:
                    authors.append(normalize_author(author_entry['text'], unreverse=False))
                    
                title = item.get('title', '')
                
                subtitle = item.get('Subtitle', None)
                if subtitle:
                    title = '%s: %s'%(title, subtitle)
                 
                title = normalize_title(title)
                
                publisher_entry = item.get('publisher', {})
                publisher = publisher_entry.get('text', '')
                
                language = item.get('language', '')
                
                formats = set()
                media_type = item.get('mediaType', '')
                if media_type == 'eBook':
                    formats.add(FORMAT_ADOBE_EPUB)
                elif media_type == 'eAudio':
                    formats.add(FORMAT_OCD)
                
                released_date = item.get('releasedDate', '0001-01-01T00:00:00Z')
                if released_date != '0001-01-01T00:00:00Z':
                    pubdate = parse_only_date(released_date, assume_utc=True)
                else:
                    pubdate = None
                
                lbook = LibraryBook(authors=authors, title=title, available=True, lib=self, book_id=isbn,
                        language=language, formats=formats, isbn=isbn, publisher=publisher, pubdate=pubdate, 
                        search_author=search_author)

                        
                self.log.info('Found %s'%repr(lbook))
                books.add(lbook)
                    
                results_processed += 1
                
            page_num += 1

        return False
