﻿#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2016, John Howell <jhowell@acm.org>'
__docformat__ = 'restructuredtext en'


import traceback
from functools import partial

try:
    from PyQt5.Qt import (QIcon, QMenu, QUrl)
except ImportError:
    from PyQt4.Qt import (QIcon, QMenu, QUrl)
    
from calibre.gui2 import (open_url)

from calibre_plugins.overdrive_link.library import (full_library_id, SearchableLibrary)
from calibre_plugins.overdrive_link.log import exception_str

# Identifiers
IDENT_AVAILABLE_LINK = 'odid'
IDENT_RECOMMENDABLE_LINK = 'odrid'
IDENT_PURCHASABLE_LINK = 'odpid'
IDENT_NEEDED_LINK = 'odnid'
IDENT_AMAZON = 'amazon'
IDENT_ISBN = 'isbn'

LIDLTYPES = {IDENT_AVAILABLE_LINK, IDENT_RECOMMENDABLE_LINK, IDENT_PURCHASABLE_LINK} # library & discovered books
CIDLTYPES = {IDENT_AVAILABLE_LINK, IDENT_RECOMMENDABLE_LINK, IDENT_PURCHASABLE_LINK, IDENT_NEEDED_LINK} # calibre books

MAX_LINKS_BEFORE_SUBMENU = 10   # excessive links for a book cause the creation of a submenu to avoid a long main menu



# Link display key
LINK_AVAILABLE = '@'
LINK_RECOMMENDABLE = '!'
LINK_PURCHASABLE = '$'
LINK_NEEDED = '>'

# link format prefixes
LINK_FORMAT_EBOOK = ''
LINK_FORMAT_AUDIOBOOK = '#'
BOOK_DISCOVERED = '*'

# separators for book-library links
SEP_EBOOK = '@'
SEP_AUDIOBOOK = '#'


class ODLink(object):
    '''
    A ODLink represents a link between a specific edition of a book and a particular lending library.

    These links are in the form of book-id@provider-id/library-id for ebooks or book-id#provider-id/library-id for audiobooks
    The book id must not contain an @ or # symbol!
    
    For OverDrive, the provider_id and trailing '/' are not present, the library is the host name of the library web site, 
    and the book is the content reserve id (36 character UUID) used by OverDrive to identify an edition of a book.
    
    All fields are lower case.
    '''

    def __init__(self, str=None, provider_id=None, library_id=None, book_id=None, is_audiobook=None, 
                    book_key=False, check=False, config=None):
                    
        self.config = config    # only needed for some operations
        
        if str:
            if SEP_EBOOK in str:
                self.book_id, full_library_id = str.split(SEP_EBOOK, 1)
                self.is_audiobook = False
            elif SEP_AUDIOBOOK in str:
                self.book_id, full_library_id = str.split(SEP_AUDIOBOOK, 1)
                self.is_audiobook = True 
            elif check:
                raise ValueError('ODLink: missing separator')
            else:
                self.book_id = ''
                full_library_id = str
                self.is_audiobook = False
                
            
            if '/' in full_library_id:
                self.provider_id, self.library_id = full_library_id.split('/', 1)
            else:
                self.provider_id = ''
                self.library_id = full_library_id
                self.book_id = self.book_id.lower() # fix case from older version of plugin
                
        else:
            self.provider_id = provider_id
            self.library_id = library_id
            self.book_id = book_id
            self.is_audiobook = is_audiobook
         
        if check or book_key:
            # Check link components for validity. Repair if possible, otherwise raise exception.
            SearchableLibrary.validate_provider_id(self.provider_id)
            provider = SearchableLibrary.provider(self.provider_id)
            
            if book_key:
                self.library_id = provider.book_key_library_id(self.library_id)
            else:
                self.library_id = provider.validate_library_id(self.library_id, config=config)
                
            self.book_id = provider.validate_book_id(self.book_id, self.library_id)
  
        

    def __unicode__(self):
        return ''.join((self.book_id, SEP_AUDIOBOOK if self.is_audiobook else SEP_EBOOK, full_library_id(self.provider_id, self.library_id)))
            
        
    def __str__(self): 
        return unicode(self)
        
        
    def __repr__(self): 
        return unicode(self)
        
        
    def url(self):
        # Generate the url to open the web page showing a book edition at a lending library
        return SearchableLibrary.provider(self.provider_id).book_url(self.library_id, self.book_id)
        
        
    def name_only(self):
        return self.config.library_name(self.provider_id, self.library_id)

        
    def format_and_name(self):
        return ''.join((LINK_FORMAT_AUDIOBOOK if self.is_audiobook else LINK_FORMAT_EBOOK, self.name_only()))

        
    def format_desc(self):
        return 'audiobook' if self.is_audiobook else 'book'

        
    def configured(self):
        return self.config.configured(self.provider_id, self.library_id)


    def enabled(self):
        return self.config.enabled(self.provider_id, self.library_id)
    

    def disabled(self):
        return self.config.disabled(self.provider_id, self.library_id)
       
       
    def __hash__(self):
        return hash((self.provider_id, self.library_id, self.book_id, self.is_audiobook))
        
        
    def key(self):
        if self.config is None:
            return (
                self.is_audiobook, 
                self.provider_id,
                self.library_id, 
                self.book_id)
            
        # Key for sorting in order of decreasing importance using configured information about libraries
        return (
            self.is_audiobook, 
            self.config.priority(self.provider_id, self.library_id), 
            self.config.library_name(self.provider_id, self.library_id), 
            self.book_id)
        
        
    def __cmp__(self, other):
        return cmp(unicode(self), unicode(other))

        
    def __eq__(self, other):
        return ((self.provider_id == other.provider_id) and (self.library_id == other.library_id) and
                (self.book_id == other.book_id) and (self.is_audiobook == other.is_audiobook))

 
    def amazon_ident(self):
        return SearchableLibrary.provider(self.provider_id).amazon_ident(self.library_id)
        
        
       
class ODLinkSet(object):
    '''
    A set of lending library links that all apply to a book in the calibre library, separated by '&'s.
    ('&' used for separator since ',' and '|' have special handling within calibre.)
    '''
    
    def __init__(self, str=None, config=None, odlink=None, odlinks=None, errors=None, desc=None, book_key=False):
        if str:
            if errors is not None:
                # check links for problems and only keep good ones
                self.odlinks = set()
                for s in str.split('&'):
                    if s:
                        try:
                            self.odlinks.add(ODLink(str=s, check=True, config=config))
                        except Exception as e:
                            errors.append('%s: %s' % (desc, exception_str(e)))
            else:
                self.odlinks = set([ODLink(str=s, check=False, config=config, book_key=book_key) for s in str.split('&') if s])
        elif odlinks:
            self.odlinks = odlinks
        elif odlink:
            self.odlinks = set([odlink])
        else:
            self.odlinks = set()
            
            

    def add(self, odlink):
        self.odlinks.add(odlink)
        return self
            

    def discard(self, odlink):
        self.odlinks.discard(odlink)
        return self
            

    def update(self, other):
        self.odlinks.update(other.odlinks)
        return self
            

    def __unicode__(self):
        return '&'.join(sorted([unicode(link) for link in self.odlinks]))
            
        
    def __str__(self):
        return unicode(self)
        
        
    def __repr__(self): 
        return unicode(self)
        
        
    def __len__(self):
        # Also used to true/false value of an object
        return len(self.odlinks)
        
        
    def __cmp__(self, other): 
        return cmp(unicode(self), unicode(other))
        

    def __or__(self, other):
        return ODLinkSet(odlinks=self.odlinks | other.odlinks)  # Union
        
        
    def __and__(self, other):
        return ODLinkSet(odlinks=self.odlinks & other.odlinks)  # Intersection
        
        
    def __sub__(self, other):
        return ODLinkSet(odlinks=self.odlinks - other.odlinks)  # Difference
        
        
    def __rxor__(self, other):
        return ODLinkSet(odlinks=self.odlinks ^ other.odlinks)  # Symmetric difference
        
        
    def __contains__(self, odlink):
        return odlink in self.odlinks
        
        
    def sorted_odlinks(self):
        # List of individual links sorted by decreasing priority
        return sorted(list(self.odlinks), key=lambda odlink: odlink.key())
        
        
    def primary_odlink(self):
        return self.sorted_odlinks()[0]


    def enabled(self):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if odlink.enabled()]))
        

    def disabled(self):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if odlink.disabled()]))


    def ebooks(self):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if not odlink.is_audiobook]))


    def audiobooks(self):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if odlink.is_audiobook]))


    def has_provider_id(self, provider_id):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if odlink.provider_id == provider_id]))


    def configured(self):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if odlink.configured()]))
        

    def unconfigured(self):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if not odlink.configured()]))
        

    def amazon(self):
        return ODLinkSet(odlinks=set([odlink for odlink in self.odlinks if SearchableLibrary.provider(odlink.provider_id).is_amazon]))
        

    def name_list(self, prefix=''):
        def name_key(link_str):
            # sort audiobook links after e-book links
            if link_str.startswith('#'):
                return (True, link_str[1:])
            return(False, link_str)
            
        return ['%s%s' % (prefix,link) for link in 
            sorted(list(set([odlink.format_and_name() for odlink in self.odlinks])), key=name_key)]
        

    def names(self, prefix=''):
        return ', '.join(self.name_list(prefix))
        

def linksets_str(idlinks):
    if not (idlinks[IDENT_AVAILABLE_LINK] or idlinks[IDENT_RECOMMENDABLE_LINK] or
            idlinks[IDENT_PURCHASABLE_LINK] or idlinks.get(IDENT_NEEDED_LINK, ODLinkSet())):
        return 'No library links'

    return ', '.join(filter(
        None, [
            idlinks[IDENT_AVAILABLE_LINK].names(LINK_AVAILABLE),
            idlinks[IDENT_RECOMMENDABLE_LINK].names(LINK_RECOMMENDABLE),
            idlinks[IDENT_PURCHASABLE_LINK].names(LINK_PURCHASABLE),
            idlinks.get(IDENT_NEEDED_LINK, ODLinkSet()).names(LINK_NEEDED),
            ]))
            
        
def linkset_menu(idlinks, menu, config, add_separator=False, save=None):
    have_links = False
    
    for ident, action, article in [
            (IDENT_AVAILABLE_LINK, 'Borrow', 'from'),
            (IDENT_RECOMMENDABLE_LINK, 'Recommend', 'to'),
            (IDENT_PURCHASABLE_LINK, 'Purchase', 'at')]:
        try:
            odxid = idlinks.get(ident, '')
            
            if isinstance(odxid, unicode):
                odxid = ODLinkSet(str=odxid, config=config)  # convert link strings to linksets
                
            odlinks = odxid.sorted_odlinks()
            if odlinks:
                have_links = True
                
                menu.addSeparator()
                
                if len(odlinks) > MAX_LINKS_BEFORE_SUBMENU:
                    # create a submenu for excessive number of links of same type
                    ac = menu.addAction(QIcon(I('store.png')), action)
                    submenu = QMenu(menu)
                    ac.setMenu(submenu)

                else:
                    submenu = menu  # put on main menu

                for odlink in odlinks:
                    ac = submenu.addAction(QIcon(I('store.png')), 
                        '%s %s %s %s'%(action, odlink.format_desc(), article, odlink.name_only()))
                    ac.setStatusTip(odlink.url())
                    ac.triggered.connect(partial(browse_to_url, odlink.url()))
                    
                    if save:
                        save.addAction(ac)  # save copy of created action to avoid problems with Qt
                        
        except:
            traceback.print_exc()
    
    return have_links
    
    

def browse_to_url(url):
    if url:
        open_url(QUrl(url))
        
