﻿#!/usr/bin/env python
#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2016, John Howell <jhowell@acm.org>'
__docformat__ = 'restructuredtext en'

import re
import cookielib
import json
import urlparse
import urllib2
import dateutil.parser
from dateutil.tz import tzutc

from calibre.ebooks.BeautifulSoup import BeautifulSoup
from calibre.utils.date import parse_only_date

from calibre_plugins.overdrive_link.book import LibraryBook
from calibre_plugins.overdrive_link.formats import (FORMAT_ADOBE_EPUB, FORMAT_ADOBE_PDF, FORMAT_MP3)
from calibre_plugins.overdrive_link.library import (SearchableLibrary, LendingLibrary)
from calibre_plugins.overdrive_link.net import open_url
from calibre_plugins.overdrive_link.match import (normalize_author, normalize_title)
from calibre_plugins.overdrive_link.parseweb import LibraryError
from calibre_plugins.overdrive_link.json import js_value


CLOUD_LIBRARY_FORMAT_NAMES = {
    'EPUB': FORMAT_ADOBE_EPUB,
    'PDF': FORMAT_ADOBE_PDF,
    'MP3': FORMAT_MP3,
    }


class CloudLibrary(SearchableLibrary):
    id = '3m'                   # Original name was 3M Cloud Library
    name = 'Cloud Library'
    
    ebook_formats_supported = {FORMAT_ADOBE_EPUB, FORMAT_ADOBE_PDF}
    audiobook_formats_supported = {FORMAT_MP3}
    
    formats_supported = ebook_formats_supported | audiobook_formats_supported
    
    supports_recommendation = True
    allow_format_merge = False
    sign_in_affects_check_obtainable = True     # can determine wait for books on hold
    
    
    @staticmethod    
    def validate_library_id(library_id, migrate=True, config=None):
        if (':' in library_id) or ('/' in library_id):
            purl = urlparse.urlparse(library_id)
            m = re.match(r'^/library/([^/]+)/', purl.path)
            if not m:
                raise ValueError('Cloud Library library id cannot be extracted from url: "%s"'%library_id)
            library_id = m.group(1)
                
        if not re.match(r'^([0-9a-zA-Z_]+)$', library_id):
            raise ValueError('Cloud Library library id must be alphanumeric: "%s"'%library_id)

        return library_id
             

    @staticmethod    
    def validate_book_id(book_id, library_id):
        if not re.match(r'^([0-9a-z]+)$', book_id):
            raise ValueError('Cloud Library book id must be alphanumeric: "%s"' % book_id)
            
        return book_id
            

    @staticmethod    
    def book_url(library_id, book_id):
        return 'http://ebook.yourcloudlibrary.com/library/%s/Featured/ItemDetail/%s' % (library_id, book_id)


    def __init__(self):
        self.cookiejar = cookielib.CookieJar()
        
        self.recommendation_allowed = False
        self.search_audiobooks_allowed = False
        
        self.holds_checked = False
        self.holds = {}
        

    def sign_in(self, use_credentials):
        self.library_url = 'https://ebook.yourcloudlibrary.com/library/%s/Featured' % self.library_id
        
        response = open_url(self.log, self.library_url, cookiejar=self.cookiejar)
            
        # Parse the html results for analysis
        soup = BeautifulSoup(response.data, convertEntities=BeautifulSoup.HTML_ENTITIES)
        
        scripts = soup.findAll('script', attrs={'type':"text/javascript"})
        for script in scripts:
            if 'mmmWebPatron.UserSession=' in unicode(script):
                library_info = js_value(self.log, unicode(script), 'mmmWebPatron.UserSession=')["Library"]
                break
        else:
            raise Exception('Missing mmmWebPatron.UserSession')
            
        if library_info["Is3MCatalogVisible"]:
            self.log.info('Search for recommendable books is supported by %s' % self.name)
            self.recommendation_allowed = True
            
        if library_info["CanSearchForAudioBooks"]:
            self.log.info('Search for audiobooks is supported by %s' % self.name)
            self.search_audiobooks_allowed = True
            
                    
        if self.card_number and use_credentials:
            self.signin_required = True
            
            url = 'https://ebook.yourcloudlibrary.com/uisvc/%s/Patron/LoginPatron' % self.library_id
                    
            data = {}
            data["UserId"] = self.card_number
            data["Password"] = self.card_pin
            
            self.log.info('Post %s' % url)
            
            result = self.open_cloud_library_url(url, data=json.dumps(data, ensure_ascii=True, separators=(',', ':')),
                                        log_request=False)
                            
            if not result.get('Success', False):
                msg = (result.get('ErrorMessage', '') or result.get('Message', '') or
                        ' '.join(result.get('Messages', [])) or result.get('FailureReason', '')
                        or 'Reason unknown')
                        
                raise LibraryError('Sign in failed: %s' % msg)
                
            self.session_id = result['SessionId']
        
            self.log.info('Sign in successful')
            self.signed_in = True
    
    
    def find_books(self, books, search_author, search_title, keyword_search, find_recommendable):
        '''
        Search Cloud Library for books that match an author/title (or subsets thereof) and 
        return the found matches with Cloud Library identifiers.
        books = Set of Books to be updated
        '''
        
        search_ebook = len(self.config.search_formats & self.ebook_formats_supported) > 0
        search_audiobook = len(self.config.search_formats & self.audiobook_formats_supported) > 0
        
        if search_ebook and search_audiobook:
            if self.search_audiobooks_allowed:
                media = 'all'
                desired_formats = self.formats_supported
            else:
                media = 'ebook'
                desired_formats = self.ebook_formats_supported
        elif search_ebook:
            media = 'ebook'
            desired_formats = self.ebook_formats_supported
        elif search_audiobook and self.search_audiobooks_allowed:
            media = 'audio'
            desired_formats = self.audiobook_formats_supported
        else:
            return False    # no Cloud Library desired formats available for search
            
        RESULTS_PER_PAGE = 20
        
        page_num = 1
        has_more = True
        search_id = None
        results_processed = 0
        
        while (has_more):
        
            url = 'https://ebook.yourcloudlibrary.com/uisvc/%s/Search%s?media=%s&src=%s' % (self.library_id,
                    '' if page_num == 1 else '/Next', media, '3m' if find_recommendable else 'lib')
                    
            data = {}
            
            if page_num == 1:
                data["SearchString"] = ' '.join([search_author, search_title]).strip()
                data["SortBy"] = ''     # title, author, publication_date, creationDate, rating
            else:
                data["SearchId"] = search_id
            
            data["from"] = (page_num - 1) * RESULTS_PER_PAGE
            data["count"] = RESULTS_PER_PAGE
            
            result = self.open_cloud_library_url(url, data=json.dumps(data, ensure_ascii=True, separators=(',', ':')))
            
            search_id = result.get('SearchId', '')
            has_more = result.get('HasMore', False)
            num_results = len(result.get('Items', []))
            
            if has_more and num_results != RESULTS_PER_PAGE:
                self.log.error('Expected %d results, but received %d' % (RESULTS_PER_PAGE, num_results))
            
            for item in result.get('Items', []):
                document_id = item['Id']
                
                title = normalize_title(item.get('Title', ''))
                subtitle = item.get('SubTitle', '')
                if subtitle:
                    title = '%s: %s' % (title, subtitle)
                    
                authors = [normalize_author(a, unreverse=True) for a in item.get('Authors', '').split(';')]
                
                publisher = item.get('Publisher', '')
                
                if 'PublicationDate' in item:
                    pubdate = dateutil.parser.parse(unicode(item['PublicationDate'])).replace(tzinfo=tzutc())  # ISO 8601 format
                elif 'PublicationYear' in item:
                    pubdate = parse_only_date(unicode(item['PublicationYear']), assume_utc=True)
                else:
                    pubdate = None
                    
                isbn = item.get('ISBN', '')
                
                format = item.get('Format', '')
                if format in CLOUD_LIBRARY_FORMAT_NAMES:
                    formats = set([CLOUD_LIBRARY_FORMAT_NAMES[format]])
                else:
                    raise LibraryError('Unexpected format: %s' % format)
                
                allowed_action = item.get('AllowedPatronAction', 'Borrow')
                
                available = False
                recommendable = False
                if allowed_action in ['Borrow', 'PlaceHold', 'RemoveHold']:
                    available = True
                elif allowed_action == 'Suggest':
                    recommendable = True
                elif allowed_action not in ['None']:
                    self.log.warn('Unknown AllowedPatronAction (%s)' % allowed_action)
                    
                    
                lbook = LibraryBook(authors=authors, title=title,
                        publisher=publisher, pubdate=pubdate, isbn=isbn, formats=formats,
                        available=available, recommendable=find_recommendable and recommendable,
                        lib=self, book_id=document_id, search_author=search_author,
                        allow_get_from_cache=False)  # prevent old cached results from being used
                        
                if formats.isdisjoint(desired_formats):
                    self.log.info('Ignoring wrong format: %s' % repr(lbook))
                elif (not available) and (not find_recommendable):
                    self.log.warn('Unavailable book found: %s' % repr(lbook))
                else:
                    self.log.info('Found: %s' % repr(lbook))
                    books.add(lbook)
                    
                results_processed += 1

            
            page_num += 1
            
        return False
            
 
    def check_book_obtainable(self, book_id):
        self.check_current_holds()
        available_copies = 0
        availability_date = None
        number_waiting_overall = 0

        if book_id in self.holds:
            self.log.info('Using availability date from %s holds list for %s' % (self.name, book_id))
            availability_date = self.holds[book_id]
            
        else:
            # Can get info for book using:
            # http://ebook.yourcloudlibrary.com/uisvc/(library-id)/Item?id=fhntpg9
            # Accept: application/json, text/plain, */*
                
            item = self.open_cloud_library_url('https://ebook.yourcloudlibrary.com/uisvc/%s/Item?id=%s&media=all&src=lib' % (self.library_id, book_id))
            
            available_copies = 0
            availability_date = None
            number_waiting_overall = 0
            allowed_action = item.get('AllowedPatronAction', 'None')
            
            if allowed_action == 'Borrow':
                available_copies = 1
                
            elif allowed_action in ['PlaceHold', 'RemoveHold']:
                if allowed_action == 'RemoveHold':
                    self.log.info('Book is on hold for current user')
            
                availability_date = item.get("GuaranteedAvilabilityDate", None)
                if availability_date:
                    availability_date = dateutil.parser.parse(availability_date).replace(tzinfo=tzutc())
                else:
                    number_waiting_overall = 1      # wait not provided, guess

            elif allowed_action not in ['None', 'Suggest']:
                self.log.warn('Unknown AllowedPatronAction (%s)' % allowed_action)

        # estimate availability
        return self.when_obtainable(library_copies=1, available_copies=available_copies,
                    number_waiting_overall=number_waiting_overall, availability_date=availability_date)
        

    def check_current_holds(self):
        if self.holds_checked or not self.signed_in:
            return
            
        self.holds_checked = True
        
        try:
            result = self.open_cloud_library_url('https://ebook.yourcloudlibrary.com/uisvc/%s/Patron' % self.library_id)
                            
            book_count = result.get('BookCount', {})
            
            for book_id,date in book_count.get('HoldEndDates', {}).items():
                self.log.info('Found hold at %s: book_id=%s available_date=%s' % (self.name, book_id, date))
                self.holds[book_id] = dateutil.parser.parse(date).replace(tzinfo=tzutc())  # ISO 8601 format

        except Exception as e:
            self.log.exception('', e)
            
            
    def open_cloud_library_url(self, url, **kwargs):
        kwargs['cookiejar'] = self.cookiejar
        kwargs['content_type'] = "application/json;charset=UTF-8"
        kwargs['referer'] = self.library_url
        kwargs['addheaders'] = [("Accept", "application/json, text/plain, */*")]
        kwargs['retry_on_internal_error'] = False
        
        try:
            response = open_url(self.log, url, **kwargs)
            
        except Exception as e:
            if not (type(e) == urllib2.HTTPError and e.code in [401, 500]):
                raise
                
            result = json.loads(e.response_data)    # some API errors returned this way
        
            msg = (result.get('ErrorMessage', '') or result.get('Message', '') or
                    ' '.join(result.get('Messages', [])) or result.get('FailureReason', '')
                    or 'Reason unknown')
                    
            raise LibraryError(msg)
            
        return json.loads(response.data)
        
    
def inventory_cloud_library_sites(abort, log, status, config):

    #Get list of sites based on http://www.yourcloudlibrary.com/bibCloud.js (from http://www.yourcloudlibrary.com/index.php/en-us/)
    #Do check library capabilities for each
    
    MASTER_JSON_URL = "https://ebook.3m.com"
    
    def jquery(method, params):
        response = open_url(log, '%s/json/rpc?json={"method":"%s","params":[%s]}' % (
            MASTER_JSON_URL, method, ','.join(['"%s"' % p for p in params])))
        json_data = json.loads(response.data)
        if "result" not in json_data:
            raise Exception('jquery(%s): %s' % (method, unicode(json_data)))
        
        return json_data['result']
        
    library_count = 0
    audiobook_count = 0
    recommendation_count = 0
    sites = set()
    checked_sites = set()
    libraries = []
    tokens = {}
    
    
    friendly_name = {"3m.au": "Australia", "3m.ca": "Canada", "3m.gb": "United Kingdom", "3m.nz": "New Zealand", "3m.us": "United States"}
    
    '''
    response = open_url(log, 'http://ebook.yourcloudlibrary.com/bibCloud.js')     # library search interface
    countries = js_value(self.log, response.data, 'var countryArrayLookup = ')
    for i in range(0, len(countries), 2):
        friendly_name[countries[i]] = countries[i+1]
    '''
    
    natures = jquery("WSReaktorMgmt.getCompany", ["3m"])['natures']
    for country in natures:
        country_id = country["name"]
        log.info("Country: %s (%s)" % (friendly_name.get(country_id, country_id), country_id))
        
        token = jquery("WSAuth.authenticateAnonymousUser", [country_id])['token']
        
        states = jquery("WSLibraryMgmt.getStates", [country_id])
        for state in states:
            state_id = state["abbreviation"]
            log.info('State: %s (%s)'%(state["name"], state_id))
            status.update(0, country_id + '/' + state_id)
            
            state_libraries = jquery("WSLibraryMgmt.getLibraryBranchesByState", [token, state_id])
            for library in state_libraries:
                tokens[library["libraryID"]] = token
            
            libraries += state_libraries
            
             
    for i,library in enumerate(libraries):
        log.info('Library: %s / %s / %s'%(library["libraryID"], library["branchID"], library["name"]))
        
        library_name = library["name"]
        status.update(i / len(libraries), library_name)
        
        cloud_library_id = library["libraryID"]
        result = jquery("WSLibraryMgmt.getLibraryByID", [tokens[cloud_library_id], cloud_library_id])
        library_id = result['urlName']
        
        # see LibraryConfig for kwargs
        lending_lib = LendingLibrary(library_id=library_id, name=library_name, enabled=True, 
            card_number='', card_pin='', branch_id='', provider_id=CloudLibrary.id)
        
        lib = SearchableLibrary.create(log, config, lending_lib)
        
        library_count += 1
    
        if lib.library_id in checked_sites:
            log.info('Already checked %s at %s'%(lib.name, lib.library_id))
        else:
            checked_sites.add(lib.library_id)
            lib.sign_in(False)
            
            if lib.recommendation_allowed:
                recommendation_count += 1
                
            if lib.search_audiobooks_allowed:
                audiobook_count += 1
                
            sites.add(library_id)

                    
    log.info('')
    log.info('Sites: %s' % ', '.join(sorted(list(sites))))
    log.info('')
    
    log.summary('Found %d libraries, %d sites, %d audiobook sites, %d sites allow recommendations'%(
                    library_count, len(sites), audiobook_count, recommendation_count))
