﻿#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2016, John Howell <jhowell@acm.org>'
__docformat__ = 'restructuredtext en'

import json
import dateutil.parser

from calibre.ebooks.metadata import (authors_to_string, author_to_author_sort, title_sort)
from calibre_plugins.overdrive_link.match import (primary_author)
from calibre_plugins.overdrive_link.link import (
    LIDLTYPES, CIDLTYPES, ODLink, ODLinkSet, linksets_str, 
    LINK_AVAILABLE, LINK_RECOMMENDABLE, LINK_PURCHASABLE,
    IDENT_RECOMMENDABLE_LINK, IDENT_NEEDED_LINK)
from calibre_plugins.overdrive_link.fixups import LEVEL_BY_BOOK_KEY
from calibre_plugins.overdrive_link.formats import (LEVEL_NONE, COMPARE_INDICIES, LEVEL_EPUB,
    ALL_READABLE_FORMATS, ALL_LISTENABLE_FORMATS, LEVELS_OF_PROVIDER, LEVEL_OF_FORMAT_AND_PROVIDER)

UNKNOWN = 'Unknown' # used by calibre for empty author or title


def level_of_format_and_provider(format, provider_id):
    key = (format, provider_id)
    if key in LEVEL_OF_FORMAT_AND_PROVIDER:
        return LEVEL_OF_FORMAT_AND_PROVIDER[key]
        
    key = ('*', provider_id)
    if key in LEVEL_OF_FORMAT_AND_PROVIDER:
        return LEVEL_OF_FORMAT_AND_PROVIDER[key]
        
    key = (format, '*')
    if key in LEVEL_OF_FORMAT_AND_PROVIDER:
        return LEVEL_OF_FORMAT_AND_PROVIDER[key]
        
    return None
    
 
def best_of_index(levels, i):
    best = LEVEL_NONE
    
    for l in levels:
        if l[i] > best[i]:
            best = l
            
    return best
        
    
def need_levels(levels_have, levels_available):
    desc = ''
    needed = False
    
    for i in COMPARE_INDICIES:
        best_have = best_of_index(levels_have, i)
        best_available = best_of_index(levels_available, i)
        
        if best_available[i] > best_have[i]:
            desc = best_available[0] + '>' + best_have[0]
            needed = True
            break
            
        desc = best_available[0] + ('=' if best_available[i] == best_have[i] else '<') + best_have[0]
            
    return needed, desc
    
    
def remove_conflicting_needed_links(odnid):
    # eliminate links that are certain to be of lower priority than other links
    max_rank = None
    for odlink in ODLinkSet(str=odnid).odlinks:
        if (not odlink.is_audiobook) and (odlink.provider_id in LEVELS_OF_PROVIDER):
            rank = LEVELS_OF_PROVIDER[odlink.provider_id][0][1]
            max_rank = rank if max_rank is None else max(max_rank, rank)
    
    kept_links = ODLinkSet()
    for odlink in ODLinkSet(str=odnid).odlinks:
        if (not odlink.is_audiobook) and (LEVELS_OF_PROVIDER.get(odlink.provider_id, LEVEL_EPUB)[0][1] >= max_rank):
            kept_links.add(odlink)

    return unicode(kept_links)

    
def add_author(old_authors, new_author):
    if new_author and (new_author not in old_authors):
        old_authors.append(new_author)  # keep order of existing authors
    
    
def merge_authors(old_authors, new_authors):
    for new_author in new_authors:
        add_author(old_authors, new_author)
        
   
def unique_authors(authors):
    new_authors = []    # create a fresh list
    for author in authors:
        add_author(new_authors, author)
            
    return new_authors
    
    
    

class BookObj(object):
    '''
    methods common to all types of books: calibre, library, discovered
    '''
    
    def get_idlinks(self, config):
        idlinks = {}
        
        for ltype in self.idltypes:
            idlinks[ltype] = ODLinkSet(str=getattr(self, ltype, ''), config=config)
            
        return idlinks
        
    def get_orig_idlinks(self, config):
        orig_idlinks = {}
        
        for ltype in self.idltypes:
            orig_idlinks[ltype] = ODLinkSet(str=getattr(self, 'orig_' + ltype, ''), config=config)
            
        return orig_idlinks
        
    def set_idlinks(self, idlinks):
        for ltype in self.idltypes:
            setattr(self, ltype, unicode(idlinks[ltype]) if ltype in idlinks else '')

    def set_orig_idlinks(self, orig_idlinks):
        for ltype in self.idltypes:
            setattr(self, 'orig_' + ltype, unicode(orig_idlinks[ltype]) if ltype in orig_idlinks else '')

    def links_have_changed(self):
        for ltype in self.idltypes:
            if getattr(self, ltype, '') != getattr(self, 'orig_' + ltype, ''):
                return True
                
        return False
            
    def update_links(self, other):    
        for ltype in self.idltypes:
            setattr(self, ltype, unicode(ODLinkSet(str=getattr(self, ltype, '')) | ODLinkSet(str=getattr(other, ltype, ''))))
        
    def preserve_all_links(self):
        for ltype in self.idltypes:
            setattr(self, ltype, unicode(ODLinkSet(str=getattr(self, ltype, '')) | ODLinkSet(str=getattr(self, 'orig_' + ltype, ''))))

    def preserve_disabled_links(self, config):
        for ltype in self.idltypes:
            setattr(self, ltype, unicode(ODLinkSet(str=getattr(self, ltype, ''), config=config) |
                ODLinkSet(str=getattr(self, 'orig_' + ltype, ''), config=config).disabled()))
        
    def migrate_links(self, config):
        for ltype in self.idltypes:
            setattr(self, ltype, getattr(self, 'orig_' + ltype, ''))

        # For the most part links migrate themselves in library_id and book_id validation.
        # Any other migration goes here.
        
        # migrate Amazon KOLL purchase links to Amazon store purchase links
        if self.odpid:
            linkset = ODLinkSet(str=self.odpid, config=config)
            for odlink in linkset.odlinks:
                if odlink.provider_id == 'ak' and (odlink.library_id in ['', 'prime']):
                    odlink.library_id = 'store'
                    
            self.odpid = unicode(linkset)
   
    def check_links(self, errors, config):
        book_name = unicode(self)
        
        for ltype in self.idltypes:
            setattr(self, ltype, unicode(ODLinkSet(str=getattr(self, ltype, ''),
                        errors=errors, desc=book_name + ', ' + ltype, config=config).configured()))

    def links_str(self, config):
        return linksets_str(self.get_idlinks(config))
            
    def remove_outdated_links(self):
        # remove recommend links for a book that has become available
        if IDENT_RECOMMENDABLE_LINK in self.idltypes:
            self.odrid = unicode((ODLinkSet(str=self.odrid)) - ODLinkSet(str=self.odid))
        
        if IDENT_NEEDED_LINK in self.idltypes:
            # remove needed links if not available
            self.odnid = unicode((ODLinkSet(str=self.odnid)) & ODLinkSet(str=self.odid))
            
            # remove conflicting needed links (those of lower priority)
            self.odnid = remove_conflicting_needed_links(self.odnid)
        
    def make_orig_links(self):
        for ltype in self.idltypes:
            setattr(self, 'orig_' + ltype, getattr(self, ltype, ''))
            setattr(self, ltype, '')

        self.is_newly_discovered = False

    def sort_key(self):
        # Establish sort order: author sort then title sort. Use unique ID as tie breaker.
        # cache sort key to speed up "sorted"
        if not hasattr(self, 'cached_sort_key'):
            self.cached_sort_key = (author_sort_key(primary_author(self.authors)) if self.authors else '', title_sort(self.title), id(self))
            
        return self.cached_sort_key
        
    def del_sort_key(self):
        # invalidate cache if author or title changes
        if hasattr(self, 'cached_sort_key'):
            del self.cached_sort_key
        
    def __cmp__(self, other):
        return cmp(self.sort_key(), other.sort_key())

   
           
class CalibreBook(BookObj):
    '''
    An object keeping track of all of the information related to a calibre book.
    
    id              calibre book id (None if this book not already in the calibre database)
    authors         list of author names (assume primary first)
    title           book title
    series          book series
    isbn            book isbn for matching instead of by title/author
    orig_odid       lending library links as of when the database was read
    orig_odrid      lending library recommendations as of when the database was read
    orig_odpid      purchasable links as of when the database was read
    orig_odnid      needed links as of when the database was read
    last_modified   last modified date/time as of when the book data was extracted from calibre database
    
    odid            new lending library links to be applied to the book 
    odrid           new lending library recommendations to be applied to the book 
    odpid           new purchasable links to be applied to the book 
    odnid           new needed links to be applied to the book 
    '''
    
    idltypes = CIDLTYPES    # calibre book link types
    
    def __init__(self, id=None, authors=[], title='', series='', series_index=0.0, isbn='', 
                orig_odid='', orig_odrid='', orig_odpid='', orig_odnid='', orig_obtainable='',
                orig_et_asin='', orig_et_status=None, epub_file_name=None,
                levels_have=[], allow_discovery=True, last_modified=None, from_book=None):
        self.id = id
        self.authors = authors
        self.title = title
        self.series = series
        self.series_index = series_index
        self.isbn = isbn
        self.orig_odid = orig_odid
        self.orig_odrid = orig_odrid
        self.orig_odpid = orig_odpid
        self.orig_odnid = orig_odnid
        self.orig_obtainable = orig_obtainable
        self.orig_et_asin = orig_et_asin
        self.orig_et_status = orig_et_status
        self.epub_file_name = epub_file_name
        self.levels_have = levels_have
        self.allow_discovery = allow_discovery
        self.last_modified = last_modified
        
        # Initialize fields set by search
        self.matched = False
        self.odid = ''
        self.odrid = ''
        self.odpid = ''
        self.odnid = ''
        self.obtainable = ''
        self.et_asin = ''
        self.et_status = None
        self.inventory = ''

        if from_book is not None:
            # copy attributes from an existing book object
            self.__dict__.update(from_book.__dict__)
                 
    def __unicode__(self): 
        # display string for a book
        return '%s by %s'%(self.title, limited_authors_to_string(self.authors))

    def __str__(self):
        return unicode(self) 
        
    def __repr__(self): 
        return unicode(self)
        
             

class InfoBook(BookObj):
    '''
    An object keeping track of all of the information related to a book at one or more lending library
    without link or presence/absence information. (Caution when changing - values will be cached!)
    '''
    
    idltypes = None    # no link types
    
    def __init__(
            self, provider_id='', library_id='', book_id='',
            authors=[], title='', series='', series_index=0.0, isbn='', publisher='', 
            pubdate=None, language='', formats=set(), lib=None, cache_allowed=True,
            from_json=None):
              
        if lib:
            self.provider_id = lib.provider.id
            self.library_id = lib.library_id
            self.book_id = lib.validate_book_id(book_id, lib.library_id)
        else:
            self.provider_id = provider_id
            self.library_id = library_id
            self.book_id = book_id
            
        
        self.authors = unique_authors(authors)  # drop any empty or non-unique author strings & force copy
        self.title = title
        self.series = series
        self.series_index = series_index
        self.isbn = isbn
        self.publisher = publisher
        self.pubdate = pubdate
        self.language = language
        self.formats = formats
        self.cache_allowed = cache_allowed
        
        if from_json is not None:
            dict_copy = json.loads(from_json)
            
            # fix types for some attributes
            dict_copy['formats'] = set(dict_copy.get('formats', []))
            
            if dict_copy['pubdate'] is not None: dict_copy['pubdate'] = dateutil.parser.parse(dict_copy['pubdate'])
            
            for attrib, value in dict_copy.items():
                self.__dict__[attrib] = value
        
        if len(self.book_id) == 0: raise AssertionError
        
        self.book_key = unicode(ODLink(provider_id=self.provider_id, library_id=self.library_id, book_id=self.book_id, book_key=True))
        

    def to_json(self):
        dict_copy = self.__dict__.copy()    # shallow copy
        
        # make json serializable
        dict_copy['formats'] = list(dict_copy['formats'])
        if dict_copy['pubdate'] is not None: dict_copy['pubdate'] = dict_copy['pubdate'].isoformat()
        
        del dict_copy['book_key']    # will be re-created on load
        
        #for attrib, value in dict_copy.items():
        #    print(' dict[%s] = "%s"' %(attrib, unicode(value)))
        
        return json.dumps(dict_copy)
        
        
    def __unicode__(self): 
        # short display string for a book
        return '%s%s'%(self.title, ' by %s'%limited_authors_to_string(self.authors) if self.authors else '')
        

    def __str__(self):
        return unicode(self)


    def __repr__(self): 
        # unambiguous string representation of a book
        return '%s%s [%s]'%(self.title, ' by %s'%authors_to_string(self.authors) if self.authors else '',
                ', '.join(filter(None, [
                        self.series if self.series_index == 0.0 else '%s [%02d]'%(self.series, int(self.series_index)),
                        self.book_key,
                        '|'.join(sorted(list(self.formats))),
                        self.isbn, 
                        self.publisher, 
                        self.pubdate.isoformat() if self.pubdate else '',
                        self.language,
                        '' if self.cache_allowed else 'no-cache',
                        ])))
                    
                    
                    
    def __cmp__(self, other):
        # Establish sort order and identity
        return cmp(self.book_key, other.book_key)
            
            

    
class LibraryBook(BookObj):
    '''
    An object keeping track of all of the information related to a book at one or more lending library,
    with the same provider.
    '''
    
    idltypes = LIDLTYPES    # library book link types
    
    def __init__(self, lib=None, book_id=None, authors=[], title='', series='', series_index=0.0, isbn='', publisher='', 
                    pubdate=None, language='', available=False, recommendable=False, purchasable=False, 
                    formats=set(), search_author='', allow_get_from_cache=True):
                    
        self.first_lib = lib
        self.provider_id = lib.provider_id
        self.book_id = lib.validate_book_id(book_id, lib.library_id)
        self.book_key = unicode(ODLink(provider_id=self.provider_id, library_id=self.first_lib.library_id, book_id=self.book_id, book_key=True))
        self.allow_get_from_cache = allow_get_from_cache
        
        self.authors = unique_authors(authors)  # drop any empty or non-unique author strings & force copy
        self.title = title
        self.series = series
        self.series_index = series_index
        self.isbn = isbn
        self.publisher = publisher
        self.pubdate = pubdate
        self.language = language
        
        self.formats = formats.copy()  # force copy
        
        self.available = set()
        self.recommendable = set()
        self.purchasable = set()
        
        if available: self.available.add(lib)
        if recommendable: self.recommendable.add(lib)
        if purchasable: self.purchasable.add(lib)
        
        self.search_authors = set()
        if search_author: self.search_authors.add(search_author)
        
        
    def __unicode__(self): 
        # short display string for a book
        return '%s%s'%(self.title, ' by %s'%limited_authors_to_string(self.authors) if self.authors else '')
        

    def __str__(self):
        return unicode(self)
        
        
    def __repr__(self): 
        # unambiguous string representation of a book
        return '%s%s [%s]'%(self.title, ' by %s'%authors_to_string(self.authors) if self.authors else '',
                ', '.join(filter(None, [
                        self.series if self.series_index == 0.0 else '%s [%02d]'%(self.series, int(self.series_index)),
                        '%s@%s'%(self.book_id, self.provider_id),
                        ' '.join([LINK_AVAILABLE+repr(lib) for lib in self.available] +
                            [LINK_RECOMMENDABLE+repr(lib) for lib in self.recommendable] +
                            [LINK_PURCHASABLE+repr(lib) for lib in self.purchasable]),
                        '|'.join(sorted(list(self.formats))), 
                        self.isbn, 
                        self.publisher, 
                        self.pubdate.isoformat() if self.pubdate else '',
                        self.language,
                        ])))

        
    def __cmp__(self, other):
        # Establish sort order: author sort then title sort. Use unique ID as tie breaker.
        return cmp(
            (author_sort_key(primary_author(self.authors)) if self.authors else '', title_sort(self.title), id(self)),
            (author_sort_key(primary_author(other.authors)) if other.authors else '', title_sort(other.title), id(other)))
            
            
    def merge_from(self, other, check_same_formats=True, check_add_formats=True):
        if other.book_key != self.book_key:
            raise ValueError('cannot merge library books with different book keys')
            
        if check_same_formats and len(self.formats ^ other.formats) != 0:
            raise ValueError('cannot merge library books with different formats')
            
        # if any formats are provided in find phase they cannot be added-to in get book details
        if check_add_formats and len(self.formats) > 0 and len(other.formats - self.formats) > 0:
            raise ValueError('cannot merge library books with additional formats')
            
        # combine fields that allow multiple values
        
        '''
        # base combined author set on the longer list of authors, giving preference to the 'other' set
        if len(other.authors) < len(self.authors):
            merge_authors(self.authors, other.authors)
        else:
            new_authors = []    # create a fresh list
            merge_authors(new_authors, other.authors)
            merge_authors(new_authors, self.authors)
            self.authors = new_authors
        '''
        
        self.formats |= other.formats
        
        if hasattr(other,'available'):
            self.available |= other.available
            self.recommendable = (self.recommendable | other.recommendable) - self.available    # remove available from recommendable
            self.purchasable |= other.purchasable
            
        if hasattr(other,'search_authors'):
            self.search_authors |= other.search_authors
        
        # although 'authors' allows multiple values use only one complete set
        if other.authors: self.authors = unique_authors(other.authors)
        
        # prefer new (other) to old (self) for single value fields
        if other.title: self.title = other.title
        if other.series: self.series = other.series
        if other.series_index: self.series_index = other.series_index
        if other.language: self.language = other.language
        if other.publisher: self.publisher = other.publisher
        if other.pubdate: self.pubdate = other.pubdate
        if other.isbn: self.isbn = other.isbn
        
        self.del_sort_key()
        

    def create_links(self, allowed_formats):
        # use existing fields to create library links for this book
            
        def create_odlinkset(book_id, libraries, is_ebook, is_audiobook):
            odlinks = set()
            for lib in libraries:
                if is_ebook:
                    odlinks.add(ODLink(provider_id=lib.provider_id, library_id=lib.library_id, book_id=book_id, is_audiobook=False))
                    
                if is_audiobook:
                    odlinks.add(ODLink(provider_id=lib.provider_id, library_id=lib.library_id, book_id=book_id, is_audiobook=True))
                
            return unicode(ODLinkSet(odlinks=odlinks))
            
        is_ebook = not self.formats.isdisjoint(ALL_READABLE_FORMATS & allowed_formats)
        is_audiobook = not self.formats.isdisjoint(ALL_LISTENABLE_FORMATS & allowed_formats)
                            
        self.odid = create_odlinkset(self.book_id, self.available, is_ebook, is_audiobook)
        self.odrid = create_odlinkset(self.book_id, self.recommendable, is_ebook, is_audiobook)
        self.odpid = create_odlinkset(self.book_id, self.purchasable, is_ebook, is_audiobook)
        
        
    def levels_available(self):
        if self.book_key in LEVEL_BY_BOOK_KEY:
            return [LEVEL_BY_BOOK_KEY[self.book_key]]  # override for specific books
            
        levels = []
        for format in self.formats:
            level = level_of_format_and_provider(format, self.provider_id)
            if level is not None and level not in levels:
                levels.append(level)
            
        return levels

    
class DiscoveredBook(BookObj):
    '''
    An object keeping track of all of the information related to a book discovered at a lending library.
    
    In order to allow storage of a discovered book list all attributes must be
    JSON convertible types: list, dict, unicode string, int/long, float, True, False, None
    

    authors         list of author names (assume primary first)
    title           book title
    odid            links for libraries carrying this book 
    odrid           links for libraries that can accept recommendations for this book
    odpid           links where this book can be purchased
    isbn            book isbn (string)
    publisher       book publisher
    pubdate         publication date as utc datetime object
    language        language of publication
    ignored         Hidden from UI (True/False)
    '''
    
    idltypes = LIDLTYPES    # library book link types
    
    def __init__(self, authors=[], title='', odid='', odrid='', odpid='', 
            series='', series_index=0.0, isbn='', publisher='', pubdate=None, language='', ignored=False):
                    
        # Required fields - will be stored for discovered books (even if empty)
        
        self.authors = unique_authors(authors)  # drop any empty or non-unique author strings & force copy
        self.title = title
        self.odid  = odid
        self.odrid = odrid
        self.odpid = odpid
        self.series = series
        self.series_index = series_index
        self.isbn = isbn
        self.publisher = publisher
        self.pubdate = pubdate
        self.language = language
        self.ignored = ignored
        
        
    def merge_from(self, other, replace_links=False, update_links=False):
        # update original book using new links and any missing data
        
        if replace_links:
            self.odid = other.odid
            self.odrid = other.odrid
            self.odpid = other.odpid
            
        elif update_links:
            self.update_links(other)

        if not self.series:
            self.series = other.series
            
        if not self.series_index:
            self.series_index = other.series_index
        
        if not self.isbn:
            self.isbn = other.isbn
        
        if not self.publisher:
            self.publisher = other.publisher
        
        if not self.pubdate:
            self.pubdate = other.pubdate
        
        if not self.language:
            self.language = other.language
            
        self.del_sort_key()
        
        
        
    def __unicode__(self): 
        # display string for a book
        return '%s%s'%(self.title, ' by %s'%limited_authors_to_string(self.authors) if self.authors else '')

        
    def __str__(self): 
        return unicode(self)

        
    def __repr__(self): 
        # unambiguous string representation of a book
        return '%s%s [%s]'%(self.title, ' by %s'%authors_to_string(self.authors) if self.authors else '',
                ', '.join(filter(None, [
                        self.series if self.series_index == 0.0 else '%s [%02d]'%(self.series, int(self.series_index)),
                        self.odid, 
                        LINK_RECOMMENDABLE+self.odrid if self.odrid else '',
                        LINK_PURCHASABLE+self.odpid if self.odpid else '', 
                        self.isbn, 
                        self.publisher, 
                        self.pubdate.isoformat() if self.pubdate else '', 
                        self.language,
                        'ignored' if self.ignored else '',
                        ])))
                    
        
   
    
'''
Assorted utility routines
'''    

MAX_AUTHORS = 3

def limited_authors_to_string(authors):
    # List only a limited number of authors when a book has many listed
    
    if len(authors) <= MAX_AUTHORS:
        return authors_to_string(authors)
        
    return "%s..."%authors_to_string(authors[:MAX_AUTHORS])
    
    

def author_sort_key(author):
    return author_to_author_sort(author).lower()
