﻿#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2016, John Howell <jhowell@acm.org>'
__docformat__ = 'restructuredtext en'

import re
import urllib
import json
import cookielib

from calibre.ebooks.BeautifulSoup import BeautifulSoup
from calibre.utils.date import parse_only_date

from calibre_plugins.overdrive_link.numbers import value_unit
from calibre_plugins.overdrive_link.formats import (FORMAT_ADOBE_EPUB, FORMAT_ADOBE_PDF, FORMAT_BLIO, FORMAT_ACOUSTIK, 
    FORMAT_AXIS360_READER, FORMAT_AXIS360_AUDIO)
from calibre_plugins.overdrive_link.book import (LibraryBook, InfoBook)
from calibre_plugins.overdrive_link.library import SearchableLibrary
from calibre_plugins.overdrive_link.match import (normalize_author, normalize_title)
from calibre_plugins.overdrive_link.net import (open_url, hostname_from_url)
from calibre_plugins.overdrive_link.parseweb import (LibraryError, class_contains, must_find, must_findAll, text_only)
from calibre_plugins.overdrive_link.language import LANGUAGE_NAME


FORMAT_OF_CLASS = {
    '_axis360ebook_': FORMAT_AXIS360_READER,
    '_axis360audio_': FORMAT_AXIS360_AUDIO, # example : sfpl.axis360.baker-taylor.com/Title?itemId=0017222541
    '_epub_': FORMAT_ADOBE_EPUB,
    '_pdf_': FORMAT_ADOBE_PDF,
    '_blio_': FORMAT_BLIO,
    '_acoustik_': FORMAT_ACOUSTIK,
    '_buttondeatil_': None,         # Not a format, just a wrapper
    }
    
# Bucks County still has EPUBs as of 9/2015: http://buckslib.axis360.baker-taylor.com/


def library_host(library_id):
    if '.' in library_id:
        return library_id
        
    return '%s.axis360.baker-taylor.com' % library_id
    

'''
http://ocls.axis360.baker-taylor.com/Search/GetContent?availabilityFilter=ALL&formatType=NONE&firstload=1&layout=ListView&term=scott+adams&searchBy=keyword&sort=-score&author=&subject=&genre=&series=&sourcePage=&includeRecommendable=False&pubDate=&dateAdded=&audience=&format=&take=10&skip=0&page=1&pageSize=10

http://ocls.axis360.baker-taylor.com/Title?itemId=0012420466
Overview: title, author, publisher, date
Details: language, isbn
Select a format (blio, epub, pdf, acoustik)

'''

    
class Axis360(SearchableLibrary):
    id = 'ax'
    name = 'Axis 360'
    formats_supported = {FORMAT_ACOUSTIK, FORMAT_ADOBE_EPUB, FORMAT_ADOBE_PDF, FORMAT_BLIO, 
                FORMAT_AXIS360_READER, FORMAT_AXIS360_AUDIO}
    
    
    @staticmethod
    def validate_library_id(library_id, migrate=True, config=None):
        if (':' in library_id) or ('/' in library_id):
            library_id = hostname_from_url(library_id)
            
        if library_id.lower().endswith('.axis360.baker-taylor.com'):
            library_id = library_id[:-len('.axis360.baker-taylor.com')]    # strip suffix
        
        if not re.match(r'^([0-9a-zA-Z]+)$', library_id):
            raise ValueError('Axis 360 library id must be alphanumeric: "%s"' % library_id)
                
        return library_id.lower()
        
         
    @staticmethod
    def validate_book_id(book_id, library_id):
        if not re.match(r'^([0-9a-z]+)$', book_id):
            raise ValueError('Axis 360 book id must be alphanumeric: "%s"' % book_id)
            
        return book_id
        
            
    @staticmethod
    def book_url(library_id, book_id):
        return 'http://%s/Title?itemId=%s' % (library_host(library_id), book_id)
        

    def __init__(self):
        self.cookiejar = cookielib.CookieJar()

        
    def find_books(self, books, search_author, search_title, keyword_search, find_recommendable):
        '''
        Search Axis 360 for books that match an author/title (or subsets thereof).
        '''
        
        page_num = 1
        total_pages = 1
        total_results = 0
        results_processed = 0
        RESULTS_PER_PAGE = 10
        
        MAX_RESULTS_ALLOWED = 500
        
        data = {}
        data['layout'] = 'ListView'
        data['pageSize'] = unicode(RESULTS_PER_PAGE)
        data['firstload'] = '1'
        
        # establish page size
        open_url(self.log, 'http://%s/Search/GetLayoutViewIndex' % library_host(self.library_id), urllib.urlencode(data),
                    cookiejar=self.cookiejar)

        
        while (page_num <= total_pages):
            data = {}
            data['availabilityFilter'] = 'ALL'
            data['formatType'] = ''     # or 'NONE'
            data['firstload'] = '1'
            data['layout'] = 'ListView'
            data['term'] = ' '.join([search_author, search_title]).strip()
            data['searchBy'] = 'keyword'
            data['sort'] = '-score'
            data['author'] = ''
            data['subject'] = ''
            data['genre'] = ''
            data['series'] = ''
            data['sourcePage'] = ''
            data['includeRecommendable'] = 'False'
            data['pubDate'] = ''
            data['dateAdded'] = ''
            data['audience'] = ''
            data['format'] = ''
            data['page'] = unicode(page_num)
            data['take'] = unicode(RESULTS_PER_PAGE)
            data['skip'] = unicode((page_num - 1) * RESULTS_PER_PAGE)
            data['pageSize'] = unicode(RESULTS_PER_PAGE)
            
            response = open_url(self.log, 'http://%s/Search/GetContent?%s' % (library_host(self.library_id), urllib.urlencode(data)),
                                cookiejar=self.cookiejar)
            
            results = json.loads(response.data)      # Parse the json results
            
            new_total_results = results["TotalHits"]
            if total_results and (new_total_results != total_results):
                raise LibraryError('Total results changed from %d to %d'%(total_results, new_total_results))
            
            total_results = new_total_results
            total_pages = ((total_results - 1) // RESULTS_PER_PAGE) + 1  # floor division
            self.log.info('Response: page %d of %d. %d total results'%(page_num, total_pages, total_results))
            
            if total_results > MAX_RESULTS_ALLOWED:
                return True
                
            for book in results["Books"]:
                # "Narrators", "Series", "Genres", "Genre", "ImageUrl", "IsAvailable", "LegacyBookModel", "Synopsis"
                # "ButtonViewHTML", "ReservationName", "ReservationDescription", "StartDate", "EndDate", "Layout",
                # "IsRecommendable", "LookInRecommendedCollection", "IsAlreadyRecommended", "Format"
                # only one format listed so ignore it
                
                book_id = book["BookID"]
                title = normalize_title(book["Title"])
                authors = [normalize_author(a, unreverse=True) for a in re.split(r'[;#]', book["Authors"])]
                
                isbn = book["ISBN"]
                
                pubdate = None
                book_pubdate = book["PublicationDate"]
                if book_pubdate:
                    pubdate = parse_only_date(book_pubdate, assume_utc=True)
                
                available = book["BelongsToLibrary"]
                           
                lbook = LibraryBook(
                    authors=authors, title=title, isbn=isbn, pubdate=pubdate,
                    available=available, lib=self, book_id=book_id, search_author=search_author)
                        
                if not available:
                    self.log.info('Ignoring unavailable: %s' % repr(lbook))
                else:
                    self.log.info('Found: %s' % repr(lbook))
                    books.add(lbook)
            
                results_processed += 1
                
            if results_processed >= total_results:
                break
                
            page_num += 1
            
        if results_processed != total_results:
            raise LibraryError('Expected %s but found %d'%(value_unit(total_results,'result'), results_processed))
            
        return False
        

    def get_book_info(self, book_id):
        response = open_url(self.log, self.book_url(self.library_id, book_id))
        
        # Parse page for: isbn, publisher, language
        
        publisher = ''
        pubdate = None
        isbn = ''
        formats = set()
        language = ''
        
        soup = BeautifulSoup(response.data, convertEntities=BeautifulSoup.HTML_ENTITIES)
        
        book_overview = must_find(soup, 'div', attrs={'id': 'overview'})
        book_title = must_find(book_overview, 'div', attrs={'id': 'divItemTitle'})
        title = normalize_title(text_only(book_title))
        
        book_author = must_find(book_overview, 'div', attrs={'id': 'divItemAuthor'})
        authors = [normalize_author(a, unreverse=True) for a in re.split(r'[;#]', text_only(book_author))]
        
        book_pubdates = book_overview.findAll('div', attrs={'class': 'bookpubdate'})
        for book_pubdate in book_pubdates:
            key, sep, value = text_only(book_pubdate).partition(':')
            if sep == ':':
                key = key.strip()
                if key == 'Publication Date':
                    pubdate = parse_only_date(value.strip(), assume_utc=True)
                elif key == 'Publisher':
                    publisher = value.strip()

        book_details = must_find(soup, 'div', attrs={'id': 'details'})
            
        book_pubdates = book_details.findAll('div', attrs={'class': 'bookpubdate'})
        for book_pubdate in book_pubdates:
            key, sep, value = text_only(book_pubdate).partition(':')
            if sep == ':':
                key = key.strip()
                if key == 'Language':
                    language = value.strip()
                    language = LANGUAGE_NAME.get(language, language)
                elif key == 'ISBN':
                    isbn = value.strip()
        
        format_divs = soup.findAll('div', attrs={'class': re.compile('icon_format_')}, recursive=True)
        for format_div in format_divs:
            format_class = format_div['class']
            for fclass in FORMAT_OF_CLASS:
                if fclass in format_class:
                    if FORMAT_OF_CLASS[fclass] is not None:
                        formats.add(FORMAT_OF_CLASS[fclass])
                    break
            else:
                self.log.warn('Unknown Axis360 format %s' % format_class)
                           
        return InfoBook(
            authors=authors, title=title, isbn=isbn, publisher=publisher, pubdate=pubdate,
            language=language, formats=formats, lib=self, book_id=book_id)
    
            
    def check_book_obtainable(self, book_id):
        response = open_url(self.log, self.book_url(self.library_id, book_id))
        
        library_copies = 1
        available_copies = 0
        number_waiting_overall = 0
        estimated_wait_days = None
        
        if "The requested title is either incorrect or not available with library inventory" in response.data:
            raise LibraryError('Book no longer present in library collection')
        
        soup = BeautifulSoup(response.data, convertEntities=BeautifulSoup.HTML_ENTITIES)
        
        for copy_info in must_findAll(soup, 'div', attrs=class_contains('CopiesInfo')):
            copy_text = text_only(copy_info)
            first,sep,rest = copy_text.partition(' ')
            
            if first == 'Owned':
                library_copies = int(rest)
            elif first == 'Available':
                available_copies = int(rest)
            elif rest in ['patron on hold list', 'patrons on hold list']:
                number_waiting_overall = int(first)
            elif copy_text.startswith('Estimated Wait: '):
                if copy_text.endswith('day') or copy_text.endswith('days'):
                    estimated_wait_days = int(copy_text.split()[2])
                else:
                    raise LibraryError('Unexpected %s' % copy_text)

        # not signed in so cannot determine if book is checked out or already on hold
        
        # estimate availability
        return self.when_obtainable(library_copies=library_copies, available_copies=available_copies,
                        number_waiting_overall=number_waiting_overall, estimated_wait_days=estimated_wait_days)
 
