﻿#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2016, John Howell <jhowell@acm.org>'
__docformat__ = 'restructuredtext en'

import time
import traceback
from functools import partial

try:
    from PyQt5.Qt import (
        QMenu, QToolButton, QProgressDialog, QDialog, QHBoxLayout, QVBoxLayout, QTextBrowser,
        QDialogButtonBox, QPushButton, QIcon, QSize, Qt)
    is_pyqt4 = False
except ImportError:
    from PyQt4.Qt import (
        QMenu, QToolButton, QProgressDialog, QDialog, QHBoxLayout, QVBoxLayout, QTextBrowser,
        QDialogButtonBox, QPushButton, QIcon, QSize, Qt)
    is_pyqt4 = True
    
from calibre.gui2 import (error_dialog, question_dialog, Dispatcher)
from calibre.gui2.actions import InterfaceAction
from calibre.ebooks.metadata.book.base import Metadata
from calibre.utils.date import (now, UNDEFINED_DATE)
from calibre.utils.config_base import tweaks

from calibre_plugins.overdrive_link.book import CalibreBook
from calibre_plugins.overdrive_link.formats import (
    ALL_READABLE_FORMATS, ALL_LISTENABLE_FORMATS,
    LEVEL_SCAN_CBZ, LEVEL_SCAN_PDF, LEVEL_PDF, LEVEL_SCRIBD_EPUB, LEVEL_SCRIBD_PDF, LEVEL_HOOPLA_EPUB, 
    LEVEL_HOOPLA_CBZ, LEVEL_OYSTER_EPUB, LEVEL_KINDLE, LEVEL_EPUB, LEVEL_SKIP, LEVEL_USER_KINDLE,
    LEVEL_USER_EPUB, LEVEL_UNATTRIB_EPUB, LEVEL_UNATTRIB_KINDLE, LEVEL_UNATTRIB_PDF,
    AZW3_FILE_TYPE, MOBI_FILE_TYPE, EPUB_FILE_TYPE)
from calibre_plugins.overdrive_link.link import (
    ODLinkSet, IDENT_NEEDED_LINK, IDENT_AVAILABLE_LINK, IDENT_RECOMMENDABLE_LINK, IDENT_PURCHASABLE_LINK,
    IDENT_ISBN, IDENT_AMAZON, linksets_str, SEP_EBOOK, SEP_AUDIOBOOK,
    LINK_AVAILABLE, LINK_RECOMMENDABLE, LINK_PURCHASABLE, LINK_NEEDED,
    LINK_FORMAT_AUDIOBOOK, BOOK_DISCOVERED, browse_to_url, linkset_menu)
from calibre_plugins.overdrive_link.common_utils import (set_plugin_icon_resources, get_icon)
from calibre_plugins.overdrive_link.jobs import (start_search_jobs, SPECIAL_JOBS)
from calibre_plugins.overdrive_link.config import (
    DEFAULT_RELEASE_NOTE_VERSION, set_defaults, PLUGIN_ICONS, HELP_FILE, ReleaseNoteVersion,
    IncrementalSearchSequences, plugin_config)
from calibre_plugins.overdrive_link.discovered import (
    have_discovered_books, get_discovered_books, new_books_discovered, DiscoveredBooksDialog)
from calibre_plugins.overdrive_link.numbers import value_unit
from calibre_plugins.overdrive_link.release_notes import (RELEASE_NOTES, LAST_RELEASE_NOTE_VERSION)
from calibre_plugins.overdrive_link.log import (exception_str, html_to_text)
from calibre_plugins.overdrive_link.tweak import (TWEAK_LIBRARY_GROUPS, TWEAK_TRACK_NEEDED_BOOKS)
from calibre_plugins.overdrive_link import ActionOverdriveLink


FIELD_OBTAINABLE = 'ol_wait_weeks'      # lookup name for custom column holding when-obtainable results

# yes/no custom column. valid values are: None, True, False
FIELD_ET_STATUS = 'et_status'           # lookup name for custom column holding et-status results

# date column
FIELD_ET_DATE = 'et_date'               # lookup name for custom column holding date et-status became True

NEEDED_GROUP_NAME = '**needed**'


def search_escape(str):
    # escape a string for inclusion in a calibre search (" and \)
    return str.replace('\\', '\\\\').replace('"', '\\"')
    
def re_escape(str):
    # like re.escape, but only escapes re special characters instead of all non-alphanumerics
    return ''.join(("\\" + c if c in '.^$*+?{}\\[]|()' else c) for c in str)
    

class OverdriveLinkAction(InterfaceAction):
    name = ActionOverdriveLink.name
    type = ActionOverdriveLink.type

    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Overdrive', None, 'Link books to lending libraries', None)
    popup_type = QToolButton.InstantPopup
    dont_add_to = frozenset(['menubar-device', 'context-menu-device'])
    action_type = 'current'

    def genesis(self):
        # initialize book tracking for search jobs
        self.next_job_id = 0
        self.last_job_start_time = None
        self.book_ids_being_searched = {}
        self.num_job_completions_in_progress = 0
        
        # load configuration
        self.config = set_defaults()

        # Load resources
        self.help_text = unicode(self.load_resources(HELP_FILE)[HELP_FILE], encoding='utf-8', errors='ignore')

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        # Initialize the gui menu for this plugin
        self.create_default_menu_actions()
        self.gui.keyboard.finalize()

        self.menu = QMenu(self.gui)
        self.set_default_menu()
        self.menu.aboutToShow.connect(self.set_customized_menu)
        self.menu.aboutToHide.connect(self.set_default_menu)

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        
        self.main_icon = get_icon(PLUGIN_ICONS[0])
        self.qaction.setIcon(self.main_icon)

        if not self.config.disable_metadata_plugin:
            self.create_metadata_plugin()

    def create_metadata_plugin(self):
        '''
        Total hack to add a metadata plugin as if it had been loaded separately.
        This plugin is used to show links in the book details panel.
        '''
        try:
            from calibre_plugins.overdrive_link.metadata import OverdriveLinkMetadata
            from calibre.customize.ui import _initialized_plugins
            metadata_plugin = OverdriveLinkMetadata(self.plugin_path, self)
            _initialized_plugins.append(metadata_plugin)
        except:
            # Just continue on, in case a change to calibre breaks the above hack
            traceback.print_exc()

    def create_default_menu_actions(self):
        self.default_menu = QMenu(self.gui)
        m = self.default_menu

        # Actions that operate on the current selection

        self.search_book_action = self.create_menu_action(
            m, 'OverdriveLinkSearchBooks',
            'Search for selected books', 'search.png',
            description='Search for selected books at configured lending libraries',
            triggered=partial(self.search, False, False, None, None))

        self.search_author_action = self.create_menu_action(
            m, 'OverdriveLinkSearchAuthors', 'Search for books by selected authors', 'search.png',
            description='Discover additional books by selected authors, missing from your calibre library',
            triggered=partial(self.search, True, False, None, None))

        self.discover_by_keywords_action = self.create_menu_action(
            m, 'OverdriveLinkSearchKeywords', 'Discover books by keyword', 'search.png',
            description='Discover books matching the first set of configured keywords',
            triggered=partial(self.search, True, False, None, 0))    # not selection-based, but fits better here

        self.browse_book_action = self.create_menu_action(
            m, 'OverdriveLinkBrowseBook', 'Browse selected book at lending library', 'store.png',
            description='Browse to the web page for the selected book at the highest priority lending library',
            triggered=self.browse_to_primary_link)

        self.obtain_book_action = self.create_menu_action(
            m, 'OverdriveLinkObtainBooks', 'Check current availability of selected books', 'search.png',
            description='Check if selected books are available now and estimate wait if not',
            triggered=self.obtain_books)
            
        self.obtain_needed_book_action = self.create_menu_action(
            m, 'OverdriveLinkObtainNeededBooks', 'Check current availability of selected needed books', 'search.png',
            description='Check if selected needed books are available now and estimate wait if not',
            triggered=partial(self.obtain_books, NEEDED_GROUP_NAME))
            
        self.unlink_book_action = self.create_menu_action(
            m, 'OverdriveLinkUnlinkBooks', 'Unlink selected books', get_icon('images/unlink.png'),
            description='Remove links for currently selected books (except for disabled libraries)',
            triggered=self.unlink_books)

        if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
            self.unlink_needed_action = self.create_menu_action(
                m, 'OverdriveLinkUnlinkNeeded', 'Remove "needed" links of selected books', get_icon('images/unlink.png'),
                description='Remove any "needed" links for currently selected books',
                triggered=self.unlink_needed)
        else:
            self.unlink_needed_action = None

        m.addSeparator()

        # Actions that are not selection-based

        self.display_new_release_notes_action = self.create_menu_action(
            m, 'OverdriveLinkNewReleaseNotes',
            'Important release notes!', 'rating.png',
            description='Display important notes about this plugin release',
            triggered=self.display_new_release_notes)

        self.select_all_action = self.create_menu_action(
            m, 'OverdriveLinkSelectAll', 'Select all available books', get_icon('images/link.png'),
            description='Select all books containing an available library book link',
            triggered=self.show_available_books)

        self.select_recommend_action = self.create_menu_action(
            m, 'OverdriveLinkSelectRecommend', 'Select all recommendable books', get_icon('images/link.png'),
            description='Select all books containing only recommendable book links',
            triggered=self.show_recommendable_books)

        self.select_purchase_action = self.create_menu_action(
            m, 'OverdriveLinkSelectPurchase', 'Select all purchasable books', get_icon('images/link.png'),
            description='Select all books containing only purchasable book links',
            triggered=self.show_purchasable_books)

        self.select_need_action = self.create_menu_action(
            m, 'OverdriveLinkSelectNeed', 'Select all needed books', get_icon('images/link.png'),
            description='Select all books containing a needed library book link',
            triggered=self.show_needed_books)

        self.select_all_audiobook_action = self.create_menu_action(
            m, 'OverdriveLinkSelectAllAudio', 'Select all available audiobooks', get_icon('images/link.png'),
            description='Select all books containing an available library audiobook link',
            triggered=self.show_available_audiobooks)

        self.select_recommend_audiobook_action = self.create_menu_action(
            m, 'OverdriveLinkSelectRecommendAudio', 'Select all recommendable audiobooks', get_icon('images/link.png'),
            description='Select all books containing only recommendable audiobook links',
            triggered=self.show_recommendable_audiobooks)

        self.select_purchase_audiobook_action = self.create_menu_action(
            m, 'OverdriveLinkSelectPurchaseAudio', 'Select all purchasable audiobooks', get_icon('images/link.png'),
            description='Select all books containing only purchasable audiobook links',
            triggered=self.show_purchasable_audiobooks)

        self.select_need_audiobook_action = self.create_menu_action(
            m, 'OverdriveLinkSelectNeedAudio', 'Select all needed audiobooks', get_icon('images/link.png'),
            description='Select all books containing a needed library audiobook link',
            triggered=self.show_needed_audiobooks)

        self.manage_results_action = self.create_menu_action(
            m, 'OverdriveLinkManageResults', 'Manage discovered books', 'merge_books.png',
            description='Manage books found through "Search by selected authors", missing from calibre library',
            triggered=self.handle_discovered_books)

        self.gutenberg_index_action = self.create_menu_action(
            m, 'OverdriveLinkBuildGutenbergIndex', 'Build index for Project Gutenberg', 'config.png',
            description='Download catalog from Project Gutenberg and build an index for searching',
            triggered=self.gutenberg_index)

        self.check_action = self.create_menu_action(
            m, 'OverdriveLinkCheckLinks', 'Check and repair book links', 'config.png',
            description='Check existing book links for problems and repair them',
            triggered=self.check_all_links)

        self.customize_action = self.create_menu_action(
            m, 'OverdriveLinkCustomize', 'Customize plugin', 'config.png',
            description='Configure the plugin\'s settings',
            triggered=self.show_configuration)

        self.help_action = self.create_menu_action(
            m, 'OverdriveLinkHelp', 'Help', 'help.png',
            description='Display help for this plugin',
            triggered=self.show_help)

        # Actions that are temporary and error messages (Not default actions)

        self.temp_menu = QMenu(self.gui)
        m = self.temp_menu

        self.customize_error_action = self.add_menu_action(
            m, '(Customize this plugin to enable search)', 'dialog_error.png',
            'Use "Customize plugin" to configure lending libraries', enabled=False)

        self.link_error_action = self.add_menu_action(
            m, '(Selected book not linked)', 'dialog_error.png',
            'There are no links for the selected book', enabled=False)

        self.select_error_action = self.add_menu_action(
            m, '(Select books to enable search)', 'dialog_error.png',
            'Search is performed based on selected books', enabled=False)

    def set_default_menu(self):
        # Copy actions from the default menu to the current menu
        self.menu.clear()

        for a in QMenu.actions(self.default_menu):
            self.menu.addAction(a)

    def set_customized_menu(self):
        # Build menu on the fly based on the number of books selected and actual links
        # Save actions in a temp menu since clearing this menu on aboutToHide sometimes causes the program
        # to crash in QT if the action is deleted just after being triggered

        m = self.menu
        m.clear()

        if self.have_new_release_notes():
            m.addAction(self.display_new_release_notes_action)
            return

        db = self.gui.current_db
        
        if self.gui.library_view.selectionModel().hasSelection():
            if self.config.any_enabled() and not self.config.configuration_error:
                # allow search for books
                m.addAction(self.search_book_action)
                m.addAction(self.search_author_action)

                if TWEAK_LIBRARY_GROUPS in tweaks:
                    sm = QMenu(m)
                    self.add_menu_action(
                        m, 'Search for books at library group', 'search.png',
                        'Search at a specific group of libraries', submenu=sm)

                    for group_name, lib_names in sorted(tweaks.get(TWEAK_LIBRARY_GROUPS).items()):
                        self.temp_menu.addAction(
                            self.add_menu_action(
                                sm, group_name, 'search.png', ', '.join(lib_names),
                                triggered=partial(
                                    self.search, '+discovery' in lib_names,
                                    '+incremental' in lib_names, group_name, None)))
                                
                if self.config.discover_keywords:
                    sm = QMenu(m)
                    self.add_menu_action(
                        m, 'Discover books by keyword', 'search.png',
                        'Discover books based on configured keywords', submenu=sm)

                    for i, keywords in enumerate(self.config.discover_keywords):
                        self.temp_menu.addAction(
                            self.add_menu_action(
                                sm, keywords, 'search.png',
                                'Discover books matching keywords ' + keywords,
                                triggered=partial(self.search, True, False, None, i)))
                                
            else:
                m.addAction(self.customize_error_action)
                
            m.addSeparator()
            show_linked_book_menu_items = False

            if len(self.gui.library_view.selectionModel().selectedRows()) == 1:
                # Only follow links for a single book
                book_id = self.gui.library_view.get_selected_ids()[0]
                identifiers = db.new_api.get_proxy_metadata(book_id).identifiers
                
                # Open browser for access to book at lending library
                
                if linkset_menu(identifiers, m, self.config, save=self.temp_menu):
                    m.addSeparator()
                    show_linked_book_menu_items = True
                else:
                    m.addAction(self.link_error_action)
            else:
                show_linked_book_menu_items = True  # multiple books selected, some may be linked
                
            if show_linked_book_menu_items and not self.config.configuration_error:
                if self.lib_has_custom_column(FIELD_OBTAINABLE) and self.config.any_enabled():
                    if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                        m.addAction(self.obtain_needed_book_action)
                    else:
                        m.addAction(self.obtain_book_action)

                    if TWEAK_LIBRARY_GROUPS in tweaks:
                        sm = QMenu(m)
                        self.add_menu_action(
                            m, 'Check current availability at library group', 'search.png',
                            'Check current availability of selected books at a specific group of libraries', submenu=sm)

                        for group_name, lib_names in sorted(tweaks.get(TWEAK_LIBRARY_GROUPS).items()):
                            self.temp_menu.addAction(
                                self.add_menu_action(
                                    sm, group_name, 'search.png', ', '.join(lib_names),
                                    triggered=partial(self.obtain_books, group_name)))
                                
                m.addAction(self.unlink_book_action)
                
                if TWEAK_LIBRARY_GROUPS in tweaks:
                    sm = QMenu(m)
                    self.add_menu_action(
                        m, 'Unlink selected books from library group', 'images/unlink.png',
                        'Remove links for currently selected books from a specific group of libraries', submenu=sm)

                    for group_name, lib_names in sorted(tweaks.get(TWEAK_LIBRARY_GROUPS).items()):
                        self.temp_menu.addAction(
                            self.add_menu_action(
                                sm, group_name, 'images/unlink.png', ', '.join(lib_names),
                                triggered=partial(self.unlink_books, group_name)))
                
                if self.unlink_needed_action:
                    m.addAction(self.unlink_needed_action)
                    
        else:
            m.addAction(self.select_error_action)

        m.addSeparator()

        sm = QMenu(m)

        for is_audio, type, sep in [(False, 'book', SEP_EBOOK), (True, 'audiobook', SEP_AUDIOBOOK)]:
            for ident, cat, all_ebook_action, all_audiobook_action in [
                    (IDENT_AVAILABLE_LINK, 'Available',
                     self.select_all_action, self.select_all_audiobook_action),
                    (IDENT_RECOMMENDABLE_LINK, 'Recommendable',
                     self.select_recommend_action, self.select_recommend_audiobook_action),
                    (IDENT_PURCHASABLE_LINK, 'Purchasable',
                     self.select_purchase_action, self.select_purchase_audiobook_action),
                    (IDENT_NEEDED_LINK, 'Needed',
                     self.select_need_action, self.select_need_audiobook_action),
                    ]:
                    
                if ident == IDENT_NEEDED_LINK and not tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                    continue
                 
                ssm = QMenu(sm)
                ssm.addAction(all_audiobook_action if is_audio else all_ebook_action)
                
                for library in sorted(self.config.libraries, key=lambda lib: lib.name):
                        
                    if cat == 'Recommendable' and not library.provider.supports_recommendation:
                        continue

                    if cat == 'Purchasable'and not library.provider.supports_purchase(library.library_id):
                        continue
                            
                    if ((is_audio and library.provider.formats_supported.isdisjoint(ALL_LISTENABLE_FORMATS)) or
                            ((not is_audio) and library.provider.formats_supported.isdisjoint(ALL_READABLE_FORMATS))):
                        continue
                        
                    self.temp_menu.addAction(self.add_menu_action(
                        ssm, 'Only at ' + library.name, 'images/link.png',
                        'Select books with %s:%s%s' % (ident, sep, library.full_library_id),
                        triggered=partial(self.show_linked, ident, sep, library.full_library_id)))

                self.add_menu_action(sm, '%s %ss' % (cat, type), 'images/link.png', '', submenu=ssm)
                
            sm.addSeparator()

        self.add_menu_action(
            m, 'Select previously linked books', 'images/link.png',
            'Actions to be taken for books linked previously', submenu=sm)

        m.addSeparator()
        if have_discovered_books(db):
            m.addAction(self.manage_results_action)
            
        if self.config.have_project_gutenberg:
            m.addAction(self.gutenberg_index_action)

        m.addAction(self.check_action)
        m.addAction(self.customize_action)
        m.addAction(self.help_action)
        
    def show_available_books(self):
        self.show_linked(IDENT_AVAILABLE_LINK, SEP_EBOOK)

    def show_recommendable_books(self):
        self.show_linked(IDENT_RECOMMENDABLE_LINK, SEP_EBOOK)

    def show_purchasable_books(self):
        self.show_linked(IDENT_PURCHASABLE_LINK, SEP_EBOOK)

    def show_needed_books(self):
        self.show_linked(IDENT_NEEDED_LINK, SEP_EBOOK)

    def show_available_audiobooks(self):
        self.show_linked(IDENT_AVAILABLE_LINK, SEP_AUDIOBOOK)

    def show_recommendable_audiobooks(self):
        self.show_linked(IDENT_RECOMMENDABLE_LINK, SEP_AUDIOBOOK)

    def show_purchasable_audiobooks(self):
        self.show_linked(IDENT_PURCHASABLE_LINK, SEP_AUDIOBOOK)
        
    def show_needed_audiobooks(self):
        self.show_linked(IDENT_NEEDED_LINK, SEP_AUDIOBOOK)

    def show_linked(self, link_type, book_type, full_library_id=''):
        # produce a search string that will find books of the right type, linked to the supplied library
                
        if full_library_id and (len(full_library_id) < 10 or full_library_id.endswith('/')):
            # There is a provider without a specific library or with a short library-id
            # This can match links for the wrong library if searched with a sub-string search so use a re search instead
            # Future: Should do this for any library id string that is a prefix of another.
            ident_re = '~%s%s(&|$)' % (book_type, re_escape(full_library_id))
            search = 'identifiers:"=%s:%s"' % (link_type, search_escape(ident_re))
        else:
            search = 'identifiers:"=%s:%s%s"' % (link_type, book_type, full_library_id)
                
        if link_type == IDENT_RECOMMENDABLE_LINK:
            search += ' and not identifiers:"=%s:%s"' % (IDENT_AVAILABLE_LINK, book_type)
            
        self.gui.search.set_search_string(search)
       

    def browse_to_primary_link(self):
        selected_ids = self.gui.library_view.get_selected_ids()
        if not selected_ids:
            error_dialog(self.gui, 'No Books Selected', 'Select a book to browse at lending library', show=True)
            return

        if len(selected_ids) > 1:
            error_dialog(
                self.gui, 'Too many Books Selected', 'Select only a single book to browse at lending library',
                show=True)
            return

        odid = self.gui.current_db.new_api.get_proxy_metadata(selected_ids[0]).identifiers.get(IDENT_AVAILABLE_LINK)
        if not odid:
            error_dialog(
                self.gui, 'Selected book not linked', 'There are no lending library links for the selected book',
                show=True)
            return

        # Open browser for access to book at lending library
        browse_to_url(ODLinkSet(str=odid, config=self.config).primary_odlink().url())
        

    def search(self, discover_books, incremental=False, group_name=None, kw_index=None):
        db = self.gui.current_db
        selected_ids = db.new_api.all_book_ids() if incremental else self.gui.library_view.get_selected_ids()
        
        if kw_index is not None:
            if len(self.config.discover_keywords) == 0:
                error_dialog(
                    self.gui, 'No keywords have been configured for book discovery.',
                    'Use "Customize plugin" to configure keywords for search.', show=True)
                return

            keywords = self.config.discover_keywords[kw_index]
            special = keywords in SPECIAL_JOBS
            discover_books = not special
            if not special: selected_ids = []
        else:
            keywords = None
            special = False

            if not selected_ids:
                error_dialog(self.gui, 'No books selected', 'Select books to enable search', show=True)
                return
                
            num_already_being_searched = self.check_search_job_books(selected_ids)
            if num_already_being_searched > 0:
                error_dialog(
                    self.gui, 'Cannot start search',
                    'Search already in progress for ' + value_unit(num_already_being_searched, 'same book'),
                    show=True)
                return
            
        if special:
            collect_et_status = (keywords == 'inventory_amazon_et') and self.lib_has_custom_column(FIELD_ET_STATUS)
        else:
            collect_et_status = False
            if not self.config.any_enabled():
                error_dialog(
                    self.gui, 'No enabled lending Library',
                    'Use "Customize plugin" to configure and enable lending libraries',
                    show=True)
                return

            if not self.config.search_formats:
                error_dialog(
                    self.gui, 'No formats have been enabled for search.',
                    'Use "Customize plugin" to configure formats for search.', show=True)
                return
            

        if not self.config.disable_confirm_dlgs:
            if not keywords:
                q = '<p>Search lending library collection %s for %s' % (
                    'and recommendable' if self.config.check_recommendable else '',
                    value_unit(len(selected_ids), 'selected book'))

                if incremental:
                    q += ' incrementally'
                    
                if discover_books:
                    q += ' and save any discovered books by the same authors'
                    
            elif special:
                q = '<p>%s' % SPECIAL_JOBS[keywords].name
                
            else:
                q = '<p>Discover books in lending library collection %s with keywords %s' % (
                    'and recommendable' if self.config.check_recommendable else '',
                    keywords)

            q += '?<p>(This will run as a background job.)'

            if not question_dialog(self.gui, 'Start search', q):
                return
                
        ignore_ids = set()
        
        if discover_books:
            # Need info on all books in the calibre library so that those already known are not found again
            all_calibre_books = self.collect_book_metadata(db.new_api.all_book_ids(), ignore_ids=ignore_ids, 
                            collect_et_status=collect_et_status)
            
            # Since checking for more discovered books, provide those known so far
            orig_discovered_books = get_discovered_books(db)
            
        else:
            all_calibre_books = self.collect_book_metadata(selected_ids, ignore_ids=ignore_ids,
                            collect_et_status=collect_et_status)
                            
            orig_discovered_books = []  # unused for this search type

        if (all_calibre_books is None) and not special:
            return
        
        if ignore_ids:
            selected_ids = [id for id in selected_ids if id not in ignore_ids]    # drop books to be ignored during search
                
        if not keywords:
            if not selected_ids:
                error_dialog(self.gui, 'Only ignored books selected', 'Select non-ignored books to enable search', show=True)
                return
                
            self.gui.status_bar.show_message(
                'Searching lending libraries for ' + value_unit(len(selected_ids), 'book'))
        elif special:
            self.gui.status_bar.show_message(SPECIAL_JOBS[keywords].name)
        else:
            self.gui.status_bar.show_message('Discovering books with keywords ' + keywords)
            
        if incremental:
            sequences = plugin_config[IncrementalSearchSequences]
            incremental_sequence = sequences.get(group_name, 0)
            sequences[group_name] = incremental_sequence + 1
            plugin_config[IncrementalSearchSequences] = sequences
        else:
            incremental_sequence = None
        
        # Start a background job as separate thread in this process to do the actual searches
        # Pass all required configuration so that changes made during the run of the job will not cause problems
        start_search_jobs(
            self, self.config, all_calibre_books, selected_ids, discover_books, orig_discovered_books,
            keywords, self.group_libraries(group_name), incremental_sequence, Dispatcher(self._search_complete))


    def _search_complete(self, job):
        '''
        Called upon completion of a search operation in a search background job
        '''
        if job.failed:
            self.gui.job_exception(job, dialog_title='%s search failed!' % self.name)
            return      # Note: unable to track book ids that were in use for this job!

        self.gui.status_bar.show_message('%s search completed' % self.name, 3000)

        if not hasattr(job, 'html_details'):
            job.html_details = job.details  # Details from ParallelJob are actually html.

        self._report_search_completion(job.result, job.html_details)
        

    def obtain_books(self, group_name=None):
        if not self.lib_has_custom_column(FIELD_OBTAINABLE):
            error_dialog(self.gui, 'Missing Custom Column', 
                    'A custom column with a lookup name of %s must exist in current calibre library to enable availability checking' % FIELD_OBTAINABLE,
                    show=True)
            return
        
        selected_ids = self.gui.library_view.get_selected_ids()
        if not selected_ids:
            error_dialog(self.gui, 'No Books Selected', 'Select books to enable availability checking', show=True)
            return

        num_already_being_searched = self.check_search_job_books(selected_ids)
        if num_already_being_searched > 0:
            error_dialog(
                self.gui, 'Cannot start search',
                'Search already in progress for ' + value_unit(num_already_being_searched, 'same book'),
                show=True)
            return
                
        if not self.config.any_enabled():
            error_dialog(
                self.gui, 'No enabled lending Library',
                'Use "Customize plugin" to configure and enable lending libraries',
                show=True)
            return
            
        if not self.config.disable_confirm_dlgs:
            q = '<p>Check %s for current availability to borrow or expected wait' % (
                value_unit(len(selected_ids), 'selected linked book'))

            q += '?<p>(This will run as a background job.)'

            if not question_dialog(self.gui, 'Start check', q):
                return
                
        selected_books = self.collect_book_metadata(selected_ids, collect_obtainable=True)  # read database
        if selected_books is None:
            return

        # Start a background job as separate thread in this process to do the actual checks
        start_search_jobs(self, self.config, selected_books, selected_ids, False, [],
                            'check_obtainable', self.group_libraries(group_name), None,
                            Dispatcher(self._search_complete))
                    
     
    def group_libraries(self, group_name):
        if not group_name:
            return None
            
        if group_name == NEEDED_GROUP_NAME:
            return []
            
        library_names = []
        # handle library definitions that include others
        for lib_name in tweaks.get(TWEAK_LIBRARY_GROUPS)[group_name]:
            if lib_name.startswith('*'):
                library_names.extend(tweaks.get(TWEAK_LIBRARY_GROUPS)[lib_name[1:]])
            elif not lib_name.startswith('+'):
                library_names.append(lib_name)
                
        return library_names


    def unlink_books(self, group_name=None):
        selected_ids = self.gui.library_view.get_selected_ids()
        if not selected_ids:
            error_dialog(self.gui, 'No Books Selected', 'Select books to enable unlinking', show=True)
            return

        selected_books = self.collect_book_metadata(selected_ids)   # Gather book data from the database
        if selected_books is None:
            return
            
        saved_enabled_libraries = self.config.enabled_libraries
        
        library_names = self.group_libraries(group_name)
        if library_names is not None:
            # use alternate set of enabled libraries
            self.config.enabled_libraries = []
            for lib in self.config.libraries:
                lib.enabled = (lib.name in library_names)
                if lib.enabled:
                    self.config.enabled_libraries.append(lib)
         
        self.config.enabled_libraries = saved_enabled_libraries
        
        for book in selected_books:
            book.preserve_disabled_links(self.config)  # Keep any links for disabled libraries

        self._report_search_completion((None, selected_books, [], [], [], ['Unlink books']), None)
        

    def unlink_needed(self):
        selected_ids = self.gui.library_view.get_selected_ids()
        if not selected_ids:
            error_dialog(self.gui, 'No Books Selected', 'Select books to enable unlinking', show=True)
            return

        selected_books = self.collect_book_metadata(selected_ids)   # Gather book data from the database
        if selected_books is None:
            return

        for book in selected_books:
            book.preserve_all_links()
            book.odnid = '' # remove needed links

        self._report_search_completion((None, selected_books, [], [], [], ['Unlink needed']), None)
        

    def check_all_links(self):
        db = self.gui.current_db
        
        num_already_being_searched = self.check_search_job_books(db.new_api.all_book_ids())
        if num_already_being_searched > 0:
            error_dialog(
                self.gui, 'Cannot check links',
                'Search in progress for ' + value_unit(num_already_being_searched, 'book'),
                show=True)
            return
            
        all_calibre_books = self.collect_book_metadata(db.new_api.all_book_ids())
        if all_calibre_books is None:
            return
            
        all_discovered_books = get_discovered_books(db)
        
        total_books = len(all_calibre_books) + len(all_discovered_books)

        progress = QProgressDialog('Checking book links for errors', 'Cancel', 0, total_books, self.gui)
        progress.setWindowTitle(self.name)
        progress.setWindowFlags(progress.windowFlags() & (~Qt.WindowContextHelpButtonHint))
        progress.setMinimumWidth(400)
        progress.setMinimumDuration(500)   # Show progress quickly
        progress.setModal(True)
        progress.setValue(0)
        
        errors = []
        
        for i, book in enumerate(all_calibre_books):
            book.migrate_links(self.config)
            book.check_links(errors, self.config)
            book.remove_outdated_links()
            
            progress.setValue(i + 1)    # calls QApplication.processEvents() for the modal dialog
            if progress.wasCanceled():
                return

        for i, book in enumerate(all_discovered_books):
            book.make_orig_links()
            book.migrate_links(self.config)
            book.check_links(errors, self.config)
            book.remove_outdated_links()
            
            progress.setValue(i+len(all_calibre_books))    # calls QApplication.processEvents() for the modal dialog
            if progress.wasCanceled():
                return

        progress.reset()
        
        job_id = self.register_search_job(db.new_api.all_book_ids())
        self._report_search_completion((job_id, all_calibre_books, all_discovered_books, errors, [], ['Check links']), None)
        
        
    def lib_has_custom_column(self, column_name):
        #print('custom columns: %s' % ', '.join(self.gui.current_db.custom_column_label_map.keys()))
        return column_name in self.gui.current_db.custom_column_label_map
            

    def collect_book_metadata(self, ids, ignore_ids=None, collect_obtainable=False, collect_et_status=False, 
                collect_epub_data=False):
        '''
        Collect and prepare book data for processing by the background job while displaying progress to the user
        '''
        progress = QProgressDialog('Collecting calibre book data', 'Cancel', 0, len(ids), self.gui)
        progress.setWindowTitle(self.name)
        progress.setWindowFlags(progress.windowFlags()&(~Qt.WindowContextHelpButtonHint))
        progress.setMinimumWidth(400)
        progress.setMinimumDuration(500)   # Show progress quickly
        progress.setModal(True)
        progress.setValue(0)

        db = self.gui.current_db
        books = []
        
        for i, id in enumerate(ids):
            mi = db.new_api.get_proxy_metadata(id)
            
            # handle special tags that indicate the quality level of the calibre book formats
            levels_have = []
            allow_discovery = True
            
            if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                for tag in mi.tags:
                    level = None
                    
                    if tag == 'ol.ignore':
                        if ignore_ids is not None:
                            ignore_ids.add(id)     # Do not allow searches for this book or its authors
                    elif tag == 'ol.no-discovery':
                        allow_discovery = False    # Do not allow discovery of news books based on this book's authors
                    elif tag == 'ol.no-need':
                        level = LEVEL_SKIP
                    elif tag.startswith('s.'):
                        if tag == 's.mobi.v5':
                            level = LEVEL_UNATTRIB_KINDLE
                        elif tag.startswith('s.mobi.v5.'):
                            level = LEVEL_KINDLE
                        elif tag.startswith('s.mobi.v3') or tag.startswith('s.mobi.v4'):
                            level = LEVEL_USER_KINDLE
                            
                        elif tag.startswith('s.pdf.v5.scribd'):
                            level = LEVEL_SCRIBD_PDF
                        elif tag.startswith('s.pdf.v5.'):
                            level = LEVEL_PDF
                        elif tag.startswith('s.pdf.v5s.'):
                            level = LEVEL_SCAN_PDF
                        elif tag == 's.pdf.v5':
                            level = LEVEL_UNATTRIB_PDF
                           
                        elif tag.startswith('s.cbr.v5s.') or tag.startswith('s.cbz.v5s.'):
                            level = LEVEL_SCAN_CBZ
                    
                        elif tag.startswith('s.sepub.v5.'):
                            level = LEVEL_SCRIBD_EPUB
                            
                        elif tag == 's.epub.v5.oyster':
                            level = LEVEL_OYSTER_EPUB
                            
                        elif tag == 's.epub.v5.hoopla':
                            level = LEVEL_HOOPLA_EPUB
                            
                        elif tag == 's.cbz.v5.hoopla':
                            level = LEVEL_HOOPLA_CBZ
                            
                        elif tag == 's.epub.v5':
                            level = LEVEL_UNATTRIB_EPUB
                        elif tag.startswith('s.epub.v5.'):
                            level = LEVEL_EPUB
                        elif tag.startswith('s.epub.v3') or tag.startswith('s.epub.v4'):
                            level = LEVEL_USER_EPUB
                            
                            
                    if level is not None and level not in levels_have:
                        levels_have.append(level)
                        
                
            if collect_obtainable:
                obtainable = db.new_api.field_for('#' + FIELD_OBTAINABLE, id, default_value='')
                if isinstance(obtainable, tuple):
                    obtainable = ', '.join(sorted(obtainable))  # convert tag-like values into single string
            else:
                obtainable = ''
            
            
            if collect_et_status:
                et_asin = mi.identifiers.get(IDENT_AMAZON, '')

                if self.config.update_amazon_idents:
                    for odlink in (ODLinkSet(str=mi.identifiers.get(IDENT_AVAILABLE_LINK, '')) |
                                   ODLinkSet(str=mi.identifiers.get(IDENT_PURCHASABLE_LINK, ''))).amazon().sorted_odlinks():
                        if odlink.amazon_ident() == IDENT_AMAZON:
                            et_asin = odlink.book_id
                            break

                et_status = db.new_api.field_for('#' + FIELD_ET_STATUS, id)
            else:
                et_asin = ''
                et_status = None
            
            
            epub_file_name = None
            if collect_epub_data:
                b_formats = self.book_formats(id)
                # choose highest priority desired format that this book has (if any)
                for input_format in [AZW3_FILE_TYPE, EPUB_FILE_TYPE, MOBI_FILE_TYPE]:       # add more as needed
                    if input_format in b_formats:
                        epub_file_name = db.new_api.format(id, input_format, as_file=False, as_path=True, preserve_filename=True)
                        break
                    
                    
            books.append(CalibreBook(
                id=id, authors=[self.remove_author_suffix(a) for a in mi.authors], 
                title=mi.title, series=mi.series, series_index=mi.series_index,
                isbn=mi.identifiers.get(IDENT_ISBN, ''),
                orig_odid=mi.identifiers.get(IDENT_AVAILABLE_LINK, ''),
                orig_odrid=mi.identifiers.get(IDENT_RECOMMENDABLE_LINK, ''),
                orig_odpid=mi.identifiers.get(IDENT_PURCHASABLE_LINK, ''),
                orig_odnid=mi.identifiers.get(IDENT_NEEDED_LINK, '') if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False) else '',
                orig_obtainable=obtainable,
                orig_et_asin=et_asin,
                orig_et_status=et_status,
                epub_file_name=epub_file_name,
                levels_have=levels_have,
                allow_discovery=allow_discovery,
                last_modified=mi.last_modified))
                
            progress.setValue(i + 1)    # calls QApplication.processEvents() for the modal dialog
            if progress.wasCanceled():
                return None
                

        progress.reset()
        return books
        
        
    def remove_author_suffix(self, author):
        # remove suffix such as " [editor]"
        if author.endswith(']'):
            return author.partition('[')[0].strip()
            
        return author
        

    def book_formats(self, book_id):
        formats = self.gui.current_db.formats(book_id, index_is_id=True) or ''
        return set([x.lower().strip() for x in formats.split(',')])
        

    def _report_search_completion(self, result, log):
        '''
        Handle search completion. Can also handle completion of check for ids to remove.
        '''

        def add_book_details(detail, book_message_list):
            for book, message, is_discovered in sorted(book_message_list, key=lambda book_message: book_message[0]):
                detail.append(''.join(
                    (BOOK_DISCOVERED if is_discovered else '', unicode(book), ' (%s)' % message if message else '')))

        def add_optional_key():
            if not self.key_used:
                keys = []
                keys.append(('Discovered book', BOOK_DISCOVERED))
                keys.append(('Available at', LINK_AVAILABLE))
                keys.append(('Recommendable at', LINK_RECOMMENDABLE))
                keys.append(('Purchasable at', LINK_PURCHASABLE))
                
                if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                    keys.append(('Needed', LINK_NEEDED))
                    
                keys.append(('Audiobook', LINK_FORMAT_AUDIOBOOK))
                
                details.append('Key: %s' % ', '.join(['%s=%s' % (k2, k1) for k1,k2 in keys]))
                self.key_used = True
                
        def report_book_changes(book, is_discovered):
            # Compare the individual links to find those added and removed
            idlinks = book.get_idlinks(self.config)
            orig_idlinks = book.get_orig_idlinks(self.config)
            
            added_idlinks = {}
            removed_idlinks = {}
            unchanged_idlinks = {}
            any_added = False
            any_removed = False
            any_unchanged = False
            any_other_change = False
            
            if not is_discovered:
                if book.orig_obtainable != book.obtainable:
                    obtainable_changed.append((book, book.obtainable or 'Not available', False))
                    any_other_change = True
                
                if unicode(book.orig_et_status) != unicode(book.et_status):
                    et_status_changed.append((book, unicode(book.et_status), False))
                    any_other_change = True
                
                if book.orig_et_asin != book.et_asin:
                    et_asin_changed.append((book, book.et_asin, False))
                    any_other_change = True
                
                if book.inventory:
                    book_inventory.append((book, book.inventory, False))
                
            for ltype in book.idltypes:
                added_idlinks[ltype] = idlinks[ltype] - orig_idlinks[ltype]
                removed_idlinks[ltype] = orig_idlinks[ltype] - idlinks[ltype]
                unchanged_idlinks[ltype] = orig_idlinks[ltype] & idlinks[ltype]
                
                any_added = any_added or len(added_idlinks[ltype]) > 0
                any_removed = any_removed or len(removed_idlinks[ltype]) > 0
                any_unchanged = any_unchanged or len(unchanged_idlinks[ltype]) > 0
            
            if any_added:
                added_to_library.append((book, linksets_str(added_idlinks), is_discovered))

            if any_removed:
                removed_from_library.append((book, linksets_str(removed_idlinks), is_discovered))

            if any_unchanged and not is_discovered:
                # Don't bother reporting unchanged links for previously discovered books
                unchanged_at_library.append((book, linksets_str(unchanged_idlinks), is_discovered))
                
            if (any_added or any_removed or any_other_change):
                if is_discovered:
                    changed_discovered_books.append(book)
                else:
                    changed_books.append(book)


        job_id, books, discovered_books, search_errors, search_warnings, search_summaries = result
        
        # all paths out of search completion must decrement this when done!
        self.num_job_completions_in_progress += 1
        
        changed_books = []
        changed_discovered_books = []

        # Digest and summarize the results by category. The same book may be in multiple categories.
        # Each of these lists holds (book, message, discovered-t/f) tuples.
        discovered_at_library = []
        added_to_library = []
        removed_from_library = []
        unchanged_at_library = []
        obtainable_changed = []
        et_status_changed = []
        et_asin_changed = []
        book_inventory = []

        for book in discovered_books:
            if book.is_newly_discovered:
                discovered_at_library.append((book, book.links_str(self.config), True))
                changed_discovered_books.append(book)
            else:
                report_book_changes(book, True)

        for book in books:
            if not (book.orig_odid or book.odid or
                    book.orig_odrid or book.odrid or
                    book.orig_odpid or book.odpid or
                    book.orig_odnid or book.odnid or
                    book.orig_obtainable or book.obtainable or
                    (book.orig_et_status is not None) or (book.et_status is not None) or
                    book.orig_et_asin or book.et_asin or
                    book.inventory):
                unchanged_at_library.append((book, 'No links', False))    # No links, old or new
            else:
                report_book_changes(book, False)
                
                
        # Show summary of books by category
        main_msg = 'Search complete'
        message = []
        details = []
        self.key_used = False
        
        if search_summaries:
            main_msg = search_summaries[0]
            
            if len(books) == 0 and len(discovered_books) == 0:
                details.extend(search_summaries)
            
        if search_errors:
            message.append('%s in searching' % value_unit(len(search_errors), 'error'))
            add_optional_key()
            details.append('<span style="color:red">---------------- ERRORS WHILE SEARCHING  ----------------</span>')
            details.extend(search_errors)

        if search_warnings:
            message.append('%s in searching' % value_unit(len(search_warnings), 'warning'))
            if self.config.report_warnings:
                add_optional_key()
                details.append('<span style="color:blue">--------------- WARNINGS WHILE SEARCHING  ---------------</span>')
                details.extend(search_warnings)
            
        if search_errors or search_warnings:
            message.append('(click "View log" for details)')

        if discovered_at_library:
            message.append('%s' % value_unit(len(discovered_at_library), 'discovered book'))
            add_optional_key()
            details.append('<span style="color:green">-------- DISCOVERED BOOKS (MISSING FROM CALIBRE) --------')
            details.append('(Choose "Manage discovered books" from the plugin menu to view these.)</span>')
            add_book_details(details, discovered_at_library)

        if added_to_library:
            message.append('%s to be added' % value_unit(len(added_to_library), 'book/library link'))
            add_optional_key()
            details.append('<span style="color:green">------ NEWLY FOUND BOOK/LIBRARY LINKS (TO BE ADDED) ------</span>')
            add_book_details(details, added_to_library)

        if removed_from_library:
            message.append('%s to be removed' % value_unit(len(removed_from_library), 'book/library link'))
            add_optional_key()
            details.append('<span style="color:green">------- REMOVED BOOK/LIBRARY LINKS (TO BE UNLINKED) ------</span>')
            add_book_details(details, removed_from_library)

        if obtainable_changed:
            message.append('%s to current availability' % value_unit(len(obtainable_changed), 'change'))
            add_optional_key()
            details.append('<span style="color:green">------------- CHANGES TO CURRENT AVAILABILITY -------------</span>')
            add_book_details(details, obtainable_changed)

        if et_asin_changed:
            message.append('%s to ET ASIN' % value_unit(len(et_asin_changed), 'change'))
            add_optional_key()
            details.append('<span style="color:green">----------- CHANGES TO ENHANCED TYPESETTING ASIN -----------</span>')
            add_book_details(details, et_asin_changed)

        if et_status_changed:
            message.append('%s to ET status' % value_unit(len(et_status_changed), 'change'))
            add_optional_key()
            details.append('<span style="color:green">---------- CHANGES TO ENHANCED TYPESETTING STATUS ----------</span>')
            add_book_details(details, et_status_changed)

        if book_inventory:
            message.append(value_unit(len(book_inventory), 'EPUB inventory'))
            add_optional_key()
            details.append('<span style="color:green">------------------------ INVENTORY ------------------------</span>')
            add_book_details(details, book_inventory)

        if unchanged_at_library:
            message.append('%s unchanged' % value_unit(len(unchanged_at_library), 'book/library link'))
            if self.config.report_unchanged:
                add_optional_key()
                details.append('<span style="color:green">-------------------- UNCHANGED LINKS ---------------------</span>')
                add_book_details(details, unchanged_at_library)

        if not message:
            message.append('No links.')

        message_str = self.name + ' found: ' + '. '.join(message)
        
        html_details = '\n\n'.join(details)
        details_str = html_to_text(html_details)
        full_log = '\n\n<table bgcolor="darkBlue" width="80%" height="4" align="center"><tr><td></td></tr></table>\n\n'.join(
                filter(None, [message_str, html_details, log]))
                
        log_msg = self.name + ' Log'
        main_msg = self.name + ': ' + main_msg
        
        proceed_param = (job_id, changed_books, changed_discovered_books)

        geom_pref = '%s:%s' % (self.type, self.name)    # Plugin-specific name for saving dialog box geometry
        
        # print('show_det=' + unicode(self.config.show_details_at_completion))
        
        if self.config.disable_job_completion_dlgs:
            self._proceed_with_update(proceed_param)
                
        elif changed_books or changed_discovered_books:
            message_str += '<p>Proceed with updating?'
            
            if changed_discovered_books:
                try:
                    self.gui.proceed_question(
                        self._proceed_with_update, proceed_param, full_log,
                        log_msg, main_msg, message_str,
                        det_msg=details_str, show_copy_button=True,
                        action_callback=self._proceed_with_update_no_discovered,
                        action_label='Non-* only', action_icon=get_icon('ok.png'), focus_action=False,
                        show_det=self.config.show_details_at_completion,
                        cancel_callback=self._proceed_do_nothing, icon=self.main_icon)
                except:
                    self.gui.proceed_question(
                        self._proceed_with_update, proceed_param, full_log,
                        log_msg, main_msg, message_str,
                        det_msg=details_str, show_copy_button=True,
                        action_callback=self._proceed_with_update_no_discovered,
                        action_label='Non-* only', action_icon=get_icon('ok.png'), focus_action=False,
                        show_det=self.config.show_details_at_completion, geom_pref=geom_pref,
                        cancel_callback=self._proceed_do_nothing)
            else:
                try:
                    self.gui.proceed_question(
                        self._proceed_with_update, proceed_param, full_log,
                        log_msg, main_msg, message_str,
                        det_msg=details_str, show_copy_button=True,
                        show_det=self.config.show_details_at_completion,
                        cancel_callback=self._proceed_do_nothing, icon=self.main_icon)
                except:
                    self.gui.proceed_question(
                        self._proceed_with_update, proceed_param, full_log,
                        log_msg, main_msg, message_str,
                        det_msg=details_str, show_copy_button=True,
                        show_det=self.config.show_details_at_completion, geom_pref=geom_pref,
                        cancel_callback=self._proceed_do_nothing)

        else:
            message_str += '<p>No changes will be made.'
            
            try:
                self.gui.proceed_question(
                    self._proceed_do_nothing, proceed_param, full_log, log_msg, main_msg, message_str,
                    det_msg=details_str, show_copy_button=True,
                    show_det=self.config.show_details_at_completion, show_ok=True,
                    cancel_callback=self._proceed_do_nothing,
                    icon=self.main_icon)
            except:
                self.gui.proceed_question(
                    self._proceed_do_nothing, proceed_param, full_log, log_msg, main_msg, message_str,
                    det_msg=details_str, show_copy_button=True,
                    show_det=self.config.show_details_at_completion, show_ok=True, geom_pref=geom_pref,
                    cancel_callback=self._proceed_do_nothing)

    def _proceed_do_nothing(self, proceed_param):
        self._proceed_with_update((proceed_param[0], [], []), save_discovered_books=False)

    def _proceed_with_update_no_discovered(self, proceed_param):
        self._proceed_with_update(proceed_param, save_discovered_books=False)

    def _proceed_with_update(self, proceed_param, save_discovered_books=True):
        # any uncaught exceptions will prevent further proceed question dialogs, so deal with everything here
        try:
            job_id, changed_books, changed_discovered_books = proceed_param

            if changed_discovered_books and save_discovered_books:
                new_books_discovered(self.gui.current_db, changed_discovered_books, self.gui, self.config)

            if changed_books:
                db = self.gui.current_db
                modified = []

                for book in changed_books:
                    if not db.new_api.has_id(book.id):
                        modified.append('%s (Removed)' % unicode(book))
                        book.id = None
                    else:
                        db_last_modified = db.new_api.get_proxy_metadata(book.id).last_modified
                        if db_last_modified > book.last_modified:
                            modified.append('%s (Modified)' % unicode(book))
                    
                apply_changes = True
                if modified and not self.config.disable_job_completion_dlgs:
                    if not question_dialog(
                            self.gui, 'Some books changed',
                            '<p>The metadata for some books in your library has changed since you started the search. If you'
                            ' proceed, some of those changes may be overwritten.<p>Click "Show details" to see the list of'
                            ' changed books.<p>Do you want to proceed?',
                            det_msg='\n'.join(sorted(modified))):
                        apply_changes = False

                if apply_changes:
                    # Apply the changes to the database while displaying a progress dialog
                    progress = QProgressDialog('Applying changes to book metadata', 'Cancel', 0, len(changed_books), self.gui)
                    progress.setWindowTitle(self.name)
                    progress.setWindowFlags(progress.windowFlags()&(~Qt.WindowContextHelpButtonHint))
                    progress.setMinimumWidth(400)
                    progress.setMinimumDuration(30000 if self.config.disable_job_completion_dlgs else 500)   # Show progress quickly
                    progress.setModal(True)
                    progress.setValue(0)

                    updated_ids = []
                    marked_ids = {}

                    for i, book in enumerate(changed_books):
                        if book.id is not None:
                            updated = False
                            
                            if book.links_have_changed() or (book.et_asin != book.orig_et_asin):
                                mi = db.new_api.get_metadata(book.id, get_user_categories=False) # get existing identifiers for book
                                
                                mi.set_identifier(IDENT_AVAILABLE_LINK, book.odid)
                                mi.set_identifier(IDENT_RECOMMENDABLE_LINK, book.odrid)
                                mi.set_identifier(IDENT_PURCHASABLE_LINK, book.odpid)
                                
                                if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                                    mi.set_identifier(IDENT_NEEDED_LINK, book.odnid)
                                
                                if self.config.update_amazon_idents:
                                    for odlink in (ODLinkSet(str=book.odid) | ODLinkSet(str=book.odrid) |
                                                   ODLinkSet(str=book.odpid) | ODLinkSet(str=book.odnid)).amazon().sorted_odlinks():
                                        mi.set_identifier(odlink.amazon_ident(), odlink.book_id)
                                        
                                if book.et_asin != book.orig_et_asin:
                                    mi.set_identifier(IDENT_AMAZON, book.et_asin)
                                        
                                db.new_api.set_field('identifiers', {book.id: mi.identifiers})  # replace all identifiers
                                updated = True

                            if book.obtainable != book.orig_obtainable:
                                db.new_api.set_field('#' + FIELD_OBTAINABLE, {book.id: book.obtainable})
                                updated = True
                            
                            if book.et_status != book.orig_et_status and self.lib_has_custom_column(FIELD_ET_STATUS):
                                db.new_api.set_field('#' + FIELD_ET_STATUS, {book.id: book.et_status})
                                updated = True
                                
                            if book.et_status is True and self.lib_has_custom_column(FIELD_ET_DATE):
                                orig_et_date = db.new_api.field_for('#' + FIELD_ET_DATE, book.id, default_value='')
                                if not orig_et_date:
                                    db.new_api.set_field('#' + FIELD_ET_DATE, {book.id: now()})
                                    updated = True
                                
                            
                            if updated:
                                updated_ids.append(book.id)

                                if self.config.mark_changed_books:
                                    marked_ids[book.id] = 'updated'

                        progress.setValue(i + 1)    # calls QApplication.processEvents() for the modal dialog
                        
                    progress.reset()

                    if marked_ids:
                        db.set_marked_ids(marked_ids)  # Mark results

                    # Update the gui view to reflect changes to the database
                    if updated_ids:
                        self.gui.library_view.model().refresh_ids(updated_ids)
                        current = self.gui.library_view.currentIndex()
                        self.gui.library_view.model().current_changed(current, current)
                        self.gui.tags_view.recount()
                    
            # must be done at end of all job completions!
            self.deregister_search_job(job_id)
            
        except Exception as e:
            traceback.print_exc()
            error_dialog(self.gui, 'Unhandled exception', exception_str(e), det_msg=traceback.format_exc(), show=True)

        # must always be done when finished with the proceed question handling
        self.num_job_completions_in_progress -= 1
        
        
    def register_search_job(self, book_ids):
        job_id = self.next_job_id
        self.next_job_id += 1
        
        self.book_ids_being_searched[job_id] = set(book_ids)   # save the list
        self.last_job_start_time = time.time()
        # print('Started job %d with %d books' % (job_id, len(self.book_ids_being_searched[job_id])))
        
        return job_id
        
    def deregister_search_job(self, job_id):
        if job_id is not None:
            # print('Completed job %d' % job_id)
            del self.book_ids_being_searched[job_id]
        
    def check_search_job_books(self,book_ids):
        all_job_books_ids = set()
        
        # clean up the record of book ids currently being searched if there are no jobs in progress
        # could be some left over from job failures or manually cancelled jobs
        if (self.num_job_completions_in_progress == 0 and (not self.gui.job_manager.has_jobs()) and
                self.last_job_start_time and (time.time() - self.last_job_start_time > 10)):
            if len(self.book_ids_being_searched) > 0:
                print('%s did not complete properly!, jobs: %s' % (
                    value_unit(len(self.book_ids_being_searched) > 0, 'search job'),
                    ', '.join([unicode(id) for id in self.book_ids_being_searched.keys()])))
                self.book_ids_being_searched = {}
        
        for job_book_ids in self.book_ids_being_searched.values():
            all_job_books_ids.update(job_book_ids)
            
        return len(set(book_ids) & all_job_books_ids)
        
        
    def gutenberg_index(self):
        if not self.config.disable_confirm_dlgs:
            q = '<p>Download Project Gutenberg catalog and build search index?<p>(This will run as a background job.)'

            if not question_dialog(self.gui, 'Start download', q):
                return
                
        self.gui.status_bar.show_message('Building Project Gutenberg search index')

        # Start a background job
        start_search_jobs(self, self.config, [], [], True, [], 'build_gutenberg_index', None, None,
                Dispatcher(self._search_complete))
        

    def handle_discovered_books(self):
        db = self.gui.current_db
        if not have_discovered_books(db):
            error_dialog(
                self.gui, 'No Discovered Books',
                'No additional books were found using "Search for books by selected authors".',
                show=True)
            return

        abd = DiscoveredBooksDialog(plugin_action=self, db=db, parent=self.gui, config=self.config)
        if not abd.result():
            return

        # DiscoveredBooksDialog accepted.
        added_books, updated_books = abd.get_results()
        new_ids = []
        updated_ids = []
        marked_ids = {}

        if added_books:
            # Add selected books to the database while displaying a progress dialog
            progress = QProgressDialog('Adding discovered books', 'Cancel', 0, len(added_books), self.gui)
            progress.setWindowTitle(self.name)
            progress.setWindowFlags(progress.windowFlags()&(~Qt.WindowContextHelpButtonHint))
            progress.setMinimumWidth(400)
            progress.setMinimumDuration(500)   # Show progress quickly
            progress.setModal(True)
            progress.setValue(0)
            
            new_books = []

            for i, dbook in enumerate(added_books):
                book = CalibreBook(from_book=dbook)
                mi = Metadata(book.title, book.authors)

                if book.publisher:
                    mi.publisher = book.publisher

                if book.pubdate:
                    mi.pubdate = book.pubdate

                if book.isbn:
                    mi.set_identifier(IDENT_ISBN, book.isbn)

                if book.odid:
                    mi.set_identifier(IDENT_AVAILABLE_LINK, book.odid)
                    
                    if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                        book.odnid = unicode(ODLinkSet(str=book.odid).ebooks())     # assume all new e-book links needed
                        book.remove_outdated_links()
                        mi.set_identifier(IDENT_NEEDED_LINK, book.odnid)

                if book.odrid:
                    mi.set_identifier(IDENT_RECOMMENDABLE_LINK, book.odrid)

                if book.odpid:
                    mi.set_identifier(IDENT_PURCHASABLE_LINK, book.odpid)

                if self.config.update_amazon_idents:
                    for odlink in (ODLinkSet(str=book.odid) | ODLinkSet(str=book.odrid) |
                                   ODLinkSet(str=book.odpid)).amazon().sorted_odlinks():
                        mi.set_identifier(odlink.amazon_ident(), odlink.book_id)

                if book.series:
                    mi.series = book.series
                    mi.series_index = book.series_index

                new_books.append((mi, {}))   # create a book with no formats (empty book)
                
                progress.setValue(i + 1)    # calls QApplication.processEvents() for the modal dialog


            new_ids, duplicates_ids = db.new_api.add_books(new_books) # add the new books to calibre

            if self.config.mark_changed_books:
                for id in new_ids:
                    marked_ids[id] = 'updated'
                
            progress.reset()

        if updated_books:
            # Update selected books in the database while displaying a progress dialog
            progress = QProgressDialog('Updating discovered books', 'Cancel', 0, len(updated_books), self.gui)
            progress.setWindowTitle(self.name)
            progress.setWindowFlags(progress.windowFlags()&(~Qt.WindowContextHelpButtonHint))
            progress.setMinimumWidth(400)
            progress.setMinimumDuration(500)   # Show progress quickly
            progress.setModal(True)
            progress.setValue(0)

            for i, dbook in enumerate(updated_books):
                book = CalibreBook(from_book=dbook)
                mi = db.new_api.get_metadata(book.id, get_user_categories=False)

                book.orig_odid = mi.identifiers.get(IDENT_AVAILABLE_LINK, '')
                book.orig_odrid = mi.identifiers.get(IDENT_RECOMMENDABLE_LINK, '')
                book.orig_odpid = mi.identifiers.get(IDENT_PURCHASABLE_LINK, '')
                
                if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                    book.orig_odnid = mi.identifiers.get(IDENT_NEEDED_LINK, '')
                    book.odnid = book.odid
                    
                book.preserve_all_links()   # only add links, don't remove
                book.remove_outdated_links()

                mi.set_identifier(IDENT_AVAILABLE_LINK, book.odid)
                mi.set_identifier(IDENT_RECOMMENDABLE_LINK, book.odrid)
                mi.set_identifier(IDENT_PURCHASABLE_LINK, book.odpid)
                
                if tweaks.get(TWEAK_TRACK_NEEDED_BOOKS, False):
                    mi.set_identifier(IDENT_NEEDED_LINK, book.odnid)
                
                if self.config.update_amazon_idents:
                    for odlink in (ODLinkSet(str=book.odid) | ODLinkSet(str=book.odrid) |
                                   ODLinkSet(str=book.odpid)).amazon().sorted_odlinks():
                        mi.set_identifier(odlink.amazon_ident(), odlink.book_id)

                if book.isbn and IDENT_ISBN not in mi.identifiers:
                    mi.set_identifier(IDENT_ISBN, book.isbn)

                if book.publisher and not mi.publisher:
                    mi.publisher = book.publisher

                if book.pubdate and (mi.pubdate is None or mi.pubdate <= UNDEFINED_DATE):
                    mi.pubdate = book.pubdate

                if book.series and not mi.series:
                    mi.series = book.series
                    mi.series_index = book.series_index
                  
                # Could probably call set_metadata with force_changes=True to force replacement of empty
                # identifiers, but I am concerned that might erase other fields that should be left untouched.
                db.new_api.set_metadata(book.id, mi, set_title=False, set_authors=False) # replace those fields changed to non-blank
                db.new_api.set_field('identifiers', {book.id: mi.identifiers})  # replace all identifiers to allow removal

                updated_ids.append(book.id)

                if self.config.mark_changed_books:
                    marked_ids[book.id] = 'updated'

                progress.setValue(i + 1)    # calls QApplication.processEvents() for the modal dialog
                
            progress.reset()

        if marked_ids:
            db.set_marked_ids(marked_ids)  # Mark results

        if new_ids:
            db.data.books_added(new_ids)  # update database view to add new books
            self.gui.library_view.model().books_added(len(new_ids))  # add rows to table for new books
            self.gui.library_view.select_rows(new_ids)  # select for user to see

        if updated_ids:
            self.gui.library_view.model().refresh_ids(updated_ids)
            current = self.gui.library_view.currentIndex()
            self.gui.library_view.model().current_changed(current, current)

        if is_pyqt4:
            self.gui.db_images.reset()  # update cover grid view (calibre 1.0)
        else:
            self.gui.db_images.beginResetModel() # update cover grid view (calibre 2.0)
            self.gui.db_images.endResetModel()

        self.gui.tags_view.recount()    # update tag viewer panel


    def add_menu_action(self, menu, text, image=None, tooltip=None, triggered=None, enabled=True, submenu=None):
        # Minimal version without keyboard shortcuts, etc.

        ac = menu.addAction(text)

        if tooltip:
            ac.setToolTip(tooltip)
            ac.setStatusTip(tooltip)    # This is the only one actually used
            ac.setWhatsThis(tooltip)

        if triggered:
            ac.triggered.connect(triggered)

        if image:
            ac.setIcon(get_icon(image))

        ac.setEnabled(enabled)

        if submenu:
            ac.setMenu(submenu)

        return ac
        

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def show_help(self):
        self.show_text(self.help_text, 'Help', contents=True)

    def show_text(self, text, title, confirm=False, contents=False):
        dialog = QDialog(self.gui)
        layout = QVBoxLayout()
        
        text_browser = QTextBrowser()
        text_browser.setOpenExternalLinks(True)
        
        text_browser.setHtml(text)
        layout.addWidget(text_browser)
        
        button_layout = QHBoxLayout()
        
        if contents:
            top_button = QPushButton('Top (Contents)')
            top_button.setIcon(QIcon(I('arrow-up.png')))
            top_button.setToolTip('Scroll to top and show table of contents links')
            top_button.clicked.connect(partial(text_browser.scrollToAnchor, "contents"))
            button_layout.addWidget(top_button)
            
        button_layout.addStretch()
        

        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(dialog.reject)
        
        if confirm:
            button_box.addButton(QDialogButtonBox.Ok)
            button_box.accepted.connect(dialog.accept)

        button_layout.addWidget(button_box)
        layout.addLayout(button_layout)

        dialog.setLayout(layout)
        dialog.setModal(True)
        dialog.resize(QSize(700, 500))
        dialog.setWindowTitle('%s %s' % (self.name, title))
        dialog.setWindowIcon(QIcon(I('help.png')))
        dialog.setWindowFlags(dialog.windowFlags()&(~Qt.WindowContextHelpButtonHint))
        dialog.exec_()
        
        return dialog.result() == QDialog.Accepted
        
    def have_new_release_notes(self):
        if tuple(plugin_config[ReleaseNoteVersion]) == DEFAULT_RELEASE_NOTE_VERSION:
            # Plugin version last installed missing. Either new install or previous version was from
            # before release note tracking.
            if self.config.any_enabled():
                plugin_config[ReleaseNoteVersion] = (1, 0, 0)  # plugin installed prior to release note tracking
            else:
                plugin_config[ReleaseNoteVersion] = ActionOverdriveLink.version  # new install of plugin
            
        return LAST_RELEASE_NOTE_VERSION > tuple(plugin_config[ReleaseNoteVersion])
        
        
    def display_new_release_notes(self):
        display_notes = []
        
        for release_version, release_note in RELEASE_NOTES:
            if release_version > tuple(plugin_config[ReleaseNoteVersion]):
                display_notes.append(release_note)
        
        if self.show_text('<p><p>'.join(display_notes), 'Release Notes', confirm=True):
            plugin_config[ReleaseNoteVersion] = ActionOverdriveLink.version  # prevent showing again

