#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

name = 'GenericIdentifiers'
description = _('Define generic identifiers for any site.')
author = 'David Forrester'

import copy
from functools import partial
try:
    from PyQt5 import Qt as QtGui
    from PyQt5.Qt import (QTableWidgetItem, QVBoxLayout, Qt, QGroupBox, QTableWidget,
                          QCheckBox, QAbstractItemView, QHBoxLayout, QIcon,
                          QInputDialog)
except ImportError:
    from PyQt4 import QtGui
    from PyQt4.Qt import (QTableWidgetItem, QVBoxLayout, Qt, QGroupBox, QTableWidget,
                          QCheckBox, QAbstractItemView, QHBoxLayout, QIcon,
                          QInputDialog)
from calibre.gui2 import get_current_db, question_dialog, error_dialog
from calibre.gui2.complete2 import EditWithComplete
from calibre.gui2.metadata.config import ConfigWidget as DefaultConfigWidget
from calibre.utils.config import JSONConfig
from calibre.devices.usbms.driver import debug_print

from calibre_plugins.generic_identifiers.common_utils import ReadOnlyTableWidgetItem

STORE_NAME = 'Options'
KEY_IDENTIFIERS = 'indentifierMappings'

DEFAULT_IDENTIFIERS = {
#                'smashwords': {'displayName': 'Smashwords', 'url':"https://www.smashwords.com/books/view/%s"}
                'mobileread': {'displayName': 'MobileRead', 'url':"http://www.mobileread.com/forums/showthread.php?t=%s"}
                }

DEFAULT_STORE_VALUES = {
    KEY_IDENTIFIERS: copy.deepcopy(DEFAULT_IDENTIFIERS)
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/GenericIdentifiers')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES


class IndentifierMappingsTableWidget(QTableWidget):
    def __init__(self, parent, all_tags):
        QTableWidget.__init__(self, parent)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tags_values = all_tags

    def populate_table(self, indentifier_mappings):
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(indentifier_mappings))
        header_labels = [_('Identifier'), _('Display Name'), _('URL')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(24)
        self.horizontalHeader().setStretchLastSection(True)

        for row, indentifier in enumerate(sorted(indentifier_mappings.keys(), key=lambda s: (s.lower(), s))):
            self.populate_table_row(row, indentifier, indentifier_mappings[indentifier])

        self.resizeColumnToContents(0)
        self.set_minimum_column_width(0, 200)
        self.setSortingEnabled(False)
        if len(indentifier_mappings) > 0:
            self.selectRow(0)

    def set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, indentifier, indentifier_map):
        self.setItem(row, 0, ReadOnlyTableWidgetItem(indentifier))
        debug_print("populate_table_row: indentifier_map=", indentifier_map)
        self.setItem(row, 1, QTableWidgetItem(indentifier_map.get('displayName', '')))
        self.setItem(row, 2, QTableWidgetItem(indentifier_map.get('url', '')))

    def create_tags_edit(self, value, row):
        tags_edit = EditWithComplete(self)
        tags_edit.set_add_separator(False)
        tags_edit.update_items_cache(self.tags_values)
        tags_edit.setText(value)
#        tags_edit.editingFinished.connect(partial(self.tags_editing_finished, row, tags_edit))
        return tags_edit

    def tags_editing_finished(self, row, tags_edit):
        # Update our underlying widget for sorting
        self.item(row, 1).setText(tags_edit.text())

    def get_data(self):
        indentifier_mappings = {}
        for row in range(self.rowCount()):
            indentifier = unicode(self.item(row, 0).text()).strip()
            indentifier_name = unicode(self.item(row, 1).text()).strip()
            url = unicode(self.item(row, 2).text()).strip()
            indentifier_mappings[indentifier] = {'url': url, 'displayName': indentifier_name}
        return indentifier_mappings

    def select_indentifier(self, indentifier_name):
        for row in range(self.rowCount()):
            if unicode(self.item(row, 0).text()) == indentifier_name:
                self.setCurrentCell(row, 1)
                return

    def get_selected_indentifier(self):
        if self.currentRow() >= 0:
            return unicode(self.item(self.currentRow(), 0).text())


class ConfigWidget(DefaultConfigWidget):

    def __init__(self, plugin):
        DefaultConfigWidget.__init__(self, plugin)
        c = plugin_prefs[STORE_NAME]
        all_tags = get_current_db().all_tags()

        self.gb.setMaximumHeight(80)
        indentifier_group_box = QGroupBox('Defined Identifiers', self)
        self.l.addWidget(indentifier_group_box, self.l.rowCount(), 0, 1, 2)
        indentifier_group_box_layout = QVBoxLayout()
        indentifier_group_box.setLayout(indentifier_group_box_layout)

        tags_layout = QHBoxLayout()
        indentifier_group_box_layout.addLayout(tags_layout)

        self.edit_table = IndentifierMappingsTableWidget(self, all_tags)
        tags_layout.addWidget(self.edit_table)
        button_layout = QVBoxLayout()
        tags_layout.addLayout(button_layout)
        add_mapping_button = QtGui.QToolButton(self)
        add_mapping_button.setToolTip(_('Add Identifier'))
        add_mapping_button.setIcon(QIcon(I('plus.png')))
        add_mapping_button.clicked.connect(self.add_mapping)
        button_layout.addWidget(add_mapping_button)
        spacerItem1 = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)
        remove_mapping_button = QtGui.QToolButton(self)
        remove_mapping_button.setToolTip(_('Delete Identifier'))
        remove_mapping_button.setIcon(QIcon(I('minus.png')))
        remove_mapping_button.clicked.connect(self.delete_mapping)
        button_layout.addWidget(remove_mapping_button)
        spacerItem3 = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem3)
        rename_indentifier_button = QtGui.QToolButton(self)
        rename_indentifier_button.setToolTip(_('Rename Identifier'))
        rename_indentifier_button.setIcon(QIcon(I('edit-undo.png')))
        rename_indentifier_button.clicked.connect(self.rename_indentifier)
        button_layout.addWidget(rename_indentifier_button)
        spacerItem2 = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)
        reset_defaults_button = QtGui.QToolButton(self)
        reset_defaults_button.setToolTip('Reset to plugin default mappings')
        reset_defaults_button.setIcon(QIcon(I('clear_left.png')))
        reset_defaults_button.clicked.connect(self.reset_to_defaults)
        button_layout.addWidget(reset_defaults_button)
        spacerItem2 = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)
        self.l.setRowStretch(self.l.rowCount()-1, 2)

        self.edit_table.populate_table(c[KEY_IDENTIFIERS])

    def commit(self):
        DefaultConfigWidget.commit(self)
        new_prefs = {}
        new_prefs[KEY_IDENTIFIERS] = self.edit_table.get_data()
        plugin_prefs[STORE_NAME] = new_prefs

    def add_mapping(self):
        new_indentifier, ok = QInputDialog.getText(self, _('Add identifier'),
                    _('Enter the new identifier:'), text='')
        if not ok:
            # Operation cancelled
            return
        new_indentifier = unicode(new_indentifier).strip()
        if not new_indentifier:
            return
        # Verify it does not clash with any other mappings in the list
        data = self.edit_table.get_data()
        for indentifier in data.keys():
            if indentifier.lower() == new_indentifier.lower():
                return error_dialog(self, _('Add Failed'), _('The identifier already exists'), show=True)
        data[new_indentifier] = {}
        self.edit_table.populate_table(data)
        self.edit_table.select_indentifier(new_indentifier)

    def delete_mapping(self):
        if not self.edit_table.selectionModel().hasSelection():
            return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected identifier?'),
                show_copy_button=False):
            return
        for row in reversed(sorted(self.edit_table.selectionModel().selectedRows())):
            self.edit_table.removeRow(row.row())

    def rename_indentifier(self):
        selected_indentifier = self.edit_table.get_selected_indentifier()
        if not selected_indentifier:
            return
        new_indentifier, ok = QInputDialog.getText(self, _('Rename identifier'),
                    _('Enter the new identifier:'), text=selected_indentifier)
        if not ok:
            # Operation cancelled
            return
        new_indentifier = unicode(new_indentifier).strip()
        if not new_indentifier or new_indentifier == selected_indentifier:
            return
        data = self.edit_table.get_data()
        if new_indentifier.lower() != selected_indentifier.lower():
            # Verify it does not clash with any other mappings in the list
            for indentifier in data.keys():
                if indentifier.lower() == new_indentifier.lower():
                    return error_dialog(self, _('Rename Failed'), _('An identifier with the same name already exists'), show=True)
        data[new_indentifier] = data[selected_indentifier]
        del data[selected_indentifier]
        self.edit_table.populate_table(data)
        self.edit_table.select_indentifier(new_indentifier)

    def reset_to_defaults(self):
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to reset to the plugin default identifiers?'),
                show_copy_button=False):
            return
        self.edit_table.populate_table(DEFAULT_IDENTIFIERS)

