# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

from __future__ import unicode_literals, division, absolute_import, print_function

import os
import sys
import zipfile
from contextlib import contextmanager


from compatibility_utils import PY2, unicode_str
from unipath import pathof
from epub_utils import unzip_epub_to_dir, epub_zip_up_book_contents
from utilities import expanduser, file_open
from epubstuff import ePubReader, fix_opf

if PY2:
    from Tkinter import Tk
    import tkFileDialog as tkinter_filedialog
    import tkMessageBox as tkinter_msgbox
else:
    from tkinter import Tk
    import tkinter.filedialog as tkinter_filedialog
    import tkinter.messagebox as tkinter_msgbox


prefs = {}

@contextmanager
def temp_epub_handle(prefix='OEBImport', suffix='.epub', delete=True):
    import tempfile
    fd, temp_file = tempfile.mkstemp(prefix=prefix, suffix=suffix)
    try:
        yield temp_file
    finally:
        if delete:
            os.close(fd)
            os.remove(temp_file)

@contextmanager
def make_temp_directory():
    import tempfile
    import shutil
    temp_dir = tempfile.mkdtemp()
    yield temp_dir
    shutil.rmtree(temp_dir)

def fileChooser():
    localRoot = Tk()
    localRoot.withdraw()
    file_opt = {}
    file_opt['parent'] = None
    file_opt['title']= 'Select EPUB'
    file_opt['defaultextension'] = '.epub'
    # retrieve the initialdir from JSON prefs
    file_opt['initialdir'] = unicode_str(prefs['use_file_path'], 'utf-8')
    file_opt['multiple'] = False
    file_opt['filetypes'] = [('EPUB', ('.epub'))]
    localRoot.quit()
    return tkinter_filedialog.askopenfilename(**file_opt)

def update_msgbox(title, msg):
    localRoot = Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Arial -12')
    localRoot.quit()
    return tkinter_msgbox.showinfo(title, msg)

def run(bk):
    global prefs
    prefs = bk.getPrefs()
    diddle_opf = False

    # set default preference values
    if 'use_file_path' not in prefs:
        prefs['use_file_path'] = expanduser('~')

    inpath = fileChooser()
    if inpath == '' or not os.path.exists(inpath):
        print('No input file selected!')
        bk.savePrefs(prefs)
        return 0

    # Save last directory accessed to prefs
    prefs['use_file_path'] = pathof(os.path.dirname(inpath))
    # Save prefs to json
    bk.savePrefs(prefs)

    epub = ePubReader(inpath)
    if not epub.hasValidMimeType() or not epub.hasValidContainer():
        print('Not a valid container.')
        return -1
    if epub.opfPath is None:
        print('No opf file found.')
        return -1
    if epub.has_xml_decl() is None:
        print('Could not read opf file.')
        return -1
    if not epub.has_xml_decl():
        diddle_opf = True
    if not epub.get_package_ver():
        print('Can\'t get package version.')
        return -1
    if epub.get_package_ver() > 1:
        if not diddle_opf:
            print('Package version is not "1.0" and xml declaration present.\nNothing to do.')
            return -1

    rel_opf_path = epub.opfPath

    with make_temp_directory() as temp_dir:
        unzip_epub_to_dir(inpath, temp_dir)
        abs_opf_path = os.path.join(temp_dir, rel_opf_path)

        if not fix_opf(abs_opf_path, diddle_opf):
            print('Couldn\'t fix opf file.')
            return -1
        with temp_epub_handle() as new_epub:
            epub_zip_up_book_contents(temp_dir, new_epub)

            with file_open(new_epub,'rb')as fp:
                data = fp.read()
            bk.addotherfile('dummy.epub', data)

    return 0

def main():
    print ('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
