# -*- coding: utf-8 -*-
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import unicode_literals, division, absolute_import, print_function

import os
import re
from zipfile import ZipFile

try:
    import xml.etree.cElementTree as ET
except ImportError:
    import xml.etree.ElementTree as ET

from utilities import file_open, find_output_encoding
from compatibility_utils import unicode_str

DECL = '<?xml version="1.0" encoding="utf-8"?>'
MIMETYPE = b'application/epub+zip'
MEDIATYPE = 'application/xhtml+xml'
NSMAP = {'opf': '{http://www.idpf.org/2007/opf}',
                 'con': '{urn:oasis:names:tc:opendocument:xmlns:container}'}


def fix_opf(opffile, add_xml_decl=False):
    with file_open(opffile, 'r', encoding='utf-8') as fp:
        newopf = ''
        if add_xml_decl:
            newopf += DECL
        for line in fp:
            if line.lstrip().startswith('<item'):
                if line.find('text/x-oeb1-document'):
                    line = line.replace('text/x-oeb1-document', 'application/xhtml+xml')
                if line.find('text/html'):
                    line = line.replace('text/html', 'application/xhtml+xml')
            if line.lstrip().startswith('<package'):
                if line.find('version="1.0"'):
                    line = line.replace('version="1.0"', 'version="2.0"')
            newopf += line
    try:
        file_open(opffile,'wb').write(newopf.encode('utf-8'))
    except:
        return False
    return True

class ePubReader:
    '''Class to check the (basic) integrity of an ePub container and to gather/read
    the files that are manifested in the OPF file as media-type application/xhtml+xml'''
    def __init__(self, inzip):
        self.inzip = ZipFile(inzip,'r')
        self.files = []
        self.opfPath = None

    def hasValidMimeType(self):
        '''Try to read the mimetype file'''
        for zinfo in self.inzip.infolist():
            if zinfo.filename == 'mimetype':
                try:
                    if self.inzip.read(zinfo.filename) == MIMETYPE:
                        return True
                except:
                    break
        return False

    def hasValidContainer(self):
        '''Try to read the container file to find the path to the OPF file'''
        for zinfo in self.inzip.infolist():
            if zinfo.filename == 'META-INF/container.xml':
                try:
                    containerData = self.inzip.read(zinfo.filename)
                    xmltree = ET.fromstring(containerData)
                    expr = './/%srootfiles/%srootfile' % (NSMAP['con'], NSMAP['con'])
                    elem = xmltree.find(expr)
                    opfPath = elem.get('full-path', None)
                    if opfPath is not None:
                        self.opfPath = opfPath
                        return True
                except:
                    break
        return False

    def get_package_ver(self):
        _version_pattern = re.compile(b'<package[^>]*version="(\d)\.\d"')
        try:
            opfData = self.inzip.read(self.opfPath)
        except:
            return 0
        m = _version_pattern.search(opfData)
        if m:
            version = m.group(1)
        else:
            return 0
        return int(version)

    def has_xml_decl(self):
        try:
            opfData = self.inzip.read(self.opfPath)
        except:
            return None
        opfData = opfData.decode('utf-8')
        if opfData.startswith('<?xml'):
            return True
        return False

    def getXhtmlFiles(self):
        '''Iterate the OPF's manifest and get all hrefs of the items with a media-type of application/xhtml+xml'''
        files = []
        try:
            opfData = self.inzip.read(self.opfPath)
        except:
            return [], 'Error reading/finding OPF file.'

        prefix = os.path.split(self.opfPath)[0]
        if prefix != '':
            prefix += '/'
        xmltree = ET.fromstring(opfData)
        manifest_expr = './/%smanifest/%sitem' % (NSMAP['opf'], NSMAP['opf'])
        try: #Iterate the manifest.
            for elem in xmltree.findall(manifest_expr):
                if elem.get('media-type') == MEDIATYPE:
                    '''Collect the files and patch the file-paths to match the paths in the zipfile.infolist()'''
                    files.append(prefix + elem.get('href'))
        except:
            return files, 'Error parsing OPF manifest.'
        if not files:
            return files, 'No items of media-type "application/xhtml+xml" found in OPF\'s manifest.'
        return files, None

    def readXhtmlFile(self, xhtml_file, encoding):
        '''Read/return the contents of the (x)html using proper encoding.'''
        data = self.inzip.read(xhtml_file).decode(encoding)
        return data

    def close(self):
        '''Close the zip (epub) file.'''
        self.inzip.close()
