#!/bin/sh

Notice () {
cat << 'ENDofNotice'
 Copyright (C) 2016 by M. S. Zick
 Licensed under the MIT license, see:
 http://www.opensource.org/licenses/mit-license.php

Usage:
  Argument 1 - from file: typically /mnt/us/rootfs.img
  Argument 2 - to device: typically /dev/mmcblk0p1
  Argument 3 - Optional block size, default: 4096 bytes
  This script will also copy&confirm file-to-file and device-to-file
  without truncation or extension if the destination file exists.
ENDofNotice
  exit 1
}

# Targets the BusyBox v-1.17.1 Ash of the Kindle (5+) Diags system.
# Also tested under Bash (v-4.2.1) and Dash (v-0.5.5.1).

# Utility dd as implemented and used here does return limited error codes 
# but does not distinguish between EOF of input or ENOSPACE on output.
#
# dd outputs:
#   $1 == 0+0 skip beyond eof
#   $1 == 1+0 complete block copied
#   $1 == 0+1 partial block copied
#   $7 == bytes copied in all cases

# Summary:
# While block copy OK -
#   Copy in-block to temporary, copy out=block to (ram) temporary, compare
#   if match, continue to next block
#   no match:
#     copy in-block to out-block (I.E: program flash)
#     copy out-block to (ram) temporary, compare
#     if match: continue to next block
#     no match: report and exit

# Requires the full featured dd from coreutils, next to this script
dd=$(dirname $0)/dd
[ -f $dd ] && [ -x $dd ] || {
    echo "Executable $dd not found"
    echo "The dd utility from coreutils must be located"
    echo "in the same directory as this script."
    exit 1
   }
 
[ $# -eq 2 ] || [ $# -eq 3 ] || Notice
[ -f $1 ] || [ -b $1 ] || Notice
[ -f $2 ] || [ -b $2 ] || Notice
InFile=$1
OutFile=$2

Chk=4096
[ $# -eq 3 ] && Chk=$3 

# Test file names in ram (tmpfs)
TmpIn='/tmp/t1.bin'
TmpOut='/tmp/t2.bin'
FragIn='/tmp/f1.bin'
FragOut='/tmp/f2.bin'

# Initial block copy
Ret1=' '
Ret2=' '
Sze1=0
Sze2=0

cBytes=0    # Bytes in common
BLK=0       # Current block index
SKP=0       # Matching blocks (whole and partial) skipped

# Don't even ask about these
a=' '
b=' '
c=' '

# Trim temp* files to same size - only bytes in common are handled.
# A rather expensive version of "truncate --size= ..." (x2)
TrimIn () {
  local cBytes=$1
  set $($dd if=$TmpIn of=$FragIn oflag=noatime bs=$cBytes count=1 2>&1)
  # Normal message is 15 tokens long - anything less is an error message
  if [ $# -lt 15 ] ; then 
    # Grab last two tokens (words), as in: permission denied
    a='$'$(($#-1)) ; b='$'$(($#)) ; c=$(eval echo $a $b)
    echo "Abort: $c on TmpIn -> FragIn copy"
    return 1
  fi
  mv $FragIn $TmpIn 2>/dev/null
  if [ $? -ne 0 ] ; then
    echo "Abort: move FragIn -> TmpIn failed"
    return 1
  fi
  return 0
}

TrimOut () {
  local cBytes=$1
  set $($dd if=$TmpOut of=$FragOut oflag=noatime bs=$cBytes count=1 2>&1)
  # Normal message is 15 tokens long - anything less is an error message
  if [ $# -lt 15 ] ; then 
    # Grab last two tokens (words), as in: permission denied
    a='$'$(($#-1)) ; b='$'$(($#)) ; c=$(eval echo $a $b)
    echo "Abort: $c on TmpOut -> FragOut copy"
    return 1
  fi
  mv $FragOut $TmpOut 2>/dev/null
  if [ $? -ne 0 ] ; then
    echo "Abort: move FragOut -> TmpOut failed"
    return 1
  fi
  return 0
}

# Find bytes in common - trimming either TmpIn or TmpOut as required
# returns the cBytes value and pass/fail error codes.
FndCom () {
  local Sze1=$1
  local Sze2=$2
  [ $Sze1 -eq 0 ] || [ $Sze2 -eq 0 ] && { echo '0' ; return 1 ; }
  [ $Sze1 -eq $Sze2 ] && { echo $Sze1 ; return 0 ;              }
  [ $Sze1 -gt $Sze2 ] && { echo $Sze2 ; return 0 ;              }
  [ $Sze2 -gt $Sze1 ] && { echo $Sze1 ; return 0 ;              }
  echo 0 ; return 1 # should not be reached
}

# Compare for equal (common) sized blocks
CmpBlk () {
  cmp -s $TmpIn $TmpOut 
  return $?
}

ReadBack () {
  local cBytes=$1      # bytes in common
  local COND
  local BYTES
  set $($dd if=$OutFile of=$TmpOut oflag=noatime bs=$Chk skip=$BLK count=1 2>&1)
  if [ $# -lt 15 ] ; then 
    a='$'$(($#-1)) ; b='$'$(($#)) ; c=$(eval echo $a $b)
    echo "Error: $c on OutFile -> TmpOut copy"
    return 1
  fi
  COND="$1" ; BYTES="$7"
  [ $COND = '0+0' ] && {
    echo "Error: Unexpected status code in ReadBack: $COND, Bytes: $BYTES"
    return 1 # Fail
  }
  if [ $BYTES -gt $cBytes ] ; then
    if TrimOut $cBytes ; then
      CmpBlk ; return $?
    else
      return 1
    fi
  else
    CmpBlk ; return $?
  fi
}

WriteBlock () {
  local cBytes=$1
  local COND
  local BYTES
  set $($dd if=$TmpIn of=$OutFile oflag=noatime conv=notrunc bs=$Chk seek=$BLK count=1 2>&1)
  if [ $# -lt 15 ] ; then 
    a='$'$(($#-1)) ; b='$'$(($#)) ; c=$(eval echo $a $b)
    echo "Error: $c on TmpIn -> OutFile copy"
    return 1
  fi
  COND="$1" ; BYTES="$7"
  [ $COND = '0+0' ] && {
    echo "Error: Unexpeced status code in WriteBlock: $COND, Bytes: $BYTES"
    return 1 # Fail
  }  
  [ $BYTES -ne $cBytes ] && {
    echo "Error: WriteBlock failure; Status code: $COND; Bytes written: $BYTES"
    return 1 # Fail
  }     
  ReadBack $cBytes ; return $?
}

TOT=0       # Total bytes processed (blocks+fragment)
COND=' '
OK='true'
while [ "$OK" = 'true' ] ; do
  # Copy '$BLK' source and destination to ram temporaries
  set $($dd if=$InFile  of=$TmpIn  oflag=noatime bs=$Chk skip=$BLK count=1 2>&1)
  if [ $# -lt 15 ] ; then 
    a='$'$(($#-1)) ; b='$'$(($#)) ; c=$(eval echo $a $b)
    echo "Abort: $c on InFile -> TmpIn copy"
    OK='false'
    continue
  fi
  Ret1=$1 ; Sze1=$7
  
  set $($dd if=$OutFile of=$TmpOut oflag=noatime bs=$Chk skip=$BLK count=1 2>&1)
  if [ $# -lt 15 ] ; then 
    a='$'$(($#-1)) ; b='$'$(($#)) ; c=$(eval echo $a $b)
    echo "Abort: $c on OutFile -> TmpOut copy"
    OK='false'
    continue
  fi
  Ret2=$1 ; Sze2=$7
  COND="$Ret1.$Ret2"
  
  # Combine original copy conditions, there are only 9 (3^2) combinations
  case $COND in
    # full block of both available
    "1+0.1+0" ) cBytes=$Chk
                if CmpBlk ; then
                    SKP=$((SKP+1)) ; BLK=$((BLK+1)) ; TOT=$((TOT+cBytes))
                else # write, read-back, and compare
                  if WriteBlock $cBytes; then
                    BLK=$((BLK+1)) ; TOT=$((TOT+cBytes))
                  else
                    OK='false'
                    echo "Abort: full block w/r/c failure; (1+0.1+0)"
                  fi
                fi 
                continue ;;
                
    # past EOF on both, normal termination when: (size mod $Chk) = 0
    "0+0.0+0" ) cBytes=0
                OK='false' 
                echo "Termination on block boundary. (0+0.0+0)" 
                continue ;; 
                
    # partial block on both, normal termination when: (size mod $Chk) != 0
    "0+1.0+1" ) if cBytes=$(FndCom $Sze1 $Sze2) ; then
                  [ $Sze1 -ne $cBytes ] && TrimIn  $cBytes
                  [ $Sze2 -ne $cBytes ] && TrimOut $cBytes
                  if CmpBlk ; then
                    SKP=$((SKP+1)) ; TOT=$((TOT+cBytes))
                    echo "Termination on partial block boundary; match (0+1.0+1)" 
                  else # write, read-back, and compare
                    if WriteBlock $cBytes ; then 
                      TOT=$((TOT+cBytes))
                      echo "Termination on partial block boundary; w/r/c (0+1.0+1)" 
                    else
                      echo "Abort: partial blocks, common bytes: $cBytes; w/r/c failure (0+1.0+1)"
                    fi
                  fi 
                else
                  echo 'Abort: Find common bytes failed (0+1.0+1)'
                fi
                OK='false'
                continue ;; 
                
    # partial block on source, full block on destination
    "0+1.1+0" ) if cBytes=$(FndCom $Sze1 $Sze2) ; then
                  TrimOut $cBytes
                  if CmpBlk ; then
                    SKP=$((SKP+1)) ; TOT=$((TOT+cBytes))
                    echo "Notice: Partial block on $InFile, common bytes: $cBytes; match (0+1.1+0)"
                  else # write, read-back, and compare
                    if WriteBlock $cBytes ; then 
                      TOT=$((TOT+cBytes))
                      echo "Notice: Partial block on $InFile, common bytes: $cBytes; w/r/c (0+1.1+0)"
                    else
                      echo "Abort: Partial block on $InFile, common bytes: $cBytes; w/r/c failed (0+1.1+0)" 
                    fi
                  fi
                else
                  echo 'Abort: Find common bytes failed (0+1.1+0)'              
                fi 
                OK='false' 
                continue ;;
                 
    # full block on source, partial block on destination
    "1+0.0+1" ) if cBytes=$(FndCom $Sze1 $Sze2) ; then
                  TrimIn $cBytes
                  if CmpBlk ; then
                    SKP=$((SKP+1)) ; TOT=$((TOT+cBytes))
                    echo "Notice: Partial block on $OutFile, common bytes: $cBytes; match (1+0.0+1)"
                  else # write, read-back, and compare
                    if WriteBlock $cBytes ; then 
                      TOT=$((TOT+cBytes))
                      echo "Notice: Partial block on $OutFile, common bytes: $cBytes; w/r/c (1+0.0+1)"
                    else
                      echo "Abort: Partial block on $OutFile, common bytes: $cBytes; w/r/c failed (1+0.0+1)" 
                    fi
                  fi 
                else
                  echo 'Abort: Find common bytes failed (1+0.0+1)'              
                fi
                OK='false'
                continue ;;  
                 
    # Various notification conditions - 
    # loop termination is by reading one too many blocks
    "1+0.0+0" ) cBytes=0
                OK='false'
                echo "Notice: full block on source, EOF on $OutFile (1+0.0+0)"
                continue ;; 
    "0+1.0+0" ) cBytes=0
                OK='false'
                echo "Notice: partial block on source, EOF on $OutFile (0+1.0+0)"
                continue ;; 
    "0+0.1+0" ) cBytes=0
                OK='false'
                echo "Notice: EOF on $InFile, full block on $OutFile (0+0.1+0)"
                continue ;; 
    "0+0.0+1" ) cBytes=0
                OK='false'
                echo "Notice: EOF on $InFile, partial block on $OutFile (0+0.0+1)"
                continue ;; 
    # Catch-all, unrecognized copy status - expected when using Busybox dd
        *     ) cBytes=0
                OK='false' 
                echo "Abort: Unrecognized copy status: $COND"
                continue ;; 
  esac
done
rm $TmpIn $TmpOut $FragIn $FragOut 2>/dev/null
echo "Blocks: $BLK, plus Bytes: $cBytes, Total skips: $SKP, Total bytes: $TOT"
exit 0
