#!/bin/sh
#
##############################################################################
#
# Controls whether the device's password is enabled or not.
#
##############################################################################

# change to directory of this script
cd "$(dirname "$0")"

# load configuration
if [ -e "config.sh" ]; then
	source config.sh
fi

# load utils
if [ -e "utils.sh" ]; then
	source utils.sh
else
	echo "Could not find utils.sh in `pwd`"
	exit
fi


###############################################################################

pwd_off() {
	rm -f $PASSCODEFILE
}

pwd_on() {
	touch $PASSCODEFILE
}

###############################################################################

get_enabled_setting () {
	CURRENTMINUTE=$(( 60*`date +%-H` + `date +%-M` ))

	for schedule in $SCHEDULE; do
	        read STARTHOUR STARTMINUTE ENDHOUR ENDMINUTE ENABLED TIMEOUT << EOF
			$( echo $schedule | sed -e 's/[:,=,-]/ /g' )
EOF
	        START=$(( 60*$STARTHOUR + $STARTMINUTE ))
	        END=$(( 60*$ENDHOUR + $ENDMINUTE ))

	        if [ $CURRENTMINUTE -ge $START ] && [ $CURRENTMINUTE -lt $END ]; then
	               	echo $ENABLED
	               	return
		fi
        done
        
        echo 1
}


###############################################################################

get_timeout_setting () {
	CURRENTMINUTE=$(( 60*`date +%-H` + `date +%-M` ))

	for schedule in $SCHEDULE; do
	        read STARTHOUR STARTMINUTE ENDHOUR ENDMINUTE ENABLED TIMEOUT << EOF
			$( echo $schedule | sed -e 's/[:,=,-]/ /g' )
EOF
	        START=$(( 60*$STARTHOUR + $STARTMINUTE ))
	        END=$(( 60*$ENDHOUR + $ENDMINUTE ))

	        if [ $CURRENTMINUTE -ge $START ] && [ $CURRENTMINUTE -lt $END ]; then
	               	echo $TIMEOUT
	               	return
		fi
        done
        
        echo 0
}


###############################################################################

get_remaining_settings_time ()  {
	CURRENTMINUTE=$(( 60*`date +%-H` + `date +%-M` ))

	for schedule in $SCHEDULE; do
	        read STARTHOUR STARTMINUTE ENDHOUR ENDMINUTE ENABLED TIMEOUT << EOF
			$( echo $schedule | sed -e 's/[:,=,-]/ /g' )
EOF
	        START=$(( 60*$STARTHOUR + $STARTMINUTE ))
	        END=$(( 60*$ENDHOUR + $ENDMINUTE ))

	        if [ $CURRENTMINUTE -ge $START ] && [ $CURRENTMINUTE -lt $END ]; then
	        	# calculate how long this setting is still active, but add
	        	# some more seconds on it to prevent any race condition on switch
	               	echo $(( 60*($END - $CURRENTMINUTE) + 15))
	               	return
		fi
        done
        
        echo 0
}


###############################################################################

# only run on supported devices
#TODO

SLEEPSTART=0

# abort if the passcode is not enabled - this may indicate that
# the user (just?) deactivated it - and we do not want to lock 
# the user out
if [ ! -e $PASSCODEFILE ]; then
	logger "Passcode file $PASSCODEFILE not found. Please set a passcode"
	logger "via the Kindle settings before enabling/running this extension"
	bash ./disable.sh
	exit
fi


# main loop
while [ true ]; do

	# check power state - screensaver or not
	ACTIVE=0
	lipc-get-prop com.lab126.powerd status | grep -q "Powerd state: Active" && ACTIVE=1

	if [ $ACTIVE -eq 1 ]; then
		# by the check above we know that password lock is enabled while we are
		# running - so a reset will not cause any problems
		
		# wait until we go to screensaver
		SLEEPSTART=0
		logger "Waiting for screensaver to start"
		lipc-wait-event com.lab126.powerd goingToScreenSaver || true

	else # we are in screensaver mode
		# should we be locked?
		SHOULDBELOCKED=$(get_enabled_setting)
		
		# time until next configuration change
		SLEEPTIME=$(get_remaining_settings_time)

		# if we just entered
		if [ $SLEEPSTART -eq 0 ]; then
			# note the time we started to sleep
			SLEEPSTART=$(date +%s)
			
			# if we are on with a timeout, we wait with locking
			if [ $SHOULDBELOCKED -eq 1 ] && [ $(get_timeout_setting) -gt 0 ]; then
				SHOULDBELOCKED=0
				SLEEPTIME=$(get_timeout_setting)
			fi
		fi

		# set the lock status
		if [ $SHOULDBELOCKED -eq 0 ]; then
			pwd_off
		else
			pwd_on
		fi

		# wait for device to go out of screensaver or until our timeout 
		# has passed 
		if [ $SLEEPTIME -gt 0 ]; then
			logger "Waiting for screensaver to stop (but wake up after $SLEEPTIME seconds)."
			lipc-wait-event -s $SLEEPTIME com.lab126.powerd outOfScreenSaver
		else
			logger "Waiting for screensaver to stop."
			lipc-wait-event com.lab126.powerd outOfScreenSaver
		fi
	fi
done
