#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2014, Jose Antonio Espinosa - BiblioEteca Technologies'
__docformat__ = 'restructuredtext en'

import socket, re, datetime
import urllib, os
from collections import OrderedDict
from threading import Thread
from lxml import etree
from lxml.html import fromstring, tostring



from calibre.ebooks.metadata.book.base import Metadata
from calibre.library.comments import sanitize_comments_html


import calibre_plugins.BIBLIOETECA.config as cfg

class Worker(Thread): # Get  details

    '''
    Get book details from BIBLIOETECA book page in a separate thread
	related to BIBLIOETECA
    '''
    name                    = 'Worker'
    description             = _('Get book details from biblioeteca.com book page in a separate thread')
    author                  = 'Jose Antonio Espinosa - BiblioEteca.com'
    version                 = (0, 3, 0)
    minimum_calibre_version = (0, 8, 0)


    def __init__(self, combo, result_queue, browser, log, relevance, plugin, timeout=20):	   
        Thread.__init__(self)
        self.daemon = True
        self.url = combo[0]
        self.rating = combo[1]
        self.result_queue =  result_queue
        self.log, self.timeout = log, timeout
        self.relevance, self.plugin = relevance, plugin
        self.browser = browser.clone_browser()
        self.cover_url = self.biblioeteca_id = self.isbn = None
        self.cover_bol = self.comment_bol = self.bol_url = None
        self.bol_root = None

    def run(self):
        try:
            self.get_details()
        except:
            self.log.exception('get_details failed for url: %r'%self.url)

    def get_details(self):

        root = self.url
        self.parse_details(root)
        


    def parse_details(self, root):      
        try:
            biblioeteca_id = self.parse_biblioeteca_id(root)
        except:
            self.log.exception('Error parsing BIBLIOETECA id for url: %r'%self.url)
            biblioeteca_id = None
                
        try:		    
            title = self.parse_title(root)
            series = None
            series_index = None
        except:
            self.log.exception('Error parsing title for url: %r'%self.url)
            title = None
        try:
            authors = self.parse_authors(root)
        except:
            self.log.exception('Error parsing authors for url: %r'%self.url)
            authors = []        	
        if not title or not authors or not biblioeteca_id:
            self.log.error('Could not find title/authors/biblioeteca_id for %r'%self.url)
            self.log.error('biblioeteca_id: %r Title: %r Authors: %r'%(biblioeteca_id, title, authors))
            return
        mi = Metadata(title, authors)
        mi.set_identifier('biblioeteca', biblioeteca_id)
        self.biblioeteca_id = biblioeteca_id
        
        try:
            (series, series_index) = self.parse_series (root)
            if series:
                mi.series = series
                mi.series_index = series_index
        except:
            self.log.exception('Error parsing series for url: %r'%self.url)

       
        try:
            isbn = self.parse_isbn(root)
            if isbn:
                self.isbn = isbn
                mi.isbn = isbn
        except:
            self.log.exception('Error parsing ISBN for url: %r'%self.url)

        try:
            mi.rating = self.parse_rating(root)
        except:
            self.log.exception('Error parsing ratings for url: %r'%self.url)
          
        try:
            mi.comments = self.parse_comments(root)
        except:
            self.log.exception('Error parsing comments for url: %r'%self.url)

        try:
            self.cover_url = self.parse_cover(root)
        except:
            self.log.exception('Error parsing cover for url: %r'%self.url)
            
        mi.has_cover = bool(self.cover_url)

        try:
            mi.publisher = self.parse_publisher(root)
        except:
            self.log.exception('Error parsing publisher for url: %r'%self.url)

        try:
            mi.pubdate = self.parse_published_date(root)
        except:
            self.log.exception('Error parsing published date for url: %r'%self.url)

        mi.source_relevance = self.relevance

        if self.biblioeteca_id:
            if self.isbn:
                self.plugin.cache_isbn_to_identifier(self.isbn, self.biblioeteca_id)

        self.plugin.clean_downloaded_metadata(mi)
        self.result_queue.put(mi)
        
    def parse_biblioeteca_id(self, root):
        meta_node = root.xpath('id')
        if meta_node:
            return meta_node[0].text
        

    def parse_series(self, root):
        series_node = root.xpath('saga')
        if not series_node:
            return (None, None)

        #Número 1 de la saga
        series_text = series_node[0].text.strip();
        series_number = root.xpath('saga_number')
        if not series_number:
            series_index=0
        else:
            series_index = float(series_number[0].text)
        
        series_name = series_text;
        
        return (series_name, series_index)
    
    def parse_title(self,root):
        title_node = root.xpath('title')
        if title_node:
            title=title_node[0].text.strip()
            return title

    def parse_authors(self, root):
        auteur_node = root.xpath('author')
        authors = []
        if auteur_node:
            autor_str = auteur_node[0].text
            autores = autor_str.split(';')
            for auteurs in autores:
                #El autor está cambiado el orden apellidos, nombre hay que darle la vuelta
                #para que esté bien
                #autor_s = auteurs.text_content().strip().split(',',2)
                autor_s=auteurs.strip().split(',',2)
                print (auteurs)
                autor_s.reverse()
                autor = ' '.join(autor_s)
                authors.append(autor)
            return authors

   
    def parse_isbn(self, root):
        isbn_node=root.xpath('isbn')
        if isbn_node:
            if isbn_node[0]:
                isbn=isbn_node[0].text.strip()
                return isbn
        

    def parse_rating(self, root):
        rating_node = root.xpath('rating')
        if rating_node:
            # Si pone ND es que no hay
            # si pone un numero esa es la valoracion
            rating = rating_node[0].text.strip()
            rating = rating.replace(',','.')
            #print ("Rating",rating)
            if rating != 'ND':
                rating_value = float(rating)/2
                return rating_value



    def parse_publisher(self, root):
        publisher = None
        return(publisher)


    def parse_published_date(self, root):
        pub_date = None
        

    def _convert_date_text(self, date_text):
        # Note that the date text could be "2003", "december 2003" or "December 10th 2003"
        year = int(date_text[-4:])
        month = 1
        day = 1
        if len(date_text) > 4:
            text_parts = date_text[:len(date_text)-5].partition(' ')
            month_name = text_parts[0]
            # Need to convert the month name into a numeric value
            month_dict = {"januari":1, "februari":2, "maart":3, "april":4, "mei":5, "juni":6,
                "juli":7, "augustus":8, "september":9, "oktober":10, "november":11, "december":12}
            month = month_dict.get(month_name, 1)
            if len(text_parts[2]) > 0:
                day = int(re.match('([0-9]+)', text_parts[2]).groups(0)[0])
        from calibre.utils.date import utc_tz
        return datetime.datetime(year, month, day, tzinfo=utc_tz)


    def parse_comments(self, root):
        comments = ''
        meta_node = root.xpath('long_desc')
        if meta_node:
            comments = meta_node[0].text
            comments= comments + '<br>(fuente: <a href="http://www.biblioeteca.com">BiblioEteca.com</a>)<br>'
        if comments:
            return comments

    def parse_cover(self, root):
        result_node=root.xpath('img_url')
        if result_node:
            img_url = result_node[0].text
            pos=img_url.count('sinportada')
            if pos==0:
                self.plugin.cache_identifier_to_cover_url(self.biblioeteca_id, img_url)
                return img_url
